/* llp.c  */

#include <sys/types.h>
#include <sys/timeb.h>
#include <time.h>
#include <stdio.h>

#define LINELEN 122
#define NEWLINE '\n'
#define FORMFEED '\014'  /*  used to skip to top of page  */
#define NEWFILE '\002' /* used to restart sequencing from 1 (STX char)*/
#define EOM -1
#define BACKSPACE '\010'
#define ESC '\033'
#define FF  '\014'
#define NUMLINES 76

/* Portrait		ESC&l0O */
/* Landscape		ESC&l1O */
/* Reset		ESCE 	*/
/* Primary Pitch	ESC(s#H  # is num of chars/inch */
/* Vert Motion Index	ESC&l#C  # is num of 1/48" incr */
/* Top Margin		ESC&l#E  # is num of lines	*/
/* Text Length		ESC&l#F  # is num of lines	*/
/* Left Margin		ESC&l#L  # is col number	*/
/* Right Margin		ESC&l#M  # is col number	*/
/* Stroke Weight	ESC(s-3B light	*/
/* Stroke Weight	ESC(s0B  medium	*/
/* Stroke Weight	ESC(s3B bold	*/
/* Cursor Position	ESC&a#R  # is row number	*/
/* Formfeed/Reset	FF ESC
/* Underline on		ESC&dD	*/
/* Underline off	ESC&d@	*/
/* Page Eject		ESC&l0H	*/
/* Font Selection	ESC(s0T	is line printer	*/
/* Prop Spacing		ESC(s1P */
/* Fixed Spacing	ESC(s0P */
/* Symbol Set		ESC(0U is ASCII	*/

char form1[] = {
    "\033E\033&l0O\033(0U\033(s0p16.66H\033&l6.0C\033&l78F\033&l5E\033(s-3B\033&a0R\033&a145M\033&a10L\r\000"};

char form6[] = {
    "\014\033&a0R\033&a145M\033&a10L\r\000"};

/* Formfeed/Reset */
char form2[] = {
    "\014\033E\000"};

char form3[] = {
    "\033&a0R\033&a90M\033&a88L\r\n\r\n\000"};

char form4[] = {
    "|\r\n\000"};

char form5[] = {
    "\033&a0R\033&a171M\033&a91L\r\000"};

char   chr;			/* character holder */
char   line[LINELEN+8],*lineptr;  /* line buffer and pointer */
int	linelen = LINELEN+6;	/* length without numbers */
int 	tabs = 0;    /*  global access to -t argument -- initialized off  */
int 	tablen = 8;
int	pageno = 1;
int	linecnt = 0;
int	newpage = 0;
int	seq;		/* sequence numbers */
char *	title;
int	skipseq = 0;

main(argc,argv)
char   **argv;

{
    register FILE *fi;
    int	i,j,count;	/* character counters */
    int	forcend;	/* line length & paging control variables */
    int	savchar;	/* temporary character holder */
    int	noline = 0;

    seq = forcend = 0;	/* initialize flags & indicators */

    while (--argc > 0) {
    	for ( ; (*++argv)[0]=='-'; argc-- ) {
    	    i = 1;
    	    while ((*argv)[i] != 0) {	/* pass through all flags */
    		switch ((*argv)[i++]) {
    		    case 's':           /* sequencing ('-s') option */
    			seq = 1;
    			linelen = LINELEN; /* length with numbers */
    			break;		/* set up for sequence numbering */
    		    case 'T':           /* tabbing ('-T') option */
    			tabs=1;
    			tablen = 8;
    			break;		/* set up for tab expansion */
    		    case 't':           /* tabbing ('-t') option */
    			tabs=1;
    			tablen = 4;
    			break;		/* set up for tab expansion */
    		}
    	    }
    	}

    	if (argc < 1) {
    	    fi = stdin;
    	    *argv = "reading input from stdin";
    	} else {
    	    if ((fi = fopen(*argv, "r")) == NULL) {
    		fprintf(stderr, "seq: can't open %s\n", *argv);
    		continue;
    	    }
    	}

    	title = *argv;
    	/* now process the input */
    	if (seq)seq = 1;
    	prolog();
    	newpage = 0;

    	line[5] = ' ';                  /* clear padding column */
    	i = 6;                          /* initialize line buffer pointer */
    	while ((savchar = getc(fi)) != EOF) {   /* read until EOF or ERROR */
    	    chr = savchar;     /*  convert getc(return) to character  */
    	    if (noline && chr == '\n') {
    		noline = 0;
    		continue;
    	    }
    	    noline = 0;
    	    if (chr == FF) {
    		noline = 1;
    		line[i++] = '\n';          /* stuff character in line */
    		line[i++] = '\r';
    		if (seq) {                /* sequencing in effect? */
    		    setln();
    		    lineptr = line;        /* write out entire line buffer */
    		    count = i;
    		} else {                  /* no sequencing */
    		    lineptr = &line[6];    /* write the text part of the line */
    		    count = i - 6;
    		}
    		newpage++;
    		if (i=6)continue;
    		lineprt(lineptr, count);
    		i = 6;			/* initialize line pointer */
    		continue;
    	    }
    	    if (chr == BACKSPACE) {
    		i--;			/* backup char count */
    		continue;		/* and do next char */
    	    }
    	    /*  Now the "regular" character input can be handled  */
    	    if (forcend && chr != '\n') {	/* do we have to 'split' this line */
    		line[i++] = '\n';         /* move newline into buffer */
    		line[i++] = '\r';
    		if (seq) {                /* sequencing in effect? */
    		    setln();
    		    lineptr = line;        /* write out entire line buffer */
    		    count = i;
    		} else {                  /* no sequencing */
    		    lineptr = &line[6];    /* write the text part of the line */
    		    count = i - 6;
    		}
    		lineprt(lineptr, count);
    		skipseq = 1;
    		/*  reset line pointer to 15 and insert leading blanks  */
    		for (i=0; i < 6+tablen; i++)
    		    line[i] = ' ';
    		chr = savchar;     /*  retrieve original character read */
    	    }
    	    forcend = 0;		/*  reset line 'split' control  */
    	    if (chr == '\t')	/* is this a tab character? */
    		i = dotab(i);	/* if so, handle it */
    	    if (chr == '\n') {	/* is character a newline? */
    		line[i++] = chr;	/* move newline into buffer */
    		line[i++] = '\r';
    		if (seq) {		/* sequencing in effect? */
    		    if(!skipseq)setln();
    		    lineptr = line;       /* write out entire line buffer */
    		    count = i;
    		} else {		/* no sequencing */
    		    lineptr = &line[6];   /* write the text part of the line */
    		    count = i - 6;
    		}
    		lineprt(lineptr, count);
    		skipseq = 0;
    		i = 6;                    /* initialize line pointer */
    	    } else {                     /* character not new line */
    		line[i++] = chr;          /* stuff character in line */
    		if (i >= linelen)	/* too many chars before new line? */
    		    forcend = 1;	/* line may have to be 'split' */
    	    }
    	}
    	if (fi != stdin) fclose(fi);
    }
    epilog();
    if (i != 6)				/* END or ERROR at the wrong place? */
    	exit(2);			/* yes, abort */
    else
    	exit(0);			/* exit gracefully */
}

setln ()
{
    int	n;
    lineptr = &line[4];    /* point to last character in sequence */
    n = seq++;          /* put sequence number in temporary */
    do {
    	/* put last digit in line buffer */
    	*(lineptr--) = n % 10 + '0';
    } while ((n /= 10) > 0); /* take last digit off number */
    while (lineptr >= line)
    	*(lineptr--) = ' '; /* pad on left with blanks */
}

lineprt(buf,len)
char * buf;
int    len;
{
    int tlen;

    if (len <= 0)return;
    if (len > 2) {
    	/* trim trailing blanks */
    	tlen = len - 2;
    	while (tlen > 0 && buf[tlen-1] == ' ')tlen--;
    	if (tlen < len-2) {
    	    buf[tlen] = buf[len-2];
    	    buf[tlen+1] = buf[len-1];
    	    len = tlen + 2;
    	}
    }
    if (skipseq && len == 2) {
    	for (tlen=0; tlen < len; tlen++)buf[tlen] = ' ';
    	return;
    }
    bumpline();
    if (newpage) pagebrk();
    output(buf, len);
    for (tlen=0; tlen < len; tlen++)buf[tlen] = ' ';
}

dotab(i)   /* subroutine to handle tabs */
int i;     /* argument is current line pointer */
{
#ifdef MAR9
    if (!tabs)   /* if tab handling not requested, forget it */
    	return(i);
#endif
    chr = ' ';             /* force tab character to be a space */
#ifdef XXX
    do {
    	line[i++] = chr;
    }
    while ((i - 5)%tablen != 0);  /* insert spaces to next tab stop */
#endif
    while ((i - 6)%tablen != 0)  /* insert spaces to next tab stop */
    	line[i++] = chr;
    return(i);		/* exit now */
}

bumpline()
{
    if(linecnt++ >= NUMLINES)newpage++;
}

prolog()
{
    output(form1, strlen(form1));
    pageno = 1;
    linecnt = 0;
    mktitle();
}

epilog()
{
    output(form2, strlen(form2));
}

output(buf, len)
char * buf;
int	len;
{
    while (len--)putc(*buf++,stdout);
}

mktitle()
{
    struct timeb tbuf;
    char *ctbuf;
    char titl [160];

    ftime(&tbuf);
    ctbuf = ctime(&(tbuf.time));
    ctbuf[24] = '\0';
    sprintf(titl, "\033&dD%-80s%-25s - page %-4d \n\n\033&d@\r\000",
    	title, ctbuf, pageno);
    output(titl, strlen(titl));
}

pagebrk()
{
    int i;
#ifdef JUNK
    if (pageno & 1) {
    	output(form3, strlen(form3));
    	for (i = 0; i < 66; i++)
    	    output(form4, strlen(form4));
    	output(form5, strlen(form5));
    } else {
    	output(form6, strlen(form6));
    }
#else
    output(form6, strlen(form6));
#endif
    linecnt = 1;
    newpage = 0;
    pageno++;
    mktitle();
}

