/* stab.c */
/*
 * 	(c) Copyright 1987 Gould Inc.
 * 	    All Rights Reserved.
 */

/*
 *	Gould, Inc. Base Register Assembler
 */

#ifndef lint
/*NOBASE*/
static char *rcsid = "@(#) (Gould) $Header: stab.c,v 5.5 89/05/12 12:37:36 pcc Rel-3_0 $";
#endif

/* for stab pseudo ops (debugger information) */

#ifndef JCB
#include "a.out.h"
#include "stab.h"
#else
#include <a.out.h>
#include <stab.h>
#endif /* JCB */

#include "struct.h"
#include "operand.h"
#include "parse.h"
#include <stdio.h>

struct      stab_tab
{
    Tuchar       *sym_name;
    struct stab_tab    *st_next;
    Tuchar   st_type;
    Tuchar        st_other;
    short       st_desc;
    SYM *symbol;
};

struct stab_tab    *stab_first = 0;
struct stab_tab   **stab_last = &stab_first;
int stab_num;

extern SYM *defstab();
extern SYM *lookstab();






/*
NAME: op_stab
PURPOSE: handle the .stabd, .stabn, and .stabs pseudo ops
PRECONDITIONS:
ptr is a pointer to the optable entry for the directive.
stabd should have 3 operands, stabn 4, and stabs should have 5.
the ocode field distinguishes the different directives.
the external stab_num is used to create unique symbol table names for
the stabn and stabd ops. it is zeroed before each pass by main.
the routines defstab and lookstab are special symbol table routines used
to guarantee that every .stab is allocated a unique symbol table entry.
POSTCONDITIONS:
a symbol is defined in the symbol table from operands as defined in
the 4.1BSD UNIX assembler reference manual by Reiser and Henry (C
compiler variant, not the original Ritchie variant) and with attributes
as defined in the system header file stab.h .
ALGORITHM:
note that the string recognition is very dumb, it knows nothing about
escape characters, but instead just looks for the delimitor (").
HISTORY:
code originally written for Compion CDS MC68000 assembler RJM
modified for use in Gould assembler RJM
Mon Aug 22 11:19:00 CDT 1983
*/






int procno;
static int var_size_items;

op_stab (ptr)
register struct opcode *ptr;
{
    register SYM *sym_ptr;
    register struct stab_tab *stab;
    register Tuchar *tmp_ptr;
    struct stab_tab *make_stab ();
    enum SEGS set_space ();
    Tuchar name[MAX_STR];
    int i, numops, ops_ok;
    extern OPND oper[];
    extern SYM first_sym;
    extern SYM *latest_sym;
    extern int Symchars;
    Tuchar *sstring();

    var_size_items = 0;
    SKIPSPACE (yylinept);
    /* parse the string for a stabs */
    if (ptr -> ocode == 0)
    {
	if (*yylinept++ != '\"')
	{
	    error ("op_stab: bad delimiter for stabs");
	    SKIPEOS (yylinept);
	    return;
	}

	tmp_ptr = &name[0];
	while (yylinept && (*yylinept != '\"'))
	{
	    /* copy stabs string, note if it has variable size items. */
	    /* if so, fix up is made below after pass 2 processing */

	    if ((*tmp_ptr++ = *yylinept++) == ';')
		if (pass == 2 && !var_size_items
			&& (*yylinept == 'T' | *yylinept == 'A'))
		    ++var_size_items;
	}
	*tmp_ptr = '\0';

	if (*yylinept++ != '\"')
	{
	    error ("op_stab: bad delimiter for stabs");
	    SKIPEOS (yylinept);
	    return;
	}

	SKIPSPACE (yylinept);

	if (*yylinept++ != ',')
	{
	    error ("op_stab: comma expected");
	    SKIPEOS (yylinept);
	    return;
	}

	SKIPSPACE (yylinept);
    }		/* end of stabs first operand handling */

    /* stabs has 4 more operands, stabn also has 4, and stabd has 3 */

#define MAX_OPERS 4
    for (i = 0; i < MAX_OPERS; ++i)
	oper[i].type = OPERROR;

    ops_ok = 1;
    numops = (ptr -> ocode == 1) ? 2 : 3;	/* really number of commas */
    for (i = 0; i < numops; i++)
    {
	ops_ok = ops_ok && validate_op (ptr, i);
	if (*yylinept == ',')
	{
	    yylinept++;
	    SKIPSPACE (yylinept);
	}
	else
	{
	    error ("op_stab: comma expected");
	    SKIPEOS (yylinept);
	    return;
	}
    }
    /* now get the last one */
    ops_ok = ops_ok && validate_op (ptr, i);

    if (!ops_ok)		/* something was wrong */
    {
	return;
    }

    switch (ptr -> ocode)
    {
	case 0:	/* .stabs */
	    dostab (name, oper[0].constval, oper[1].constval,
			oper[2].constval, &oper[3]);
	    break;
	case 1:	/* .stabd */
	    dostab ((char *) 0, oper[0].constval, oper[1].constval,
			oper[2].constval, (OPND *) 0);
	    break;
	case 2:	/* .stabn */
	    dostab ((char *) 0, oper[0].constval, oper[1].constval,
			oper[2].constval, &oper[3]);
	    break;
	default:		/* should NEVER happen */
	    error ("op_stab: unknown stab");
	    break;
    }
}






/*
NAME: make_stab
PURPOSE: allocate and link a stab table structure into the stab table.
PRECONDITIONS:
name is the symbol name for the stab.
the global stab_last points to the last stab entry made to the list.
POSTCONDITIONS:
the structure is created, linked to the end of the stab list, and the
name and st_next fields are initialized.
if calloc fails, the assembler aborts, otherwise a pointer to the new
structure is returned.
ALGORITHM:
the stab list is just a linear list with a begin (stab_first which is
not used here) and end (stab_last) pointer. searching is needed only
when the symbol table is built by post_pass2, although that can be
expensive (see Stab_Fix).
HISTORY:
code originally written for Compion CDS MC68000 assembler RJM
modified for use in Gould assembler RJM
Mon Aug 22 11:19:00 CDT 1983
*/






struct stab_tab *
make_stab (sym_ptr)
register SYM  *sym_ptr;
{
    register struct stab_tab   *stab;

/* create one */
    if ((stab = (struct stab_tab *) calloc (1, sizeof *stab)) == 0)
    {
	error ("Fatal assembler error (stab storage exceeded)");
	exit (2);
    }

    (*stab_last) = stab;
    stab_last = &stab -> st_next;
    stab -> sym_name = sym_ptr -> sname;
    stab -> st_next = 0;
    return (stab);
}






/*
NAME: set_space
PURPOSE: determine what space to put this symbol in.
PRECONDITIONS:
parameter type is the st_type field from the source line (first operand
for a stabd, second for a stabn or stabs).
parameter sym_ptr will be 0 for a stabd and the symbol table pointer
from oper[3] for stabs and stabn. this is needed when a relocatable
address is used to identify a symbol that the stab is associated with.
POSTCONDITIONS:
a hopefully appropriate space (SEGS) value is returned after making a
wild guess about what it should be. the decision is complicated
considerably by the far space implementation.
HISTORY:
code originally written for Compion CDS MC68000 assembler RJM
modified for use in Gould assembler RJM
Mon Aug 22 11:19:00 CDT 1983
*/






enum SEGS
set_space (type, sym_ptr)
Tuchar   type;
SYM *sym_ptr;
{
    switch (type) {

	case N_FUN: 
	case N_SO: 
	case N_ENTRY: 
	case N_LBRAC: 
	case N_RBRAC: 
	case N_STSYM: 
	case N_LCSYM: 
	case N_ECOML: 
	case N_SLINE: 
	case N_SOL: 
	case N_NBLCS:
	case N_NBSTS:
	    if (sym_ptr)
	    {
		return (sym_ptr -> sspace);
	    }
	    else
	    {
		return (dot -> sspace);
	    }

	default: 
	    return (last_SEGS);

    }
}






/*
NAME: Stab_Fix
PURPOSE: set a.out nlist (symbol table) fields for stab symbols
PRECONDITIONS:
parameter a_sym is a pointer to the nlist structure, sym_ptr is a
pointer to the SYM structure of the assembler symbol table.
this routine is called by post_pass2 whenever it encounters a stab
symbol (as per sflags S_STABS and S_STABDN attributes).
POSTCONDITIONS:
the symbol is looked up in the stab list and attributes are set from the
stab entry in the nlist structure. if the symbol is not found, an error
is returned.
ALGORITHM:
the search is linear, which will be slow if the -g option is used
because the compiler puts out MANY stab directives in that case, but
then, what can one expect with this style of debugging?
HISTORY:
code originally written for Compion CDS MC68000 assembler RJM
modified for use in Gould assembler RJM
Mon Aug 22 11:19:00 CDT 1983
*/






Stab_Fix (a_sym, sym_ptr)
struct nlist   *a_sym;
SYM *sym_ptr;
{
    register struct stab_tab   *stab;

    for (stab = stab_first; stab; stab = stab -> st_next)
	if (stab -> symbol == sym_ptr)
	    break;

    if (stab) {
	a_sym -> n_un.n_strx = 0;
	a_sym -> n_other = stab -> st_other;
	a_sym -> n_desc = stab -> st_desc;
	a_sym -> n_value = sym_ptr -> svalue;
	if (sym_ptr -> sspace != last_SEGS)
	{
	    a_sym -> n_type = (stab -> st_type & N_STAB) |
		seginf[sym_ptr -> sspace].si_ntype;
	    a_sym -> n_value += seginf[sym_ptr -> sspace].si_start;
#if defined(COFF)
	    a_sym -> n_scnum = seginf[sym_ptr -> sspace].si_scnnum;
#endif /* COFF */
	}
	else
	{
	    a_sym -> n_type = stab -> st_type;
#if defined(COFF)
	    if (stab -> st_type & N_STAB)
		a_sym -> n_scnum = -2;
	    else
		a_sym -> n_scnum = -1;
#endif /* COFF */
	}
    }
    else
	error ("Stab_Fix: stab %s not found", sym_ptr -> sname);
}






/*
 * do the actual work to make a stab symbol
 */

dostab (name, type, other, desc, op)
Tuchar	*name;			/* name to use with symbol (stabs), or NULL 
				   for stabd or stabn */
int	type,other,desc;	/* fields to go into symbol table */
OPND	*op;			/* operand for stabs or stabn, or NULL
				   for stabd */
{
    register SYM *sym_ptr;
    register struct stab_tab *stab;
    Tuchar tmpnm[16];
    int stype = S_STABS;	/* assume stabs */
    SYM *tsymp;

    /* .stabd and .stabn need a fake name for symbol table routines */
    if (name == (Tuchar *) 0)
    {
	stype = S_STABDN;	/* it isn't a stabs */
	name = tmpnm;
	sprintf (name, "!%d", stab_num++);
    }

    if (pass == 1)
    {
	sym_ptr = defstab (name);
/* SPR0592 (revised treatment) */
/* stab entry may have a symbolic operand; if so, and the symbolic operand */
/* has already been declared but not defined, the stab entry must be */
/* relinked to precede the entry for the symbolic operand */
/* !!! only for relocatable symbolic operands */
	if (oper[3].type && (oper[3].haveseen & NAME)
		&& (tsymp = oper[3].symname) != 0
		&& !(tsymp->sflags & S_DEF)) {
	    switch (oper[0].constval) {
		default: break;

		case N_SO:
		case N_ENTRY:
		case N_LBRAC:
		case N_RBRAC:
		case N_STSYM:
		case N_LCSYM:
		case N_ECOML:
		case N_SLINE:
		case N_SOL:
		case N_NBLCS:
		case N_NBSTS:
		    {
		/* Note: latest_sym == sym_ptr */
	    sym_ptr = &first_sym;
	    while(sym_ptr && sym_ptr->s_taborder != tsymp) {
		sym_ptr = sym_ptr->s_taborder;
	    }
	    if (!sym_ptr) {
		error("dostab: Fatal; taborder botch1");
		dexit(2);
	    }
	    tsymp = sym_ptr;	/* tsymp is predecessor to oper[3].symname */
	    while(sym_ptr && sym_ptr->s_taborder != latest_sym)
		sym_ptr = sym_ptr->s_taborder; /* find latest_sym predecessor */
	    if (!sym_ptr) {
		error("dostab: Fatal; taborder botch2");
		dexit(2);
	    }
				/* sym_ptr is predecessor to latest_sym */
	    latest_sym->s_taborder = tsymp->s_taborder;	/* relink */
	    tsymp->s_taborder = latest_sym;
	    tsymp = sym_ptr;
	    sym_ptr = latest_sym;	/* restore sym_ptr to defstab() value */
	    latest_sym = tsymp;		/* new latest_sym */
	    latest_sym->s_taborder = 0;
	}
		    }
	    }
/* end SPR0592 code */
	stab = make_stab (sym_ptr);
	stab -> st_type = (Tuchar) type;
	stab -> st_other = (Tuchar) other;
	stab -> st_desc = (short) desc;
	stab -> symbol = sym_ptr;
	sym_ptr -> sflags |= stype | S_DECL | S_DEF;
    }
    else
    {
	sym_ptr = lookstab (name);
	/* these must be done in pass 2 to get correct dot */
	/* note what happens if the address is relocatable */
	if (sym_ptr)
	{
	    if (op == (OPND *) 0)
	    {
		sym_ptr -> sspace = set_space ((Tuchar) type, 0);
		sym_ptr -> svalue = dot -> svalue;
	    }
	    else
	    {
		sym_ptr -> sspace = set_space ((Tuchar) type,
		    op->symname);
		sym_ptr -> svalue = op->constval;
		/* relocatable expression, parser already has address */
		if (op->haveseen & NAME)
		{
		    sym_ptr -> svalue += op->symname -> svalue;
		}
	    }
	    if (var_size_items) {
		int lval;
		Tuchar lname[10];
		register Tuchar *d;
		SYM *tsp;
		register Tuchar *tmp_ptr;

		d = &name[0];
		tmp_ptr = sym_ptr->sname;
		while ((*d++ = *tmp_ptr++) != '\0') {
		    if (tmp_ptr[-1] == ';') {
			switch (*tmp_ptr) {
			    case 'A':
				sprintf(lname, "LA%d", procno);
				break;
			    case 'T':
				sprintf(lname, "LOC%d", procno);
				break;
			    default:
				continue;
			}
			++tmp_ptr;
			lval = 0;
			while (isdigit(*tmp_ptr))
			    lval = (((lval << 2) + lval) << 1)
					+ (*tmp_ptr++ - '0');
			/*
			 * Leave the A or the T alone.  Also, don't need
			 * the 32 byte increment since it is already
			 * figured into the offset.
			 *   --whb
			 */
			d++;
				/* Kernel P bit */
			if ((tsp = lookup(lname)) != 0)
			    d += strlen(sprintf(d, "%d", tsp->svalue + lval));
		    }
		}
		/* adjust string table size - computed before pass 2 */
		Symchars += strlen(name) - strlen(sym_ptr->sname);
		/* substitute fixed-up string */
		sym_ptr->sname = sstring(name);
	    }
	}
	else
	{
	    error ("dostab: symbol not defined %s", name);
	}
    }
}

/*
 * 	(c) Copyright 1987 Gould Inc.
 * 	    All Rights Reserved.
 */
