/*****************************************************************************

  siemotif.c: Example of using SIE in a simple Motif application

  This program demonstrates the use of the NCD Simple Imaging Extension
  (SIE) in a Motif application.  A Motif drawing area widget is used to
  display the image.  On the right side and bottom of the drawing area are
  scroll bars to select the portion of the image to be display in the
  drawing area.  Above the drawing area is a slider to select the scaling
  of the image.  The value of the scale is a percentage of enlargement or
  reduction of the image.  A value of 100 means that each pixel of the image
  corresponds to one pixel on the screen, i.e. "actual size."  Also above
  the drawing area are a set of four toggle buttons which select one of
  four 90-degree orientations for the displayed image.

  Images for the program are limited to the two image files provided by
  NCD, big.g4 and comic.g3.  The program can be modified to read other
  file formats by modifying the ReadFile function.

  Also demonstrated in this program are techniques for querying the NCD
  terminal for the amount of available memory, and handling errors
  returned by the X server that might occur when using SIE.

  To compile this program, place in a directory containing lib/XncdSie.c,
  extensions/ncdsie.h, and extensions/ncdsiestr.h (from the
  src/Contributed/siedemoTIFF directory on the NCD Contrib 1.0 CD),
  then give the following commands:

	cc -c XncdSie.c
	cc -c siemotif.c
	cc -o siemotif siemotif.o XncdSie.o -lXm -lXt -lX11 -lXext

*****************************************************************************/


#include <errno.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <Xm/Xm.h>
#include <Xm/DrawingA.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/RowColumn.h>
#include <Xm/Scale.h>
#include <Xm/ScrollBar.h>
#include <Xm/ToggleB.h>
#include <X11/Xlibint.h>
#include <X11/Xatom.h>

#include "ncdsiestr.h"
#include "ncdquery.h"


#define  MIN(x,y)  ((x)<(y) ? (x) : (y))


XtAppContext	app_context;


/* variables used to identify X errors caused by SIE */

int sieOpcode, sieBaseEvent, sieBaseError;


/* variables to store memory statistics retrieved from terminal */

int gotMemoryStats;
long ncdTotalData, ncdTotalDataFree, ncdLargestDataFree;


/* structure to store data passed amongst callback functions */

typedef struct {
  char *filename;

  unsigned char *data;
  unsigned int dataLen;
  int dataFormat;

  Pixmap imagePixmap;
  GC gc;

  int imageWidth, imageHeight;
  int scaleWidth, scaleHeight;

  int xoffset, yoffset;

  int scale, orientation;

  int needNewPixmap;

  Widget outputWidget, vsBar, hsBar;

  Atom queryAtom, targetAtom;

} imageStruct;


String fallback_resources[] = {
    "*scaleLbl.labelString: Image Enlargement/Reduction (%)",
    "*scale.minimum: 10",
    "*scale.maximum: 300",
    "*scale.value: 50",
    "*scale.leftLbl.labelString: 10",
    "*scale.ctrLbl.labelString: 145",
    "*scale.rightLbl.labelString: 300",
    "*rotateLbl.labelString: Image Rotation (degrees CCW)",
    "*sieView.width: 600",
    "*sieView.height: 600",
    "*btn0.labelString: 0",
    "*btn1.labelString: 90",
    "*btn2.labelString: 180",
    "*btn3.labelString: 270",
    "*btn0.topOffset: 10",
    "*btn1.topOffset: 10",
    "*btn2.topOffset: 10",
    "*btn3.topOffset: 10",
    NULL
};



main( argc, argv )
int argc;
char *argv[];
{
  Widget toplevel, form, scale, view;
  Widget btns[4];
  Widget scaleLbl, rotateLbl, w;
  imageStruct img;
  unsigned long gcMask, memNeeded;
  int base, error;
  XGCValues gcValues;
  Dimension viewWidth, viewHeight;
  extern XErrorHandler handleXError();
  extern void hsbarMoved(), vsbarMoved(), scaleMoved(),
              viewExpose(), viewResized(), rotateButton(), handleSelection(),
	      computeScale();

  toplevel = XtAppInitialize( &app_context, "Siemotif", NULL, 0, &argc, argv,
			      fallback_resources, NULL, 0 );

  /* Check for SIE extension in server */

  if ( False == XQueryExtension( XtDisplay(toplevel), SIENAME, &sieOpcode,
                                 &sieBaseEvent, &sieBaseError ) ) {
    fprintf( stderr, "X server does not support the %s extension.\n", SIENAME );
    fprintf( stderr, "Program ending.\n" );
    exit(1);
  }

  if ( 2 != argc ) {
    fprintf( stderr, "Usage: %s <image file>\n", argv[0] );
    exit(1);
  }

  /* Read image file.  Abort if data format is not recognized */

  img.filename = strdup( argv[1] );
  if ( ReadFile( &img ) < 0 ) {
    fprintf( stderr, "Program ending.\n" );
    exit(1);
  }

  /* Set up handler for SIE errors */

  XSetErrorHandler( handleXError );

  /* Build display area and controls */

  form = XtVaCreateWidget( "form", xmFormWidgetClass, toplevel,
			   XmNfractionBase, 12,
			   NULL );

  scaleLbl = XtVaCreateWidget( "scaleLbl", xmLabelWidgetClass, form,
			       XmNtopAttachment, XmATTACH_FORM,
			       XmNleftAttachment, XmATTACH_FORM,
			       XmNrightAttachment, XmATTACH_POSITION,
			       XmNrightPosition, 6,
			       NULL );

  scale = XtVaCreateWidget( "scale", xmScaleWidgetClass, form,
			    XmNtopAttachment, XmATTACH_WIDGET,
			    XmNtopWidget, scaleLbl,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNrightAttachment, XmATTACH_POSITION,
			    XmNrightPosition, 6,
			    XmNorientation, XmHORIZONTAL,
			    XmNshowValue, True,
			    NULL );

  /* Get initial scale value from "scale" resource. */

  XtVaGetValues( scale, XmNvalue, &img.scale, NULL );

  w = XtVaCreateManagedWidget( "leftLbl", xmLabelWidgetClass, scale, NULL );
  w = XtVaCreateManagedWidget( "ctrLbl", xmLabelWidgetClass, scale, NULL );
  w = XtVaCreateManagedWidget( "rightLbl", xmLabelWidgetClass, scale, NULL );

  XtAddCallback( scale, XmNvalueChangedCallback, scaleMoved, &img );

  rotateLbl = XtVaCreateWidget( "rotateLbl", xmLabelWidgetClass, form,
			        XmNtopAttachment, XmATTACH_FORM,
			        XmNrightAttachment, XmATTACH_FORM,
				XmNleftAttachment, XmATTACH_POSITION,
				XmNleftPosition, 6,
			        NULL );

  /* I have used individual toggle button widgets rather than a radio box
   * widget because the radio box would not arrange the buttons the way
   * I wanted, centered under the rotate label.  So, put the buttons
   * directly on the form, and handle the radio button behavior in the
   * callback function.
   */

  btns[0] = XtVaCreateManagedWidget( "btn0", xmToggleButtonWidgetClass, form,
				     XmNset, True,
				     XmNuserData, &img,
				     XmNindicatorType, XmONE_OF_MANY,
				     XmNtopAttachment, XmATTACH_WIDGET,
				     XmNtopWidget, rotateLbl,
				     XmNleftAttachment, XmATTACH_POSITION,
				     XmNleftPosition, 7,
				     NULL );
  XtAddCallback( btns[0], XmNvalueChangedCallback, rotateButton, btns );

  btns[1] = XtVaCreateManagedWidget( "btn1", xmToggleButtonWidgetClass, form,
				     XmNset, False,
				     XmNuserData, &img,
				     XmNindicatorType, XmONE_OF_MANY,
				     XmNtopAttachment, XmATTACH_WIDGET,
				     XmNtopWidget, rotateLbl,
				     XmNleftAttachment, XmATTACH_POSITION,
				     XmNleftPosition, 8,
				     NULL );
  XtAddCallback( btns[1], XmNvalueChangedCallback, rotateButton, btns );

  btns[2] = XtVaCreateManagedWidget( "btn2", xmToggleButtonWidgetClass, form,
				     XmNset, False,
				     XmNuserData, &img,
				     XmNindicatorType, XmONE_OF_MANY,
				     XmNtopAttachment, XmATTACH_WIDGET,
				     XmNtopWidget, rotateLbl,
				     XmNleftAttachment, XmATTACH_POSITION,
				     XmNleftPosition, 9,
				     NULL );
  XtAddCallback( btns[2], XmNvalueChangedCallback, rotateButton, btns );

  btns[3] = XtVaCreateManagedWidget( "btn3", xmToggleButtonWidgetClass, form,
				     XmNset, False,
				     XmNuserData, &img,
				     XmNindicatorType, XmONE_OF_MANY,
				     XmNtopAttachment, XmATTACH_WIDGET,
				     XmNtopWidget, rotateLbl,
				     XmNleftAttachment, XmATTACH_POSITION,
				     XmNleftPosition, 10,
				     NULL );
  XtAddCallback( btns[3], XmNvalueChangedCallback, rotateButton, btns );

  /* Create scrollbars */

  img.hsBar = XtVaCreateWidget( "hsbar", xmScrollBarWidgetClass, form,
			    XmNleftAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNorientation, XmHORIZONTAL,
			    XmNminimum, 0,
			    XmNmaximum, 110,
			    XmNsliderSize, 10,
			    NULL );

  XtAddCallback( img.hsBar, XmNvalueChangedCallback, hsbarMoved, &img );

  img.vsBar = XtVaCreateWidget( "vsbar", xmScrollBarWidgetClass, form,
			    XmNtopAttachment, XmATTACH_WIDGET,
			    XmNtopWidget, scale,
			    XmNrightAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNorientation, XmVERTICAL,
			    XmNminimum, 0,
			    XmNmaximum, 110,
			    XmNsliderSize, 10,
			    NULL );

  XtAddCallback( img.vsBar, XmNvalueChangedCallback, vsbarMoved, &img );

  /* Define DRAG_SCROLL if you want the image to scroll while
   * the scroll bars are being dragged.  If not defined, the
   * image will not be updated until the scroll bar is released.
   */

#define DRAG_SCROLL
#if defined(DRAG_SCROLL)
  XtAddCallback( img.hsBar, XmNdragCallback, hsbarMoved, &img );
  XtAddCallback( img.vsBar, XmNdragCallback, vsbarMoved, &img );
#endif /* DRAG_SCROLL */

  /* A Motif drawing area widget is used to display the SIE image */

  view = XtVaCreateWidget( "sieView", xmDrawingAreaWidgetClass, form,
			   XmNtopAttachment, XmATTACH_WIDGET,
			   XmNtopWidget, scale,
			   XmNleftAttachment, XmATTACH_FORM,
			   XmNbottomAttachment, XmATTACH_WIDGET,
			   XmNbottomWidget, img.hsBar,
			   XmNrightAttachment, XmATTACH_WIDGET,
			   XmNrightWidget, img.vsBar,
			   NULL );

  XtAddCallback( view, XmNexposeCallback, viewExpose, &img );
  XtAddCallback( view, XmNresizeCallback, viewResized, &img );

  img.outputWidget = view;

  XtVaSetValues( img.hsBar,
                 XmNrightAttachment, XmATTACH_WIDGET,
		 XmNrightWidget, img.vsBar,
		 NULL );

  XtManageChild( scaleLbl );
  XtManageChild( scale );
  XtManageChild( rotateLbl );
  XtManageChild( img.hsBar );
  XtManageChild( img.vsBar );
  XtManageChild( view );
  XtManageChild( form );

  XtRealizeWidget( toplevel );

  /* Calculate horizontal and vertical scale factors based on initial
   * scale value, size of image, and size of display window.
   */

  computeScale( &img );

  /* This next section, where the GC is initialized and the
   * pixmap for the image created, should really go in ReadFile,
   * but it has to be done after the view widget is created.
   */

  gcMask = GCForeground | GCBackground | GCFunction;

  gcValues.foreground = WhitePixelOfScreen( XtScreen(view) );
  gcValues.background = BlackPixelOfScreen( XtScreen(view) );
  gcValues.function = GXcopy;

  /* use a sharable gc */

  img.gc = XtGetGC( view, gcMask, &gcValues );

  /* Query the NCD terminal for memory statistics */

  queryNcdMemory( toplevel, &img );

  /* Decide if there is enough memory to store the image in the server.
   * An accurate threshold depends on the range of image sizes, number of
   * simultaneous images, amount of memory needed by other applications, and
   * total amount of terminal memory.  Determining a suitable threshold for
   * your application will require some experimenting.
   *
   * The intent demonstrated here is to maximize the use of the terminal's
   * memory, while not monopolizing all of it.  We leave enough free memory
   * for one image, so when the terminal runs low, we switch from using
   * XSieCreateImage to XSieSendImage.  This maximizes performance while
   * not limiting the number of simultaneous images.
   *
   * A further refinement would be to recheck the amount of free memory
   * at each redraw, and if enough memory becomes available, grab it.
   *
   * A completely different technique would be to store in the server the
   * pixmap of only the "currently used" image.  Once the user's attention
   * is directed elsewhere, discard the pixmap.  This could be done based
   * on which window has current focus.
   */

  /* Make sure there is free memory for 2 images before we suck up one */

  memNeeded = (img.imageWidth * img.imageHeight / 8) * 2;
  /* fudge factor, adjust as desired */
  memNeeded = memNeeded + memNeeded/5;
  fprintf( stderr, "memNeeded = %ld\n", memNeeded );

  /* Substitute ncdLargestDataFree if you have a lot of fragmentation */

  if ( memNeeded < ncdTotalDataFree ) {
      img.imagePixmap = XSieCreateImage( XtDisplay(view), XtWindow(view),
                            img.imageWidth, img.imageHeight, img.dataFormat,
                            img.data, img.dataLen );
  }
  else {
      img.imagePixmap = NULL;
  }

  img.imagePixmap = NULL;
  img.needNewPixmap = True;

  /* Begin main loop */

  XtAppMainLoop( app_context );

}     /* main() */



int
ReadFile( img )
imageStruct *img;
/*
 * Read CCITT Group 3 or Group 4 demo files supplied by NCD.  Read
 * image width and height from header, and raw image data into
 * buffer.  Return 0 on success, -1 on any failure.
 */
{
  FILE *fp;
  struct stat statbuf;
  char buf[82];
  unsigned char *p;
  int count;
  extern char *sys_errlist[];

  if ( NULL == img->filename ) {
    fprintf( stderr, "ReadFile: filename is null.\n" );
    return(-1);
  }

  fp = fopen( img->filename, "r" );
  if ( NULL == fp ) {
    fprintf( stderr, "ReadFile: unable to open file '%s': %s\n", img->filename,
                                                       sys_errlist[errno] );
    return(-1);
  }

  /* Check for known data formats */

  fgets( buf, 80, fp );
  if ( 0 == strcmp( buf, "CCITT Group4 2d\n" ) )
    img->dataFormat = SieCompressGroup42d;
  else {
    if ( 0 == strcmp( buf, "CCITT Group3 1d\n" ) )
      img->dataFormat = SieCompressGroup31d;
    else {
      fprintf( stderr, "File in unknown format.\n" );
      return(-1);
    }
  }

  fgets( buf, 80, fp );
  img->imageWidth = atoi( buf );

  fgets( buf, 80, fp );
  img->imageHeight = atoi( buf );

  fstat( fileno(fp), &statbuf );

  /* allocate space for image data */

  img->data = (unsigned char *)malloc( statbuf.st_size );
  if ( NULL == img->data ) {
    fprintf( stderr, "ReadFile: could not allocate space for image buffer.\n" );
    return(-1);
  }

  /* read image data */

  p = img->data;
  img->dataLen = 0;
  while ( (count = fread( p, 1, 1024, fp )) > 0 ) {
    p += count;
    img->dataLen += count;
  }

  /* scale units are percent of full-size image */

  img->scale = 100;

  img->orientation = SieRotation0;

  img->xoffset = 0;
  img->yoffset = 0;

  return(0);

}     /* ReadFile() */


void
rotateButton( w, btns, cbs )
Widget w;
Widget *btns;
XmToggleButtonCallbackStruct *cbs;
/*
 * This is the callback function for the four image rotation buttons.
 * It sets the value of img->orientation, and also handles the radio
 * button behavior for all of the toggle buttons.
 */
{
  imageStruct *img;

  XtVaGetValues( w, XmNuserData, &img, NULL );

  switch ( img->orientation ) {
    case SieRotation0:   XtVaSetValues( btns[0], XmNset, False, NULL ); break;
    case SieRotation90:  XtVaSetValues( btns[1], XmNset, False, NULL ); break;
    case SieRotation180: XtVaSetValues( btns[2], XmNset, False, NULL ); break;
    case SieRotation270: XtVaSetValues( btns[3], XmNset, False, NULL ); break;
  }

  if ( w == btns[0] )
    img->orientation = SieRotation0;
  if ( w == btns[1] )
    img->orientation = SieRotation90;
  if ( w == btns[2] )
    img->orientation = SieRotation180;
  if ( w == btns[3] )
    img->orientation = SieRotation270;

  computeScale(img);

  img->needNewPixmap = True;

  viewExpose( img->outputWidget, img, NULL );
}


void
vsbarMoved( w, img, cbs )
Widget w;
imageStruct *img;
XmScrollBarCallbackStruct *cbs;
/*
 * Callback for vertical scrollbar motion.
 * Set yoffset, and call viewExpose to redraw.
 */
{
  extern void viewExpose();

  img->yoffset = cbs->value;

  viewExpose( img->outputWidget, img, NULL );
}


void
hsbarMoved( w, img, cbs )
Widget w;
imageStruct *img;
XmScrollBarCallbackStruct *cbs;
/*
 * Callback for horizontal scrollbar.  Same as for vsbarMoved.
 */
{
  extern void viewExpose();

  img->xoffset = cbs->value;

  viewExpose( img->outputWidget, img, NULL );
}


void
computeScale(img)
imageStruct *img;
/*
 * Compute size of scaled image based on current scale value.  Adjust
 * scrollbar slider sizes.  Adjust x and y offsets. Compute SIE source
 * and destination rectangle dimensions based
 * on scale value in img struct and size of display window.  Perform
 * bounds checks to keep rectangles within image and window.
 */
{
  Dimension viewWidth, viewHeight;
  int current;

  /* Get current size of output window */

  XtVaGetValues( img->outputWidget,
                 XmNwidth, &viewWidth, XmNheight, &viewHeight, NULL );

  /* Compute size of scaled image, maintaining aspect
   * ration for 90 and 270 degree rotations
   */

  if ((img->orientation==SieRotation90) || (img->orientation==SieRotation270)) {
    img->scaleWidth = img->imageHeight * img->scale / 100;
    img->scaleHeight = img->imageWidth * img->scale / 100;
  }
  else {
    img->scaleWidth = img->imageWidth * img->scale / 100;
    img->scaleHeight = img->imageHeight * img->scale / 100;
  }

  /* adjust scrollbar values and size of sliders */

  if ( viewWidth >= img->scaleWidth ) {
    XtVaSetValues( img->hsBar,
		   XmNmaximum, viewWidth,
		   XmNsliderSize, viewWidth,
		   XmNvalue, 0,
		   NULL );
    img->xoffset = 0;
  }
  else {
    XtVaGetValues( img->hsBar, XmNvalue, &current, NULL );
    XtVaSetValues( img->hsBar,
		   XmNmaximum, img->scaleWidth,
		   XmNsliderSize, viewWidth,
		   XmNvalue, MIN( current, (img->scaleWidth - viewWidth) ),
		   NULL );
  }

  if ( viewHeight >= img->scaleHeight ) {
    XtVaSetValues( img->vsBar,
		   XmNmaximum, viewHeight,
		   XmNsliderSize, viewHeight,
		   XmNvalue, 0,
		   NULL );
    img->yoffset = 0;
  }
  else {
    XtVaGetValues( img->vsBar, XmNvalue, &current, NULL );
    XtVaSetValues( img->vsBar,
		   XmNmaximum, img->scaleHeight,
		   XmNsliderSize, viewHeight,
		   XmNvalue, MIN( current, (img->scaleHeight - viewHeight) ),
		   NULL );
  }

  /* Adjust x and y offsets if needed */

  if ( (viewWidth + img->xoffset) > img->scaleWidth ) {
    img->xoffset = img->scaleWidth - viewWidth;
    if ( img->xoffset < 0 )
      img->xoffset = 0;
  }

  if ( (viewHeight + img->yoffset) > img->scaleHeight ) {
    img->yoffset = img->scaleHeight - viewHeight;
    if ( img->yoffset < 0 )
      img->yoffset = 0;
  }

}     /* computeScale() */


void
scaleMoved( w, img, cbs )
Widget w;
imageStruct *img;
XmScaleCallbackStruct *cbs;
/*
 * Callback for scale widget.  Get new scale value from widget value,
 * call computeScale to do calculations, then call viewExpose to draw
 * the rescaled image.
 */
{
  Dimension viewWidth, viewHeight;
  extern void viewExpose();

  img->scale = cbs->value;

  computeScale(img);

  img->needNewPixmap = True;

  viewExpose( img->outputWidget, img, NULL );

}     /* scaleMoved() */


void
viewExpose( w, img, cbs )
Widget w;
imageStruct *img;
XmDrawingAreaCallbackStruct *cbs;
{
  Dimension viewWidth, viewHeight;
  unsigned long gcMask;
  XGCValues gcValues;
  GC tmpGc;

  /* If scale has been changed, new scaled pixmap of image is needed */

  if ( img->needNewPixmap ) {
    if ( NULL != img->imagePixmap )
      XFreePixmap( XtDisplay(w), img->imagePixmap );

    if ( DefaultDepthOfScreen(XtScreen(w)) > 1 ) {

      /* Create pixmap to contain scaled image */

      img->imagePixmap = XCreatePixmap( XtDisplay(w), XtWindow(w),
				        img->scaleWidth, img->scaleHeight, 1 );

      /* Create graphics context for SIE to use to draw in 1-deep pixmap */

      gcMask = GCForeground | GCBackground | GCFunction | GCPlaneMask;

      gcValues.foreground = BlackPixelOfScreen( XtScreen(w) );
      gcValues.background = WhitePixelOfScreen( XtScreen(w) );
      gcValues.function = GXcopy;
      gcValues.plane_mask = 1;

      tmpGc = XCreateGC( XtDisplay(w), img->imagePixmap, gcMask, &gcValues );

      /* Decode and scale image into pixmap */

      XSieSendImage( XtDisplay(w), img->imagePixmap, tmpGc, img->imageWidth,
		     0, 0, img->imageWidth, img->imageHeight,
		     0, 0, img->scaleWidth, img->scaleHeight, img->orientation,
		     img->dataFormat, img->data, img->dataLen );

      XFreeGC( XtDisplay(w), tmpGc );
    }
    else {

      /* Create pixmap to contain scaled image */

      img->imagePixmap = XCreatePixmap( XtDisplay(w), XtWindow(w),
					img->scaleWidth, img->scaleHeight, 1 );

      /* Decode and scale image into pixmap */

      XSieSendImage( XtDisplay(w), img->imagePixmap, img->gc, img->imageWidth,
		     0, 0, img->imageWidth, img->imageHeight,
		     0, 0, img->scaleWidth, img->scaleHeight, img->orientation,
		     img->dataFormat, img->data, img->dataLen );
    }
    img->needNewPixmap = False;
  }

  /* Get current size of output window */

  XtVaGetValues( w, XmNwidth, &viewWidth, XmNheight,  &viewHeight, NULL );

  /* If screen depth is greater than 1, then copy pixmap into first
   * plane of display window.  Otherwise, copy pixmap into window directly.
   */

  if ( DefaultDepthOfScreen(XtScreen(w)) > 1 ) {
    XCopyPlane( XtDisplay(w), img->imagePixmap, XtWindow(w), img->gc,
               img->xoffset, img->yoffset, viewWidth, viewHeight, 0, 0, 1 );
  }
  else {
    XCopyArea( XtDisplay(w), img->imagePixmap, XtWindow(w), img->gc,
               img->xoffset, img->yoffset, viewWidth, viewHeight, 0, 0 );
  }
}    /* viewExpose() */


void
viewResized( w, img, cbs )
Widget w;
imageStruct *img;
XmDrawingAreaCallbackStruct *cbs;
/*
 * Callback for resize event on view widget.  Basically does the same
 * thing as scaleMoved, except the existing scale value is used.
 */
{
  Widget view;
  Dimension viewWidth, viewHeight;
  extern void viewExpose();

  computeScale(img);

  img->needNewPixmap = True;

  viewExpose( w, img, NULL );

}     /* viewResized() */


XErrorHandler
handleXError( display, event )
Display *display;
XErrorEvent *event;
/*
 * This function demonstrates how to intercept X errors, and
 * handle non-fatal SIE extension errors.
 */
{

  /* If event->request_code does not match sieOpcode, then error
   * was not caused by an SIE operation.  Call the default error
   * handler to display the appropriate message.
   */

  if ( sieOpcode != event->request_code ) {
    if (_XDefaultError(display, event) == 0)
      return 0;
  }
  else {

    /* Handle SIE errors.  This code is an example only; replace it
     * with your own error handling code based on how your application
     * should deal with the various error conditions.  The most common
     * errors will be BadAlloc, insufficient memory for creating the
     * image, and BadValue, fax data is corrupted.  Application programs
     * should be able to handle these errors without completely bailing
     * out.  Other types of errors are usually the result of
     * programming errors.
     */

    switch ( event->minor_code ) {

      case X_SieCreateImage:
	if ( BadAlloc == event->error_code ) {
	  fprintf( stderr, "SieCreateImage: X server has insufficient " );
	  fprintf( stderr, "memory to store image.\n" );
	  return(0);
	}
	if ( BadValue == event->error_code ) {
	  fprintf(stderr,"SieCreateImage: data does not match compression ");
	  fprintf(stderr, "format.\nUnable to display image.\n" );
	  exit(1);
	}
	if (_XDefaultError(display, event) == 0)
	  return 0;
	break;

      case X_SieSendImage:
	if ( BadAlloc == event->error_code ) {
	  fprintf( stderr, "SieSendImage: X server has insufficient " );
	  fprintf( stderr, "memory to store image.\n" );
	  return(0);
	}
	if ( BadValue == event->error_code ) {
	  fprintf(stderr,"SieSendImage: data does not match compression ");
	  fprintf(stderr, "format.\nUnable to display image.\n" );
	  exit(1);
	}
	if (_XDefaultError(display, event) == 0)
	  return 0;
	break;

      case X_SieCopyImage:
	if ( BadAlloc == event->error_code ) {
	  fprintf( stderr, "SieCopyImage: X server has insufficient " );
	  fprintf( stderr, "memory to display image.\n" );
	  return(0);
	}
	if (_XDefaultError(display, event) == 0)
	  return 0;
	break;

      case X_SieFetchImage:
	if ( BadAlloc == event->error_code ) {
	  fprintf( stderr, "SieFetchImage: X server has insufficient " );
	  fprintf( stderr, "memory to fetch image.\n" );
	  return(0);
	}
	if (_XDefaultError(display, event) == 0)
	  return 0;
	break;
	
      default:
	if (_XDefaultError(display, event) == 0)
	  return 0;
	break;
    }
  }
  return(0);

}     /* handleXError() */


Bool
queryNcdMemory( toplevel, img )
Widget toplevel;
imageStruct *img;
/*
 * Query the NCD terminal for memory statistics using the X selection
 * mechanism.  Terminal must be running NCDware 3.2.0 or later for
 * this to work.  Return True if successful, False otherwise.
 */
{
  static Bool firstTime = True;
  extern Bool gotMemoryStats;
  XEvent event;
  void handleSelection();

  img->queryAtom = XInternAtom( XtDisplay(toplevel), "_NCD_QUERY", True );
  if ( !img->queryAtom ) {
      fprintf( stderr, "XInternAtom failed for _NCD_QUERY.\n" );
      return( False );
  }

  img->targetAtom = XInternAtom( XtDisplay(toplevel),
                                 "_NCD_MEMORY_STATISTICS", True );
  if ( !img->targetAtom ) {
      fprintf( stderr, "XInternAtom failed for _NCD_MEMORY_STATISTICS.\n" );
      return( False );
  }

  /* Add an event handler function to handle SelectionNotify events */

  if ( firstTime ) {
      XtAddEventHandler( toplevel, 0, True, handleSelection, img );
      firstTime = False;
  }

  gotMemoryStats = False;

  /* Ask for the data */

  XConvertSelection( XtDisplay(toplevel), img->queryAtom, img->targetAtom,
                     img->targetAtom, XtWindow(toplevel), CurrentTime );

  /* In this example, data is returned sychonously, so we wait for
   * a SelectionNotify event to occur.  Alternatively, we could simply
   * return and continue with other tasks until the SelectionNotify
   * event handler is called.
   */

  while( !gotMemoryStats )
    if (True == XCheckTypedEvent(XtDisplay(toplevel), SelectionNotify, &event))
      XtDispatchEvent(event);

  return( True );

}     /* queryNcdMemory() */


void
handleSelection( w, img, ev, unused )
Widget w;
XEvent *ev;
imageStruct *img;
Boolean unused;
{
    extern void reportNcdMemory();

    if (ev->type == SelectionNotify &&
	ev->xselection.selection == img->queryAtom &&
	ev->xselection.target == img->targetAtom )
            reportNcdMemory ( XtDisplay(w), &ev->xselection );
}


void
reportNcdMemory( dpy, sep )
Display *dpy;
XSelectionEvent *sep;
{
    Atom actual_type;
    int actual_format;
    unsigned long nitems, bytesAfter;
    unsigned long *data = NULL;

    (void) XGetWindowProperty (dpy, sep->requestor, sep->property,
			       0L, NCD_Q_MEM_count, True, XA_INTEGER, 
			       &actual_type, &actual_format, &nitems, 
			       &bytesAfter, (unsigned char **) &data);
    if (!data)
	return;

    if (data[NCD_Q_MEM_MASK] & (1L << NCD_Q_MEM_TOTAL_DATA)) {
	fprintf(stderr, "Total data memory: %ld\n", data[NCD_Q_MEM_TOTAL_DATA]);
	ncdTotalData = data[NCD_Q_MEM_TOTAL_DATA];
    }

    if (data[NCD_Q_MEM_MASK] & (1L << NCD_Q_MEM_TOTAL_DATA_FREE)) {
	fprintf(stderr, "Total free data memory: %ld\n",
                        data[NCD_Q_MEM_TOTAL_DATA_FREE]);
	ncdTotalDataFree = data[NCD_Q_MEM_TOTAL_DATA_FREE];
    }

    if (data[NCD_Q_MEM_MASK] & (1L << NCD_Q_MEM_LARGEST_DATA_FREE)) {
	fprintf(stderr, "Largest free data chunk: %ld\n",
                        data[NCD_Q_MEM_LARGEST_DATA_FREE]);
	ncdLargestDataFree = data[NCD_Q_MEM_LARGEST_DATA_FREE];
    }

    /* Only 88K-based terminals (19c, MCX, etc) have seperate code and
     * data memory.  Code memory will be reported as 0 for all others.
     */

    if (data[NCD_Q_MEM_MASK] & (1L << NCD_Q_MEM_TOTAL_CODE))
	fprintf(stderr, "Total code memory: %ld\n", data[NCD_Q_MEM_TOTAL_CODE]);

    gotMemoryStats = True;

    XFree ((char *) data);

    return;

}     /* reportNcdMemory() */


/* The IBM AIX libXext.a library seems to be missing this funtion */
XMissingExtension()
{
}

