Page	60,132
Title	-Morrow Clock Interrupt Routines
;
;	Copyright (c) 1985
;	Morrow Designs, Inc.
;	San Leandro, California
;
;	Last Update 30_Jan_86
;
%Out	ClkInt.asm
page
;----------------------------------------------------------------------
; Equates (20_Apr_85)
;--------------------
;
; RTC definitions for National MM58167A chip
;-------------------------------------------
;
RTC_Base	equ	0C0h		;base port for real time clock
RTC_Hundreds	equ	RTC_Base+1	;	hundredths and tenths 
RTC_Seconds	equ	RTC_Base+2	;	seconds count (0-59)
RTC_Minutes	equ	RTC_Base+3	;	minutes count (0-59)
RTC_Hours	equ	RTC_Base+4	;	hours count (0-23)
RTC_Day		equ	RTC_Base+6	;	Day of month (1-31)
RTC_Month	equ	RTC_Base+7	;	Month	(1-12)
RTC_Ram_Base	equ	RTC_Base+8	;Base of Ram in RTC chip (8 Bytes)
RTC_Int_Satus	equ	RTC_Base+10h	;RTC interrupt status reg (Read only)
RTC_Int_Control	equ	RTC_Base+11h	;RTC interrupt control reg (Write " )
RTC_Status	equ	RTC_Base+14h	;roll-over status flag
RTC_Check	equ	0AA55h		;compare values for validity check

; Include Files
;--------------
;
;	Icon.lit
;	Rom.lit
;	Intr.lit
;	Sys.lit
;	Disk.lit
;	IO.lit
;	Video.lit
;
	.xlist
	Include	../Rom/Rom.lit
	Include	../Rom/Intr.lit
	Include	../Rom/Sys.lit
	Include ../Icon/Icon.lit
	Include ../Disk/Disk.lit
	Include	../IO/IO.lit
	Include ../Video/Video.lit
	.list

page
;======================================================================
Rom_Data Segment At (40H)
;========================
;
	extrn	Drive_Motor_Count:Byte
	extrn	Drive_Status:Byte
	extrn	Shift_Status:Byte
Rom_Data EndS

;======================================================================
Screen_Segment Segment at (0B800H)
;=================================
;
;	Define the Base of the Video Ram

Screen_Segment	EndS

;======================================================================
Monitor_Segment Segment Word Public
;==================================
;
Assume	cs:Monitor_Segment, ds:Rom_Data, ss:Nothing, es:Nothing

; Include Files
;--------------
;	Disk.ext
;	Video.ext
;	Sys.ext

	.xlist
	Include	../Disk/Disk.ext
	Include ../Video/Video.ext
	Include	../Rom/Sys.ext
	.list

	extrn	R_W_Rtc_Ram:Near
	extrn	In_Morrow_6845:Near
	extrn	Out_Morrow_6845:Near
	extrn	Init_Key_Buff:Near
	extrn	Delay:Near
 	extrn	Minute:Near
	extrn	DS_to_Rom_Data:Near

	extrn	Disp_Day:Byte
	extrn	Disp_Month:Byte
	extrn	Disp_Year:Word

	extrn	Clock_Hdsec:Byte	;Current 1/100th of a Second
	extrn	Clock_Second:Byte	;Current Second
	extrn	Clock_Minute:Byte	;Current Minute
	extrn	Clock_Hour:Byte		;Current Hour
	extrn	Clock_Day:Byte		;Day of the Month
	extrn	Clock_Month:Byte	;Current Month
	extrn	Clock_Year:Word		;Current Year
	extrn	Clock_Last_Day:Byte	;Day of the Month Read on Last Read
	extrn	N_System_Setup:Word	;Saved system setup
	extrn	N_Serial_Setup:Word	;Serial port setup
	extrn	N_Modem_Setup:Word	;Saved modem setup
	extrn	N_Year:Word		;Current year from setup
	extrn	N_Month:Byte		;Month machine turned on
	extrn	Old_Shift_Status:Byte	; detects progs that change shift_Status
page
;----------------------------------------------------------------------
; Fixed Data Area (20_Apr_85)
;----------------------------
;
Tod_Func_Lookup	Label Word
;-------------------------
;	1) This is a vector table for the extended Time of Day interrupt
;	   function (int 1A).
;	2) Functions are arranged in ascending order by their Function
;	   code. The Function numbers must be contiguous.
;
	dw	Get_Date		;(2Ah) -- Get the Date (in Bcd)
	dw	Set_Date		;(2Bh) -- Set the Date (in Bcd)
	dw	Get_Time		;(2Ch) -- Get the Time (in Bcd)
	dw	Set_Time		;(2Dh) -- Set the Time (in Bcd)
	dw	Get_Disp_Date		;(2Eh) -- Get Displayed Date
	dw	Set_Disp_Date		;(2Fh) -- Set Displayed Date
	dw	R_W_Rtc_Ram		;(30h) -- Enable / Disable RTC and Ram

page
;----------------------------------------------------------------------
Time_of_Day_Interrupt Proc Far; (1_Jul_85)
;-----------------------------------------
;
; TIME_OF_DAY_INTERRUPT: (FUNCTION, PARAMETERS)	     Interrupt 1AH (26)
;
;	Time_of_Day_Interrupt is called to set or reset the current
; time-of-day.  It uses special units of time which are a multiple of
; the system timer, which interrupts the CPU at an 18.206482 Hz rate.
; This count is kept as a word pair (long unsigned integer), which
; contains the number of time periods since midnight.  An hour is
; represented as 0:65543, while a day is 24:176.
;	To execute a time-of-day function, load register AH with the
; function code which follows (as well as any parameters required by
; the particular operation desired).  Next, perform an 'INT 1AH' - 
; the function will be performed, and any results will be returned
; in the registers specified.
;
;	NOTE: Only functions 0 and 1 are standard function calls. 
;		Function calls 2A-2F are extended calls for the
;		Morrow Pivot II.
;
; Function Code 0 - Read the Current Time of Day in 'TICKS'
;   Output:
;	CX: Most significant word of timer count
;	DX: Least significant word of timer count
;	AL: Number of days which have passed since last time read
;
; Function Code 1 - Set the Time of Day
;    Input:
;	CX: Most significant count word
;	DX: Least significant count word
;
; Function Code 2A - Return the Date from the RTC
;   Output:
;	CX: Clock Year (1984-2025)
;	DH: Month (1-12)
;	DL: Day of Month (1-31)
;
; Function Code 2B - Set Date in RTC
;    Input:
;	CX: Clock Year (1984-2025 is valid range)
;	DH: Month (1-12)
;	DL: Day of Month (1-31)
;
; Function Code 2C - Get Time of Day from RTC
;   Output:
;	CH: Clock Hour (0-23)
;	CL: Clock Minute (0-59)
;	DH: Clock Second (0-59)
;	DL: Clock Hundredths of seconds (0-99)
;
; Function Code 2D - Set Time of Day in RTC
;    Input:
;	CH: Clock Hour (0-23)
;	CL: Clock Minute (0-59)
;	DH: Clock Second (0-59)
;	DL: Clock Hundredths of seconds (0-99)
;
; Function Code 2E - Return the diplayed Date 
;   Output:
;	CX: Clock Year (1984-2025)
;	DH: Month (1-12)
;	DL: Day of Month (1-31)
;
; Function Code 2F - Set Displayed Date 
;    Input:
;	CX: Clock Year (1984-2025 is valid range)
;	DH: Month (1-12)
;	DL: Day of Month (1-31)
;
; Function Code 30 - Enable / Disable RTC and NovRam
;    Input:
;	AL:	0 --> Disable RTC and RAM Writes
;		1 --> Enable RTC and RAM Writes
;
Assume	cs:Monitor_Segment, ds:Rom_Data, ss:Nothing, es:Nothing
Public	Time_of_Day_Interrupt

	PushReg	<bx,ds>
	call	DS_to_Rom_Data		;DS:= Rom Data

	cmp	ah,Tod_Get		;If (Function eq Get Time)
	jnz	Tod11

	mov	al,Tod.Days		;	read the day counter
	mov	Tod.Days,0		;	Clear day count for next read
	mov	cx,Tod.MsCount		;	CX:= M.S.Word of Time
	mov	dx,Tod.LsCount		;	DX:= L.S.Word of Time
	jmp	Short Tod2

Tod11:	cmp	ah,Tod_Set		;Else If (Function eq Set Time)
	jnz	Short Tod12

	mov	Tod.Days,0		;	Force day count to be 0
	mov	Tod.MsCount,cx		;	Write M.S.Word of Time
	mov	Tod.LsCount,dx		;	Write L.S.Word of Time
	jmp	Short Tod2

Tod12:	cmp	ah,Tod_Get_Date		;Else If ( (Function ge Get_Date) And
	jb	Tod2
	cmp	ah,Tod_RTC_Access	;	   (Function le RTC_Access) )
	ja	Tod2

	sub	ah,Tod_Get_Date		;	AL:= Zero Based Function Number
	shl	ah,1			;	(convert to a word offset)
	mov	bl,ah
	mov	bh,0			;	BX:= Index to Vector Table

	Assume	ds:Monitor_Segment
	push	ax
	mov	ax,Monitor_Segment
	mov	ds,ax			;	DS:= Monitor_Segment
	pop	ax

	pushf
	cli
	call	Tod_Func_Lookup[bx]	;	Execute the Function
	popf

	mov	al,Clock_Day		;	Get date
	xchg	al,Clock_Last_Day	;	Set last day
	sub	al,Clock_Last_Day	;	Set al to 0 for same day

Tod2:	PopReg	<ds,bx>			;(restore registers)
	IRet				;Return

Time_of_Day_Interrupt	EndP

page
;----------------------------------------------------------------------
Get_Date Proc Near; (21_Apr_85)
;------------------------------
;	1) Function Code 2A - Return the Date from the RTC
;		CX: Clock Year (1984-2025)
;		DH: Month (1-12)
;		DL: Day of Month (1-31)
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, ss:Nothing, es:Nothing

	push	bx
	call	Get_Rtc_Time		;	Read the time from the Rtc
	pop	bx

	mov	cx,Clock_Year		;CX:= Clock Year
	mov	dh,Clock_Month		;DH:= Clock Month
	mov	dl,Clock_Day		;DL:= Clock Day
	ret				;Return

Get_Date	EndP

;----------------------------------------------------------------------
Get_Time Proc Near; (25_Jun_85)
;------------------------------
;	1) Function Code 2C - Get Time of Day from RTC
;		CH: Clock Hour (0-23)
;		CL: Clock Minute (0-59)
;		DH: Clock Second (0-59)
;		DL: Clock Hundredths of seconds (0-99)
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, ss:Nothing, es:Nothing
Public	Get_Time

	push	bx
	call	Get_Rtc_Time		;Read the time from the Rtc
	pop	bx

	mov	ch,Clock_Hour		;CH:= Clock Hour
	mov	cl,Clock_Minute		;CL:= Clock Minute
	mov	dh,Clock_Second		;DH:= Clock Seconds
	mov	dl,Clock_HdSec		;DL:= Clock 1/100's of Seconds
	ret				;Return

Get_Time	EndP

page
;----------------------------------------------------------------------
Set_Date Proc Near; (21_Apr_85)
;------------------------------
;	1) Function Code 2B - Set Date in RTC
;		CX: Clock Year (1984-2025 is valid range)
;		DH: Month (1-12)
;		DL: Day of Month (1-31)
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, ss:Nothing, es:Nothing

	PushReg	<ax,dx>
	mov	Clock_Year,cx		;Year:= CX
	mov	Clock_Month,dh		;Clock Month:= DH
	mov	Clock_Day,dl		;Clock Day:= DL

	mov	al,1
	call	R_W_Rtc_Ram		;EnAble Rtc and NovRam Accesses

	mov	dx,RTC_Day		;DX:= Rtc Ram Day
	mov	al,Clock_Day		;AL:= Clock Day
	call	Bin_2_BCD		;Convert to BCD
	out	dx,al			;Write Rtc Ram Day of month

	inc	dx			;DX:= Rtc Ram Month
	mov	al,Clock_Month		;AL:= Clock Month
	mov	N_Month,al		;Update NovRam Month
	call	Bin_2_BCD		;Convert Month to BCD
	out	dx,al			;Write Rtc Ram Month

	mov	ax,Clock_Year		;AX:= Clock Year
	mov	N_Year,ax		;Update NovRam Year

	mov	al,0
	call	R_W_Rtc_Ram		;DisAble Rtc and NovRam Accesses

	PopReg	<dx,ax>
	ret				;Return

Set_Date	EndP

page
;----------------------------------------------------------------------
Set_Time Proc Near; (21_Apr_85)
;------------------------------
;	1) Function Code 2D - Set Time of Day in RTC
;		CH: Clock Hour (0-23)
;		CL: Clock Minute (0-59)
;		DH: Clock Second (0-59)
;		DL: Clock Hundredths of seconds (0-99)
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, ss:Nothing, es:Nothing

	PushReg	<ax,dx>
	mov	Clock_Hour,ch	 	;Clock Hour:= CH
	mov	Clock_Minute,cl 	;Clock Minutes:= CL
	mov	Clock_Second,dh		;Clock Seconds:= DH
	mov	Clock_HdSec,dl		;Clock Seconds:= DL
	call	Set_TOD_Count		;set 'TICKS' since midnight

	mov	al,1
	call	R_W_Rtc_Ram		;EnAble Rtc and NovRam Accesses

	mov	dx,RTC_Seconds		;DX:= Rtc Ram Seconds
	mov	al,Clock_Second		;AL:= Clock Seconds
	call	Bin_2_BCD		;Convert to BCD
	out	dx,al			;Write Seconds to Rtc Ram

	inc	dx			;DX:= Rtc Ram Minutes
	mov	al,Clock_Minute		;AL:= Clock Minutes
	call	Bin_2_BCD		;Convert Minutes to BCD
	out	dx,al			;Write Minutes to Rtc Ram

	inc	dx			;DX:= Rtc Ram Hours
	mov	al,Clock_Hour		;AL:= Clock Hours
	call	Bin_2_BCD		;Convert Hours to BCD
	out	dx,al			;Write Hours to Rtc Ram

	mov	al,0
	call	R_W_Rtc_Ram		;DisAble Rtc and NovRam Accesses

	PopReg	<dx,ax>
	ret				;Return

Set_Time	EndP

;----------------------------------------------------------------------
Get_Disp_Date Proc Near; (6_May_85)
;----------------------------------
;	1) Function Code 2E - Get Displayed date
;		CX: Displayed Year (1985-2025)
;		DH: Displayed Month (1-12)
;		DL: Displayed Day (1-31)
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, ss:Nothing, es:Nothing


	mov	cx,Disp_Year		;CX:= Displayed year
	mov	dh,Disp_Month		;DH:= Displayed Month
	mov	dl,Disp_Day		;DL:= Displayed Day
	ret				;Return

Get_Disp_Date 	EndP


;----------------------------------------------------------------------
Set_Disp_Date Proc Near; (6_May_85)
;----------------------------------
;	1) Function Code 2F - Set Displayed date
;	On Entry:
;		CX: Displayed Year (1985-2025)
;		DH: Displayed Month (1-12)
;		DL: Displayed Day (1-31)
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, ss:Nothing, es:Nothing


	mov	Disp_Year,cx		;Displayed year:=CX
	mov	Disp_Month,dh		;Displayed Month:=DH
	mov	Disp_Day,dl		;Displayed Day:=DL
	ret				;Return

Set_Disp_Date 	EndP

page
;----------------------------------------------------------------------
Timer_Interrupt Proc Far; (1_Jul_85)
;-----------------------------------
; TIMER_INTERRUPT: 				     Interrupt 08H (08)
;
;	Timer_Interrupt is the system real-time clock entry.  Roughly
; 18 times a second it updates a set of real-time variables which can
; be set or read by the user 'TIME_OF_DAY_INTERRUPT' function call.
; This routine is responsible for turning the floppy disk drive motors
; off after they have not been used for a while, and for flashing the
; 'software' cursor in high-res graphics mode (if it is enabled).
;
; On the PIVOT, this routine is also responsible for setting the RTC
; if time or date changed, and if appointments are enabled, for checking
; every minute for pending appointments.
;
Assume	cs:Monitor_Segment, ds:Rom_Data, ss:Nothing, es:Nothing
Public	Timer_Interrupt

	sti				; turn interrupts back on!!!!!
	PushReg	<ds,ax,dx>

	mov	al,End_of_Intr		;Get the end-of-interrupt command
	out	Interrupt_Ctrl,al	;Let other interrupts through
	in	al,Interrupt_Mask	;But dis-allow timer ints until done
	or	al,1			; by disbling int 8
	out	Interrupt_Mask,al


	call	DS_to_Rom_Data		;DS:= Rom Data

	;Update Time-of-Day
	inc	Tod.LsCount		;Increment the L.S. timer word
	jnz	TI1			;No overflow - time O.K.

	inc	Tod.MsCount		;Increment the M.S. timer word
TI1:	cmp	Tod.MsCount,Day_CountM	;M.S. count reached a day?
	jnz	TI2			;No, time all set
	cmp	Tod.LsCount,Day_CountL	;L.S. count reached a day?
	jnz	TI2			;No, finished with time update

	inc	Tod.Days		;Yes - increment day count
	mov	Tod.MsCount,0		;Set the most-significant word to 0
	mov	Tod.LsCount,0		;Set the least-significant word to 0


; Turn off floppy drive motors if unused
;---------------------------------------
; This piece of code is designed to execute as fast as possible.
; Cleanliness and comprehensability were secondary considerations.
;
TI2:	
	cmp	FDC_Power_Flag,1	;check if counts enabled
	jnz	Ti25			;jump if disabled

	mov	dx,disk_controller_port	;765 status address
	in	al,dx			;get 765 status
	test	al,13h			;see if 765 is in use
	jz	dp3			;jump if not in use
	mov	disk_use_status,1		;set flag that 765 is in use

Dp3:	mov	dx,word ptr Disk0_Turn_Off_Limit	;load both limits
Dp2:	dec	Disk_Turn_Off_Count	;dec counter
	jz	Dp2			;don't allow zero

	cmp	dl,disk_turn_off_count	;compare drive 0 limit
	jnz	Dp0			;jump if not at limit
	cmp	disk_use_status,0	;see if 765 is in use
	jnz	dp10			;jump if in use

	xor	dl,dl			;clear drive 0 limit
	mov	ax,10feh		;Turn off drive 0
	call	Turn_Off_Drive

Dp10:	mov	disk_use_status,0	;clear 765 in use flag
Dp0:	cmp	dh,disk_turn_off_count	;compare drive 1 limit
	jnz	Dp1			;jump if not at limit
	cmp	disk_use_status,0	;see if 765 is in use
	jnz	dp11			;jump if in use

	xor	dh,dh			;clear drive 1 limit
	mov	ax,20fdh		;Turn off drive 1
	call	Turn_Off_Drive

Dp11:	mov	disk_use_status,0	;clear 765 in use flag
Dp1:	mov	word ptr Disk0_Turn_Off_Limit,dx	;save limits
	or	dh,dl			;see if both limits are 0
	jnz	Ti25

	mov	FDC_Power_Flag,2	;Set FDC power flag
	call	test_power
	jnz	TI25

	mov	FDC_Power_Flag,0	;Clear FDC power flag
	mov	dx,Disk_Control_Port	;Point to the disk hardware port
	mov	al,0			;Get the command to shut down
	out	dx,al			;Disable the FDC
	mov	ax,219h			;Prepare to turn off FDC power
	call	Out_Morrow_6845

TI25:	mov	ah,0			;Set compatible data value
	dec	Drive_Motor_Count	;Has the floppy been used for awhile?
	jnz	TI3			;Yes - let it keep spinning
	mov	ah,0CH			;No, set compatible data for drive timeout
	cmp	FDC_Power_Flag,0	;Is power already off to the drives?
	jz	TI3			;Yes, don't do I/O
	and	Drive_Status,NOT Disk_Motor_Masks ;No, flag all motors as off
	mov	dx,Disk_Control_Port	;Point to the disk hardware port
	mov	al,Disk_IO_Enable	;Get the command to shut motors
	out	dx,al			;Disable the motors


; Execute user-supplied interrupt service routine
;------------------------------------------------
;
TI3:	push	ds
	mov	al,ah			;Get compatible disk timeout value
	mov	ah,0			;Set compatible value in AH
	int	User_Clock_Intr		;Execute user clock routine
	pop	ds

; Check if a program has changed keyboard shift status	(22_July_85)
;-------------------------------------------------------------------
;
	mov	ah,Shift_Status		; get Shift_Status
	cmp	ah,cs:Old_Shift_Status	; if (Shift_Status .NE. Old_Shift_Status)
	jz	NoChange		; then (Changed by Software)
	mov	cs:Old_Shift_Status,ah	;   set new status
	mov	al,Kbrd_State_Register	;   send new state to keyboard processor
	call	Out_Morrow_6845


; Check if a minute has passed, and handle alarm checking if so
;--------------------------------------------------------------
;
NoChange:
	call	Minute			; check for pending appointments

	in	al,Interrupt_Mask	;Let Timer Int's through now
	And	al,NOT 1
	out	Interrupt_Mask,al

	PopReg	<dx,ax,ds>		;Restore registers
	iret

Timer_Interrupt EndP

page
;----------------------------------------------------------------------
Turn_Off_Drive Proc Near; (18_Jan_86)
;------------------------------------
;
;	Turn_Off_Drive is a pair of subroutines used by Timer_Interupt
; to control the power to the floppy drives. The routine Test_Power checks
; if the system is currently on AC or Battery power. It returns a zero
; status if on battery or a non-zero status if on AC. Turn_Off_Drive 
; calls Test_Power and if on battery, it turns off the drive specified by
; AH.
;
Assume	cs:Monitor_Segment, ds:Rom_Data, ss:Nothing, es:Nothing
	push	ax
	call	test_power
	pop	ax
	jnz	Td1
	and	Recal_Status,al
	mov	al,19h
	call	Out_Morrow_6845
Td1:	ret

Test_Power:
	mov	al,System_Status_Register ;Get status reg.
	call	In_Morrow_6845
	test	al,AC_Power_On		;Test ac_on bit.
	ret

Turn_Off_Drive endp

page
;----------------------------------------------------------------------
Get_Rtc_Time Proc Near; (20_Apr_85)
;----------------------------------
;	1) This routine is used to initalize the variables kept up
;	   by the 18.2 Hz. interrupt.  These variables are set on
;	   power-up from the RTC.  They are then kept up in the back-
;	   ground, and used by the ROM Software, most notably CLOCK.
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, ss:Nothing, es:Nothing
Public	Get_Rtc_Time

	PushReg	<ax,cx,dx,ds>
	mov	ax,Monitor_Segment	; ds must reference data segment
	mov	ds,ax
	mov	al,1			; enable RTC access
	call	R_W_Rtc_Ram

GrLp1:	mov	dx,Rtc_Hundreds		;Loop	DX:= Pointer to Rtc hundredths of Seconds
	in	al,dx			;       Read hundredths of seconds
	call	Bcd_2_Bin		;       Convert from BCD to binary
	mov	Clock_HdSec,al		;       Update Hundredths

	inc	dx			;	DX:= Pointer to Rtc Seconds
	in	al,dx			;	Read the Seconds
	call	Bcd_2_Bin		;	Convert BCD to binary
	mov	Clock_Second,al		;	Update Seconds

	inc	dx			;	DX:= Pointer to Rtc Minutes
	in	al,dx
	call	Bcd_2_Bin		;	Convert from BCD to binary
	mov	Clock_Minute,al		;	Update Minutes

	inc	dx			;	DX:= Pointer to Rtc Hours
	in	al,dx
	call	Bcd_2_Bin		;	Convert from BCD to binary
	mov	Clock_Hour,al		;	Update Hours

	inc	dx			;	(Skip Day of the Week)

	inc	dx			;	DX:= Pointer - Rtc Day of Month
	in	al,dx
	call	Bcd_2_Bin		;	Convert from BCD to binary
	mov	Clock_Day,al

	inc	dx			;	DX:= Pointer to Month
	in	al,dx
	push	ax
	in	al,Rtc_Status		;	If (clock rolling over)
	test	al,1
	pop	ax
	jnz	GrLp1			;		Restart Loop
	call	Bcd_2_Bin		;Convert from BCD to binary
	mov	Clock_Month,al		;Update Month

	mov	cx,N_Year		;get year as saved in NovRam
	mov	ah,N_Month		;get last month out of NovRam
	cmp	ah,Clock_Month		;If (Rtc_Month < NovRam_Month)
	jle	GrSk1

	inc	N_Year			;	Increment year
	mov	cx,N_Year

GrSk1:	mov	Clock_Year,cx		;Update Year
	mov	al,Clock_Month
	mov	N_Month,al		;set month in NovRam for New Year check

	xor	al,al
	call	R_W_Rtc_Ram		;Disable RTC and NovRam
	PopReg	<ds,dx,cx,ax>
	ret

Get_Rtc_Time	EndP

page
;----------------------------------------------------------------------
Bcd_2_Bin Proc Near; (21_Apr_85)
;-------------------------------
;	1) Takes a BCD number in al and returns the binary equivalent in al.
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment 

	PushReg	<bx,cx>
	mov	bl,al
	and	al,0F0h			; get 10's digit
	mov	cl,4
	shr	al,cl			; now ready to multiply by 10
	mov	bh,10
	mul	bh
	and	bl,0Fh			; adding 1's so mask MS nibble
	add	al,bl			; add 1's digits
	PopReg	<cx,bx>
	ret

Bcd_2_Bin	EndP

;----------------------------------------------------------------------
Bin_2_Bcd Proc Near; (21_Apr_85)
;-------------------------------
;	1) Takes an 8 bit binary number in al and returns the BCD 
;	   equivalent in the al.
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment

	PushReg	<bx,cx>
	mov	bl,10
	and	ah,00h			; clear ah
	div	bl			; convert digits ah = 1's, al = 10's
	mov	cl,4
	shl	al,cl			; Get 10's in high nibble
	and	al,0F0h			; save MS BCD digit, clear LS digit
	or	al,ah
	PopReg	<cx,bx>
	ret

BIN_2_Bcd 	EndP

page
;----------------------------------------------------------------------
Set_TOD_Count Proc Near; (1_Jul_85)
;----------------------------------
;	1) This routine initializes the Time Of Day counter
;	   that is kept up by the 18.2 Hz interrupt.  It sets the
;	   count of the TOD variables to contain the correct number
;	   of 'TICKS' since midnight for the correct time.
;	2) Register Usage: AX changed, all others preserved
;
Assume	cs:Monitor_Segment, ds:Monitor_Segment, ss:Nothing, es:Nothing
Public	Set_ToD_Count

	PushReg	<bx,cx,dx,ds>
	mov	ax,Monitor_Segment
	mov	ds,ax			;DS:= Base of Monitor Segment

;	mov	ah,Clock_Second		; Read current second count
;Sync:	mov	al,Clock_Second		; Repeat
;	cmp	al,ah			;	Read Clock_Second
;	jz	sync			; Until Roll-Over occurs

	mov	al,60			
	mul	Clock_Hour		; convert hours to # of minutes
	add	al,Clock_Minute		; add minute time
	adc	ah,0
	mov	dx,60
	mul	dx			; DX:AX = # of seconds past midnight
	add	al,Clock_Second
	adc	ah,0			; propogate carry through DX:AX
	adc	dx,0

	mov	cx,dx			; save high word of seconds
	mov	bx,4661			; multiply seconds past midnight...
	mul	bx			; ..by 256 * 18.2064
	xchg	cx,ax
	xchg	bx,dx
	mul	dx
	add	ax,bx
	mov	dl,ch			; now, divide result by 256 to get...
	mov	dh,al			; ...actual ticks past midnight
	mov	al,ah			;AX = MsCount
	mov	ah,0			;DX = LsCount for timer

	Assume	ds:Rom_Data
	call	DS_to_Rom_Data		;DS:= Rom Data
	pushf
	cli				;disable int's while updating
	mov	Tod.MsCount,ax		;set the Msb of Ticks since midnight
	mov	Tod.LsCount,dx		; "   "  Lsb "    "     "      "
	mov	Tod.Days,0		;clear days elapsed counter
	popf

	PopReg	<ds,dx,cx,bx>
	ret				;Return

Set_TOD_Count	EndP

Monitor_Segment	EndS
		End
