// 'Windows CE 3.0 Programming' Source Code Samples (Prentice Hall, 2000)
// Source Code Author: Nick Grattan (nick@softwarepaths.com)
// Version 1.00

// ListDB.cpp: implementation of the CListDB class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
//#include "clDesktop.h"
#include "ListDB.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CListDB::CListDB()
{
	m_bChanged = TRUE;			// data file may have changed since last access?
	hFile = INVALID_HANDLE_VALUE;
}

CListDB::~CListDB()
{
	if(hFile != INVALID_HANDLE_VALUE)
		CloseHandle(hFile);
}

BOOL CListDB::OpenCreateDatabase()
{
	hFile = CreateFile(_T("\\ActiveSynNotes.dat"),
			GENERIC_WRITE | GENERIC_READ,
			FILE_SHARE_DELETE | FILE_SHARE_READ | FILE_SHARE_WRITE,
			NULL,
			OPEN_ALWAYS,
			FILE_ATTRIBUTE_NORMAL,
			NULL);
	if(hFile == NULL)
		return FALSE;
	else
		return TRUE;
}

#ifdef CL_DESKTOP
BOOL CListDB::ListNotes(CListCtrl &lstCtrl)
{
	int nNote = 0;
	LV_ITEM lvi;
	NOTE aNote;
	DWORD dwRead;

	lstCtrl.DeleteAllItems();
	lvi.mask = LVIF_TEXT | LVIF_STATE;
	lvi.state = 0;
	lvi.stateMask = 0;
	
	// move to start of file
	SetFilePointer(hFile, 0, NULL, FILE_BEGIN);

	while(ReadFile(hFile, (LPVOID)&aNote, sizeof(NOTE), &dwRead, NULL)
			&& dwRead > 0)
	{
		CTime ctime(aNote.ftOriginal);
		TCHAR szBuffer[100];
		wsprintf(szBuffer, _T("%02d/%02d/%02d %02d:%02d:%02d"), 
			ctime.GetDay(), ctime.GetMonth(), ctime.GetYear(), 
			ctime.GetHour(), ctime.GetMinute(), ctime.GetSecond());
		lvi.iItem = nNote;
		lvi.iSubItem = 0;
		lvi.pszText = szBuffer;
		
		lstCtrl.InsertItem(&lvi);

		CString sNote;
		lstCtrl.SetItemText(nNote, 1, aNote.szNote);
		nNote++;
	}
	return TRUE;
}
#endif

// add new note, using specified time (which may have come from device)
BOOL CListDB::AddNote(FILETIME* ftCreateTime, FILETIME* ftModifyTime, LPCWSTR sNote)
{
	NOTE aNote;
	DWORD dwWritten;

	aNote.ftOriginal = *ftCreateTime;
	aNote.ftLastUpdate = *ftModifyTime;
	wcscpy(aNote.szNote, sNote);
	SetFilePointer(hFile, 0, NULL, FILE_END);
	if(!WriteFile(hFile, (LPVOID)&aNote, sizeof(NOTE), &dwWritten, NULL) ||
			dwWritten != sizeof(NOTE))
		return FALSE;
	m_bChanged = TRUE;
	return TRUE;
}

// add new note, using current time
BOOL CListDB::AddNote(LPCWSTR sNote)
{
	// add a note for the current time
	SYSTEMTIME sysTime;
	FILETIME fileTime;
	NOTE aNote;
	DWORD dwWritten;

	GetSystemTime(&sysTime);
	sysTime.wMilliseconds = 0;
	SystemTimeToFileTime(&sysTime, &fileTime);
	aNote.ftOriginal = fileTime;
	aNote.ftLastUpdate = fileTime;

	wcscpy(aNote.szNote, sNote);
	// move to the end of the file
	SetFilePointer(hFile, 0, NULL, FILE_END);
	if(!WriteFile(hFile, (LPVOID)&aNote, sizeof(NOTE), &dwWritten, NULL) ||
			dwWritten != sizeof(NOTE))
		return FALSE;
	m_bChanged = TRUE;
	return TRUE;
}

BOOL CListDB::DeleteNote(FILETIME* ftNoteTime)
{
	NOTE aNote;
	DWORD dwRead, dwWritten;

	// find the note
	if(!FindNote(ftNoteTime))
	{
		MessageBox(NULL, _T("Could not find delete note"), NULL, MB_OK);
		return FALSE;
	}
	// shuffle all records up
	while(TRUE)
	{
		// move to start of next note
		if(SetFilePointer(hFile, sizeof(NOTE), NULL, FILE_CURRENT) ==
					0xFFFFFFFF)
			break; 			// at the end of file
		else		// read next note
			ReadFile(hFile, (LPVOID)&aNote, sizeof(NOTE), &dwRead, NULL);
		if(dwRead == 0)	// no more records to read
			break;
		// move back two notes
		SetFilePointer(hFile, (LONG)sizeof(NOTE) * -2, NULL, FILE_CURRENT);
		// write out note
		WriteFile(hFile, (LPVOID)&aNote, sizeof(NOTE), &dwWritten, NULL);
	}
	// set end of file pointer to one note from end
	SetFilePointer(hFile, (LONG)sizeof(NOTE) * -1, NULL, FILE_END);
	// set end of file pointer
	SetEndOfFile(hFile);
	m_bChanged = TRUE;
	return TRUE;
}

BOOL CListDB::DeleteNote(LPCTSTR sNoteTime)
{
	FILETIME fileTime;

	StringToFileTime(&fileTime, sNoteTime);
	return DeleteNote(&fileTime);

}

BOOL CListDB::DeleteAll()
{
	// move to start of file
	SetFilePointer(hFile, 0, NULL, FILE_BEGIN);
	// set end of file pointer
	SetEndOfFile(hFile);
	m_bChanged = TRUE;
	return TRUE;
}

BOOL CListDB::UpdateNote(FILETIME* ftNoteTime, LPCWSTR sNote)
{
	NOTE aNote;
	DWORD dwWritten;
	SYSTEMTIME sysTime;
	FILETIME fileTime;

	if(!FindNote(ftNoteTime))
		return FALSE;
	// upate the record
	aNote.ftOriginal = *ftNoteTime;

	// record last modification time
	GetSystemTime(&sysTime);
	sysTime.wMilliseconds = 0;
	SystemTimeToFileTime(&sysTime, &fileTime);
	aNote.ftLastUpdate = fileTime;

	wcscpy(aNote.szNote, sNote);
	if(!WriteFile(hFile, (LPVOID)&aNote, sizeof(NOTE), &dwWritten, NULL) ||
			dwWritten != sizeof(NOTE))
		return FALSE;
	m_bChanged = TRUE;
	return TRUE;
}

BOOL CListDB::UpdateNote(LPCTSTR sNoteTime, LPCWSTR sNote)
{
	FILETIME fileTime;

	StringToFileTime(&fileTime, sNoteTime);
	// find the note
	return UpdateNote(&fileTime, sNote);
}

// sets the file pointer to the start of the note with the given FILETIME 
// time stamp
BOOL CListDB::FindNote(FILETIME* ftToFind, NOTE* lpNote)
{
	NOTE aNote;
	DWORD dwRead;

	SetFilePointer(hFile, 0, NULL, FILE_BEGIN);

	while(ReadFile(hFile, (LPVOID)&aNote, sizeof(NOTE), &dwRead, NULL) &&
			dwRead == sizeof(NOTE))
	{
		if(CompareFileTime(&aNote.ftOriginal, ftToFind) == 0)
		{
			// move back to start of record
			SetFilePointer(hFile, (LONG)sizeof(NOTE) * -1, NULL, FILE_CURRENT);
			if(lpNote != NULL)
				memcpy(lpNote, &aNote, sizeof(NOTE));
			return TRUE;
		}
	}
	return FALSE;
}

// converts our string format to a FILETIME
void CListDB::StringToFileTime(FILETIME* fileTime, LPCTSTR sNoteTime)
{
	SYSTEMTIME sysTime;

#ifdef UNICODE
	sysTime.wDay = (WORD)_wtoi(sNoteTime);
	sysTime.wMonth = (WORD)_wtoi(&sNoteTime[3]);
	sysTime.wYear = (WORD)_wtoi(&sNoteTime[6]);
	sysTime.wHour = (WORD)_wtoi(&sNoteTime[11]);
	sysTime.wMinute = (WORD)_wtoi(&sNoteTime[14]);
	sysTime.wSecond = (WORD)_wtoi(&sNoteTime[17]);
#else
	sysTime.wDay = (WORD)atoi(sNoteTime);
	sysTime.wMonth = (WORD)atoi(&sNoteTime[3]);
	sysTime.wYear = (WORD)atoi(&sNoteTime[6]);
	sysTime.wHour = (WORD)atoi(&sNoteTime[11]);
	sysTime.wMinute = (WORD)atoi(&sNoteTime[14]);
	sysTime.wSecond = (WORD)atoi(&sNoteTime[17]);
#endif
	sysTime.wMilliseconds = 0;
	SystemTimeToFileTime(&sysTime, fileTime);
}

// Returns the first note in the .DAT file
BOOL CListDB::GetFirstNote(FILETIME* ftCreateTime, LPWSTR szNote, FILETIME* ftModifyTime)
{
	dwGetNextNote = 0;
	SetFilePointer(hFile, 0, NULL, FILE_BEGIN);
	return GetNextNote(ftCreateTime, szNote, ftModifyTime);	
}

// Returns the next note in the .DAT file
BOOL CListDB::GetNextNote(FILETIME* ftCreateTime, LPWSTR szNote, FILETIME* ftModifyTime)
{
	NOTE aNote;
	DWORD dwRead;

	SetFilePointer(hFile, dwGetNextNote, NULL, FILE_BEGIN);
	if(!ReadFile(hFile, (LPVOID)&aNote, sizeof(NOTE), &dwRead, NULL))
	{
		return FALSE;
	}
	if(dwRead != sizeof(NOTE))
	{
		return FALSE;
	}
	wcscpy(szNote, aNote.szNote);
	*ftCreateTime = aNote.ftOriginal;
	*ftModifyTime = aNote.ftLastUpdate;
	dwGetNextNote = SetFilePointer(hFile, 0, NULL, FILE_CURRENT);
	return TRUE;
}
