// 'Windows CE 3.0 Programming' Source Code Samples (Prentice Hall, 2000)
// Source Code Author: Nick Grattan (nick@softwarepaths.com)
// Version 1.00

// Component implmenentation

#include "stdafx.h"
#include "component.h"
#include "resource.h"
#include ".\cldesktop\ListDB.h"		// for access to data file

////////////////////////////////////////////////////////////////////////////////
// IReplStore implementation in CActiveSyncEng

///////////////////////////////////////////////////////////////////////////////////
// Initialize access to the object store

STDMETHODIMP CActiveSyncEg::Initialize(IReplNotify *pNotify, UINT uFlags )
{
	m_bInitialized = TRUE;
	return NOERROR;
}

// returns information about the current store
STDMETHODIMP CActiveSyncEg::GetStoreInfo(PSTOREINFO pStoreInfo )
{
	// Check correct version of StoreInfo structure
	if(pStoreInfo->cbStruct < sizeof(*pStoreInfo))
	{
		MessageBox(NULL, _T("GetStoreInfo - Invalid Arg"), NULL, 0);
		return E_INVALIDARG;
	}
	// we only support single threaded operation
	pStoreInfo->uFlags = SCF_SINGLE_THREAD;
	// Set store's progid and description
	strcpy(pStoreInfo->szProgId, g_szVerIndProgID);
	strcpy(pStoreInfo->szStoreDesc, g_szFriendlyName);
	// this is as far as we get if we're not Initialized
	if(!m_bInitialized)
	{
		return NOERROR;
	}
	// Create the store's unique identifier - Set the length of the store identifier
	pStoreInfo->cbStoreId = (strlen(g_szStoreFile) + 1) * sizeof(TCHAR);
	// ActiveSync calls GetStoreInfo twice. Once to get the size of the store id 
	// (when lpbStoreId is NULL), and a second time, providing a buffer pointed to
	// by lpbStoreId where the store id can be placed.
	if(pStoreInfo->lpbStoreId == NULL)
		return NOERROR;
	memcpy(pStoreInfo->lpbStoreId, g_szStoreFile, 
				(strlen(g_szStoreFile) + 1) * sizeof(TCHAR));
	return NOERROR;
}

// Called by ActiveSync to report status/errors in the sync. process
STDMETHODIMP CActiveSyncEg::ReportStatus(HREPLFLD hFld, HREPLITEM hItem, UINT uStatus, UINT uParam )
{
	// not implemented at present
	return NOERROR;
}

// Called to determine if two store ids refer to the same store, or a different store
// In our case there is only ever one store. Returns 0 if store ids represent the same
// store. 

STDMETHODIMP_(int) CActiveSyncEg::CompareStoreIDs(LPBYTE lpbID1, UINT cbID1, 
									LPBYTE lpbID2, UINT cbID2)
{
	if(cbID1 < cbID2)
		return -1;	// first store is smaller than the second store
	if(cbID1 > cbID2)
		return 1;	// first store is larger than the second store
	// now compare the store ids byte by byte.
	return memcmp(lpbID1, lpbID2, cbID1);
}

///////////////////////////////////////////////////////////////////////////////////
// Item related routines

// Determine if these HREPLITEMs are acutally the same. Use the ftCreated time to determine this.
STDMETHODIMP_(int) CActiveSyncEg::CompareItem(HREPLITEM hItem1, HREPLITEM hItem2 )
{

	LPREPLOBJECT lpRepObj1 = (LPREPLOBJECT)hItem1;
	LPREPLOBJECT lpRepObj2 = (LPREPLOBJECT)hItem2;

	int nRet = CompareFileTime(&lpRepObj1->ftCreated, &lpRepObj2->ftCreated);
	return nRet;
}

// Compare two HREPLITEMs and see if they represent a changed item. Use ftModified to determine this.
STDMETHODIMP_(BOOL) CActiveSyncEg::IsItemChanged(HREPLFLD hFld, 
												 HREPLITEM hItem, HREPLITEM hItemComp )
{
	LPREPLOBJECT lpRepObj1 = (LPREPLOBJECT)hItem;

	if(hItemComp != NULL)
	{
		LPREPLOBJECT lpRepObj2 = (LPREPLOBJECT)hItemComp;
		return CompareFileTime(&lpRepObj1->ftModified, &lpRepObj2->ftModified);
	}
	else
	{
		// need to compare this object with the one in the .DAT file
		NOTE aNote;
		if(m_ListDB.FindNote(&lpRepObj1->ftCreated, &aNote))
			return CompareFileTime(&lpRepObj1->ftModified, &aNote.ftLastUpdate);
		else
		{
			MessageBox(NULL, _T("Could not find record for IsItemChanged"), NULL, 0);
			return FALSE;
		}
	}
}

// Determine if this object should be replicated. All our objects should be replicated
STDMETHODIMP_(BOOL) CActiveSyncEg::IsItemReplicated(HREPLFLD hFld, HREPLITEM hItem )
{
	return TRUE;
}

// Called by ActiveSync to ensure that our timestamps etc are up to date
STDMETHODIMP_(void) CActiveSyncEg::UpdateItem(HREPLFLD hFld, HREPLITEM hItemDst, 
													HREPLITEM hItemSrc )
{
	LPREPLOBJECT lpRepDest = (LPREPLOBJECT)hItemDst;
	LPREPLOBJECT lpRepSrc = (LPREPLOBJECT)hItemSrc;
	if(hItemSrc != NULL)
	{
		*lpRepDest = *lpRepSrc;
	}
	else
	{
		// need to find the given record and update the modified flag
		NOTE aNote;
		if(m_ListDB.FindNote(&lpRepDest->ftCreated, &aNote))
		{
			lpRepDest->ftModified = aNote.ftLastUpdate;
		}
		else
			MessageBox(NULL, _T("Could not find record for UpdateItem"), NULL, 0);
	}
	return;
}

///////////////////////////////////////////////////////////////////////////////////
// Folder related routines

// We only support a single folder for our items (the .DAT file). The HREPFLFLD
// handle will represent the CListDB class (which provides access to the .DAT file)
STDMETHODIMP CActiveSyncEg::GetFolderInfo(LPSTR lpszObjType, 
							HREPLFLD *phFld, IUnknown ** ppObjHandler)
{
	LPREPLOBJECT pFolder = (LPREPLOBJECT) *phFld;
	if(pFolder == NULL)		// new folder required
	{
		pFolder = new REPLOBJECT;
	}
	pFolder->uType = RT_FOLDER;
	pFolder->fChanged = TRUE;
	*phFld = (HREPLFLD)pFolder;
	// CDataHandler member m_DataHandler implements IReplObjHandler
	*ppObjHandler = &m_DataHandler;
	return NOERROR;
}

// Called by ActiveSync to implement real time updates. 
// Return true as this is not implemented
STDMETHODIMP CActiveSyncEg::IsFolderChanged(HREPLFLD hFld, BOOL *pfChanged )
{
	*pfChanged = TRUE;
	return NOERROR;
}

///////////////////////////////////////////////////////////////////////////////////
// Enumeration of folders

// Returns an HREPLITEM structure for the first item in the .DAT file. The data in HREPITEM
// is the OriginalTime (the unique identifier) and ModifyTime (to determine if the item has
// changed);
STDMETHODIMP CActiveSyncEg:: FindFirstItem(HREPLFLD hFld,  HREPLITEM *phItem, BOOL *pfExist )
{
	WCHAR szNote[STRLEN_NOTE];
	FILETIME ftCreateTime, ftModifyTime;

	// attempt to get first record
	*pfExist = m_ListDB.GetFirstNote(&ftCreateTime, szNote, &ftModifyTime);
	if(!*pfExist)
		return NOERROR;
	// now make up the HREPLFLD
	LPREPLOBJECT lpRepl = new REPLOBJECT;
	lpRepl->uType = RT_ITEM;
	lpRepl->ftCreated = ftCreateTime;
	lpRepl->ftModified = ftModifyTime;
	*phItem = (HREPLITEM)lpRepl;		// set our pointer into HREPLITEM
	return NOERROR;
}

// Find the next item from the .DAT file
STDMETHODIMP CActiveSyncEg::FindNextItem(HREPLFLD hFld,  HREPLITEM *phItem, BOOL *pfExist )
{
	WCHAR szNote[STRLEN_NOTE];
	FILETIME ftCreateTime, ftModifyTime;

	// attempt to get first record
	*pfExist = m_ListDB.GetNextNote(&ftCreateTime, szNote, &ftModifyTime);
	if(!*pfExist)
	{
		return NOERROR;
	}
	// now make up the HREPLFLD
	LPREPLOBJECT lpRepl = new REPLOBJECT;
	lpRepl->uType = RT_ITEM;
	lpRepl->ftCreated = ftCreateTime;
	lpRepl->ftModified = ftModifyTime;
	*phItem = (HREPLITEM)lpRepl;		// set our pointer into HREPLITEM
	return NOERROR;
}

// Finished going through all records. Nothing to do in this case.

STDMETHODIMP CActiveSyncEg::FindItemClose(HREPLFLD hFld )
{
	return NOERROR;
}

///////////////////////////////////////////////////////////////////////////////////
// Object management routines

// Converts handle to object or folder to a stream of bytes. Called twice, first time
// to get the size of the stream and a second time with a lpb buffer large enough
// for the stream of bytes.

STDMETHODIMP_(UINT) CActiveSyncEg::ObjectToBytes(HREPLOBJ hObject, LPBYTE lpb )
{
	if(lpb != NULL)			// buffer has been created to requested size
		memcpy(lpb, (LPREPLOBJECT)hObject, sizeof(REPLOBJECT));
	return sizeof(REPLOBJECT);
}

// Converts an array of bytes to an HREPLOBJ. This can be a folder or an item buffer

STDMETHODIMP_(HREPLOBJ) CActiveSyncEg::BytesToObject(LPBYTE lpb, UINT cb )
{
	if(cb != sizeof(REPLOBJECT))
		MessageBox(NULL, _T("Not correct size in Bytes to object"), NULL,0);
	LPREPLOBJECT lpReplObject = new REPLOBJECT;
	// perform the copy
	memcpy(lpReplObject, lpb, cb);
	return (HREPLOBJ)lpReplObject;
}

STDMETHODIMP_(void) CActiveSyncEg::FreeObject(HREPLOBJ hObject )
{
	LPREPLOBJECT pItem = (LPREPLOBJECT)hObject;
	delete (LPREPLOBJECT) hObject;
}

// Copies HREPLOBJ from source to destination
STDMETHODIMP_(BOOL) CActiveSyncEg::CopyObject(HREPLOBJ hObjSrc, HREPLOBJ hObjDest)
{
	LPREPLOBJECT lpRepObjSrc = (LPREPLOBJECT)hObjSrc;
	LPREPLOBJECT lpRepObjDest = (LPREPLOBJECT)hObjDest;
	*lpRepObjDest = *lpRepObjSrc;
	return TRUE;
}

STDMETHODIMP CActiveSyncEg::IsValidObject(HREPLFLD hFld, HREPLITEM hItem, UINT uFlags )
{
	LPREPLOBJECT lpRepObj;

	if(hFld != NULL)
	{
		lpRepObj = (LPREPLOBJECT)hFld;
		if(lpRepObj->uType == RT_FOLDER)
			return NOERROR;
		else
			return RERR_CORRUPT;
	}

	if(hItem!= NULL)
	{
		lpRepObj = (LPREPLOBJECT)hItem;
		if(lpRepObj->uType != RT_ITEM)
			return RERR_CORRUPT;
		NOTE aNote;
		// attempt to find the item
		if(m_ListDB.FindNote(&lpRepObj->ftCreated, &aNote))
			return NOERROR;
		else
			return RERR_OBJECT_DELETED;
	}
	return NOERROR;
}

///////////////////////////////////////////////////////////////////////////////////
// UI related routines
STDMETHODIMP CActiveSyncEg::ActivateDialog(UINT uidDialog, HWND hwndParent, HREPLFLD hFld, IEnumReplItem *penumItem )
{
	MessageBox(NULL, _T("No settings for this ActiveSync provider"), NULL, 0);
	return NOERROR;
}

STDMETHODIMP CActiveSyncEg::GetObjTypeUIData(HREPLFLD hFld, POBJUIDATA pData )

{    
	// make sure we have the right version of OBJUIDATA
    if ( pData->cbStruct != sizeof( OBJUIDATA ) )        
	{
		MessageBox(NULL, _T("GetObjTypeUIData returns E_INVALIDARG"), NULL, 0);
		return E_INVALIDARG;
	}
    pData->hIconLarge = (HICON)LoadImage( g_hModule, 
				MAKEINTRESOURCE( IDI_ICON ), 
				IMAGE_ICON, 32, 32, 0 );
    pData->hIconSmall = (HICON)LoadImage( g_hModule, 
				MAKEINTRESOURCE( IDI_ICON ), 
				IMAGE_ICON, 16, 16, 0 );
    strcpy( pData->szName, g_szFriendlyName );
    strcpy( pData->szTypeText, _T("Note"));
    strcpy( pData->szPlTypeText, _T("Notes") );
    strcpy( pData->szSyncText, _T("\\ActiveSynNotes.dat")); 
	return NOERROR;
}

STDMETHODIMP CActiveSyncEg::GetConflictInfo(PCONFINFO pConfInfo )
{
	LPREPLOBJECT lpRepObjLocal = (LPREPLOBJECT)pConfInfo->hLocalItem;
	LPREPLOBJECT lpRepObjRemote = (LPREPLOBJECT)pConfInfo->hRemoteItem;

	if(CompareFileTime(&lpRepObjLocal->ftModified, &lpRepObjRemote->ftModified) == 0)
	{
		MessageBox(NULL, _T("Objects are the same"), NULL, MB_OK);
		return RERR_IGNORE;	// objects are the same
	}
	NOTE aNote;
	if(m_ListDB.FindNote(&lpRepObjLocal->ftCreated, &aNote))
	{
		strcpy(pConfInfo->szLocalName, _T("Note"));
		wcstombs(pConfInfo->szLocalDesc, aNote.szNote, 512);
		// Find information on temp. object copied from device.
		strcpy(pConfInfo->szRemoteName, _T("Note"));
		FILETIME ft;
		memset(&ft, 0, sizeof(ft));
		if(m_ListDB.FindNote(&ft, &aNote))
		{
			wcstombs(pConfInfo->szRemoteDesc, aNote.szNote, 512);
		}
		else
			strcpy(pConfInfo->szRemoteDesc, _T("Unknown Note"));
	}
	return NOERROR;
}

STDMETHODIMP CActiveSyncEg::RemoveDuplicates(LPSTR lpszObjType, UINT uFlags )
{
	return E_NOTIMPL;	// we don't implemement this
}

