// 'Windows CE 3.0 Programming' Source Code Samples (Prentice Hall, 2000)
// Source Code Author: Nick Grattan (nick@softwarepaths.com)
// Version 1.00

// Chapter 9: Serial Communications. Sample Code

#include "stdafx.h"
#include "examples.h"

// *** Listing 9.1
//
// Open COM1 serial port.
// 

void ReportCommError(LPTSTR lpszMessage);
DWORD WINAPI CommReadThreadFunc(LPVOID lpParam);
BOOL SendText(HWND hWnd);

HANDLE hCommPort = INVALID_HANDLE_VALUE;

void Listing9_1()
{
	hCommPort = CreateFile (_T("COM1:"), // Port Name (Unicode compatible)
			GENERIC_READ | GENERIC_WRITE, // Open for Read-Write
            0,             // COM port cannot be shared
            NULL,          // Always NULL for Windows CE
            OPEN_EXISTING, // For communication resource
            0,             // Non-overlapped operation only
            NULL);         // Always NULL for Windows CE
	if(hCommPort == INVALID_HANDLE_VALUE)
	{
		ReportCommError(_T("Opening Comms Port."));
		return;
	}
	// set the timeouts to specify the behavor of reads and writes.
	COMMTIMEOUTS ct;
	ct.ReadIntervalTimeout = MAXDWORD; 
	ct.ReadTotalTimeoutMultiplier = 0; 
    ct.ReadTotalTimeoutConstant = 0; 
    ct.WriteTotalTimeoutMultiplier = 10; 
    ct.WriteTotalTimeoutConstant = 1000; 
	if(!SetCommTimeouts(hCommPort, &ct))
	{
		ReportCommError(_T("Setting comm. timeouts."));
		Listing9_2(); // close comm port
		return;
	}
	// get the current communications parameters, and configure baud rate
	DCB dcb;
	dcb.DCBlength = sizeof(DCB);
	if(!GetCommState(hCommPort, &dcb))
	{
		ReportCommError(_T("Getting Comms. State."));
		Listing9_2(); // close comm port
		return;
	}
	dcb.BaudRate = CBR_19200;		// set baud rate to 19,200
	dcb.fOutxCtsFlow	= TRUE;
	dcb.fRtsControl     = RTS_CONTROL_HANDSHAKE;
	dcb.fDtrControl     = DTR_CONTROL_ENABLE;
	dcb.fOutxDsrFlow    = FALSE;
	dcb.fOutX           = FALSE; // no XON/XOFF control
	dcb.fInX            = FALSE;
	dcb.ByteSize        = 8;
	dcb.Parity          = NOPARITY;
	dcb.StopBits        = ONESTOPBIT;
	if(!SetCommState(hCommPort, &dcb))
	{
		ReportCommError(_T("Setting Comms. State."));
		Listing9_2(); // close comm port
		return;
	}
	// now need to create the thread that will be reading the comms port
	HANDLE hCommReadThread = CreateThread(NULL, 0, CommReadThreadFunc, NULL, 0, NULL);
	if(hCommReadThread == NULL)	
	{
		ReportCommError(_T("Creating Thread."));
		Listing9_2(); // close comm port
		return;
	}
	else
	{
		CloseHandle(hCommReadThread);
	}
}

// reports any errors encountered in these communications functions.

void ReportCommError(LPTSTR lpszMessage)
{
	TCHAR szBuffer[200];
	wsprintf(szBuffer, _T("Communications Error %d \r\n%s"), 
				GetLastError(),
				lpszMessage);
	cout << szBuffer << endl;
}

// Thread function to read communications port. This thread is responsible
// for reading any incoming text and adding it to the text box.

DWORD WINAPI CommReadThreadFunc(LPVOID lpParam)
{
	DWORD dwBytesRead;
	DWORD fdwCommMask;
	TCHAR szwcsBuffer[1000];
	char szBuffer[1000];

	SetCommMask (hCommPort, EV_RXCHAR);
	szBuffer[0] = '\0';
	while(hCommPort != INVALID_HANDLE_VALUE)
	{
		if(!WaitCommEvent (hCommPort, &fdwCommMask, 0))
		{
			// has the comms port been closed?
			if(GetLastError() != ERROR_INVALID_HANDLE)
				ReportCommError(_T("WaitCommEvent."));
			return 0;		// terminate thread on first error
		}
		SetCommMask (hCommPort, EV_RXCHAR);
		// Read ANSI characters
		if(!ReadFile(hCommPort,		// where to read from (the open comm port)
				 szBuffer,			// where to put the character
				 999,				// number of bytes to read
				 &dwBytesRead,		// how many bytes were actually read
				 NULL))				// overlapped I/O not supported
		{
			ReportCommError(_T("Reading comms port."));
			return 0;		// terminate thread on first error
		}
		szBuffer[dwBytesRead] = '\0';	// NULL terminate
		// now convert to Unicode
		mbstowcs(szwcsBuffer, szBuffer, 1000);
		cout << szwcsBuffer;
	}
	return 0;
}

// *** Listing 9.2
//
// Closes COM1 serial port.
// 

void Listing9_2()
{
	if(hCommPort != INVALID_HANDLE_VALUE)
	{
		CloseHandle(hCommPort);
		hCommPort = INVALID_HANDLE_VALUE;
		cout << _T("Com. port closed") << endl;
	}
	else
		cout << _T("Com. port was not open") << endl;
}

// *** Listing 9.3
//
// Write text to serial port.
// 

#define BUFF_SIZE 200

void Listing9_3()
{
	DWORD dwBytesToWrite;
	DWORD dwBytesWritten;

	TCHAR szwcsBuffer[BUFF_SIZE];
	char szBuffer[BUFF_SIZE];

	if(!GetTextResponse(_T("Text to send:"), szwcsBuffer, 200))
		return;
	// convert to ANSI character set
	dwBytesToWrite = wcstombs(szBuffer, szwcsBuffer, BUFF_SIZE);
	// append a carrage return/line feed pair
	szBuffer[dwBytesToWrite++] = '\r';
	szBuffer[dwBytesToWrite++] = '\n';
	if(!WriteFile(hCommPort,	// where to write to (the open comm port)
			szBuffer,			// what to write
			dwBytesToWrite,		// number of bytes to be written to port
			&dwBytesWritten,	// number of bytes that were actually written
			NULL))				// overlapped I/O not supported			
	{
		ReportCommError(_T("Sending text."));
		return;
	}
}

// *** Listing 9.4
//
// Receiving NMEA output from a GPS device.
// 

DWORD WINAPI GPSReadThreadFunc(LPVOID);
HANDLE hGPSPort = INVALID_HANDLE_VALUE;
void ParseRMC(LPTSTR szSentence);

void Listing9_4()
{
	hGPSPort = CreateFile (_T("COM1:"), // Port Name (Unicode compatible)
			GENERIC_READ | GENERIC_WRITE, // Open for Read-Write
            0,             // COM port cannot be shared
            NULL,          // Always NULL for Windows CE
            OPEN_EXISTING, // For communication resource
            0,             // Non-overlapped operation only
            NULL);         // Always NULL for Windows CE
	if(hGPSPort == INVALID_HANDLE_VALUE)
	{
		ReportCommError(_T("Opening Comms Port."));
		return;
	}
	// set the timeouts to specify the behavior of reads and writes.
	COMMTIMEOUTS ct;
	ct.ReadIntervalTimeout = 1000; 
	ct.ReadTotalTimeoutMultiplier = 0; 
    ct.ReadTotalTimeoutConstant = 0; 
    ct.WriteTotalTimeoutMultiplier = 10; 
    ct.WriteTotalTimeoutConstant = 1000; 
	if(!SetCommTimeouts(hGPSPort, &ct))
	{
		ReportCommError(_T("Setting comm. timeouts."));
		Listing9_2(); // close comm port
		return;
	}
	// get the current communications parameters, and configure baud rate
	DCB dcb;
	dcb.DCBlength = sizeof(DCB);
	if(!GetCommState(hGPSPort, &dcb))
	{
		ReportCommError(_T("Getting Comms. State."));
		Listing9_2(); // close comm port
		return;
	}
	dcb.BaudRate = CBR_9600;		// set baud rate to 9600
	dcb.fOutxCtsFlow	= FALSE;
	dcb.fRtsControl     = RTS_CONTROL_DISABLE;
	dcb.fDtrControl     = DTR_CONTROL_DISABLE;
	dcb.fOutxDsrFlow    = FALSE;
	dcb.fOutX           = TRUE; // XON/XOFF control
	dcb.fInX            = TRUE;
	dcb.ByteSize        = 8;
	dcb.Parity          = NOPARITY;
	dcb.StopBits        = ONESTOPBIT;
	if(!SetCommState(hGPSPort, &dcb))
	{
		ReportCommError(_T("Setting Comms. State."));
		Listing9_2(); // close comm port
		return;
	}
	// now need to create the thread that will be reading the comms port
	HANDLE hCommReadThread = CreateThread(NULL, 0, 
			GPSReadThreadFunc, NULL, 0, NULL);
	if(hCommReadThread == NULL)	
	{
		ReportCommError(_T("Creating Thread."));
		Listing9_2(); // close comm port
		return;
	}
	else
	{
		CloseHandle(hCommReadThread);
	}
}

// Thread function reads NMEA output from GPS device
DWORD WINAPI GPSReadThreadFunc(LPVOID)
{
	DWORD dwBytesRead;
	char szSentence[1000], c;
	TCHAR szwcsSentence[1000];
	int nc = 0;

	SetThreadPriority(GetCurrentThread(), THREAD_PRIORITY_BELOW_NORMAL);
	while(hGPSPort != INVALID_HANDLE_VALUE)
	{
		if(!ReadFile(hGPSPort, &c, 1, &dwBytesRead, NULL))
		{
			ReportCommError(_T("Reading comms port."));
			return 0;		// terminate thread on first error
		}
		if(dwBytesRead == 1)
		{
			if(c == '\n') // LF marks end of sentance
			{
				szSentence[nc - 1] = '\0';// remove trailing CR
				nc = 0;			// ready to read next sentence
				if(strlen(szSentence) < 6)
					cout << _T("Corrupt sentence") << endl;
				else if(szSentence[0] != '$')
					cout << _T("Sentence did not start with $") << endl;
				else 
				{
					// have a sentence. convert to Unicode
					mbstowcs(szwcsSentence, szSentence, 1000);
					// find sentence ID
					if(wcsncmp(&szwcsSentence[3], _T("RMC"), 3) == 0)
						ParseRMC(szwcsSentence);
				}
			}
			else
				szSentence[nc++] = c;
		}
	}
	return 0;
}

// returns the next token from the sentence.
LPTSTR GetNextToken(LPTSTR lpSentence, LPTSTR lpToken)
{
	lpToken[0] = '\0';
	if(lpSentence == NULL) // empty sentence
		return NULL;
	if(lpSentence[0] == '\0') // end of sentence
		return NULL;
	if(lpSentence[0] == ',') // empty token
		return lpSentence + 1;
	while(*lpSentence != ',' && *lpSentence != '\0' && *lpSentence != '*')
	{
		*lpToken = *lpSentence;
		lpToken++;
		lpSentence++;
	}
	lpSentence++;  // skip over comma
	*lpToken = '\0';
	return lpSentence;
}

// Parses a RMC sentence which has the format:
// RMC,225446,A,4916.45,N,12311.12,W,000.5,054.7,191194,020.3,E*68

void ParseRMC(LPTSTR szSentence)
{
	TCHAR szToken[20];
	DWORD dwCheckSum = 0, dwSentenceCheckSum;

	cout.CLS();
	// Calculate the checksum. Exclude $ and work up to *
	for(UINT i = 1; i < wcslen(szSentence) && szSentence[i] != '*'; i++)
		dwCheckSum ^= szSentence[i];

	// lpNextTok points at ID $GPRMS, ignore this
	szSentence = GetNextToken(szSentence, szToken);
	// Time of Fix, convert to Unicode
	szSentence = GetNextToken(szSentence, szToken);
	cout << _T("Time (UTC hhmmss):") << szToken << endl;
	// Navigation receiver (GPS) warning
	szSentence = GetNextToken(szSentence, szToken);
	if(szToken[0] == 'A')
		cout << _T("Receiving OK") << endl;
	else
		cout << _T("Suspect signal") << endl;
	// Latitude
	szSentence = GetNextToken(szSentence, szToken);
	cout << _T("Latitude (ddmm.ss): ") << szToken;
	// Latitude N or S
	szSentence = GetNextToken(szSentence, szToken);
	cout << szToken << endl;
	// Longitude
	szSentence = GetNextToken(szSentence, szToken);
	cout << _T("Longitude dddmm.ss: ") << szToken;
	// Longitude W or E
	szSentence = GetNextToken(szSentence, szToken);
	cout << szToken << endl;
	// Speed in Knots
	szSentence = GetNextToken(szSentence, szToken);
	cout << _T("Speed (Knots): ") << szToken << endl;
	// Course made good
	szSentence = GetNextToken(szSentence, szToken);
	cout << _T("Course made good: ") << szToken << _T(" deg") << endl;
	// Date
	szSentence = GetNextToken(szSentence, szToken);
	cout << _T("Date (ddmmyyyy):") <<  szToken << endl;
	// Magnetic Variation
	szSentence = GetNextToken(szSentence, szToken);
	cout << _T("Mag. Var (Deg): ") << szToken;
	// Magnetic Variation W or E
	szSentence = GetNextToken(szSentence, szToken);
	cout << szToken << endl;
	// do the check sum
	szSentence = GetNextToken(szSentence, szToken);
	LPTSTR lpEnd;
	dwSentenceCheckSum = wcstoul(szToken, &lpEnd, 16);
	if(dwCheckSum != dwSentenceCheckSum)
		cout << _T("Error in checksum");
	else
		cout << _T("Checksum OK");
}

// *** Listing 9.5
//
// Closes COM1 serial port used for reading GPS device.
// 

void Listing9_5()
{
	if(hGPSPort != INVALID_HANDLE_VALUE)
	{
		CloseHandle(hGPSPort);
		hGPSPort = INVALID_HANDLE_VALUE;
		cout << _T("GPS port closed") << endl;
	}
	else
		cout << _T("GPS port was not open") << endl;
}


