@chapter(Tasks)

@b[There is a new paragraph describing stack overflows and the use of
guardwords.]

This chapter presents the workings of the tasking package and the interface
specifications for it. The tasking package is actually an impressively small
piece of code which was originally written for TCP/Telnet on a Version 6
Unix system by Larry Allen for use in TCP, and then ported to the PC by me,
modified by Lou Konopelski, and then changed again by me.

Each task has a stack and a task control block. The task control block is
actually at the top of the stack. The pointer which @i{tk@us2[]fork()}
returns is a pointer to this structure. The stack is allocated by simply
calling @i{calloc()} with the desired stack size. If really weird things
seem to be happening to a program, try increasing the stack size of some
tasks. Also, the stack size that's passed on to @i{tk@us2()init()} is
meaningless; the routine which calls @i{tk@us2()init()} still runs on the
system stack. Sometimes @i{printf()} will blow the stack since it uses a
recursive routine to print numbers.

@section(Include Files and New Types)

To use the tasking package, you should include the file @i[<task.h>].
Likewise, include @i[<timer.h>] to use the timer package. Neither of these
files includes the other one. @i[<ip.h>] @i(does) include @i[<task.h>]
(after immense indirection). Also, timers need the @i[<q.h>] include file,
too. This one comes for free with @i[<ip.h>].

The file @i[<task.h>] gives us several new types. There is a task control
block associated with every task; it is a structure which will be looked at
later, and the new type @i(task) is simply the same as that structure. The
type @i(Task) is a pointer to a task structure. We also gain @i(event) and
@i(stack). @i(stack) is an int; an array of them make space for a task stack.
@i(event) is intended as a binary semaphore to allow communication between
tasks, and to allow a task to determine why it was woken up.

It is more tasteful to use the @i(task) type than the @i(Task) type and the
latter will probably vanish before this manual is done.

To link in tasking, specify @i[-ltask] on the linker command line.

@section(Task Control Blocks)

@begin(figure)
@begin(verbatim)
struct	task	{
	stack	*tk@us()fp;		/* task's current frame ptr */
	char	*tk@us()name;	/* the task's name */
	int	ev@us()flg;	/* flag set if task is scheduled */
	struct	task *tk@us()nxt;	/* pointer to next task */
	unsigned tk@us()count;		/* number of wakeups */
	unsigned *tk@us()guard;		/* pointer to lowest guardword */
	unsigned tk@us()size;		/* stack size */
	stack	tk@us()stack[1];	/* top of task's stack */
	};

@end(verbatim)
@caption(The task control block structure)
@tag(taskstruct)
@end(figure)

The task control block structure is presented in figure @ref(taskstruct). A
pointer to a task control block is like a process id in this system. Task
control blocks form a circular list, chained by the @i(tk@us2[]nxt) member
of the structure. The scheduler is a round robin scheduler; it simply loops
through the circular list of tasks until it finds one which is runnable
(that is, a task whose @i(ev@us2[]flg) is TRUE) and then switches to that
task. A context switch merely consists of saving a small amount of state on
the stack and calling the routine @i[tk@us2()swtch()] (which is called by
@i[tk@us2()block()] which you'll read about later). When the task later
runs again, it is by that call to @i[tk@us2()swtch()] returning.

When a task stack is allocated, it is filled with guardwords. The
@i[tk@us2()guard] field points to the lowest word on the stack. On every
context switch, this word is checked to verify that it still contains a
guardword. If it doesn't, the tasking package decides that the task had a
stack overflow, prints an informational message and exits. Guardwords are
also used by the @i[tk@us2()stats()] function to determine how much of the
stack has been used.

The actual task stack is directly above the control block in memory; storage
for both of them is allocated at the same time. The reason the structure
declares one entry in the stack is because the C compiler we used did not
allow zero length arrays at the end of structures.

Finally, there is a global variable, @i[tk@us2()cur], which is a pointer to
the task control block of the task which is currently running.

@section[Tastefulness in Usage]

There are certain conventions which should be followed when using tasks. The
most important one has to do with how to use or treat task wakeups.
Generally, task wakeups should be treated only as hints. When a task runs,
it should try to discover why it was woken up and do the right thing. This
includes being able to cope with a seemingly reasonless wakeup. Since tasks
are forked as runnable, this is the correct structure for a task:

@begin[verbatim]
task@us()definition() {
	local variables;

	while(1) {
		if(no@us()work) {
			tk@us()block();
			continue;
			}

		do@us()the@us()right@us()thing;
		}
	}
@end[verbatim]

@section(User Tasking Functions)

Descriptions of functions in the tasking system follow.

@subsection<@i[tk@us2()fork()]>

@begin(verbatim)
task *tk@us()fork(prev@us()task, entry@us()point, stack@us()size, name, arg)
	task *prev@us()task;
	int (*entry@us()point)();
	unsigned stack@us()size;
	char *name;
	unsigned arg;

@end(verbatim)

This call creates a new task, setting up a task control block and a stack
for it. It inserts it in the linked list of tasks directly after the task
specified by @i(prev@us2[]task). This replaces task priorities but is
unfortunately slightly brain damaged. The idea was that you could control
the order in which tasks ran if you specified their positions in the
circular list, thereby cutting down on useless task wakeups. This works
all right in a small system with two or three tasks, done by one person, but
falls to pieces in a larger system with such things as the timer task and
tasks handling user input, etc. Generally it's safe to specify
@i(tk@us2()cur) as @i(prev@us2()task). A problem with replacing this system
with something more flexible and/or less damaged is that the current system
is very efficient and can do fast task schedulings and context switches. A
larger system would probably have to deal with more issues of atomicity and
would be less efficient.

Note that tasks are not really forked in the Unix sense of forking, but the
word stuck.

@i(entry@us2()point) is the pointer to the function which is this task. That
function must never return. @i[tk@us2()fork()] builds a stack frame for this
function so that the first time the task runs, the function appears to be
called. The stack frame is built by calling the function @i[tk@us2()frame()]
with the argument @i[arg]. It can store its state on the stack, have local
variables, and in general, do functionish things, except that it must never
return. At the top of the stack frame is the address of a routine called
@i[@us2()cdump()], which is a "simulation of the Unix core dump facility".
If you ever get a message about core dumps and horrible errors and having a
nice day, don't worry (and don't look for a core dump, it's only a
simulation), it just means that a task returned@footnote[Actually, this
message changed in the new release but it's still historical fun. The new
message says that a task returned and tries to print the task queue.]

@begin(group)
This is an example of a task which will print a message and then block
forever. Note that a task is set to runnable when forked, so it does not need
to be woken up after it is forked.
@begin(verbatim)

task1() {

	printf("Hello world.\n");
	printf("I'm a task, and I've just run.\n");

	while(1) tk@us()block();
	}

@end(verbatim)
@end(group)

It is generally good form to enclose the main body of the task in a
@i(while) or @i(for) loop.

The @i[stack@us2()size] parameter indicates the number of byte of stack the
task wants. Remember, this stack is used by the tasks local variables, by
functions called by the task, and as the interrupt stack for any interrupts
which occur while that task is running. 800 to 1000 bytes seem to be good
minimum sizes. This size does not include the size of the task control
block; @i[tk@us2()fork()] adds this in itself.

Finally, the @i(name) parameter is a string which has a textual name for the
task and is sometimes useful for debugging. There is a global variable called
@i(TDEBUG) which is normally set to FALSE. When it is TRUE, every time a
task runs or blocks, a message is printed on the display saying what it did
and what its name was. Of course, this makes it hard to see anything else
which might be going on, like a telnet, but it can be useful to see what task
is crashing the program, or what task runs when, and things of this nature.

Interrupt handlers should never fork tasks.

@subsection<@i[tk@us2()init()]>

@begin(verbatim)
task *tk@us()init(stack@us()size)
	unsigned stack@us()size;

@end(verbatim)

@b[The @i[stack@us()size] parameter is now very important and should be
carefully selected. In older systems it was ignored.]

This routine initializes the tasking system. It builds a task control block
for the "currently running" task, gives it a name ("Main") and leaves it
running on the system stack. The @i[stack@us2()size] parameter is now used
and should be the maximum number of bytes you expect to be used on the main
stack. This function should be called before any other routine in the
tasking system, or in the timer system. It returns a pointer to the task
control block that it built.

@subsection<@i[tk@us2()block()]>

@begin(verbatim)
tk@us()block()

@end(verbatim)

@i[tk@us2()block()] takes no arguments and returns no value. It blocks the
currently running task. Since the tasking system is non-preemptive, this is
the only way for another task to gain control of the processor. This routine
returns the next time the task which blocked runs.

@i[tk@us2()block()] contains the heart of the scheduler. It basically runs
through the circular list of tasks until it finds a runnable task and then
does a context switch to that task, which then sees its last call to
@i[tk@us2()block()] return. @i[tk@us2()block()] calls @i[tk@us2()swtch()] to
perform the actual context switch.

Interrupt handlers should never call this routine.

@subsection<@i[tk@us2()exit()]>

@begin[verbatim]

tk@us()exit()

@end[verbatim]

This function causes the current task to die. When another task becomes
runnable, this task's task control block and stack will be deallocated. This
routine should not be called from interrupt level and no further references
should be made to this task's task control block after this call is made.
@i[tk@us2()exit()] never returns.

@subsection<@i[tk@us2()wake()]>

@begin(verbatim)

tk@us()wake(tk)
	task *tk;

@end(verbatim)

This routine is the counterpart to @i[tk@us2()block()]. It merely sets the
"event flag" in the task control block which @i(tk) points to to TRUE so
that the next time the scheduler runs, this task will be considered runnable.
This is such a simple routine that it is actually a macro. It can also be
called from interrupt handlers since the code which is generated is atomic
(a single @i(mov) instruction suffices). An example of a usage of this
routine is by the ethernet interrupt handler, which, when a good packet is
received, wakes up the ethernet task and enqueues the packet on the ethernet
received packet queue.

@subsection<@i[tk@us2()yield()]>

@begin(verbatim)

tk@us()yield()

@end(verbatim)

@i[tk@us2()yield()] performs a @i[tk@us2()wake(tk@us2()cur)] and a
@i[tk@us2()block()]. It essentially yields the processor to let other tasks
run, but wakes up the current task before yielding so that it regains
control of the processor after an unspecified amount of time. This is also a
macro, and also should never be called from interrupt level.

@subsection<@i[tk@us2()kill()]>

@begin[verbatim]

tk@us()kill(t)
	task *t;

@end[verbatim]

This function kills a task. The task is immediately removed from the list of
tasks and its stack is deallocated.

@subsection<@i[tk@us2()sleep()]>

@begin[verbatim]

tk@us()sleep(t)
	task *t;

@end[verbatim]

This macro puts a task to sleep. It forgets about any wakeups waiting. If
the task is the current task, it @i[does not] block the task.

@subsection<@i[tk@us2()stats()]>

@begin[verbatim]

tk@us()stats()

@end[verbatim]

This function prints statistics about all tasks in the list. It prints the
name of each task, the allocated stack size and the number of bytes of stack
which have actually been used by the task.

@section(Internal Tasking Functions)

@begin(b)
The following functions are internal to the tasking system only and should
not be called from outside of it. They are only included here for
thoroughness.
@end(b)

@subsection<@i[tk@us2()frame()]>

@begin(verbatim)

tk@us()frame(tk, tos, entry)
	task *tk;
	unsigned *tos;
	int (*entry)();

@end(verbatim)

This routine builds the actual stack frame for a task. It is the first
assembly language routine in the tasking system. It builds the frame for the
task whose control block is pointed to by @i(tk). @i(tos) is the highest
location in this task's stack. @i(entry) is the entry point of the task.
A guardword pointing at the @i[@us2()cdump()] routine is placed on the top
of the stack in case the task attempts to return. The task is not set to
runnable by @i[tk@us2()frame()]; this must be done by the caller if desired.

@subsection<@i[tk@us2()swtch()]>

@begin(verbatim)

tk@us()swtch(tk)
	task *tk;

@end(verbatim)

This routine performs the actual context switch to the task whose control
block is pointed to by @i(tk). When the current task runs again, the call
that was made to @i[tk@us2()swtch()] will return. This is the second
assembly language routine in the tasking system. The only routines that
should need to be rewritten for the tasking system to be ported to another
machine are this one, @i[tk@us2()frame()], and the stack filling routines
below.

@subsection<@i[@us2()stk@us2()sys@us2()fill()]>

@begin[verbatim]

@us()stk@us()sys@us()fill(number)
	unsigned number;

@end[verbatim]

This function fills the system (main) stack with @i[number] guardwords.

@subsection<@i[@us2()stk@us2()fill()]>

@begin[verbatim]

@us()stk@us()fill(number, stack)
	unsigned number;
	unsigned *stack;

@end[verbatim]

This function fills the stack at address @i[stack] with @i[number]
guardwords.

