#ifndef __UCGEMATRIX_H__
#define __UCGEMATRIX_H__

/*
 * Declarations for unsigned character general matrices.
 *
 * $Id: ucgemat.h,v 3.4.2.1 90/10/22 15:52:34 keffer Rel $
 *
 ****************************************************************************
 *
 * Rogue Wave 
 * P.O. Box 2328
 * Corvallis, OR 97339
 *
 * Copyright (C) 1989. This software is subject to copyright protection under 
 * the laws of the United States and other countries.
 *
 ***************************************************************************
 *
 */

/*
 * This class is derived from class UCharVec.  Data is stored
 * in column major order.
 *
 * Defining the preprocessor directive "BOUNDS_CHECK" will invoke
 * bounds checking.
 *
 */

#include "ucvec.h"
class IGEMatrix;
class UCGEMatrix;

/*
 * The UCGEPick class allows selected elements of the matrix to be addressed.
 * There are no public constructors.
 */

class UCGEPick {
private:
  const UCGEMatrix*	M;
  const IntVec*		rowPick;
  const IntVec*		colPick;
  UCGEPick(const UCGEMatrix* m, const IntVec* xrow, const IntVec* xcol);
  UCGEPick(const UCGEPick& p)	{M=p.M; rowPick=p.rowPick; colPick=p.colPick;}
  friend		UCGEMatrix;
protected:
  void			assertRowCol(unsigned, unsigned) Const;
  void			assertElements(unsigned, const IntVec&) Const;
  void			assertColRange(unsigned) Const;
  void			assertRowRange(unsigned) Const;
public:
  void			operator=(const UCGEMatrix&);
  void			operator=(const UCGEPick&);
  void			operator=(UChar);

  UChar&		operator()(int i, int j) Const;
  unsigned		rows() Const { return rowPick->length(); }
  unsigned		cols() Const { return colPick->length(); }
};

class UCGEMatrix : private UCharVec {
  unsigned ncols;	/* Number of columns*/
  unsigned nrows;	/* Number of rows*/
protected:
  void 			assertColRange(unsigned) Const;
  void 			assertRowRange(unsigned) Const;
  void			assertRowCol(unsigned, unsigned) Const;
  void			assertLength(const UCharVec&) Const;
  void			assertSquare() Const;
  void			assertProduct(const UCGEMatrix&) Const;
  void			assertProduct(const UCharVec&) Const;
public:
  UCGEMatrix();
  UCGEMatrix(unsigned rows, unsigned cols);
  UCGEMatrix(unsigned rows, unsigned cols, UChar initval);
  UCGEMatrix(const UChar* dat, unsigned, unsigned);  /* Copy of dat will be made*/
  UCGEMatrix(const UCharVec& v, unsigned, unsigned); /* Reference to v will be made*/
  UCGEMatrix(const UCGEMatrix& m);	   /* Reference to m will be made*/
  UCGEMatrix(const UCGEPick& p);

  operator		IGEMatrix()    Const;  /* Convert to IGEMatrix*/

  UChar*		data() Const	{return UCharVec::data();}
  unsigned		cols() Const;
  unsigned		rows() Const;

  UCGEMatrix&		reference(const UCGEMatrix& m); /* Reference self to m*/
  UCGEMatrix		deepCopy() Const;	/* copy of self with distinct instance variables */
  UCGEMatrix		copy() Const	{return deepCopy();} /* Synonym for deepCopy()*/
  void			deepenShallowCopy();	/* Guarantee that references==1:*/
  void			resize(unsigned,unsigned);

  void			scanFrom(istream& s);	/* Format 2 x 2 [ 1 3 2 4 ]*/
  void			printOn(ostream& s) Const;
  void			readFrom(istream&);
  void			storeOn(ostream&) Const;
  void			readFrom(fileDescTy&);
  void			storeOn(fileDescTy&) Const;
  void			readFrom(RWFile*); /* Internal binary formatting*/
  void			storeOn(RWFile*) Const;
  unsigned		binaryStoreSize() Const;

  UCharVec		operator[](unsigned j) Const;	/* Return a col as a slice*/
  UCharVec		col(unsigned j) Const;		/* Return a col as a slice*/
  UCharVec		row(unsigned i) Const;		/* Return a row as a slice*/
  UCharVec		diagonal(int idiag=0) Const;	/* Return a diagonal as a slice*/
  inline UChar&		operator()(int i, int j) Const; /* Subscripting*/
  inline UChar&		sub(int, int) Const;		/* Assumes stride==1; use with care*/
  UCGEPick		operator()(const IntVec&, const IntVec&) Const; /* Picking*/

/* Math functions*/
  UCGEMatrix		product(const UCGEMatrix&) Const; /* Inner product*/
  UCharVec		product(const UCharVec&) Const;

/* Assignment operators --- self must be same size as m*/
  UCGEMatrix&		operator=(const UCGEMatrix& m);
  UCGEMatrix&		operator=(const UCGEPick& p);
  UCGEMatrix&		operator=(UChar);
  UCGEMatrix&		operator+=(const UCGEMatrix& m);
  UCGEMatrix&		operator+=(UChar);
  UCGEMatrix&		operator-=(const UCGEMatrix& m);
  UCGEMatrix&		operator-=(UChar);
  UCGEMatrix&		operator*=(const UCGEMatrix& m);
  UCGEMatrix&		operator*=(UChar);
  UCGEMatrix&		operator/=(const UCGEMatrix& m);
  UCGEMatrix&		operator/=(UChar);

  /* Boolean operators:*/
  RWBoolean		operator==(const UCGEMatrix&) Const;
  RWBoolean		operator!=(const UCGEMatrix&) Const;

/* Increment/decrement operators*/
  UCGEMatrix&		operator++();

  UCGEMatrix&		operator--();

/* Friendly arithmetic operators.*/
  friend UCGEMatrix	operator-(const UCGEMatrix&);	/* Unary minus*/
#ifndef NO_UNARY_PLUS
  friend UCGEMatrix	operator+(const UCGEMatrix&);	/* Unary plus*/
#endif
  friend UCGEMatrix	operator*(const UCGEMatrix&, const UCGEMatrix&);
  friend UCGEMatrix	operator/(const UCGEMatrix&, const UCGEMatrix&);
  friend UCGEMatrix	operator+(const UCGEMatrix&, const UCGEMatrix&);
  friend UCGEMatrix	operator-(const UCGEMatrix&, const UCGEMatrix&);
  friend UCGEMatrix	operator*(const UCGEMatrix&, UChar);
  friend UCGEMatrix	operator*(UChar, const UCGEMatrix&);
  friend UCGEMatrix	operator/(const UCGEMatrix&, UChar);
  friend UCGEMatrix	operator/(UChar, const UCGEMatrix&);
  friend UCGEMatrix	operator+(const UCGEMatrix&, UChar);
  friend UCGEMatrix	operator+(UChar, const UCGEMatrix&);
  friend UCGEMatrix	operator-(const UCGEMatrix&, UChar);
  friend UCGEMatrix	operator-(UChar, const UCGEMatrix&);

  friend UCGEMatrix	abs(const UCGEMatrix&);
  friend UChar		dot(const UCGEMatrix&,const UCGEMatrix&);
  friend UChar		prod(const UCGEMatrix&);
  friend UChar		sum(const UCGEMatrix&);
};

/* Other (related) declarations:*/
istream&		operator>>(istream& s, UCGEMatrix& m);
ostream&		operator<<(ostream& s, const UCGEMatrix& m);
UCGEMatrix		transpose(const UCGEMatrix&);

/******************* I N L I N E S **************************/

Inline unsigned UCGEMatrix::cols() Const { return ncols;}
Inline unsigned UCGEMatrix::rows() Const { return nrows;}
Inline void UCGEMatrix::deepenShallowCopy(){UCharVec::deepenShallowCopy();}

#ifndef NO_UNARY_PLUS
  Inline UCGEMatrix operator+(const UCGEMatrix& m)		{ return m; }
#endif
#ifndef NO_INLINED_TEMP_DESTRUCTORS
  inline UCGEMatrix operator*(UChar d, const UCGEMatrix& m){ return m*d; }
  inline UCGEMatrix operator+(UChar d, const UCGEMatrix& m){ return m+d; }
#endif

/* Return a column*/
Inline UCharVec
UCGEMatrix::operator[](unsigned j) Const{
#ifdef BOUNDS_CHECK
  assertColRange(j);
#endif
  return UCharVec::slice(j*nrows,nrows,1);
}

Inline UCharVec
UCGEMatrix::col(unsigned j) Const{	/* Same as above*/
#ifdef BOUNDS_CHECK
  assertColRange(j);
#endif
  return UCharVec::slice(j*nrows,nrows,1);
}

Inline UCharVec
UCGEMatrix::row(unsigned i) Const{
#ifdef BOUNDS_CHECK
  assertRowRange(i);
#endif
  return UCharVec::slice(i, ncols, nrows);
}

Inline UCharVec
UCGEMatrix::diagonal(int i) Const{
  register int iabs=ABS(i);
#ifdef BOUNDS_CHECK
  assertSquare();
  assertRowRange(iabs);
#endif
  return UCharVec::slice(i>0 ? i*nrows : iabs, nrows-iabs, nrows+1);
}

inline UChar&
UCGEMatrix::operator()(int i, int j) Const{
#ifdef BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return UCharVec::operator()(j*nrows+i);
}

inline UChar&
UCGEMatrix::sub(int i, int j) Const {
#ifdef BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return UCharVec::sub(j*nrows+i);
}

/********************  Pick inlines *****************************/
 
Inline
UCGEPick::UCGEPick(const UCGEMatrix* m, const IntVec* xrow, const IntVec* xcol)
{
#ifdef BOUNDS_CHECK
  assertElements(m->rows(), *xrow);
  assertElements(m->cols(), *xcol);
#endif
  M = m;  rowPick = xrow; colPick = xcol;
}

Inline UChar&
UCGEPick::operator()(int i, int j) Const {
  return (*M)( (*rowPick)(i), (*colPick)(j) );
}

#endif /* __UCGEMATRIX_H__ */
