#ifndef __SCGEMATRIX_H__
#define __SCGEMATRIX_H__

/*
 * Declarations for signed character general matrices.
 *
 * $Id: scgemat.h,v 3.4.2.1 90/10/22 15:52:31 keffer Rel $
 *
 ****************************************************************************
 *
 * Rogue Wave 
 * P.O. Box 2328
 * Corvallis, OR 97339
 *
 * Copyright (C) 1989. This software is subject to copyright protection under 
 * the laws of the United States and other countries.
 *
 ***************************************************************************
 *
 */

/*
 * This class is derived from class SCharVec.  Data is stored
 * in column major order.
 *
 * Defining the preprocessor directive "BOUNDS_CHECK" will invoke
 * bounds checking.
 *
 */

#include "scvec.h"
class IGEMatrix;
class SCGEMatrix;

/*
 * The SCGEPick class allows selected elements of the matrix to be addressed.
 * There are no public constructors.
 */

class SCGEPick {
private:
  const SCGEMatrix*	M;
  const IntVec*		rowPick;
  const IntVec*		colPick;
  SCGEPick(const SCGEMatrix* m, const IntVec* xrow, const IntVec* xcol);
  SCGEPick(const SCGEPick& p)	{M=p.M; rowPick=p.rowPick; colPick=p.colPick;}
  friend		SCGEMatrix;
protected:
  void			assertRowCol(unsigned, unsigned) Const;
  void			assertElements(unsigned, const IntVec&) Const;
  void			assertColRange(unsigned) Const;
  void			assertRowRange(unsigned) Const;
public:
  void			operator=(const SCGEMatrix&);
  void			operator=(const SCGEPick&);
  void			operator=(SChar);

  SChar&		operator()(int i, int j) Const;
  unsigned		rows() Const { return rowPick->length(); }
  unsigned		cols() Const { return colPick->length(); }
};

class SCGEMatrix : private SCharVec {
  unsigned ncols;	/* Number of columns*/
  unsigned nrows;	/* Number of rows*/
protected:
  void 			assertColRange(unsigned) Const;
  void 			assertRowRange(unsigned) Const;
  void			assertRowCol(unsigned, unsigned) Const;
  void			assertLength(const SCharVec&) Const;
  void			assertSquare() Const;
  void			assertProduct(const SCGEMatrix&) Const;
  void			assertProduct(const SCharVec&) Const;
public:
  SCGEMatrix();
  SCGEMatrix(unsigned rows, unsigned cols);
  SCGEMatrix(unsigned rows, unsigned cols, SChar initval);
  SCGEMatrix(const SChar* dat, unsigned, unsigned);  /* Copy of dat will be made*/
  SCGEMatrix(const SCharVec& v, unsigned, unsigned); /* Reference to v will be made*/
  SCGEMatrix(const SCGEMatrix& m);	   /* Reference to m will be made*/
  SCGEMatrix(const SCGEPick& p);

  operator		IGEMatrix()    Const;  /* Convert to IGEMatrix*/

  SChar*  		data() Const	{return SCharVec::data();}
  unsigned		cols() Const;
  unsigned		rows() Const;

  SCGEMatrix&		reference(const SCGEMatrix& m); /* Reference self to m*/
  SCGEMatrix		deepCopy() Const;	/* copy of self with distinct instance variables */
  SCGEMatrix		copy() Const	{return deepCopy();} /* Synonym for deepCopy()*/
  void			deepenShallowCopy();	/* Guarantee that references==1:*/
  void			resize(unsigned,unsigned);

  void			scanFrom(istream& s);	/* Format 2 x 2 [ 1 3 2 4 ]*/
  void			printOn(ostream& s) Const;
  void			readFrom(istream&);
  void			storeOn(ostream&) Const;
  void			readFrom(fileDescTy&);
  void			storeOn(fileDescTy&) Const;
  void			readFrom(RWFile*); /* Internal binary formatting*/
  void			storeOn(RWFile*) Const;
  unsigned		binaryStoreSize() Const;

  SCharVec		operator[](unsigned j) Const;	/* Return a col as a slice*/
  SCharVec		col(unsigned j) Const;		/* Return a col as a slice*/
  SCharVec		row(unsigned i) Const;		/* Return a row as a slice*/
  SCharVec		diagonal(int idiag=0) Const;	/* Return a diagonal as a slice*/
  inline SChar&		operator()(int i, int j) Const; /* Subscripting*/
  inline SChar&		sub(int, int) Const;		/* Assumes stride==1; use with care*/
  SCGEPick		operator()(const IntVec&, const IntVec&) Const; /* Picking*/

/* Math functions*/
  SCGEMatrix		product(const SCGEMatrix&) Const; /* Inner product*/
  SCharVec		product(const SCharVec&) Const;

/* Assignment operators --- self must be same size as m*/
  SCGEMatrix&		operator=(const SCGEMatrix& m);
  SCGEMatrix&		operator=(const SCGEPick& p);
  SCGEMatrix&		operator=(SChar);
  SCGEMatrix&		operator+=(const SCGEMatrix& m);
  SCGEMatrix&		operator+=(SChar);
  SCGEMatrix&		operator-=(const SCGEMatrix& m);
  SCGEMatrix&		operator-=(SChar);
  SCGEMatrix&		operator*=(const SCGEMatrix& m);
  SCGEMatrix&		operator*=(SChar);
  SCGEMatrix&		operator/=(const SCGEMatrix& m);
  SCGEMatrix&		operator/=(SChar);

  /* Boolean operators:*/
  RWBoolean		operator==(const SCGEMatrix&) Const;
  RWBoolean		operator!=(const SCGEMatrix&) Const;

/* Increment/decrement operators*/
  SCGEMatrix&		operator++();
  SCGEMatrix&		operator--();

/* Friendly arithmetic operators.*/
  friend SCGEMatrix	operator-(const SCGEMatrix&);	/* Unary minus*/
#ifndef NO_UNARY_PLUS
  friend SCGEMatrix	operator+(const SCGEMatrix&);	/* Unary plus*/
#endif
  friend SCGEMatrix	operator*(const SCGEMatrix&, const SCGEMatrix&);
  friend SCGEMatrix	operator/(const SCGEMatrix&, const SCGEMatrix&);
  friend SCGEMatrix	operator+(const SCGEMatrix&, const SCGEMatrix&);
  friend SCGEMatrix	operator-(const SCGEMatrix&, const SCGEMatrix&);
  friend SCGEMatrix	operator*(const SCGEMatrix&, SChar);
  friend SCGEMatrix	operator*(SChar, const SCGEMatrix&);
  friend SCGEMatrix	operator/(const SCGEMatrix&, SChar);
  friend SCGEMatrix	operator/(SChar, const SCGEMatrix&);
  friend SCGEMatrix	operator+(const SCGEMatrix&, SChar);
  friend SCGEMatrix	operator+(SChar, const SCGEMatrix&);
  friend SCGEMatrix	operator-(const SCGEMatrix&, SChar);
  friend SCGEMatrix	operator-(SChar, const SCGEMatrix&);

  friend SCGEMatrix	abs(const SCGEMatrix&);
  friend SChar		dot(const SCGEMatrix&, const SCGEMatrix&);
  friend SChar		prod(const SCGEMatrix&);
  friend SChar		sum(const SCGEMatrix&);
};

/* Other (related) declarations:*/
istream&		operator>>(istream& s, SCGEMatrix& m);
ostream&		operator<<(ostream& s, const SCGEMatrix& m);
SCGEMatrix		transpose(const SCGEMatrix&);

/******************* I N L I N E S **************************/

Inline unsigned SCGEMatrix::cols() Const { return ncols;}
Inline unsigned SCGEMatrix::rows() Const { return nrows;}
Inline void SCGEMatrix::deepenShallowCopy(){SCharVec::deepenShallowCopy();}

#ifndef NO_UNARY_PLUS
  Inline SCGEMatrix operator+(const SCGEMatrix& m)		{ return m; }
#endif
#ifndef NO_INLINED_TEMP_DESTRUCTORS
  inline SCGEMatrix operator*(SChar d, const SCGEMatrix& m){ return m*d; }
  inline SCGEMatrix operator+(SChar d, const SCGEMatrix& m){ return m+d; }
#endif

/* Return a column*/
Inline SCharVec
SCGEMatrix::operator[](unsigned j) Const{
#ifdef BOUNDS_CHECK
  assertColRange(j);
#endif
  return SCharVec::slice(j*nrows,nrows,1);
}

Inline SCharVec
SCGEMatrix::col(unsigned j) Const{	/* Same as above*/
#ifdef BOUNDS_CHECK
  assertColRange(j);
#endif
  return SCharVec::slice(j*nrows,nrows,1);
}

Inline SCharVec
SCGEMatrix::row(unsigned i) Const{
#ifdef BOUNDS_CHECK
  assertRowRange(i);
#endif
  return SCharVec::slice(i, ncols, nrows);
}

Inline SCharVec
SCGEMatrix::diagonal(int i) Const{
  register int iabs=ABS(i);
#ifdef BOUNDS_CHECK
  assertSquare();
  assertRowRange(iabs);
#endif
  return SCharVec::slice(i>0 ? i*nrows : iabs, nrows-iabs, nrows+1);
}

inline SChar&
SCGEMatrix::operator()(int i, int j) Const{
#ifdef BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return SCharVec::operator()(j*nrows+i);
}

inline SChar&
SCGEMatrix::sub(int i, int j) Const {
#ifdef BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return SCharVec::sub(j*nrows+i);
}

/********************  Pick inlines *****************************/
 
Inline
SCGEPick::SCGEPick(const SCGEMatrix* m, const IntVec* xrow, const IntVec* xcol)
{
#ifdef BOUNDS_CHECK
  assertElements(m->rows(), *xrow);
  assertElements(m->cols(), *xcol);
#endif
  M = m;  rowPick = xrow; colPick = xcol;
}

Inline SChar&
SCGEPick::operator()(int i, int j) Const {
  return (*M)( (*rowPick)(i), (*colPick)(j) );
}

#endif /* __SCGEMATRIX_H__ */
