#ifndef __INTVEC_H__
#define __INTVEC_H__

/*
 * Declarations for Integer vectors
 *
 * $Id: ivec.h,v 3.4.2.1 90/10/22 15:52:19 keffer Rel $
 *
 ****************************************************************************
 *
 * Rogue Wave 
 * P.O. Box 2328
 * Corvallis, OR 97339
 *
 * Copyright (C) 1989. This software is subject to copyright protection under 
 * the laws of the United States and other countries.
 *
 ***************************************************************************
 *
 */

/*
 * Defining the preprocessor directive BOUNDS_CHECK will
 * cause bounds checking on the subscripting operator.
 */

#include "rwblock.h"
STARTWRAP
#include <math.h>
ENDWRAP

class DoubleVec;
class IntVec;
class FloatVec;

/*
 * The IntPick class allows selected elements to be addressed.
 * There are no public constructors.
 */

class IntPick {
private:
  const IntVec*		V;
  const IntVec*		X;
  IntPick(const IntVec* v, const IntVec* x); /* Constructor is private*/
  IntPick(const IntPick& p) {V=p.V; X=p.X;}
  friend		IntVec;
protected:
  void			assertElements(unsigned, const IntVec&) Const;
  void			lengthCheck(unsigned) Const;
public:
  void			operator=(const IntVec&);
  void			operator=(const IntPick&);
  void			operator=(int);

  inline int&		operator()(int i) Const;
  unsigned		length() Const;
};

class IntVec {
  RWBlock*		block;
  int*			begin;
  unsigned		npts;
  int			step;

  static int		numberPerLine;		/* For printing*/
  IntVec(const IntVec&, int, unsigned, int);	/* For slices*/
protected:
  void			boundsCheck(int) Const;
  void			boundsErr(int) Const;
  void			emptyErr(const char* fname) Const;
  void			lengthCheck(int i) Const {if(npts!=i) lengthErr(i);}
  void			lengthErr(int) Const;
  void			strideCheck() Const;
  void			sliceErr(unsigned, int, unsigned, int) Const;
public:
  IntVec();
  IntVec(unsigned n);
  IntVec(unsigned n, int val);
  IntVec(unsigned n, int val, int by);
  IntVec(const IntVec& a);
  IntVec(const IntPick& p);
  IntVec(const int* dat, unsigned n);  /* Copy of dat will be made*/
  ~IntVec();
  
  operator		DoubleVec() Const;	/* Convert to DoubleVec*/

  IntVec		slice(int start, unsigned lgt, int strider=1) Const;
  
  int*			data() Const	{return begin;}
  unsigned		length() Const	{return npts;}
  int			stride() Const	{return step;}

  unsigned		binaryStoreSize() Const;	/* Storage requirements.*/
  IntVec		copy() Const {return deepCopy();}/* Synonym for deepCopy()*/
  IntVec		deepCopy() Const;		/* copy  with distinct instance variables */
  void			deepenShallowCopy();		/* Insures only 1 reference to data*/
  void			printOn(ostream& s) Const;	/* Pretty print*/
  void			readFrom(RWFile*);		/* Internal binary formatting*/
  void			readFrom(fileDescTy&);		/* Internal binary formatting*/
  void			readFrom(istream&);		/* Internal ASCII formatting*/
  IntVec&		reference(const IntVec& v);	/* Reference self to v*/
  void			resize(unsigned);		/* Will pad with zeroes if necessary*/
  void			scanFrom(istream& s);		/* Read to eof or delimit with []*/
  int			setFormatting(int) Const;	/* Change # items per line*/
  void			storeOn(RWFile*) Const;
  void			storeOn(fileDescTy&) Const;
  void			storeOn(ostream&) Const;
  
  /* Indexing:*/
  int&			operator[](int i) Const;	/* With bounds checking*/
  inline int&		operator()(int i) Const;	/* With optional bounds checking*/
  IntPick		operator()(const IntVec& x) Const;
  inline int&		sub(int i) Const;		/* Assumes stride==1; use carefully*/
  
  /* Assignment:*/
  IntVec&		operator=(const IntVec& v); /* Must be same length as v*/
  IntVec&		operator=(const IntPick&);
  IntVec&		operator=(int);
  
  /* Boolean operators:*/
  RWBoolean		operator==(const IntVec&) Const;
  RWBoolean		operator!=(const IntVec&) Const;
  
  /* Arithmetic operators:*/
  IntVec&		operator++();
  IntVec&		operator--();
  IntVec&		operator+=(const IntVec&);
  IntVec&		operator+=(int);
  IntVec&		operator-=(const IntVec&);
  IntVec&		operator-=(int);
  IntVec&		operator*=(const IntVec&);
  IntVec&		operator*=(int);
  IntVec&		operator/=(const IntVec&);
  IntVec&		operator/=(int);
  
  /* Friendly arithmetic operators:*/
  friend IntVec	operator-(const IntVec&);
#ifndef NO_UNARY_PLUS
  friend IntVec	operator+(const IntVec&);
#endif
  friend IntVec	operator*(const IntVec&,const IntVec&);
  friend IntVec	operator/(const IntVec&,const IntVec&);
  friend IntVec	operator+(const IntVec&,const IntVec&);
  friend IntVec	operator-(const IntVec&,const IntVec&);
  friend IntVec	operator*(const IntVec&,int);
  friend IntVec	operator*(int,const IntVec&);
  friend IntVec	operator/(const IntVec&,int);
  friend IntVec	operator/(int,const IntVec&);
  friend IntVec	operator+(const IntVec&,int);
  friend IntVec	operator+(int,const IntVec&);
  friend IntVec	operator-(const IntVec&,int);
  friend IntVec	operator-(int,const IntVec&);
  
  
#ifndef NO_VECTOR_MATHFUN
  /* Math functions:*/
  IntVec		apply(mathFunTy) Const;
  friend IntVec		abs(const IntVec&);
/*friend IntVec		acos(const IntVec&);*/
/*friend IntVec		asin(const IntVec&);*/
/*friend IntVec		atan(const IntVec&);*/
/*friend IntVec		atan2(const IntVec&,const IntVec&);*/
/*friend IntVec		ceil(const IntVec&);*/
/*friend IntVec		cos(const IntVec&);*/
/*friend IntVec		cosh(const IntVec&);*/
  friend IntVec		cumsum(const IntVec&);
  friend IntVec		delta(const IntVec&);
  friend int		dot(const IntVec&,const IntVec&);
/*friend IntVec		exp(const IntVec&); */
/*friend IntVec		floor(const IntVec&);*/
/*friend IntVec		log(const IntVec&);*/
  friend int		max(const IntVec&);
  friend int		min(const IntVec&);
/*friend int		mean(const IntVec&);*/
  friend int		prod(const IntVec&);
/*friend IntVec 	pow(const IntVec&,const IntVec&);*/
  friend IntVec 	reverse(const IntVec&);
/*friend IntVec 	sin(const IntVec&);*/
/*friend IntVec 	sinh(const IntVec&);*/
/*friend IntVec 	sqrt(const IntVec&);*/
  friend int		sum(const IntVec&);
/*friend IntVec		tan(const IntVec&);*/
/*friend IntVec		tanh(const IntVec&);*/
/*friend int		variance(const IntVec&);*/

#endif
  
};

/* Other (related) declarations:*/
ostream&	operator<<(ostream&, const IntVec&);
istream&	operator>>(istream&, IntVec&);
IntVec		toInt(const DoubleVec&);
IntVec		toInt(const FloatVec&);

/******************* I N L I N E S **************************/

Inline void
  IntVec::boundsCheck(int i) Const {
	if(i<0 || i>npts) boundsErr(i);
  }

Inline int&
  IntVec::operator[](int i) Const {
	boundsCheck(i); return begin[i*step];
  }

inline int&
  IntVec::operator()(int i) Const {
#ifdef BOUNDS_CHECK    
	boundsCheck(i);
#endif
	return begin[i*step];
  }

inline int&
IntVec::sub(int i) Const {
#ifdef BOUNDS_CHECK
  boundsCheck(i);
#endif
#ifdef DEBUG
  strideCheck();
#endif
  return begin[i];
}

Inline IntVec
  IntVec::slice(int start, unsigned n, int str) Const {
	return IntVec(*this, start, n, str);
  }

#ifndef NO_UNARY_PLUS
  Inline IntVec		operator+(const IntVec& a)	{return a;}
#endif
#ifndef NO_INLINED_TEMP_DESTRUCTORS
  inline IntVec		operator*(int a, const IntVec& b)	{return b*a;}
  inline IntVec		operator+(int a, const IntVec& b)	{return b+a;}
#endif

/********************  Pick inlines *****************************/
Inline
unsigned IntPick::length() Const {
  return X->length();
}

Inline
IntPick::IntPick(const IntVec* v, const IntVec* x)
{
#ifdef BOUNDS_CHECK
  assertElements(v->length(), *x);
#endif
  V = v;  X = x;
}

inline int&
IntPick::operator()(int i) Const {
  return (*V)( (*X)(i) );
}

#endif /* __INTVEC_H__ */
