#ifndef __DOUBLEVEC_H__
#define __DOUBLEVEC_H__

/*
 * Declarations for Double Precision Vectors
 *
 * $Id: dvec.h,v 3.4.2.1 90/10/22 15:52:11 keffer Rel $
 *
 ****************************************************************************
 *
 * Rogue Wave 
 * P.O. Box 2328
 * Corvallis, OR 97339
 *
 * Copyright (C) 1989, 1990. This software is subject to copyright 
 * protection under the laws of the United States and other countries.
 *
 ***************************************************************************
 *
 */

/*
 * Defining the preprocessor directive BOUNDS_CHECK will
 * cause bounds checking on the subscripting operator.
 */

#include "ivec.h"

class DoubleVec;
class DComplexVec;

/*
 * The DoublePick class allows selected elements to be addressed.
 * There are no public constructors.
 */

class DoublePick {
private:
  const DoubleVec*	V;
  const IntVec*		X;
  DoublePick(const DoubleVec* v, const IntVec* x); /* Constructor is private*/
  DoublePick(const DoublePick& p) {V=p.V; X=p.X;}
  friend		DoubleVec;
protected:
  void			assertElements(unsigned, const IntVec&) Const;
  void			lengthCheck(unsigned) Const;
public:
  void			operator=(const DoubleVec&);
  void			operator=(const DoublePick&);
  void			operator=(double);

  inline double&	operator()(int i) Const;
  unsigned		length() Const	{ return X->length(); }
};

class DoubleVec {
  RWBlock*		block;
  double*		begin;
  unsigned		npts;
  int			step;

  static int		numberPerLine;			/* For printing*/
  DoubleVec(const DoubleVec&, int, unsigned, int);	/* For slices*/
  DoubleVec(RWBlock*, double*, unsigned, int);
  friend DoubleVec	real(const DComplexVec&);
  friend DoubleVec	imag(const DComplexVec&);
protected:
  void			boundsCheck(int) Const;
  void			boundsErr(int) Const;
  void			emptyErr(const char* fname) Const;
  void			lengthCheck(int i) Const {if(npts!=i) lengthErr(i);}
  void			lengthErr(int) Const;
  void			strideCheck() Const;
  void			sliceErr(unsigned, int, unsigned, int) Const;
public:
  DoubleVec();
  DoubleVec(unsigned n);
  DoubleVec(unsigned n, double val);
  DoubleVec(unsigned n, double val, double by);
  DoubleVec(const DoubleVec& a);
  DoubleVec(const DoublePick& p);
  DoubleVec(const double* dat, unsigned n);  /* Copy of dat will be made*/
  ~DoubleVec();
  
  DoubleVec		slice(int start, unsigned lgt, int strider=1) Const;
  
  double*		data() Const	{return begin;}
  unsigned		length() Const	{return npts;}
  int			stride() Const	{return step;}

  unsigned		binaryStoreSize() Const;	/* Storage requirements.*/
  DoubleVec		copy() Const {return deepCopy();}/* Synonym for deepCopy()*/
  DoubleVec		deepCopy() Const;		/* copy with distinct instance variables */
  void			deepenShallowCopy();		/* Insures only 1 reference to data*/
  void			printOn(ostream& s) Const;	/* Pretty print*/
  void			readFrom(RWFile*);		/* Internal binary formatting*/
  void			readFrom(fileDescTy&);		/* Internal binary formatting*/
  void			readFrom(istream&);		/* Internal ASCII formatting*/
  DoubleVec&		reference(const DoubleVec& v);	/* Reference self to v*/
  void			resize(unsigned);		/* Will pad with zeroes if necessary*/
  void			scanFrom(istream& s);		/* Read to eof or delimit with []*/
  int			setFormatting(int) Const;	/* Change # items per line*/
  void			storeOn(RWFile*) Const;
  void			storeOn(fileDescTy&) Const;
  void			storeOn(ostream&) Const;

  /* Indexing:*/
  double&		operator[](int i) Const;	/* With bounds checking*/
  inline double&	operator()(int i) Const;	/* With optional bounds checking*/
  DoublePick		operator()(const IntVec& x) Const;
  inline double&	sub(int i) Const;		/* Assumes stride==1; use carefully*/

  /* Assignment:*/
  DoubleVec&		operator=(const DoubleVec& v);	/* Must be same length as v*/
  DoubleVec&		operator=(const DoublePick&);
  DoubleVec&		operator=(double);
  
  /* Boolean operators:*/
  RWBoolean		operator==(const DoubleVec&) Const;
  RWBoolean		operator!=(const DoubleVec&) Const;
  
  /* Arithmetic operators:*/
  DoubleVec&		operator++();
  DoubleVec&		operator--();
  DoubleVec&		operator+=(const DoubleVec&);
  DoubleVec&		operator+=(double);
  DoubleVec&		operator-=(const DoubleVec&);
  DoubleVec&		operator-=(double);
  DoubleVec&		operator*=(const DoubleVec&);
  DoubleVec&		operator*=(double);
  DoubleVec&		operator/=(const DoubleVec&);
  DoubleVec&		operator/=(double);
  
  /* Friendly arithmetic operators:*/
  friend DoubleVec	operator-(const DoubleVec&);
  /* Zortech does not allow overloading of unary plus operator*/
#ifndef NO_UNARY_PLUS
  friend DoubleVec	operator+(const DoubleVec&);
#endif
  friend DoubleVec	operator*(const DoubleVec&,const DoubleVec&);
  friend DoubleVec	operator/(const DoubleVec&,const DoubleVec&);
  friend DoubleVec	operator+(const DoubleVec&,const DoubleVec&);
  friend DoubleVec	operator-(const DoubleVec&,const DoubleVec&);
  friend DoubleVec	operator*(const DoubleVec&,double);
  friend DoubleVec	operator*(double,const DoubleVec&);
  friend DoubleVec	operator/(const DoubleVec&,double);
  friend DoubleVec	operator/(double,const DoubleVec&);
  friend DoubleVec	operator+(const DoubleVec&,double);
  friend DoubleVec	operator+(double,const DoubleVec&);
  friend DoubleVec	operator-(const DoubleVec&,double);
  friend DoubleVec	operator-(double,const DoubleVec&);
  
  
#ifndef NO_VECTOR_MATHFUN
  /* Math functions:*/
  DoubleVec		apply(mathFunTy) Const;
  friend DoubleVec	abs(const DoubleVec&);
  friend DoubleVec	acos(const DoubleVec&);
  friend DoubleVec	asin(const DoubleVec&);
  friend DoubleVec	atan(const DoubleVec&);
  friend DoubleVec	atan2(const DoubleVec&,const DoubleVec&);
  friend DoubleVec	ceil(const DoubleVec&);
  friend DoubleVec	cos(const DoubleVec&);
  friend DoubleVec	cosh(const DoubleVec&);
  friend DoubleVec	cumsum(const DoubleVec&);
  friend DoubleVec	delta(const DoubleVec&);
  friend double    	dot(const DoubleVec&,const DoubleVec&);
  friend DoubleVec	exp(const DoubleVec&); 
  friend DoubleVec	floor(const DoubleVec&);
  friend DoubleVec	log(const DoubleVec&);
  friend DoubleVec	log10(const DoubleVec&);
  friend int		max(const DoubleVec&);
  friend int		min(const DoubleVec&);
  friend double		mean(const DoubleVec&);
  friend double		prod(const DoubleVec&);
  friend DoubleVec	pow(const DoubleVec&,const DoubleVec&);
  friend DoubleVec	reverse(const DoubleVec&);
  friend DoubleVec	sin(const DoubleVec&);
  friend DoubleVec	sinh(const DoubleVec&);
  friend DoubleVec	sqrt(const DoubleVec&);
  friend double		sum(const DoubleVec&);
  friend DoubleVec	tan(const DoubleVec&);
  friend DoubleVec	tanh(const DoubleVec&);
  friend double		variance(const DoubleVec&);
#endif
  
};

/* Other (related) declarations:*/
DoubleVec		expandEven(const DoubleVec&);
DoubleVec		expandOdd(const DoubleVec&);
ostream&		operator<<(ostream&, const DoubleVec&);
istream&		operator>>(istream&, DoubleVec&);

/******************* I N L I N E S **************************/

Inline void
DoubleVec::boundsCheck(int i) Const{
  if(i<0 || i>npts) boundsErr(i);
}

Inline double&
DoubleVec::operator[](int i) Const{
  boundsCheck(i); return begin[i*step];
}

inline double&
DoubleVec::operator()(int i) Const {
#ifdef BOUNDS_CHECK    
  boundsCheck(i);
#endif
  return begin[i*step];
}

inline double&
DoubleVec::sub(int i) Const {
#ifdef BOUNDS_CHECK
  boundsCheck(i);
#endif
#ifdef DEBUG
  strideCheck();
#endif
  return begin[i];
}

Inline DoubleVec
DoubleVec::slice(int start, unsigned n, int str) Const {
  return DoubleVec(*this, start, n, str);
}

#ifndef NO_UNARY_PLUS
  Inline DoubleVec	operator+(const DoubleVec& a)	{return a;}
#endif
#ifndef NO_INLINED_TEMP_DESTRUCTORS
  inline DoubleVec	operator*(double a, const DoubleVec& b)	{return b*a;}
  inline DoubleVec	operator+(double a, const DoubleVec& b)	{return b+a;}
#endif

#ifndef NO_VECTOR_MATHFUN
Inline DoubleVec abs(const DoubleVec& V)	{ return V.apply(::fabs);}
Inline DoubleVec acos(const DoubleVec& V)	{ return V.apply(::acos); }
Inline DoubleVec asin(const DoubleVec& V)	{ return V.apply(::asin); }
Inline DoubleVec atan(const DoubleVec& V)	{ return V.apply(::atan); }
Inline DoubleVec ceil(const DoubleVec& V)	{ return V.apply(::ceil); }
Inline DoubleVec cos(const DoubleVec& V)	{ return V.apply(::cos); }
Inline DoubleVec cosh(const DoubleVec& V)	{ return V.apply(::cosh); }
Inline DoubleVec exp(const DoubleVec& V)	{ return V.apply(::exp); }
Inline DoubleVec floor(const DoubleVec& V )	{ return V.apply(::floor); }
Inline DoubleVec log(const DoubleVec& V)	{ return V.apply(::log); }
Inline DoubleVec log10(const DoubleVec& V)	{ return V.apply(::log10); }
Inline DoubleVec sin(const DoubleVec& V)	{ return V.apply(::sin); }
Inline DoubleVec sinh(const DoubleVec& V)	{ return V.apply(::sinh); }
Inline DoubleVec sqrt(const DoubleVec& V)	{ return V.apply(::sqrt); }
Inline DoubleVec tan(const DoubleVec& V)	{ return V.apply(::tan); }
Inline DoubleVec tanh(const DoubleVec& V)	{ return V.apply(::tanh); }
Inline double    mean(const DoubleVec& V)	{ return sum(V)/V.length(); }
#endif
 
/********************  Pick inlines *****************************/
 
Inline
DoublePick::DoublePick(const DoubleVec* v, const IntVec* x)
{
#ifdef BOUNDS_CHECK
  assertElements(v->length(), *x);
#endif
  V = v;  X = x;
}

inline double&
DoublePick::operator()(int i) Const {
  return (*V)( (*X)(i) );
}

#endif /* __DVEC_H__ */
