#ifndef __DCOMPLEXVEC_H__
#define __DCOMPLEXVEC_H__

/*
 * Declarations for DComplex Precision Vectors
 *
 * $Id: cvec.h,v 3.4.2.1 90/10/22 15:52:01 keffer Rel $
 *
 ****************************************************************************
 *
 * Rogue Wave 
 * P.O. Box 2328
 * Corvallis, OR 97339
 *
 * Copyright (C) 1989. This software is subject to copyright protection under 
 * the laws of the United States and other countries.
 ***************************************************************************
 *
 */

/*
 * Defining the preprocessor directive BOUNDS_CHECK will
 * cause bounds checking on the subscripting operator.
 */

#include "dvec.h"
#include "dcomplex.h"

class DComplexVec;

/*
 * The DComplexPick class allows selected elements to be addressed.
 * There are no public constructors.
 */

class DComplexPick {
private:
  const DComplexVec*	V;
  const IntVec*		X;
  DComplexPick(const DComplexVec* v, const IntVec* x); /* Constructor is private*/
  DComplexPick(const DComplexPick& p) {V=p.V; X=p.X;}
  friend		DComplexVec;
protected:
  void			assertElements(unsigned, const IntVec&) Const;
  void			lengthCheck(unsigned) Const;
public:
  void			operator=(const DComplexVec&);
  void			operator=(const DComplexPick&);
  void			operator=(DComplex);

  inline DComplex&	operator()(int i) Const;
  unsigned		length() Const	{ return X->length(); }
};

class DComplexVec {
  RWBlock*		block;
  DComplex*		begin;
  unsigned		npts;
  int			step;

  static int		numberPerLine;			/* For printing*/
  DComplexVec(const DComplexVec&, int, unsigned, int);	/* For slices*/
protected:
  void			boundsCheck(int) Const;
  void			boundsErr(int) Const;
  void			emptyErr(const char* fname) Const;
  void			lengthCheck(int i) Const {if(npts!=i) lengthErr(i);}
  void			lengthErr(int) Const;
  void			strideCheck() Const;
  void			sliceErr(unsigned, int, unsigned, int) Const;
public:
  DComplexVec();
  DComplexVec(unsigned n);
  DComplexVec(unsigned n, DComplex val);
  DComplexVec(unsigned n, DComplex val, DComplex by);
  DComplexVec(const DComplexVec& a);
  DComplexVec(const DComplexPick& p);
  DComplexVec(const DoubleVec& re); /* Conversion from DoubleVec*/
  DComplexVec(const DoubleVec& re, const DoubleVec& im);
  DComplexVec(const DComplex* dat, unsigned n);  /* Copy of dat will be made*/

  ~DComplexVec();
  
  DComplexVec		slice(int start, unsigned lgt, int strider=1) Const;
  
  DComplex*		data() Const    {return begin;}
  unsigned		length() Const  {return npts;}
  int			stride() Const  {return step;}

  unsigned		binaryStoreSize() Const;	/* Storage requirements.*/
  DComplexVec		copy() Const {return deepCopy();}/* Synonym for deepCopy()*/
  DComplexVec		deepCopy() Const;		/* copy with distinct instance variables */
  void			deepenShallowCopy();		/* Insures only 1 reference to data*/
  void			printOn(ostream& s) Const;	/* Pretty print*/
  void			readFrom(RWFile*);		/* Internal binary formatting*/
  void			readFrom(fileDescTy&);		/* Internal binary formatting*/
  void			readFrom(istream&);		/* Internal ASCII formatting*/
  DComplexVec&		reference(const DComplexVec& v);/* Reference self to v*/
  void			resize(unsigned);		/* Will pad with zeroes if necessary*/
  void			scanFrom(istream& s);		/* Read to eof or delimit with []*/
  int			setFormatting(int) Const;	/* Change # items per line*/
  void			storeOn(RWFile*) Const;
  void			storeOn(fileDescTy&) Const;
  void			storeOn(ostream&) Const;

  /* Indexing:*/
  DComplex&		operator[](int i) Const;	/* With bounds checking*/
  inline DComplex&	operator()(int i) Const;	/* With optional bounds checking*/
  DComplexPick		operator()(const IntVec& x) Const;
  inline DComplex&	sub(int i) Const;		/* Assumes stride==1; use carefully*/
  
  /* Assignment:*/
  DComplexVec&		operator=(const DComplexVec& v); /* Must be same length as v*/
  DComplexVec&		operator=(const DComplexPick&);
  DComplexVec&		operator=(DComplex);

  /* Boolean operators:*/
  RWBoolean		operator==(const DComplexVec&) Const;
  RWBoolean		operator!=(const DComplexVec&) Const;
  
  /* Arithmetic operators:*/
/*DComplexVec&		operator++();*/
/*DComplexVec&		operator--();*/
  DComplexVec&		operator+=(const DComplexVec&);
  DComplexVec&		operator+=(DComplex);
  DComplexVec&		operator-=(const DComplexVec&);
  DComplexVec&		operator-=(DComplex);
  DComplexVec&		operator*=(const DComplexVec&);
  DComplexVec&		operator*=(DComplex);
  DComplexVec&		operator/=(const DComplexVec&);
  DComplexVec&		operator/=(DComplex);
  
  /* Friendly arithmetic operators:*/
  friend DComplexVec	operator-(const DComplexVec&);
#ifndef NO_UNARY_PLUS
  friend DComplexVec	operator+(const DComplexVec&);
#endif
  friend DComplexVec	operator*(const DComplexVec&,const DComplexVec&);
  friend DComplexVec	operator/(const DComplexVec&,const DComplexVec&);
  friend DComplexVec	operator+(const DComplexVec&,const DComplexVec&);
  friend DComplexVec	operator-(const DComplexVec&,const DComplexVec&);
  friend DComplexVec	operator*(const DComplexVec&,DComplex);
  friend DComplexVec	operator*(DComplex,const DComplexVec&);
  friend DComplexVec	operator/(const DComplexVec&,DComplex);
  friend DComplexVec	operator/(DComplex,const DComplexVec&);
  friend DComplexVec	operator+(const DComplexVec&,DComplex);
  friend DComplexVec	operator+(DComplex,const DComplexVec&);
  friend DComplexVec	operator-(const DComplexVec&,DComplex);
  friend DComplexVec	operator-(DComplex,const DComplexVec&);
  
  
#ifndef NO_VECTOR_MATHFUN
  /* Math functions (not all implemented by complex):*/
  DComplexVec		apply(CmathFunTy) Const;
  DoubleVec		apply2(CmathFunTy2) Const;
  friend DoubleVec	abs(const DComplexVec&);
/*friend DComplexVec	acos(const DComplexVec&);*/
/*friend DComplexVec	asin(const DComplexVec&);*/
/*friend DComplexVec	atan(const DComplexVec&);*/
/*friend DComplexVec	atan2(const DComplexVec&,const DComplexVec&);*/
/*friend DComplexVec	ceil(const DComplexVec&);*/
  friend DComplexVec	cos(const DComplexVec&);
  friend DComplexVec	cosh(const DComplexVec&);
  friend DComplexVec	cumsum(const DComplexVec&);
  friend DComplexVec	delta(const DComplexVec&);
  friend DComplex	dot(const DComplexVec&,const DComplexVec&);
  friend DComplexVec	exp(const DComplexVec&); 
/*friend DComplexVec	floor(const DComplexVec&);*/
  friend DComplexVec	log(const DComplexVec&);
/*friend int		max(const DComplexVec&);*/
/*friend int		min(const DComplexVec&);*/
  friend DComplex	mean(const DComplexVec&);
  friend DComplex	prod(const DComplexVec&);
  friend DComplexVec	pow(const DComplexVec&,const DComplexVec&);
  friend DComplexVec	reverse(const DComplexVec&);
  friend DComplexVec	sin(const DComplexVec&);
  friend DComplexVec	sinh(const DComplexVec&);
  friend DComplexVec	sqrt(const DComplexVec&);
  friend DComplex	sum(const DComplexVec&);
/*friend DComplexVec	tan(const DComplexVec&);*/
/*friend DComplexVec	tanh(const DComplexVec&);*/
  friend double		variance(const DComplexVec&);

/* Complex specific functions:*/
  friend DoubleVec	arg(const DComplexVec& V);
  friend DComplexVec	conj(const DComplexVec& V);
  friend DoubleVec	imag(const DComplexVec& V);
  friend DoubleVec	norm(const DComplexVec& V);
  friend DoubleVec	real(const DComplexVec& V);

#endif
  
};

/* Other (related) declarations:*/
DComplexVec		rootsOfOne(int N, unsigned nterms);
Inline DComplexVec	rootsOfOne(int N) {return rootsOfOne(N, ABS(N));}
DComplexVec		expandConjugateEven(const DComplexVec&);
DComplexVec		expandConjugateOdd(const DComplexVec&);
ostream&		operator<<(ostream&, const DComplexVec&);
istream&		operator>>(istream&, DComplexVec&);

/******************* I N L I N E S **************************/

Inline void
DComplexVec::boundsCheck(int i) Const{
  if(i<0 || i>npts) boundsErr(i);
}

Inline DComplex&
DComplexVec::operator[](int i) Const{
  boundsCheck(i); return begin[i*step];
}

inline DComplex&
DComplexVec::operator()(int i) Const{
#ifdef BOUNDS_CHECK    
  boundsCheck(i);
#endif
  return begin[i*step];
}

inline DComplex&
DComplexVec::sub(int i) Const {
#ifdef BOUNDS_CHECK
  boundsCheck(i);
#endif
#ifdef DEBUG
  strideCheck();
#endif
  return begin[i];
}

Inline DComplexVec
DComplexVec::slice(int start, unsigned n, int str) Const {
  return DComplexVec(*this, start, n, str);
}

#ifndef NO_UNARY_PLUS
  Inline DComplexVec	operator+(const DComplexVec& a)	{return a;}
#endif
#ifndef NO_INLINED_TEMP_DESTRUCTORS
  inline DComplexVec	operator*(DComplex a, const DComplexVec& b) {return b*a;}
  inline DComplexVec	operator+(DComplex a, const DComplexVec& b) {return b+a;}
#endif

#ifndef NO_VECTOR_MATHFUN

Inline DComplex
  mean(const DComplexVec& V)	{ DComplex s = sum(V);
				  return s/DComplex(double(V.length())); }

#ifndef POINTER_TO_OVERLOAD_BUG
  /* Zortech doesn't allow this construct:*/
  Inline DComplexVec cos(const DComplexVec& V)	{ return V.apply(::cos); }
  Inline DComplexVec cosh(const DComplexVec& V) { return V.apply(::cosh); }
  Inline DComplexVec exp(const DComplexVec& V)	{ return V.apply(::exp); }
  Inline DComplexVec log(const DComplexVec& V)	{ return V.apply(::log); }
  Inline DComplexVec sin(const DComplexVec& V)	{ return V.apply(::sin); }
  Inline DComplexVec sinh(const DComplexVec& V) { return V.apply(::sinh); }
  Inline DComplexVec sqrt(const DComplexVec& V) { return V.apply(::sqrt); }
#endif

/* Complex specific:*/
Inline DoubleVec arg(const DComplexVec& V)	{ return V.apply2(::arg); }
Inline DComplexVec conj(const DComplexVec& V)	{ return V.apply(::conj); }
Inline DoubleVec norm(const DComplexVec& V)	{ return V.apply2(::norm); }
#endif
 
/********************  Pick inlines *****************************/
 
Inline
DComplexPick::DComplexPick(const DComplexVec* v, const IntVec* x)
{
#ifdef BOUNDS_CHECK
  assertElements(v->length(), *x);
#endif
  V = v;  X = x;
}

inline DComplex&
DComplexPick::operator()(int i) Const {
  return (*V)( (*X)(i) );
}

#endif /* __DCOMPLEXVEC_H__ */
