/* dbxLib.c - UniWorks ptrace server for dbxtool */

static char *copyright = "Copyright 1987,1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This module provides facilities necessary for using dbxtool, SUN's source
language debugger, to debug target processes running under UniWorks.  The
support consists of ptrace, which works like UNIX's ptrace, and an interface
to make it available via RPC, so it can be used by a remote machine.

The service is provided by a task, dbxTask, which is spawned by dbxInit.
This module may be excluded by removing the call to dbxInit made in
usrRoot (2) in usrConfig (1).

SEE ALSO: rpcLib (1), SUN dbxtool documentation
*/

#include "UniWorks.h"
#include "wait.h"
#include "rpc.h"
#include "ptrace.h"
#include "errno.h"
#include "xdr_ptrace.h"
#include "xdr_dbx.h"
#include "sysSymTbl.h"
#include "a_out.h"
#include "dbxLib.h"
#include "ioLib.h"
#include "ldLib.h"
#include "lstLib.h"
#include "memLib.h"
#include "strLib.h"
#include "taskLib.h"
#include "fppLib.h"


#define	NUM_FP_REGS	8	/* MC68881 has 8 fp registers */

/* dbxTask parameters */

int dbxTaskId;
int dbxTaskPriority	= 20;
int dbxTaskOptions	= VX_SUPERVISOR_MODE | VX_UNBREAKABLE;
int dbxTaskStackSize	= 9000;

/* parameters for tasks run for debugging */

int dbxRunTaskPriority  = 100;
int dbxRunTaskOptions	= VX_SUPERVISOR_MODE;
int dbxRunTaskStackSize = 5000;

/* LOCAL variables */

LOCAL LIST ptraceEventList;
LOCAL SEM_ID ptraceEventListSemId;


/* forward declarations */

STATUS dbxTask ();

LOCAL struct ldfile *getLdTableEntry ();
LOCAL BOOL findModule ();
LOCAL VOID remote_dbxsvc ();
LOCAL VOID ptraceRecordBreak ();
LOCAL VOID ptraceRecordExit ();

/*******************************************************************************
*
* dbxInit - initialize UniWorks support for dbxtool
*
* This routine initializes the UniWorks support package for dbxtool, SUN's
* source language debugger. It spawns dbxTask (2).  It is normally
* called from usrRoot (2), in usrConfig (1).
*
* RETURNS
*  OK if dbxTask spawned successfully,
*  ERROR otherwise
*/

STATUS dbxInit ()

    {
    dbxTaskId = taskSpawn ("dbxTask", dbxTaskPriority, dbxTaskOptions,
			   dbxTaskStackSize,
			   dbxTask, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);

    return (dbxTaskId == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* dbxTask - dbx Server task
*
* This routine is spawned by dbxInit (2).  It creates the remote debugging
* service for dbxtool, available via RPC.
*/

VOID dbxTask ()

    {
    SVCXPRT *transp;

    if (rpcTaskInit () == ERROR)
	{
	printErr ("dbxTask: rpcTaskInit failure\n");
	return;
	}

    lstInit (&ptraceEventList);
    ptraceEventListSemId = semCreate ();
    semGive (ptraceEventListSemId);

    dbgBreakNotifyInstall (ptraceRecordBreak);

    if (taskDeleteHookAdd (ptraceRecordExit) == ERROR)
	{
	printErr ("dbxTask: failure to install ptrace delete routine\n");
	return;
	}

    transp = svctcp_create (RPC_ANYSOCK, 1024, 1024);

    if (transp == NULL)
	{
	printErr ("dbxTask: couldn't create an RPC server\n");
	return;
	}

    pmap_unset (RDBXPROG, RDBXVERS);

    if (!svc_register (transp, RDBXPROG, RDBXVERS, remote_dbxsvc, IPPROTO_TCP))
	{
	printErr ("dbxTask: couldn't register RDBX service\n");
	return;
	}

    svc_run ();

    printErr ("dbxTask: svc_run returned\n");
    }
/*******************************************************************************
*
* remote_dbxsvc - the dispatch routine for dbxTask
*
* RPC calls this dispatch routine to service the remote ptrace call.
*/

LOCAL VOID remote_dbxsvc (rqstp, transp)
    struct svc_req *rqstp;	/* type of request to service */
    SVCXPRT *transp;

    {
    static char *msgErrReply =
	"remote_dbxsvc: couldn't reply to RPC call\n";
    static char *msgErrRequest =
	"remote_dbxsvc: ptrace error %d in request %d\n";
    static char *msgErrFree =
	"remote_dbxsvc: ptrace_server: error freeing args\n";

    Rptrace ptrace_in;
    Ptrace_return ptrace_out;
    int	status;
    int pid;
    struct regs out_regs;
    struct fp_status out_fpregs;
    struct fpa_regs out_fparegs;
    C_bytes out_data;
    Arg_info in_args;
    Wait_info wait_rsp;
    struct ldtabl table;
    struct ldfile ldstruct;
    PTRACE_EVENT *thisEvent;

    switch ((int)rqstp->rq_proc)
	{
	case NULLPROC:
	    if (!svc_sendreply (transp, (xdrproc_t) xdr_void, (caddr_t) 0))
		{
		printErr (msgErrReply);
		exit (1);
		}
	    return;

	case PTRACE_PEEKTEXT:
	case PTRACE_PEEKDATA:
	case PTRACE_PEEKUSER:
	case PTRACE_POKETEXT:
	case PTRACE_POKEDATA:
	case PTRACE_POKEUSER:
	case PTRACE_CONT:
	case PTRACE_KILL:
	case PTRACE_SINGLESTEP:
	case PTRACE_ATTACH:
	case PTRACE_DETACH:

	    bzero ((char *) &ptrace_in, sizeof (ptrace_in));
	    if (!svc_getargs (transp, xdr_rptrace, &ptrace_in))
		{
		svcerr_decode (transp);
		return;
		}
	    errnoSet (0);

	    ptrace_out.status = ptrace ((int) rqstp->rq_proc, ptrace_in.pid,
		    (char *) ptrace_in.addr, ptrace_in.data, (char *) NULL);

	    if ((ptrace_out.status == -1) && (errnoGet () != 0))
		{
		/* if the procedure was PTRACE_KILL, ignore the error */
		if (rqstp->rq_proc != (int)PTRACE_KILL)
		    {
		    printErr (msgErrRequest, errnoGet (), rqstp->rq_proc);
		    perror ("remote server");
		    }
		}

	    ptrace_out.errno = errnoGet () & 0xffff;
	    ptrace_out.info.ttype = NOINFO;

	    if (!svc_sendreply (transp, (xdrproc_t) xdr_ptrace_return,
			       (caddr_t) &ptrace_out))
		{
		printErr (msgErrReply);
		exit (1);
		}
 
	    if (!svc_freeargs (transp, (xdrproc_t) xdr_rptrace,
			       (char *) &ptrace_in))
		{
		printErr (msgErrFree);
		}

	    return;

	case PTRACE_GETREGS:

	    bzero ((char *) &ptrace_in, sizeof (ptrace_in));

	    if (!svc_getargs (transp, xdr_rptrace, &ptrace_in))
		{
		svcerr_decode (transp);
		return;
		}
	    errnoSet (0);

	    ptrace_out.status = ptrace ((int) rqstp->rq_proc, ptrace_in.pid,
		    (char *) &out_regs, ptrace_in.data, (char *) NULL);

	    if ((ptrace_out.status == -1) && (errnoGet () != 0))
		{
		printErr (msgErrRequest, errnoGet (), rqstp->rq_proc);
		perror ("remote server");
		}

	    ptrace_out.errno = errnoGet () & 0xffff;
	    ptrace_out.info.ttype = REGS;
	    ptrace_out.info.more_data = (caddr_t)&out_regs;

	    if (!svc_sendreply(transp, xdr_ptrace_return, (caddr_t)&ptrace_out))
		{
		printErr (msgErrReply);
		exit (1);
		}

	    if (!svc_freeargs (transp, (xdrproc_t) xdr_rptrace,
			       (char *) &ptrace_in))
		{
		printErr (msgErrFree);
		}

	    return;

	case PTRACE_SETREGS:
	case PTRACE_SETFPREGS:
	case PTRACE_SETFPAREGS:

	    bzero ((char *) &ptrace_in, sizeof (ptrace_in));
	    if (!svc_getargs (transp, xdr_rptrace, &ptrace_in))
		{
		svcerr_decode (transp);
		return;
		}
	    errnoSet (0);

	    ptrace_out.status = ptrace ((int) rqstp->rq_proc, ptrace_in.pid,
		                        (char *) ptrace_in.info.more_data,
					ptrace_in.data, (char *) NULL);

	    if ((ptrace_out.status == -1) && (errnoGet () != 0))
		{
		printErr (msgErrRequest, errnoGet (), rqstp->rq_proc);
		perror ("remote server");
		}

	    ptrace_out.errno = errnoGet () & 0xffff;
	    ptrace_out.info.ttype = NOINFO;

	    if (!svc_sendreply(transp, xdr_ptrace_return, (caddr_t)&ptrace_out))
		{
		printErr (msgErrReply);
		exit (1);
		}
 
	    if (!svc_freeargs (transp, (xdrproc_t) xdr_rptrace,
			       (char *) &ptrace_in))
		{
		printErr (msgErrFree);
		}

	    return;

	case PTRACE_GETFPREGS:

	    bzero ((char *) &ptrace_in, sizeof (ptrace_in));
	    if (!svc_getargs (transp, xdr_rptrace, &ptrace_in))
		{
		svcerr_decode (transp);
		return;
		}
	    errnoSet (0);

	    ptrace_out.status = ptrace ((int) rqstp->rq_proc, ptrace_in.pid,
		    (char *) &out_fpregs, ptrace_in.data, (char *) NULL);

	    if ((ptrace_out.status == -1) && (errnoGet () != 0))
		{
		printErr (msgErrRequest, errnoGet (), rqstp->rq_proc);
		perror ("remote server");
		}

	    ptrace_out.errno = errnoGet () & 0xffff;
	    ptrace_out.info.ttype = FPREGS;
	    ptrace_out.info.more_data = (caddr_t)&out_fpregs;

	    if (!svc_sendreply(transp, xdr_ptrace_return, (caddr_t)&ptrace_out))
		{
		printErr (msgErrReply);
		exit (1);
		}
 
	    if (!svc_freeargs (transp, (xdrproc_t) xdr_rptrace,
			       (char *) &ptrace_in))
		{
		printErr (msgErrFree);
		}

	    return;

	case PTRACE_GETFPAREGS:

	    bzero ((char *) &ptrace_in, sizeof (ptrace_in));
	    if (!svc_getargs (transp, xdr_rptrace, &ptrace_in))
		{
		svcerr_decode (transp);
		return;
		}
	    errnoSet (0);

	    ptrace_out.status = ptrace ((int) rqstp->rq_proc, ptrace_in.pid,
		    (char *) &out_fparegs, ptrace_in.data, (char *) NULL);

	    if ((ptrace_out.status == -1) && (errnoGet () != 0))
		{
		printErr (msgErrRequest, errnoGet (), rqstp->rq_proc);
		perror ("remote server");
		}

	    ptrace_out.errno = errnoGet () & 0xffff;
	    ptrace_out.info.ttype = FPAREGS;
	    ptrace_out.info.more_data = (caddr_t)&out_fparegs;

	    if (!svc_sendreply(transp, xdr_ptrace_return, (caddr_t)&ptrace_out))
		{
		printErr (msgErrReply);
		exit (1);
		}
 
	    if (!svc_freeargs (transp, (xdrproc_t) xdr_rptrace,
			       (char *) &ptrace_in))
		{
		printErr (msgErrFree);
		}

	    return;

	case PTRACE_READDATA:
	case PTRACE_READTEXT:

	    bzero ((char *) &ptrace_in, sizeof (ptrace_in));
	    if (!svc_getargs (transp, xdr_rptrace, &ptrace_in))
		{
		svcerr_decode (transp);
		return;
		}
	    errnoSet (0);

	    out_data.bytes = (caddr_t) malloc ((unsigned) ptrace_in.data);
	    out_data.len = ptrace_in.data;

	    ptrace_out.status = ptrace ((int) rqstp->rq_proc, ptrace_in.pid,
		    (char *) ptrace_in.addr, ptrace_in.data,
		    (char *) out_data.bytes);

	    if ((ptrace_out.status == -1) && (errnoGet () != 0))
		{
		printErr (msgErrRequest, errnoGet (), rqstp->rq_proc);
		perror ("remote server");
		}

	    ptrace_out.errno = errnoGet () & 0xffff;
	    ptrace_out.info.ttype = DATA;
	    ptrace_out.info.more_data = (caddr_t)&out_data;

	    if (!svc_sendreply(transp, xdr_ptrace_return, (caddr_t)&ptrace_out))
		{
		printErr (msgErrReply);
		exit (1);
		}

	    if (!svc_freeargs (transp, (xdrproc_t) xdr_rptrace,
			       (char *) &ptrace_in))
		{
		printErr (msgErrFree);
		}

	    return;

	case PTRACE_WRITEDATA:
	case PTRACE_WRITETEXT:

	    bzero ((char *) &ptrace_in, sizeof (ptrace_in));
	    if (!svc_getargs (transp, xdr_rptrace, &ptrace_in))
		{
		svcerr_decode (transp);
		return;
		}
	    errnoSet (0);

	    ptrace_out.status = ptrace ((int) rqstp->rq_proc,
		    ptrace_in.pid, (char *) ptrace_in.addr, ptrace_in.data,
		    (char *) ((C_bytes *)ptrace_in.info.more_data)->bytes);

	    if ((ptrace_out.status == -1) && (errnoGet () != 0))
		{
		printErr (msgErrRequest, errnoGet (), rqstp->rq_proc);
		perror ("remote server");
		}

	    ptrace_out.errno = errnoGet () & 0xffff;
	    ptrace_out.info.ttype = NOINFO;

	    if (!svc_sendreply(transp, xdr_ptrace_return, (caddr_t)&ptrace_out))
		{
		printErr (msgErrReply);
		exit (1);
		}

	    if (!svc_freeargs (transp, (xdrproc_t) xdr_rptrace,
			       (char *) &ptrace_in))
		{
		printErr (msgErrFree);
		}
 
	    return;

	case PROCESS_START:

	    bzero ((char *) &in_args, sizeof (in_args));
	    if (!svc_getargs (transp, xdr_arg_info, &in_args))
		{
		svcerr_decode (transp);
		return;
		}

	    pid = ptraceStart (in_args.rargc, in_args.rargv);

	    if (!svc_sendreply (transp, xdr_int, (caddr_t) &pid))
		{
		printErr (msgErrReply);
		exit (1);
		}

	    /* WARNING: DO NOT CALL svc_freeargs with xdr_arg_info!
	       It will try to free a static array */

	    return;

	case PROCESS_WAIT:

	    if (!svc_getargs (transp, xdr_int, &pid))
		{
		svcerr_decode (transp);
		return;
		}

	    wait_rsp.pid = wait3 (&wait_rsp.status, pid);

	    if (!svc_sendreply (transp, xdr_wait_info, (caddr_t) &wait_rsp))
		{
		printErr (msgErrReply);
		exit (1);
		}

	    return;

	case VX_STATE_INQ:

	    /* don't really need svc_getargs as input is void */

	    makeModTable (&table);

	    if (!svc_sendreply (transp, xdr_ldtabl, (caddr_t) &table))
		{
		printErr (msgErrReply);
		exit (1);
		}

	    return;

	case VX_FP_INQUIRE:

	    /* don't really need svc_getargs as input is void */
	    {
	    BOOL fppPresent;

	    if (fppProbe () == OK)
		fppPresent = TRUE;
	    else
		fppPresent = FALSE;

	    if (!svc_sendreply (transp, xdr_bool, (caddr_t) &fppPresent))
		{
		printErr (msgErrReply);
		exit (1);
		}

	    return;
	    }
	

	case VX_LOAD:
	    {
	    char *filename = NULL;
	    char *pText    = LD_NO_ADDRESS;
	    char *pData    = LD_NO_ADDRESS;
	    char *pBss     = LD_NO_ADDRESS;
	    char *strIndex;
	    int fd;

	    if (!svc_getargs (transp, xdr_wrapstring, &filename))
		{
		svcerr_decode (transp);
		return;
		}

	    fd = open (filename, READ);

	    if (fd == ERROR ||
		ldLoadModule (fd, ALL_SYMBOLS, &pText, &pData, &pBss) == ERROR)
	       {
	       ldstruct.name      = (char *) malloc ((unsigned) 1);
	       *ldstruct.name     = NULL;
	       ldstruct.txt_addr  = NULL;
	       ldstruct.data_addr = NULL;
	       ldstruct.bss_addr  = NULL;
	       }
	    else
	       {
	       /* chop the path (if any) off of the module name */

	       if ((strIndex = (char *) rindex (filename, '/')) != NULL)
		   strcpy (filename, (char *) (strIndex + 1));
	       else
		   if ((strIndex = (char *) index (filename, ':')) != NULL)
		       strcpy (filename, (char *) (strIndex + 1));

	       ldstruct.name = (char *) malloc ((unsigned) (strlen (filename)
	       						    + 10));
	       strcpy (ldstruct.name, filename);
	       ldstruct.txt_addr  = (int) pText;
	       ldstruct.data_addr = (int) pData;
	       ldstruct.bss_addr  = (int) pBss;
	       }

	    close (fd);

	    if (!svc_sendreply (transp, xdr_ldfile, (caddr_t) &ldstruct))
		{
		printErr (msgErrReply);
		exit (1);
		}
 
	    if (!svc_freeargs (transp, (xdrproc_t) xdr_wrapstring,
			       (char *) &filename))
		{
		printErr (msgErrFree);
		}
 
	    return;
	    }

	case VX_SYMBOL_INQ:
	    {
	    char *address;
	    UTINY type;
	    char *name = NULL;

	    if (!svc_getargs (transp, xdr_wrapstring, &name))
		{
		svcerr_decode (transp);
		return;
		}

	    if (symFind (sysSymTbl, name, &address, &type) == ERROR)
		status = ERROR;
	    else
		status = (int) address;

	    if (!svc_sendreply (transp, xdr_int, (caddr_t) &status))
		{
		printErr (msgErrReply);
		exit (1);
		}
 
	    if (!svc_freeargs (transp, (xdrproc_t) xdr_wrapstring,
			       (char *) &name))
		{
		printErr (msgErrFree);
		}

	    return;
	    }

	case VX_BREAK_ADD:

	    bzero ((char *) &ptrace_in, sizeof (ptrace_in));
	    if (!svc_getargs (transp, xdr_rptrace, &ptrace_in))
		{
		svcerr_decode (transp);
		return;
		}

	    status = b ((INSTR *)ptrace_in.addr, ptrace_in.pid, 0, TRUE);

	    if (!svc_sendreply (transp, xdr_int, (caddr_t) &status))
		{
		printErr (msgErrReply);
		exit (1);
		}
 
	    if (!svc_freeargs (transp, (xdrproc_t) xdr_rptrace,
			       (char *) &ptrace_in))
		{
		printErr (msgErrFree);
		}

	    return;

	case VX_BREAK_DELETE:

	    bzero ((char *) &ptrace_in, sizeof (ptrace_in));
	    if (!svc_getargs (transp, xdr_rptrace, &ptrace_in))
		{
		svcerr_decode (transp);
		return;
		}

	    status = bd ((INSTR *)ptrace_in.addr, ptrace_in.pid);

	    if (!svc_sendreply (transp, xdr_int, (caddr_t) &status))
		{
		printErr (msgErrReply);
		exit (1);
		}
 
	    if (!svc_freeargs (transp, (xdrproc_t) xdr_rptrace,
			       (char *) &ptrace_in))
		{
		printErr (msgErrFree);
		}

	    return;
	
	case VX_TASK_SUSPEND:

	    bzero ((char *) &ptrace_in, sizeof (ptrace_in));
	    if (!svc_getargs (transp, xdr_rptrace, &ptrace_in))
		{
		svcerr_decode (transp);
		return;
		}

	    pid = ptrace_in.pid;
	    status = taskSuspend (ptrace_in.pid);

	    if (!svc_sendreply (transp, xdr_int, (caddr_t) &status))
		{
		printErr (msgErrReply);
		exit (1);
		}

	    if (!svc_freeargs (transp, (xdrproc_t) xdr_rptrace,
			       (char *) &ptrace_in))
		{
		printErr (msgErrFree);
		}

	    /* tell dbx next time it calls wait3 */

	    thisEvent = (PTRACE_EVENT *) malloc ((unsigned)
						 sizeof (*thisEvent));
	    if (thisEvent == NULL)
		{
		logMsg ("VX_TASK_SUSPEND: OUT OF MEMORY\n");
		return;
		}

	    thisEvent->taskId = pid;
	    thisEvent->eventType = EVENT_SUSPEND;

 	    semTake (ptraceEventListSemId);
	    lstAdd (&ptraceEventList, (NODE *) thisEvent);
 	    semGive (ptraceEventListSemId);
	    return;


	default:
	    printErr ("remote_dbxsvc: invalid request (%d)\n", rqstp->rq_proc);
	    svcerr_noproc (transp);
	    return;
	}
    }
/*******************************************************************************
*
* ptraceStart - create a new task
*
* ptraceStart parses an argument list, spawns a new task, and immediately
* suspends it.
*
* RETURNS: pid | ERROR
*/

LOCAL STATUS ptraceStart (nArgs, args)
    int nArgs;
    char *args[];	/* arg0 is function name, up to MAX_TASK_ARGS args */

    {
    int pid;
    UTINY type;
    FUNCPTR address;
    PTRACE_EVENT *thisEvent;
    int argValue [MAX_TASK_ARGS+1];
    int i;

    /* find the routine in the system symbol table */

    if (symFind (sysSymTbl, args [0], (char **)&address, &type) == ERROR)
	{
	/* prepend a '_' to the symbol and look again */
	char *tmpString = (char *) malloc ((unsigned) (strlen (args[0]) + 5));
	tmpString[0] = '_';
	(VOID) strcpy (&tmpString[1], args[0]);
	if (symFind (sysSymTbl, tmpString, (char **)&address, &type) ==
									ERROR)
	    {
	    free (tmpString);
	    return (ERROR);
	    }
	free (tmpString);
	}

    if ((type & N_TYPE) != N_TEXT)
	return (ERROR);

    bzero ((char *) argValue, sizeof (argValue));

    /* parse the arguments - we support only decimal or hex (0x...) constants,
       string constants ("..."), and references to symbols in the
       system symbol table */

    for (i = 1; i < nArgs && i < MAX_TASK_ARGS + 1; i++)
	{
	/* is it a hex constant */

	if ((args[i][0] == '0') && (args[i][1] == 'x'))
	    {
	    if (sscanf (&args[i][2], "%x", &argValue[i]) != 1)
		return (ERROR);
	    continue;
	    }

	/* see if we can parse it as a decimal constant */

	if (sscanf (args[i], "%d", &argValue[i]) == 1)
	    continue;

	/* is it a string constant ? */
	if (args[i][0] == '"')
	    {
	    if (args[i][strlen (args[i]) - 1] != '"')
		return (ERROR);

	    argValue[i] = (int) malloc ((unsigned) (strlen (args[i]) + 10));
	    strcpy ((char *) argValue[i], &args[i][1]);

	    /* mask out the ending quote */
	    ((char *) argValue[i]) [strlen ((char *)argValue[i]) - 1] = NULL;

	    continue;
	    }

	/* look it up in the symbol table */

	if (symFind (sysSymTbl, args[i], (char **)&argValue[i], &type) ==
									ERROR)
	    {
	    /* prepend a '_' to the symbol and look again */
	    char *tmpString = (char *) malloc ((unsigned) (strlen (args[i])
							   + 5));
	    tmpString[0] = '_';
	    (VOID) strcpy (&tmpString[1], args[i]);
	    if (symFind (sysSymTbl, tmpString, (char **) &argValue[i],
			       &type) == ERROR)
		{
		free (tmpString);
		return (ERROR);
		}
	    free (tmpString);
	    continue;
	    }
	}

    pid = taskSpawn ("dbxRun", dbxRunTaskPriority, dbxRunTaskOptions,
		     dbxRunTaskStackSize, address,
		     argValue[1], argValue[2], argValue[3], argValue[4],
		     argValue[5], argValue[6], argValue[7], argValue[8],
		     argValue[9], argValue[10]);

    taskSuspend (pid);

    /* make sure there are no events in the event list with this taskId */

    deleteEvents (pid);

    /* tell dbx next time it calls wait3 */

    if ((thisEvent = (PTRACE_EVENT *) malloc ((unsigned) sizeof (*thisEvent)))
	== NULL)
	{
	logMsg ("ptraceStart: OUT OF MEMORY\n");
	return (pid);
	}

    thisEvent->taskId = pid;
    thisEvent->eventType = EVENT_STOP;

    semTake (ptraceEventListSemId);
    lstAdd (&ptraceEventList, (NODE *) thisEvent);
    semGive (ptraceEventListSemId);
    return (pid);
    }
/*******************************************************************************
*
* ptrace - UNIX ptrace
*
* ptrace is used to debug a process.
* See the UNIX ptrace documentation for further details.
*
* RETURNS: OK or ERROR
*/

LOCAL STATUS ptrace (request, pid, addr, data, addr2)
    int request;	/* type of request */
    int pid;		/* id of task to act upon */
    char *addr;		/* request-dependent parameter */
    int data;		/* request-dependent parameter */
    char *addr2;	/* request-dependent parameter */

    {
    int status;

    switch (request)
	{
	case PTRACE_PEEKTEXT:
        case PTRACE_PEEKDATA:
        case PTRACE_PEEKUSER:
	    if (vxMemProbe (addr, READ, 4, (char *) &status) == ERROR)
		{
		status = ERROR;
		netErrnoSet (EFAULT);
		}
	    break;

        case PTRACE_POKETEXT:
       	case PTRACE_POKEDATA:
        case PTRACE_POKEUSER:
	    if ((status = vxMemProbe (addr, WRITE, 4, (char *) &data)) == ERROR)
		netErrnoSet (EFAULT);
	    break;

        case PTRACE_CONT:
	    if (addr != (char *) 1)
		{
		panic ("PTRACE_CONT: addr != 1\n");
		status = ERROR;
		netErrnoSet (EFAULT);
		}
	    else
		{
		status = taskResume (pid);
		if (status == ERROR)
		    netErrnoSet (ESRCH);
		}
	    break;

        case PTRACE_KILL:
	    status = taskDelete (pid);

	    /* delete all the events generated by the task */

	    deleteEvents (pid);

	    if (status == ERROR)
		netErrnoSet (ESRCH);
	    break;


        case PTRACE_SINGLESTEP:
	    if ((status = dbgStepQuiet (pid)) == ERROR)
		netErrnoSet (ESRCH);
	    break;

        case PTRACE_ATTACH:
        case PTRACE_DETACH:
	    status = OK;
	    break;

	case PTRACE_GETREGS:
	    {
	    struct regs *myRegs = (struct regs *) addr;

	    bzero ((char *) myRegs, sizeof (struct regs));

	    if ((taskIdVerify (pid)) == OK)
		{
		short sr;
		taskRegsGet (pid, &myRegs->r_dreg[0], &myRegs->r_areg[0],
			     (char **) &myRegs->r_sp, (USHORT *) &sr,
			     (INSTR **) &myRegs->r_pc);
		myRegs->r_sr = sr;
		status = OK;
		}
	    else
		{
		status = ERROR;
		netErrnoSet (ESRCH);
		}
	    break;
	    }

	case PTRACE_SETREGS:
	    {
	    struct regs *myRegs = (struct regs *) addr;
	    
	    if ((taskIdVerify (pid)) == OK)
		{
		taskRegsSet (pid, &myRegs->r_dreg[0], &myRegs->r_areg[0],
			       (char *) myRegs->r_sp, (USHORT) myRegs->r_sr,
			       (INSTR *) myRegs->r_pc);
		status = OK;
		}
	    else
		{
		status = ERROR;
		netErrnoSet (ESRCH);
		}
	    break;
	    }

	case PTRACE_GETFPREGS:
	    {
	    double fpregs [NUM_FP_REGS];
	    int i;
	    struct fp_status *myFpRegs = (struct fp_status *) addr;

	    bzero ((char *) myFpRegs, sizeof (struct fp_status));
	    
	    if ((taskIdVerify (pid)) == OK)
		{
		(void) fppTaskRegsGet (pid, fpregs, &myFpRegs->fps_control,
			        &myFpRegs->fps_status, &myFpRegs->fps_iaddr);

		/* convert from double to 96 bit */

		for (i = 0; i < NUM_FP_REGS; i++)
		    fppDtoDx ((DOUBLEX *) &myFpRegs->fps_regs[i], &fpregs[i]);

		status = OK;
		}
	    else
		{
		status = ERROR;
		netErrnoSet (ESRCH);
		}
	    break;
	    }

	case PTRACE_SETFPREGS:
	    {
	    double fpregs [NUM_FP_REGS];
	    int i;
	    struct fp_status *myFpRegs = (struct fp_status *) addr;

	    /* convert from 96-bit to double */

	    for (i = 0; i < NUM_FP_REGS; i++)
		fppDxtoD (&fpregs[i], (DOUBLEX *) &myFpRegs->fps_regs[i]);
	    
	    if ((taskIdVerify (pid)) == OK)
		{
		(void) fppTaskRegsSet (pid, fpregs, myFpRegs->fps_control,
				       myFpRegs->fps_status,
				       myFpRegs->fps_iaddr);
		status = OK;
		}
	    else
		{
		status = ERROR;
		netErrnoSet (ESRCH);
		}
	    break;
	    }

	case PTRACE_READDATA:
	case PTRACE_READTEXT:

	    /* check the boundries */
	    if (vxMemProbe (addr, READ, 4, (char *) &status) != ERROR &&
		vxMemProbe (addr + data, READ, 4, (char *) &status) != ERROR)
		{
		bcopy (addr, addr2, data);
		status = OK;
		}
	    else
		{
		status = ERROR;
		netErrnoSet (EFAULT);
		}
	    break;

	case PTRACE_WRITEDATA:
	case PTRACE_WRITETEXT:

	    if (vxMemProbe (addr, READ, 4, (char *) &status) != ERROR &&
		vxMemProbe (addr + data, READ, 4, (char *) &status) != ERROR)
		{
		bcopy (addr2, addr, data);
		status = OK;
		}
	    else
		{
		status = ERROR;
		netErrnoSet (EFAULT);
		}
	    break;

	default:
	    status = ERROR;
	}

    return (status);
    }
/*******************************************************************************
*
* ptraceRecordBreak - record a breakpoint
*
* ptraceRecordBreak inserts a breakpoint event in the ptrace event list.
*/

LOCAL VOID ptraceRecordBreak (task)
    int task;

    {
    FAST PTRACE_EVENT *thisEvent;

    if ((thisEvent = (PTRACE_EVENT *) malloc ((unsigned) sizeof (*thisEvent)))
	== NULL)
	{
	logMsg ("ptraceRecordBreak: OUT OF MEMORY\n");
	return;
	}

    thisEvent->taskId = task;
    thisEvent->eventType = EVENT_BREAK;

    semTake (ptraceEventListSemId);
    lstAdd (&ptraceEventList, (NODE *) thisEvent);
    semGive (ptraceEventListSemId);
    }
/*******************************************************************************
*
* ptraceRecordExit - record a task exit event
*
* ptraceRecordExit inserts an exit event in the ptrace event list.
*/

LOCAL VOID ptraceRecordExit (pTcbX)
    TCBX *pTcbX;

    {
    FAST PTRACE_EVENT *thisEvent;

    if ((thisEvent = (PTRACE_EVENT *) malloc ((unsigned) sizeof (*thisEvent)))
	== NULL)
	{
	logMsg ("ptraceRecordBreak: OUT OF MEMORY\n");
	return;
	}

    thisEvent->taskId = pTcbX->taskId;
    thisEvent->eventType = EVENT_EXIT;

    semTake (ptraceEventListSemId);
    lstAdd (&ptraceEventList, (NODE *) thisEvent);
    semGive (ptraceEventListSemId);
    }
/*******************************************************************************
*
* wait3 - inquire task status
*
* wait3 examines the ptrace event list to see if a given task has
* done something interesting.
*
* RETURNS: taskid of interesting task or NULL
*/

LOCAL int wait3 (status, pid)
    union wait *status;
    int pid;		/* process we're interested in */

    {
    FAST PTRACE_EVENT *thisEvent;
    PTRACE_EVENT pidEvent;
    BOOL eventFound = FALSE;

    /* see if there's been any breakpoints or exceptions for the given pid */
    /* discard all but last event associated with pid			   */

    semTake (ptraceEventListSemId);

    for (thisEvent = (PTRACE_EVENT *) lstFirst (&ptraceEventList);
	 (thisEvent != NULL);
	 thisEvent = (PTRACE_EVENT *) lstNext ((NODE *)thisEvent)) 
	 {
	 if (thisEvent->taskId == pid) 
	    {
	    pidEvent = *thisEvent;
	    eventFound = TRUE;
	    lstDelete (&ptraceEventList, (NODE *) thisEvent);
 	    semGive (ptraceEventListSemId);
	    free ((char *) thisEvent);
	    }
	}

    semGive (ptraceEventListSemId);

    if (!eventFound)
	return (0);

    switch (pidEvent.eventType)
	{
	case EVENT_BREAK:
	    status->w_stopval = WSTOPPED;
	    status->w_stopsig = SIGTRAP; 
	    break;

	case EVENT_STOP:
	    status->w_stopval = WSTOPPED;
	    status->w_stopsig = SIGCHLD; 
	    break;

	case EVENT_SUSPEND:
	    status->w_stopval = WSTOPPED;
	    status->w_stopsig = SIGINT; 
	    break;

	case EVENT_EXIT:
	    status->w_stopval = NULL;
	    status->w_stopsig = NULL; 
	    break;

	default:
	    printf ("wait3: unknown event type\n");
	    return (NULL);
	}

    return (pid);
    }
/*******************************************************************************
*
* makeModTable - make a table of all loaded modules
*
* makeModTable constructs a table of all modules in the symbol table
*/

LOCAL VOID makeModTable (pLdTable)
    struct ldtabl *pLdTable;

    {
    pLdTable->tbl_size = 0;
    pLdTable->tbl_ent = NULL;

    symEach (sysSymTbl, findModule, pLdTable);
    }
/*******************************************************************************
*
* findModule - find a module name in the symbol table.
*
* findModule is used by makeModTable to construct a module table.
*
* ARGSUSED
*/

LOCAL BOOL findModule (name, val, type, pLdTable)
    FAST char *name;
    int val;
    UTINY type;
    FAST struct ldtabl *pLdTable;

    {
    FAST char *modName;
    FAST int nameLen;
    struct ldfile *thisLdTableEntry;

    if ((nameLen = substr (name, "_text")) != ERROR)
	{
	modName = (char *) malloc ((unsigned) (nameLen + 10));
	strncpy (modName, name, nameLen);
	modName [nameLen] = NULL;

	thisLdTableEntry = getLdTableEntry (pLdTable, modName);
	if ((int) thisLdTableEntry == ERROR)
	    {
	    logMsg ("findModule: couldn't contstuct ldTable\n");
	    return (FALSE);
	    }

	if (thisLdTableEntry->txt_addr == NULL)
	    thisLdTableEntry->txt_addr = val;

	return (TRUE);
	}

    if ((nameLen = substr (name, "_data")) != ERROR)
	{
	modName = (char *) malloc ((unsigned) (nameLen + 10));
	strncpy (modName, name, nameLen);
	modName [nameLen] = NULL;

	thisLdTableEntry = getLdTableEntry (pLdTable, modName);
	if ((int) thisLdTableEntry == ERROR)
	    {
	    logMsg ("findModule: couldn't contstuct ldTable\n");
	    return (FALSE);
	    }

	if (thisLdTableEntry->data_addr == NULL)
	    thisLdTableEntry->data_addr = val;

	return (TRUE);
	}

    if ((nameLen = substr (name, "_bss")) != ERROR)
	{
	modName = (char *) malloc ((unsigned) (nameLen + 10));
	strncpy (modName, name, nameLen);
	modName [nameLen] = NULL;

	thisLdTableEntry = getLdTableEntry (pLdTable, modName);
	if ((int) thisLdTableEntry == ERROR)
	    {
	    logMsg ("findModule: couldn't contstuct ldTable\n");
	    return (FALSE);
	    }

	if (thisLdTableEntry->bss_addr == NULL)
	    thisLdTableEntry->bss_addr = val;

	return (TRUE);
	}

    return (TRUE);
    }
/*******************************************************************************
*
* getLdTableEntry - get Load Table Entry
*/

LOCAL struct ldfile *getLdTableEntry (pLdTable, modName)
    FAST struct ldtabl *pLdTable;
    char *modName;

    {
    FAST int i;

    for (i = 0; i < pLdTable->tbl_size; i++)
	{
	if (strcmp (pLdTable->tbl_ent[i].name, modName) == 0)
	    {
	    free (modName);
	    return (&pLdTable->tbl_ent[i]);
	    }
	}

    /* not found - make the array bigger */

    if (pLdTable->tbl_size == 0)
	pLdTable->tbl_ent = (struct ldfile *) malloc ((unsigned)
						      sizeof (struct ldfile));
    else
	pLdTable->tbl_ent =
	    (struct ldfile *) 
	    realloc ((char *) pLdTable->tbl_ent,
		     (unsigned) ((pLdTable->tbl_size + 1) *
				 sizeof (struct ldfile)));

    bzero ((char *) &pLdTable->tbl_ent[pLdTable->tbl_size],
	   sizeof (struct ldfile));

    pLdTable->tbl_ent[pLdTable->tbl_size].name = modName;
    pLdTable->tbl_size++;

    return (&pLdTable->tbl_ent[pLdTable->tbl_size - 1]);
    }
/*******************************************************************************
*
* substr - test if substring is a substring of string
*
* RETURNS:
*    index into string where substring begins, or
*    ERROR if not found
*/

LOCAL int substr (string, substring)
    FAST char *string;
    FAST char *substring;

    {
    FAST int strDif = strlen (string) - strlen (substring);
    FAST int i;

    for (i = 0; i <= strDif; i++)
	{
	if (strcmp (string++, substring) == 0)
	    return (i);
	}

    return (ERROR);
    }
/*******************************************************************************
*
* deleteEvents - delete all the events for a given task
*
* This routine is called to insure that all the events generated
* by a given task are removed from the ptrace event list.
* (Specifically when a task is deleted, and when a new task
* is started up to insure a clean start).
*/

LOCAL VOID deleteEvents (taskId)
    int taskId;

    {
    FAST PTRACE_EVENT *thisEvent;

    semTake (ptraceEventListSemId);

    for (thisEvent = (PTRACE_EVENT *) lstFirst (&ptraceEventList);
	 (thisEvent != NULL);
	 thisEvent = (PTRACE_EVENT *) lstNext ((NODE *)thisEvent)) 
	 {
	 if (thisEvent->taskId == taskId) 
	    {
	    lstDelete (&ptraceEventList, (NODE *) thisEvent);
	    free ((char *) thisEvent);
	    }
	 }

    semGive (ptraceEventListSemId);
    }
/*******************************************************************************
*
* dbxInterrupt - suspend a task running under dbx
*
* dbxWorks currently does not implement '^C' properly - hitting a control C
* in the dbxWorks window will not cause the running task to suspend.
* This routine sends a "SIGINT" to dbxWorks which will cause it to suspend
* the task it is running, and fall into its command interpreter.
*/

VOID dbxInterrupt (taskId)
    int taskId;		/* task to be interrupted */
    {

    FAST PTRACE_EVENT *thisEvent;

    taskSuspend (taskId);

    thisEvent = (PTRACE_EVENT *) malloc ((unsigned) sizeof (*thisEvent));

    if (thisEvent == NULL)
	{
	logMsg ("VX_TASK_SUSPEND: OUT OF MEMORY\n");
	return;
	}

    thisEvent->taskId = taskId;
    thisEvent->eventType = EVENT_SUSPEND;

    semTake (ptraceEventListSemId);
    lstAdd (&ptraceEventList, (NODE *) thisEvent);
    semGive (ptraceEventListSemId);
    }
