/* tyCoDrv.c - The Microbar DBC68K2 USART tty handler */

static char *copyright = "Copyright 1984-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
04i,30may88,dnw  changed to v4 names.
04h,11feb88,gae  clean up.
04g,08feb88,dpg  added FIOBAUDRADE ioctl ability to tyCoIoctl - with caveat
		 that BOTH tyCo channels get changed at once
04f,25mar87,jlf  documentation
04e,21dec86,dnw  changed to not get include files from default directories.
04d,03mar86,jlf  changed tyIoctrl to tyIoctl.
04c,06aug85,jlf  removed EXITTYPE.  added copyright. Removed include
                 of intLib.h.
03g,11sep84,jlf  removed GLOBAL
03f,09aug84,ecs  I couldn't help myself, but there's probably very little
		   damage done.
		 Changed to install tyCoOpen as the create routine as well
		   as the open routine to allow "create" calls to work.
03e,02aug84,dnw  Changed include of vxLib.h to intLib.h.
		 Changed vxInt... to int...
03d,24jul84,ecs  Appeased lint by including memLib.h.
03c,23jul84,ecs  Removed dependency on hwConfig.h.
03b,11jul84,ecs  Changed calls to vxIntLock/vxIntUnlock to restore level.
03a,22may84,dnw  Changed to work with new i/o system.
		 Changed to work with new tyLib.
02f,16sep83,dnw  Changed tyCoStartup to return semaphore pointer.
02e,09aug83,dnw  Added create and delete args to drvTskLevel call.
02d,28jul83,dnw  Added connection to MAJ_SERIAL.
		 Changed to EOI at end of interrupt processing instead of
		   beginning, which seems to have cured various hang-ups.
		 Changed initialization to use tyDevInit in tyLib.
		   Note that all channels are now initialized to RAW mode,
		   and that vxshell sets terminal mode and smpteMstrTsk
		   sets smpte mode for their respective devices via ioctrl.
		   Also note that buffers are allocated from the memory pool.
02c,22jul83,ecs  Added initialization of options & status.
		 Added tyCoiEx, external interrupt handler.
02b,21jul83,jlf  Increased size of WT_BUF_SIZE_0 from 32 to 64.
02a,12jul83,ecs  changed to use tyLib.
01a,18may83,jlf  Written, by modifying ty534Drv.c,v01h.
*/

/*
DESCRIPTION
This is the driver for the Microbar DBC68K2 on-board serial ports.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv () to
initialize the driver, and tyCoDevCreate () to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:

.CS
    tyCoDrv ()
.CE

This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS

    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device *
	int channel;	/* Physical channel for this device (0 or 1) *
	int rdBufSize;	/* Read buffer size, in bytes *
	int wrtBufSize;	/* Write buffer size, in bytes *

.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:

.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);

.CE

IOCTL
This driver responds to the same ioctl codes as a normal ty driver.
When using the FIOBAUDRATE code, be aware that the Microbar USARTs
share a common timer so changing the baudrate on one port changes
it for the other.
*/

#include "UniWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "wdLib.h"
#include "tyLib.h"
#include "dbc68k.h"

#define IV_CO		((FUNCPTR *) 0x0120)	/* DBC68K2 on-board ports */

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    char *uart_ptr;
    char *uart_stat;
    } TY_CO_DEV;

typedef struct	/* BAUD */
    {
    int rate;
    USHORT pitValue;
    } BAUD;

LOCAL BAUD baudTable[] =
    {
    {75, 1024}, {110, 698}, {150, 512}, {300, 256}, {600, 128},
    {1200, 64}, {2400, 32}, {4800, 16}, {9600, 8}, {19200, 4},
    {38400, 2},
    };


LOCAL TY_CO_DEV *tyCoDv [N_USART_CHANNELS];	/* device descriptors */
LOCAL int tyCoDrvNum;		/* driver number assigned to this driver */


/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL int tyCoRead ();
LOCAL int tyCoWrite ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID tyCoInt ();


/*******************************************************************************
*
* tyCoDrv - install ty console driver
*
* This routine initializes the Microbar serial driver.  It should be
* called exactly once, before any other serial driver routines.  It
* installs the driver, and sets up hardware, but does not create any
* devices.
*
* RETURNS: OK or ERROR
*/

STATUS tyCoDrv ()

    {
    int ix;

    if (tyCoDrvNum > 0)
	return (OK);	/* driver already installed */

    /* set up 8 interrupt vectors to go to tyCoInt with interrupt number.
     * tyCoHrdInit sets up the 8274 to interrupt with 'status affects vector'
     * to these vectors */

    for (ix = 0; ix < 8; ix++)
	intConnect (IV_CO + ix, tyCoInt, ix);

    tyCoHrdInit ();

    tyCoDrvNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen,
				(FUNCPTR) NULL, tyCoRead, tyCoWrite, tyCoIoctl);

    return (tyCoDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device for the specified serial port. 
* The baud rate defaults to 9600.
*
* RETURNS: OK or ERROR
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* name of device */
    int channel;	/* port on which to create device */
    int rdBufSize;	/* read buffer size */
    int wrtBufSize;	/* write buffer size */

    {
    STATUS status;
    TY_CO_DEV *pTyCoDv;

    if (tyCoDrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}

    if ((pTyCoDv = (TY_CO_DEV *) malloc (sizeof (TY_CO_DEV))) == NULL)
	return (ERROR);

    /* initialize device descriptor */

    status = tyDevInit ((TY_DEV_ID) pTyCoDv, 
			rdBufSize, wrtBufSize, tyCoStartup);

    if (status != OK)
	return (ERROR);

    pTyCoDv->uart_ptr  = (channel == 0) ? DBC_UD_A : DBC_UD_B;
    pTyCoDv->uart_stat = (channel == 0) ? DBC_US_A : DBC_US_B;

    tyCoDv [channel] = pTyCoDv;

    if (iosDevAdd ((DEV_HDR *) pTyCoDv, name, tyCoDrvNum) == ERROR)
	return (ERROR);

    tyCoIoctl (pTyCoDv, FIOBAUDRATE, 9600);

    return (OK);
    }
/*******************************************************************************
*
* tyCoHrdInit - initialize the USART
*
* This routine initializes the Microbar on-board USART.
* The initialization is performed as suggested by the VRTX
* package, though the terminal characteristics are set to be the
* same as for the Microbar UNIX system.
*/

LOCAL VOID tyCoHrdInit ()

    {
    int i;
    int oldlevel;		/* current interrupt level mask */

    oldlevel = intLock ();

    /* Reset the USART, both channels, waiting a bit after each. */

    *DBC_US_A = DBC_U_CHAN_RST;
    for (i=0; i<250; i++)
	;
    *DBC_US_B = DBC_U_CHAN_RST;
    for (i=0; i<250; i++)
	;

    /* Set up the USARTs.  Both channels are set up for x16 clock, no parity,
       8 data bits, 1 stop bit. DTR and RTS are set active (low). */

    *DBC_US_A = DBC_U_WR4;
    *DBC_US_A = DBC_U_X16 | DBC_U_1_STOP;
    *DBC_US_B = DBC_U_WR4;
    *DBC_US_B = DBC_U_X16 | DBC_U_1_STOP;

    *DBC_US_A = DBC_U_WR3;
    *DBC_US_A = DBC_U_R_8_BITS | DBC_U_RX_EN;
    *DBC_US_B = DBC_U_WR3;
    *DBC_US_B = DBC_U_R_8_BITS | DBC_U_RX_EN;

    *DBC_US_A = DBC_U_WR5;
    *DBC_US_A = DBC_U_DTR | DBC_U_T_8_BITS | DBC_U_TX_EN | DBC_U_RTS;
    *DBC_US_B = DBC_U_WR5;
    *DBC_US_B = DBC_U_DTR | DBC_U_T_8_BITS | DBC_U_TX_EN | DBC_U_RTS;

    /* Write WR2a to set for 8086 mode, vectored interrupt, RxB > TxA priority,
       both A and B interrupt.  Write the vector to WR2b.  The vector is
       shifted left 2 bits because the 68000 shifts it right 2 bits in
       order to generate its vector address.  The address of the interrupt
       routine must be placed at IV_CO.  That is done in main. */

    *DBC_US_A = DBC_U_WR2;
    *DBC_US_A = DBC_U_VEC_INT | DBC_U_8086 | DBC_U_PRI_1 | DBC_U_BOTH_INT;
    *DBC_US_B = DBC_U_WR2;
    *DBC_US_B = (int) IV_CO >> 2;
    
    /* Write WR1 - Interrupt on all received chars, variable vector, 
       interrupt on tx empty. VAR_VEC need only be written to WR1b.*/

    *DBC_US_A = DBC_U_WR1;
    *DBC_US_A = DBC_U_RX_INT_ALL | DBC_U_TX_INT_EN;
    *DBC_US_B = DBC_U_WR1;
    *DBC_US_B = DBC_U_RX_INT_ALL | DBC_U_TX_INT_EN | DBC_U_VAR_VEC;

    intUnlock (oldlevel);
    } 

/*******************************************************************************
*
* tyCoOpen - open file to Microbar USART
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/*******************************************************************************
*
* tyCoRead - task level read routine for Microbar USART
*
* This routine fields all read calls to the Microbar USART.
*/

LOCAL int tyCoRead (pTyCoDv, buffer, maxbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pTyCoDv, buffer, maxbytes));
    }
/*******************************************************************************
*
* tyCoWrite - task level write routine for Microbar USART
*
* This routine fields all write calls to the Microbar USART.
*/

LOCAL int tyCoWrite (pTyCoDv, buffer, nbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pTyCoDv, buffer, nbytes));
    }
/*******************************************************************************
*
* tyCoIoctl - special device control
*
* This driver responds to the same ioctl codes as a normal ty driver.
* When using the FIOBAUDRATE code, be aware that the Microbar USARTs
* share a common timer so changing the baudrate on one port changes
* it for the other.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* argument pertinent to request */

    {
    int ix;

    if (request == FIOBAUDRATE)
	{
	for (ix = 0; ix < NELEMENTS (baudTable); ix++)
	    {
	    if (baudTable[ix].rate == arg)
		{
		*DBC_PIT_2 = LSB(baudTable[ix].pitValue);
		*DBC_PIT_2 = MSB(baudTable[ix].pitValue);

		return (OK);
		}
	    }

	return (ERROR);
	}

    return (tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg));
    }

/*******************************************************************************
*
* tyCoInt - interrupt level processing
*
* This routine handles an interrupt from the on-board USART.  The interrupt
* is decoded and the appropriate routine invoked.  The interrupt is
* terminated with an 'EOI'.
*
* RETURNS: type of interrupt exit to take
*/

LOCAL VOID tyCoInt (int_level)
    char int_level;

    {
    FAST TY_CO_DEV *pTyCoDv = (int_level & DBC_U_INT_A) ? tyCoDv[0] : tyCoDv[1];
    char outchar;

    if (pTyCoDv != NULL)
	{
	switch (int_level & DBC_U_INT_TYPE)
	    {
	    case DBC_U_INT_TX:		/* transmitter buffer empty */

		/* get character to output, if any, from I/O system */

		if (tyITx ((TY_DEV_ID) pTyCoDv, &outchar) == OK)
		    *pTyCoDv->uart_ptr = outchar;		/* output char*/
		else
		    *pTyCoDv->uart_stat = DBC_U_RST_TXINT;	/* shut xmit */

		break;


	    case DBC_U_INT_RX:		/* received character */

		/* get character from USART and give it to I/O system */

		tyIRd ((TY_DEV_ID) pTyCoDv, *pTyCoDv->uart_ptr);
		break;


	    default:			/* special or external interrupt */

		/* reset the error status */

		*pTyCoDv->uart_stat = DBC_U_ERR_RST;
		break;
	    }
	}

    /* issue EOI */

    *DBC_US_A = DBC_U_EOI;
    }
/*******************************************************************************
*
* tyCoStartup - transmitter startup routine
*
* Call interrupt level character output routine for Microbar USART.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    char outchar;

    /* any character to send ? */

    if (tyITx ((TY_DEV_ID) pTyCoDv, &outchar) == OK)
	*pTyCoDv->uart_ptr = outchar;		/* output the character */
    }
