/* sysLib.c - Mizar mz7122 and mz7124 system dependent library */

static char *copyright = "Copyright 1988, Mizar Digital Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which UniWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts
	  
    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel
				       
    clock/timer functions:
	- enable/disable timer interrupts
	- set timer periodic rate

    mailbox/location monitor functions:
	- enable mailbox/location monitor interrupts

*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "vme.h"
#include "ioLib.h"
#include "memLib.h"
#include "sysLib.h"
#include "config.h"
#include "iv68k.h"

IMPORT char end;		/* end of system, created automatically by ld */

IMPORT VOID logMsg ();

#define	HZ	3686400		/* clock rate */

/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */

int   safeLoc;


/* locals */
 
LOCAL FUNCPTR sysClkRoutine    = NULL;
LOCAL int sysClkArg;
LOCAL int clkTicksPerSecond    = 60;
LOCAL int sysClkIsRunning      = FALSE;
  
LOCAL FUNCPTR auxClkRoutine    = NULL;
LOCAL int auxClkArg;
LOCAL int auxClkTicksPerSecond = 60;
LOCAL int auxClkIsRunning      = FALSE;
   
LOCAL FUNCPTR M562TxInt        = NULL;
LOCAL FUNCPTR M562RxInt        = NULL;
LOCAL BOOL M562IsConnected     = FALSE;

LOCAL ULONG localRamSize = 0x00100000;
				/* size of local ram (7122: 1Mb, 7124: 4Mb) */

LOCAL ULONG bcl [2];		/* current value of board control latches */

/* external declarations */

VOID sysClkIntA ();	/* Handle heartbeat interrupt in assembly */
VOID sysNMIIntA ();	/* Handle NMI interrupt in assembly during boot */

/* forward declarations */

LOCAL VOID sysM562Int ();		/* The handler of autovector 4 ints. */
LOCAL VOID sysNMIInt ();		/* Handle Non Maskable Interrupt */


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "Mizar mz7122" or "Mizar mz7124"
*/

char *sysModel ()

    {
    if (localRamSize == 0x00400000)
	return ("Mizar mz7124");
    else
	return ("Mizar mz7122");
    }
/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
*
* The timers are initialized and turned off.
* Bus Control latch is initialized and set.
* Based on memory size, the board is determined to be a mz7122 or mz7124.
* The DUSCC is initialized for use by the ty driver.
*/

VOID sysHwInit ()

    {
    char readButNotUsed;
    UINT *ramClearAdrs;

    *MZ_STATUS_LATCH;	/* clear any pending INT7 bits */

    /* until we are through booting */
    intSetVec (INT_VEC_MISC, sysNMIIntA);

    sysBCLInit ();	/* init latches and software images (done here for
				the benefit of BP init) */

    sysBCLSet ((ULONG) MZ_USRLED, (ULONG) MZ_USRLED, 1); /* turn on user LED */

    /* if we are an mz7124, we have 3 Mbytes of RAM that we never
	cleared in romInit.s, because we couldn't run vxMemProbe then
	(which tells us whether we're an mz7122 or mz7124).  Why?
	Because we would have needed a stack to do so. */

    localRamSize = 0x00100000;	/* size of local ram (7122: 1Mb, 7124: 4Mb) */

    if (vxMemProbe ((char *)(LOCAL_MEM_LOCAL_ADRS + MZ7124_RAM_SIZE - 1),
	READ, 1, &readButNotUsed) == OK)
	{
	localRamSize = 0x00400000;	/* claim we're an mz7124: 4Mb */

	/* zero remaining memory */

	bzero ((char *)(LOCAL_MEM_LOCAL_ADRS + MZ7122_RAM_SIZE),
		MZ7124_RAM_SIZE - MZ7122_RAM_SIZE );
	}

    intSetVec (INT_VEC_HBTIMER, sysClkIntA);

    /* set up initial DUSCC interrupt vector */

    *M562_IVR   = 0;
    *M562_ICR	= M562_ICR_CHAN_B_MAST_INT_EN
		| M562_ICR_CHAN_A_MAST_INT_EN
		| M562_ICR_VECT_INCLDS_STATUS
		| M562_ICR_VECT_MODE_NOT_VECT;
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* INTERNAL
* We do not probe memory but rely on sysHwInit (2) to have
* established memory size.  This is used to distinguish between the
* two versions of the board.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    return (char *) (LOCAL_MEM_LOCAL_ADRS + localRamSize);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-x, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor).
*
* INTERNAL
* Note that the "WARM" restart address is at (ROM_BASE_ADRS + 16) bytes.
*/

STATUS sysToMonitor (startType)
    int startType;	/* parameter is passed to ROM to tell it how to boot.
			   The possible types are defined in h/sysLib.h */

    {
/* BEGIN DEBUG (VRTX BUG): */
    *M562_CCRA = M562_CCR_CT_STOP;		/* stop M562 timer A */
    safeLoc++;
    safeLoc++;
    *M562_ICTSRA = M562_ICTSR_CT_ZERO_COUNT;	/* ack M562 timer bits */
    safeLoc++;
    safeLoc++;
    *M562_CCRB = M562_CCR_CT_STOP;		/* stop M562 timer B */
    safeLoc++;
    safeLoc++;
    *M562_ICTSRB = M562_ICTSR_CT_ZERO_COUNT;	/* ack M562 timer bits */
    safeLoc++;
    safeLoc++;
    *M562_IERA = 0;				/* clear M562 int A */
    safeLoc++;
    safeLoc++;
    *M562_IERB = 0;				/* clear M562 int B */
    safeLoc++;
    safeLoc++;
    *M562_GSR = 0;				/* clear pending M562 bits */

    /* turn off interrupts */

    sysBCLSet ((ULONG) (MZ_PAREN | MZ_DCEN | MZ_SYSFEN | MZ_BCLREN | MZ_MBOX),
	(ULONG) 0, 1);
    sysBCLSet ((ULONG) MZ_IRQALLEN, (ULONG) MZ_IRQALLEN, 1);
    sysBCLSet ((ULONG) MZ_TIMEREN, (ULONG) 0, 1);

    *MZ_STATUS_LATCH;				/* clear pending INT7 bits */

/* :END DEBUG (VRTX BUG) */

    (* ((FUNCPTR) (ROM_BASE_ADRS + 16))) (startType);

    return (OK);	/* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine to be called at each clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    /* Set up the nonmaskable interrupt here instead of in sysHwInit
	because the malloc'ed area created would get wiped out before
	it could be used.  */

    *MZ_STATUS_LATCH;	/* clear any pending INT7 bits */

    (void) intConnect (INT_VEC_MISC, sysNMIInt, NULL);

    /* turn on SYSFAIL, BCLR acceptance */
    /* XXX
    sysBCLSet ((ULONG) MZ_SYSFEN | MZ_BCLREN, (ULONG) MZ_SYSFEN | MZ_BCLREN, 1);
    */

    if (!M562IsConnected &&
	intConnect (INT_VEC_SIRQ, sysM562Int, NULL) == ERROR)
	{
	return (ERROR);
	}

    sysClkRoutine   = routine;
    sysClkArg       = arg;

    M562IsConnected = TRUE;

    return (OK
    );
    }
/********************************************************************************
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()

    {
    if (sysClkIsRunning)
	{
	/* stop timer */
     
	*M562_CCRA = M562_CCR_CT_STOP;

	sysClkIsRunning = FALSE;
	}
    }
/********************************************************************************
* sysClkEnable - turn system clock interrupts on
*/
 
VOID sysClkEnable ()
 
    {
    int tc;

    /* calculate the divide ratio, and write it to the timer chip */

    tc = HZ / clkTicksPerSecond;

    *M562_CTPRHA = MSB(tc);
    safeLoc++;
    safeLoc++;
    *M562_CTPRLA = LSB(tc);
    safeLoc++;
    safeLoc++;

    /* enable interrupts */
 
    *M562_CTCRA  = M562_CTCR_ZERO_DET_INTR_ENABLE | M562_CTCR_CLK_X1_CLK;
    safeLoc++;
    safeLoc++;
    *M562_CCRA	 = M562_CCR_CT_START;
 
    sysClkIsRunning = TRUE;
    }
/*******************************************************************************
*
* sysClkGetRate - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkSetRate (2)
*/

int sysClkGetRate ()
    
    {
    return (clkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkSetRate - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* System clock interrupts are not enabled.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkGetRate (2), sysClkEnable (2)
*/

VOID sysClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {                                                                   
    if (ticksPerSecond > 0)
        clkTicksPerSecond = ticksPerSecond;
 
    if (sysClkIsRunning)
        {
        sysClkDisable ();
        sysClkEnable ();
        }
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connnect to interrupt
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    if (!M562IsConnected &&
	intConnect (INT_VEC_SIRQ, sysM562Int, NULL) == ERROR)
	{
	return (ERROR);
	}

    auxClkRoutine   = routine;
    auxClkArg       = arg;

    M562IsConnected = TRUE;

    return (OK);
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - clear the auxiliary clock routine
* 
* This routine disables the auxiliary clock interrupt and disconnects
* the routine currently connected to the auxiliary clock interrupt.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

VOID sysAuxClkDisconnect ()

    {
     /* disable the auxiliary clock interrupt */
     
    sysAuxClkDisable ();
 
    /* connect the dummy routine, just in case */
 
    sysAuxClkConnect (logMsg, (int)"auxiliary clock interrupt\n");
    }
/********************************************************************************
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/
 
VOID sysAuxClkDisable ()
 
    {
    if (auxClkIsRunning)
	{
	/* stop timer */
     
	*M562_CCRB = M562_CCR_CT_STOP;

	auxClkIsRunning = FALSE;
	}
    }
/********************************************************************************
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/
 
VOID sysAuxClkEnable ()
 
    {
    int tc;

    /* calculate the divide ratio, and write it to the timer chip */

    tc = HZ / auxClkTicksPerSecond;

    *M562_CTPRHB = MSB(tc);
    safeLoc++;
    safeLoc++;
    *M562_CTPRLB = LSB(tc);
    safeLoc++;
    safeLoc++;

    /* enable and start clock interrupts */

    *M562_CTCRB  = M562_CTCR_ZERO_DET_INTR_ENABLE | M562_CTCR_CLK_X1_CLK;
    safeLoc++;
    safeLoc++;
    *M562_CCRB	 = M562_CCR_CT_START;

    auxClkIsRunning = TRUE;
    }
/*******************************************************************************
*
* sysAuxClkGetRate - get the auxiliary timer frequency
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkSetRate (2)
*/

int sysAuxClkGetRate ()

    {
    return (auxClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysAuxClkSetRate - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: sysAuxClkGetRate (2)
*/

VOID sysAuxClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
        auxClkTicksPerSecond = ticksPerSecond;
     
    if (auxClkIsRunning)
        {
        sysAuxClkDisable ();
        sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VMEbus address
* that would have to be accessed to get to that byte.
*
* NOTE MZ7122:
* In A24 space, only the lower 24 address bits are examined.  In A16 space,
* only the lower 16 address bits are examined.  Unused bits are set to zeros
* on exit.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus
*
* SEE ALSO: sysBusToLocalAdrs (2)
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides;
			 * use address modifier codes as defined in vme.h,
			 * such as VME_AM_STD_SUP_DATA */
    char *localAdrs;	/* local address to convert */
    char **pBusAdrs;	/* where to return bus address */

    {
    UINT temp;

    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pBusAdrs = (char *) ((UINT) localAdrs & 0x0000ffff);
	    return (OK);	/* no on-board-only A16 space */

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    temp = (UINT) localAdrs;
	    if ((temp >= 0xff000000) && (temp <= 0xfffeffff))
		{
		temp = (UINT) localAdrs & 0x00ffffff;
		/* mcl note: we are not preventing the original address from
			having been our DPRAM address, here. */
		*pBusAdrs = (char *) temp;
		return (OK);	/* off-board A24 access */
		}
	    else
		if ((temp >= LOCAL_MEM_LOCAL_ADRS) &&
		    (temp < (UINT) sysMemTop ()) && (sysBCLGet (2) & MZ_A24EN))
		    {
		    temp = (UINT) localAdrs & 0x00ffffff;
		    *pBusAdrs = (char *)
				 (temp | (sysBCLGet (2) & MZ_DP_RAM_A24_ADRS));

		    return (OK);	/* on-board A24 access to DPRAM */
		    }
		else
		    return (ERROR);

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    temp = (UINT) localAdrs;
	    /* mcl note: we are not preventing the original address from
		having been our DPRAM address, here. */
	    if ((temp >= 0x00100000) && (temp <= 0x7fffffff))
		{
		*pBusAdrs = (char *) temp;
		return (OK);	/* off-board A32 access */
		}
	    else
		if ((temp >= 0x80000000) && (temp <= 0xfdffffff))
		    if (sysBCLGet (2) & MZ_DCEN)
			{
			*pBusAdrs = (char *) temp;
			return (OK);	/* off-board A32 access */
			}
		    else
			return (ERROR); /* daughter card is on, forget it */

	        else
		    if ((temp >= LOCAL_MEM_LOCAL_ADRS) &&
			(temp < (UINT) sysMemTop ()) &&
			(sysBCLGet (2) & MZ_A32EN))
			{
			*pBusAdrs = (char *)
			    (temp | (sysBCLGet (2) & MZ_DP_RAM_A32_ADRS));

			return (OK);	/* on-board A32 access to DPRAM */
			}
		    else
			return (ERROR);

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VMEbus memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* NOTE MZ7122:
* In A24 space, only the lower 24 address bits are examined.  In A16 space,
* only the lower 16 address bits are examined.  Unused bits are set to zeros
* on exit.
*
* RETURNS: OK, or ERROR if unknown address space.
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides;
			 * use address modifier codes as defined in
			 * vme.h, such as VME_AM_STD_SUP_DATA */
    char *busAdrs;	/* bus address to convert */
    char **pLocalAdrs;	/* where to return local address */

    {
    UINT temp;
    UINT temp1;

    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pLocalAdrs = (char *) (0xffff0000 | ((UINT) busAdrs & 0x0000ffff));
	    return (OK);

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    temp = (UINT) busAdrs & 0x00ffffff;
	    if (temp >= 0x00ff0000)
		return (ERROR);		/* small glitch in A24 space */
	    else
		{
		temp1 = temp | (sysBCLGet (2) & MZ_DP_RAM_A24_ADRS);

		if ((temp >= temp1) && (temp < (temp1 + localRamSize)))
		    {
		    if (sysBCLGet (2) & MZ_A24EN)
			{
			*pLocalAdrs = (char *)
			   ((UINT) temp - (sysBCLGet (2) & MZ_DP_RAM_A24_ADRS));

			return (OK);	/* adjust to point to DPRAM */
			}
		    else
			return (ERROR);	/* no A24 access enabled to DPRAM */
		    }
		else
		    {
		    *pLocalAdrs = (char *) (0xff000000 | (UINT) temp);
		    return (OK);	/* adjust to point to A24 space */
		    }
		}

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    temp = (UINT) busAdrs;
	    if ((temp < 0x00100000) || (temp > 0xfdffffff))
		return (ERROR);		/* limits to A32 space */
	    else
		{
		temp1 = temp | (sysBCLGet (2) & MZ_DP_RAM_A32_ADRS);
		if ((temp >= temp1) && (temp < (temp1 + localRamSize)))
		    {
		    if (sysBCLGet (2) & MZ_A32EN)
			{
			*pLocalAdrs = (char *)
			   ((UINT) temp - (sysBCLGet (2) & MZ_DP_RAM_A32_ADRS));

			return (OK);	/* adjust to point to DPRAM */
			}
		    else
			return (ERROR);	/* no A32 access enabled to DPRAM */
		    }
		else
		    {
		    *pLocalAdrs = (char *) temp;

		    /* already pointing to rest of A32 space */
		    return (OK);
		    }
		}

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* sysIntDisable - disable VMEbus interrupt level
*
* This routine disables the specified VMEbus interrupt level.
*
* NOTE MZ7122:
* All interrupt levels are handled by board control latch one.
* In addition to the seven VMEbus interrupts
* the mailbox, BCLR, and SYSFAIL interrupts are defined as #8, #9 and #10.
* The IRQ interrupt enables are active low; the others are active high.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-10
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    int val;
    int mask = 0;

    switch (intLevel)
	{
	case 1:
	    val  = MZ_IRQ1EN;
	    mask = MZ_IRQ1EN;
	    break;

	case 2:
	    val  = MZ_IRQ2EN;
	    mask = MZ_IRQ2EN;
	    break;

	case 3:
	    val  = MZ_IRQ3EN;
	    mask = MZ_IRQ3EN;
	    break;

	case 4:
	    val  = MZ_IRQ4EN;
	    mask = MZ_IRQ4EN;
	    break;

	case 5:
	    val  = MZ_IRQ5EN;
	    mask = MZ_IRQ5EN;
	    break;

	case 6:
	    val  = MZ_IRQ6EN;
	    mask = MZ_IRQ6EN;
	    break;

	case 7:
	    val = MZ_IRQ7EN;
	    mask = MZ_IRQ7EN;
	    break;

	case 8:
	    val = MZ_MBOXEN;
	    break;

	case 9:
	    val = MZ_BCLREN;
	    break;

	case 10:
	    val = MZ_SYSFEN;
	    break;

	default:
	    return (ERROR);
	}

    sysBCLSet ((ULONG)val, (ULONG)mask, 1);

    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable VMEbus interrupt level
*
* This routine enables the specified VMEbus interrupt level.
*
* NOTE MZ7122:
* All interrupt levels are handled by board control latch one.
* In addition to the seven VMEbus interrupts
* the mailbox, BCLR, and SYSFAIL interrupts are defined as #8, #9 and #10.
* The IRQ interrupt enables are active low; the others are active high.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-10.
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    int val;
    int mask = 0;

    switch (intLevel)
	{
	case 1:
	    val = MZ_IRQ1EN;
	    break;

	case 2:
	    val = MZ_IRQ2EN;
	    break;

	case 3:
	    val = MZ_IRQ3EN;
	    break;

	case 4:
	    val = MZ_IRQ4EN;
	    break;

	case 5:
	    val = MZ_IRQ5EN;
	    break;

	case 6:
	    val = MZ_IRQ6EN;
	    break;

	case 7:
	    val = MZ_IRQ7EN;
	    break;

	case 8:
	    val  = MZ_MBOXEN;
	    mask = MZ_MBOXEN;
	    break;

	case 9:
	    val  = MZ_BCLREN;
	    mask = MZ_BCLREN;
	    break;

	case 10:
	    val  = MZ_SYSFEN;
	    mask = MZ_SYSFEN;
	    break;

	default:
	    return (ERROR);
	}

    sysBCLSet ((ULONG) val, (ULONG) mask, 1);

    return (OK);
    }
/*******************************************************************************
*
* sysIntAck - acknowledge VMEbus interrupt
*
* This routine acknowledges VMEbus interrupts.
*
* NOTE MZ7122:
* This routine has no effect.
* On some boards this routine returns the vector put on
* bus by interrupting device.
*
* RETURNS: NULL
*
* ARGSUSED
*/

STATUS sysIntAck (intLevel)
    int intLevel;	/* interrupt level to acknowledge */

    {
    return (NULL);
    }
/*******************************************************************************
*
* sysIntGen - generate VMEbus interrupt
*
* This routine generates a VMEbus interrupt.
*
* RETURNS: OK
*/

STATUS sysIntGen (intLevel, intVector)
    int intLevel;	/* interrupt level to generate */
    int intVector;	/* interrupt vector to generate */

    {
    sysBCLSet ((ULONG) MZ_VIRQ, (ULONG) intLevel, 2);
    *MZ_INT_VEC_LATCH = intVector;
    return (OK);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2)
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine to be called at each mailbox interrupt*/
    int arg;		/* argument with which to call routine */

    {
    sysIntDisable (8);
    return (intConnect (INT_VEC_MBOX, routine, arg));
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* RETURNS: OK
*/

STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;		/* mailbox address */

    {
    /* set (or reset) mailbox address */

    sysBCLSet ((ULONG) MZ_MBOX,
		(ULONG) (0x80000000 | ((sysProcNum << 24) & 0x3f000000)), 1);

    /* enable mailbox interrupt */

    sysIntEnable (8);

    return (OK);
    }
/*******************************************************************************
*
* sysGetProcNum - get processor number
*
* This routine returns the processor number previously set with
* sysSetProcNum (2).
*
* RETURNS: processor number
*/
 
int sysGetProcNum ()
 
    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysSetProcNum - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*/

VOID sysSetProcNum (procNum)
    int procNum;		/* processor number */

    {
    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set across VMEbus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* RETURNS: TRUE (successful set) or FALSE (failure)
*/

BOOL sysBusTas (addr)
    char *addr;		/* address to be tested and set */

    {
    int retVal;

    retVal = *addr;
    retVal = vxTas (addr);
    return (retVal);
    }


/* miscellaneous support routines */
 
/*******************************************************************************
*
* sysMailboxAddressSet - set mailbox address as determined by sysProcNum.
*
* This routine sets the mailbox address as determined by sysProcNum.
*
* We arbitrarily have selected the mailbox addresses to be A16 addresses
* 8x00 through bx00, where x is CPU number, 0 <= x < 0x3f.  (0x3f
* is used as a null at initialization time).
*
* This routine is called by sleight of hand, as a macro in the call to
* bpattach in netInit in usrConfig.c.  The reason why is that we cannot
* set the address until we know which CPU number we are, but we have
* to have the address available for netInit.
*
* RETURNS: VMEbus address of the mailbox interrupt
*/

int sysMailboxAddressSet ()

    {
    sysBCLSet ((ULONG) MZ_MBOX,
    (ULONG) (0x80000000 | ((sysProcNum << 24) & 0x3f000000)), 1);

    return (sysMailboxAddressGet ());
    }
/*******************************************************************************
*
* sysMailboxAddressGet - get currently defined mailbox address.
*
* This routine gets the currently defined mailbox address by reading
* the value from the hardware.
*
* We arbitrarily have selected the mailbox addresses to be A16 addresses
* 8x00 through bx00, where x is CPU number, 0 <= x < 0x3f.  (0x3f
* is used as a null at initialization time).
*
* RETURNS: VMEbus address of the mailbox interrupt
*/

int sysMailboxAddressGet ()

    {
    return ((sysBCLGet (1) >> 16) & 0x0000ff00);
    }
/*******************************************************************************
*
* sysBCLInit - set initial values of both board control latches
*
* This routine sets bits in the board control latch.
*/

VOID sysBCLInit ()

    {
    bcl [0] = INITIAL_LATCH_ONE;
    *MZ_CONTROL_LATCH_ONE = bcl [0];
    bcl [1] = INITIAL_LATCH_TWO;
    *MZ_CONTROL_LATCH_TWO = bcl [1];
    }
/*******************************************************************************
*
* sysBCLGet - return the current value of the board control latch
*
* This routine returns the last value to which the board control latch
* was set.  It is only effective if all changes to the board control
* latch are done through sysBCLSet (2).
*
* RETURNS: latch value or ERROR if latchNum not in 1-2
*
* SEE ALSO: sysBCLSet (2)
*/

STATUS sysBCLGet (latchNum)
    int latchNum;	/* Board control latch to set */

    {
    if (latchNum == 1)
    	return (bcl [0]);
    else if (latchNum == 2)
    	return (bcl [1]);
    else
	return (ERROR);
    }
/*******************************************************************************
*
* sysBCLSet - set bits in the board control latch
*
* This routine sets bits in the board control latch on the mz7122/24 board.
* This routine sets and clears bits in a local copy of the latch, then
* writes that local copy to the latch.  This means that all changes to
* the latch must go through this routine.  Otherwise, any direct changes
* to the latch would be lost the next time this routine is called.
*
* The parameter "mask" determines which bits will be set, and "value"
* determines the value to which those bits will be set.
* In  other words, newBCL = (oldBCL & ~mask) | (value & mask).
*
* This routine must be called in supervisor mode.
*
* SEE ALSO: sysBCLGet (2)
*/

VOID sysBCLSet (mask, value, latchNum)
    ULONG mask;		/* which bits to change */
    ULONG value;	/* what to change bits to */
    int latchNum;	/* Board control latch to set */

    {
    if (latchNum == 1)
	{
        bcl [0] = (bcl [0] & (~mask)) | (value & mask);
	*MZ_CONTROL_LATCH_ONE = bcl [0];
	}
    else if (latchNum == 2)
	{
        bcl [1] = (bcl [1] & (~mask)) | (value & mask);
	*MZ_CONTROL_LATCH_TWO = bcl [1];
	}
    }

/*******************************************************************************
*
* sysM562Int - interrupt level processing for mz7122/24 DUSCC
*
* This routine handles an interrupt from the on-board m68562 (M562).  The
* interrupt is decoded by checking the M562's interrupt status reg, and
* the appropriate routine invoked.  
*
* This routine is located here because we use the extra timer on the M562
* chip as the system clock, which is handled here.
*/

LOCAL VOID sysM562Int ()

    {
    FAST int channel = 0;			/* channel = A */
    FAST int ivrm    = *M562_IVRM & M562_IVRM_MASK;

    if (ivrm & M562_IVRM_MASK_CHANNEL)
	{
	ivrm   &= ~M562_IVRM_MASK_CHANNEL;
	channel = 1;			/* channel = B */
	}

    switch (ivrm)
	{
	case M562_IVRM_RXRDY:
	    if (M562RxInt == NULL)
		if (channel)
		    {
		    *M562_RXFIFOB;
		    safeLoc++;
		    safeLoc++;
		    *M562_GSR = M562_GSR_A_RXRDY ;
		    }
		else
		    {
		    *M562_RXFIFOA;
		    safeLoc++;
		    safeLoc++;
		    *M562_GSR = M562_GSR_A_RXRDY ;
		    }
	    else
		(*M562RxInt) (channel);
	    break;

	case M562_IVRM_TXRDY:
	    if (M562TxInt == NULL)
		if (channel)
		    {
		    *M562_TXFIFOB = NULL;
		    safeLoc++;
		    safeLoc++;
		    *M562_GSR = M562_GSR_A_TXRDY ;
		    }
		else
		    {
		    *M562_TXFIFOA = NULL;
		    safeLoc++;
		    safeLoc++;
		    *M562_GSR = M562_GSR_A_TXRDY ;
		    }
	    else
		(*M562TxInt) (channel);
	    break;

	case M562_IVRM_RX_TX_EXCEPTION:
	    break;

	case M562_IVRM_EXTERN_OR_CT:
	    if (channel == 0)
		{
		if (sysClkRoutine != NULL)
		    (*sysClkRoutine) (sysClkArg);
		*M562_ICTSRA = M562_ICTSR_CT_ZERO_COUNT;
		}
	    else
		{
		if (auxClkRoutine != NULL)
		    (*auxClkRoutine) (auxClkArg);
		*M562_ICTSRB = M562_ICTSR_CT_ZERO_COUNT;
		}
	    break;

	/*NOTREACHED*/
	default:
	    ;
	}
    }
/*******************************************************************************
*
* sysM562Connect - set interrupt routines for the mz7122/24 DUSCC
*
* This routine connects the serial driver interrupt routines to the
* interrupt routine.  It should be called once, from the serial driver.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*/

STATUS sysM562Connect (recvRoutine, xmitRoutine)
    FUNCPTR recvRoutine;	/* receive routine */
    FUNCPTR xmitRoutine;	/* transmit routine */

    {
    if (!M562IsConnected &&
	intConnect (INT_VEC_SIRQ, sysM562Int, NULL) == ERROR)
	{
	return (ERROR);
	}

    M562TxInt = xmitRoutine;
    M562RxInt = recvRoutine;

    M562IsConnected = TRUE;

    return (OK);
    }
/*******************************************************************************
*
* sysNMIInt - interrupt level processing for mz7122/24 NMI Interrupt
*
* This routine handles the nonmaskable interrupt by reading the board
* status register and decoding it to see which of the four possible
* sources generated the interrupt.
*
* NOTE:
* None of these sources are latched.  They are all active low.
*/

LOCAL VOID sysNMIInt ()

    {
    unsigned long temp_latch;

    temp_latch = *MZ_STATUS_LATCH;

    if ((temp_latch & MZ_STATUS_BCLR) == 0)
	{
	/* pulse RALW high */
	sysBCLSet ((ULONG) MZ_VMEBUS_RALW, (ULONG) MZ_VMEBUS_RALW, 1);

	/* then low */
	sysBCLSet ((ULONG) MZ_VMEBUS_RALW, (ULONG) 0, 1);
	}

    if ((temp_latch & MZ_STATUS_ABORT) == 0)
	{
	sysToMonitor (BOOT_WARM_NO_AUTOBOOT);
	}
    if ((temp_latch & MZ_STATUS_ACFAIL) == 0)
 	{
	sysToMonitor (BOOT_COLD);		/* DEBUG */
	}
    if ((temp_latch & MZ_STATUS_SYSFAIL) == 0)
	{
	sysToMonitor (BOOT_COLD);		/* DEBUG */
	}
    }
/*******************************************************************************
*
* sysFrontPanelSwitches - read DIP switches
*
* RETURNS: 10 bit number of switch settings
*/

int sysFrontPanelSwitches ()
    {
    return ((*MZ_STATUS_LATCH & 0x0003ff00) >> 8);
    }
