/* sysLib.c - Ironics IV-1624A system dependent library */

static char *copyright = "Copyright 1986-88, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which UniWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts

NOTE
There are 4 jumpers which may be read with sysJumpers (2).
The first jumper J10 is used to determine whether the board is
at 10 or 12 Mhz.
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "config.h"
#include "iv68k.h"

IMPORT char end;		/* end of system, created automatically by ld */

IMPORT VOID logMsg ();

/* Timer 2 on the timer chip is dedicated to the bus timeout.
 * The following is the required timer control bits for timer 2.
 */

#define TMR_2_CTRL	(CLK_CTRL_OUTPUT_ENABLE | CLK_CTRL_CONTINUOUS_1 | \
			 CLK_CTRL_USE_ENABLE_CLK)

/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */

int  sysClkFreq;			/* used in tyCoDrv for baud rate */

/* locals */

LOCAL FUNCPTR sysClkRoutine    = NULL;
LOCAL int sysClkArg;
LOCAL int clkTicksPerSecond    = 60;
LOCAL int sysClkIsRunning      = FALSE;
LOCAL BOOL sysClkIsConnected   = FALSE;

LOCAL FUNCPTR auxClkRoutine    = NULL;
LOCAL int auxClkArg;
LOCAL int auxClkTicksPerSecond = 60;
LOCAL int auxClkIsRunning      = FALSE;

LOCAL VOID sysClkInt ();

/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* This routine returns a pointer to a string with the model name of
* this CPU.
*
* RETURNS: pointer to the string "Ironics IV-1624A"
*/

char *sysModel ()

    {
    return ("Ironics IV-1624A");
    }
/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode.
*
* Timers are initialized and turned off.
* IVHAN and IGOR are setup.
* The control port is zeroed (top 8 address lines to zero).
* Determine clock frequency by reading software jumpers.
*/

VOID sysHwInit ()

    {
    /* Set up Interrupt controller, disabling all interrupts */

    *IVH_LRQM = 0;	/* mask out all local interrupts */
    *IVH_IRQM = 0;	/* clear interrupt request mask */

    *IVH_PTR = INT_SIOIRQ01;	/* write SCC 0&1 interrupt control */
    *IVH_CTL = IVH_EDGE_SENSITIVE;

    *IVH_PTR = INT_SIOIRQ23;	/* write SCC 2&3 interrupt control */
    *IVH_CTL = IVH_EDGE_SENSITIVE;

    *IVH_LRQV = INT_VEC_LOCAL_BASE;	/* local int vectors start at 64 */

    /* initialize interrupt generator */

    *IGOR_IVR = IGOR_IVR_CLEAR;

    /* initialize timers to quiescent state */

    *CLK_REG2  = TMR_2_CTRL | CLK_CTRL_SELECT_R1;	/* select r1 */
    *CLK_REG13 = 0;					/* init timer 1 */
    *CLK_REG2  = 0;					/* select r3 */
    *CLK_REG13 = 0;					/* init timer 3 */

    /* NOTE: channel 2 is bus timeout -
     *       set for BTOCLK = 4 usec  ==  bus timeout = 60 usec.  */

    *CLK_WMSB2	= 0;	/* MSB of count */
    *CLK_CHAN2	= 1;	/* LSB of count */

    *IV24_CONTROL_PORT = 0;		/* Turn off FAIL */

    sysClkFreq = (sysJumpers () & 1) ? SYS_CLK_FREQ_12 : SYS_CLK_FREQ_10;
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* INTERNAL
* This routine currently does not check memory at all, but just returns
* LOCAL_MEM_LOCAL_ADRS + 1 meg, which is always top of memory for this board.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    return ((char *) LOCAL_MEM_LOCAL_ADRS + 0x100000);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-X, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor)
*
* INTERNAL
* Note that the "WARM" restart address is at (ROM_BASE_ADRS + 16) bytes.
*/

STATUS sysToMonitor (startType)
    int startType;	/* parameter is passed to ROM to tell it how to boot */

    {
    (* ((FUNCPTR) (ROM_BASE_ADRS + 16))) (startType);

    return (OK);	/* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURN: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each system clock interrupt */
    int arg;		/* argument with which to call with routine */

    {
    if (!sysClkIsConnected &&
    	intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), sysClkInt, 0) == ERROR)
	{
	return (ERROR);
	}

    sysClkIsConnected = TRUE;

    sysClkRoutine = routine;
    sysClkArg     = arg;

    return (OK);
    }
/*******************************************************************************
*
* sysClkInt - handle clock interrupt
*
* This routine handles the clock interrupt.  It is attached to the clock
* interrupt vector by the routine sysClkConnect.  It handles both the
* system clock and the auxiliary clock since these are multiplexed through
* the same interrupt.  The appropriate routines are called and the interrupts
* are acknowleged.
*
* INTERNAL
* The status register contains 4 bits:
*     0 = timer 1
*     1 = timer 2 (bus timeout - used)
*     2 = timer 3
*     7 = composite
* The status register must be re-read after an acknowledgment.
*/

LOCAL VOID sysClkInt ()

    {
    FAST char reg;

    /* test for both clocks: 1 = system, 3 = auxiliary */

    while ((reg = *CLK_STATUS) & (CLK_STATUS_INT_1|CLK_STATUS_INT_3))
	{
	if (reg & CLK_STATUS_INT_1)
	    {
	    if (sysClkRoutine != NULL)
		(* sysClkRoutine) (sysClkArg);

	    reg = *CLK_WMSB1;	/* acknowledge system clock interrupt */

	    reg = *CLK_STATUS;	/* get new status */
	    }

	if (reg & CLK_STATUS_INT_3)
	    {
	    if (auxClkRoutine != NULL)
		(* auxClkRoutine) (auxClkArg);

	    reg = *CLK_WMSB3;	/* acknowledge auxiliary clock interrupt */
	    }
	}
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()

    {
    if (sysClkIsRunning)
	{
	*CLK_REG2   = TMR_2_CTRL | CLK_CTRL_SELECT_R1;	/* select ctrl reg 1 */
	*CLK_REG13  = 0;				/* disable timer 1 */
	sysClkIsRunning = FALSE;
	}
    }
/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
*/

VOID sysClkEnable ()

    {
    unsigned int tc;
    
    sysClkDisable ();		/* just in case */

    /* Calculate count:
     *  input clock is (system clock / 10); output period = 2n + 2; so...
     *      (sys-clk-freq / 10) / ticks-per-second  =  2n + 2
     */

    tc = ((sysClkFreq / 20) / clkTicksPerSecond) - 1;

    if (tc < 1 || tc > 65535)		/* 16 bits */
	{
	printErr ("sysClkEnable: %d out of range.\n", clkTicksPerSecond);
	return;
	}

    *CLK_REG2  = TMR_2_CTRL | CLK_CTRL_SELECT_R1; /* select control reg 1 */
    *CLK_REG13 = 0;				  /* disable timer 1 */

    /* set up timer 1 on the timer chip */

    *CLK_WMSB1	= MSB(tc);
    *CLK_CHAN1	= LSB(tc);

    *CLK_REG13  = CLK_CTRL_OUTPUT_ENABLE | CLK_CTRL_INT_ENABLE |
		  CLK_CTRL_CONTINUOUS_2 | CLK_CTRL_USE_ENABLE_CLK;


    /* Enable the clock interrupt on the IVHAN chip */

    *IVH_PTR   = INT_TMRIRQ;	/* write timer interrupt control */
    *IVH_CTL   = IVH_EDGE_SENSITIVE | IVH_VEC_ENABLE;
    *IVH_LRQM |= (1 << INT_TMRIRQ);

    sysClkIsRunning = TRUE;
    }
/*******************************************************************************
*
* sysClkGetRate - get rate of system clock
*
* RETURNS: number of ticks per second of the system clock
*/

int sysClkGetRate ()
    
    {
    return (clkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkSetRate - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* System clock interrupts are not enabled.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkGetRate (2), sysClkEnable (2)
*/

VOID sysClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	clkTicksPerSecond = ticksPerSecond;

    if (sysClkIsRunning)
	{
	sysClkDisable ();
	sysClkEnable ();
	}
    }
/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects the user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call with routine */

    {
    auxClkRoutine = routine;
    auxClkArg     = arg;

    return (OK);
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - disconnect a routine from the auxiliary clock interrupt
*
* This routine disables the auxiliary clock interrupt and disconnects
* the routine currently connected to the auxiliary clock interrupt.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

VOID sysAuxClkDisconnect ()

    {
    /* disable auxiliary clock */

    sysAuxClkDisable ();

    /* connect dummy routine, just in case */

    sysAuxClkConnect (logMsg, (int) "auxiliary clock interrupt\n");
    }
/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/

VOID sysAuxClkDisable ()

    {
    if (auxClkIsRunning)
	{
	*CLK_REG2  = TMR_2_CTRL;	/* select control register 3 */
	*CLK_REG13 = 0;			/* disable timer 3 */
	auxClkIsRunning = FALSE;
	}
    }
/*******************************************************************************
*
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/

VOID sysAuxClkEnable ()

    {
    unsigned int tc;

    sysAuxClkDisable ();

    /* Calculate tc:
     *  input clock is (system clock / 10); output period = 2n + 2; so...
     *      (sys-clk-freq / 10) / ticks-per-second  =  2n + 2
     */

    tc = ((sysClkFreq / 20) / auxClkTicksPerSecond) - 1;

    if (tc < 1 || tc > 65535)		/* 16 bits */
	{
	printErr ("sysAuxClkEnable: %d out of range.\n", auxClkTicksPerSecond);
	return;
	}

    *CLK_REG2  = TMR_2_CTRL;	/* select control register 3 */
    *CLK_REG13 = 0;		/* disable timer 3 */

    /* set up timer 3 on the timer chip */

    *CLK_WMSB3	= MSB(tc);
    *CLK_CHAN3	= LSB(tc);

    *CLK_REG13  = CLK_CTRL_OUTPUT_ENABLE | CLK_CTRL_INT_ENABLE |
		  CLK_CTRL_CONTINUOUS_2 | CLK_CTRL_USE_ENABLE_CLK;


    /* Enable the clock interrupt on the IVHAN chip */

    *IVH_PTR = INT_TMRIRQ;	/* write timer interrupt control */
    *IVH_CTL = IVH_EDGE_SENSITIVE| IVH_VEC_ENABLE;
    *IVH_LRQM |= (1 << INT_TMRIRQ);

    auxClkIsRunning = TRUE;
    }
/*******************************************************************************
*
* sysAuxClkGetRate - get rate of auxiliary clock
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkSetRate (2)
*/

int sysAuxClkGetRate ()
    
    {
    return (auxClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysAuxClkSetRate - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkGetRate (2)
*/

VOID sysAuxClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	auxClkTicksPerSecond = ticksPerSecond;

    if (auxClkIsRunning)
	{
	sysAuxClkDisable ();
	sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus
*
* SEE ALSO: sysBusToLocalAdrs (2)
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides */
    char *localAdrs;	/* local address to convert                   */
    char **pBusAdrs;	/* where to return bus address                */

    {
    if ((int) localAdrs < LOCAL_MEM_LOCAL_ADRS ||
	localAdrs >= sysMemTop ())
	{
	/* this is off-board memory - just return local address */

	*pBusAdrs = localAdrs;
	}
    else
	{
	/* this is on-board memory - map to bus address space;
	 *   the following memory mapping is established by hardware jumpers:
	 *   - only processor 0 has memory on bus,
	 *   - the memory is placed in STD DATA space at address
	 *	 LOCAL_MEM_BUS_ADRS.
	 */

	if ((sysProcNum != 0) ||
	     (adrsSpace != VME_AM_STD_SUP_DATA) &&
	     (adrsSpace != VME_AM_STD_USR_DATA))
	    return (ERROR);

	*pBusAdrs =  localAdrs  + LOCAL_MEM_BUS_ADRS - LOCAL_MEM_LOCAL_ADRS;
	}

    return (OK);
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a bus memory address, this routine returns the local address that
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides */
    char *busAdrs;	/* bus address to convert                     */
    char **pLocalAdrs;	/* where to return local address              */

    {
    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pLocalAdrs = (char *) (0x00ff0000 | (int) busAdrs);
	    return (OK);

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    if ((busAdrs <   (char*)0x010000) ||
		((busAdrs >= (char*)0xf00000) && (busAdrs < (char*)0xf40000)) ||
		((busAdrs >= (char*)0xf70000) && (busAdrs < (char*)0xf80000)) ||
		(busAdrs >= (char*)0xfe0000))
		return (ERROR);

	    *pLocalAdrs = busAdrs;
	    return (OK);

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    return (ERROR);

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* sysIntEnable - enable interrupt level
*
* This routine enables the specified interrupt level.
*
* NOTE IV-1624A:
* The appropriate bit in the bus request mask is set
* on the interrupt vector handler (IVHAN) chip.
*
* RETURNS: OK (always)
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    *IVH_IRQM |= (1 << intLevel);
    return (OK);
    }
/*******************************************************************************
*
* sysIntDisable - disable interrupt level
*
* This routine disables the specified interrupt level.
*
* NOTE IV-1624A:
* The appropriate bit in the bus request mask is cleared
* on the interrupt vector handler (IVHAN) chip.
*
* RETURNS: OK (always)
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    *IVH_IRQM &= ~(1 << intLevel);
    return (OK);
    }
/*******************************************************************************
*
* sysIntAck - acknowledge interrupt
*
* This routine acknowledges the specified interrupt.
*
* NOTE IV-1624A:
* If jumpered correctly, VME interrupts will be acknowledged automatically.
* This routine has no effect.
* On some boards this routine returns the vector put on
* bus by interrupting device.
*
* RETURNS: NULL
*
* ARGSUSED
*/

int sysIntAck (intLevel)
    int intLevel;	/* interrupt level to acknowledge */

    {
    return (NULL);
    }
/*******************************************************************************
*
* sysIntGen - generate interrupt
*
* This routine generates a VME bus interrupt of the specified level with
* the specified vector.
*
* NOTE IV-1624A:
* The interrupt generator uses the interrupt level as the
* least significant 3 bits of the vector it will generate during the IACK cycle.
* Thus this routine will report an error if the least significant 3 bits of the
* specified vector is not equal to the specified interrupt level.
*
* RETURNS:
*  OK, or
*  ERROR if least significant 3 bits of the specified vector is not
*  equal to the specified level
*/

STATUS sysIntGen (level, vector)
    int level;		/* VME bus interrupt level to generate (1-7) */
    int vector;		/* interrupt vector to generate (0-255)      */

    {
    /* check for valid interrupt vector */

    if ((vector & ~IGOR_IVR_VEC_MASK) != level)
	return (ERROR);


    /* wait for any previous interrupt to be acknowleged */

    while (*IGOR_IRR & IGOR_IRR_REQ_MASK)
	;

    *IGOR_IVR = (vector & IGOR_IVR_VEC_MASK) | IGOR_IVR_ENABLE;
    *IGOR_IRR = (1 << level);

    return (OK);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the specified routine to the mailbox interrupt.
*
* RETURNS: OK, or ERROR if unable to connect interrupt handler
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine to connect to mailbox interrupt */
    int arg;		/* arg to pass to routine */

    {
    return (intConnect (INUM_TO_IVEC (INT_VEC_MAILBOX), routine, arg));
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* NOTE IV-1624A:
* The mailbox address is ignored because it is not settable in software.
*
* RETURNS: OK (always)
*
* ARGSUSED
*/

STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;	/* address of mailbox */

    {
    *IVH_PTR = INT_MAILBOXIRQ;		/* write mailbox interrupt control */
    *IVH_CTL = IVH_EDGE_SENSITIVE| IVH_VEC_ENABLE;

    *IVH_LRQM |= (1 << INT_MAILBOXIRQ);

    return (OK);
    }
/*******************************************************************************
*
* sysGetProcNum - get processor number
*
* This routine returns the processor number previously set with 
* sysSetProcNum (2).
*
* RETURNS: processor number
*/

int sysGetProcNum ()

    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysSetProcNum - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*/

VOID sysSetProcNum (procNum)
    int procNum;	/* processor number */

    {
    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set across backplane
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* NOTE IV-1624A:
* The test-and-set is done only after acquiring exclusive access to the
* VME bus by using the LOCKVME bit in the control port.
*
* RETURNS: TRUE (successful set), or FALSE (failure)
*/

BOOL sysBusTas (addr)
    char *addr;

    {
    int retVal;

    *IV24_CONTROL_PORT = CTRL_LOCKVME;	/* lock bus */

    if (*addr);				/* access address */

    retVal = vxTas (addr);

    *IV24_CONTROL_PORT = 0;		/* release bus */

    return (retVal);
    }

/* miscellaneous support routines */

/*******************************************************************************
*
* sysJumpers - read PCL jumper settings
*
* This routines returns the value of the 4 PCL jumpers,
* available for software use.
*
* Jumpers are connected to PCL lines of four channels and can be
* read via the Channel Status Register.  They are binary coded in order
* (J10-13) with near board-edge being zero and away from edge being one.
* Manufacturer's default setting is toward board edge, or zero.
*
* NOTE IV-1624A:
* The 12Mhz model must have J10 set to the left, i.e. value 1.
*
* RETURNS: 4 bits of data
*/

int sysJumpers ()

    {
#define	BIT(addr,n)		(((*addr) & 1) << n)
    int result = BIT(J10,0) | BIT(J11,1) | BIT(J12,2) | BIT(J13,3);
    
    /* invert bits of interest */

    return (result ^ 0x0f);
    }
