/* tyCoDrv.c - Force SYS68K/CPU-21/29/32 on-board MPCC tty handler */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
01f,01sep88,gae  fixed tyCoInt again.
01e,24aug88,gae  fixed tyCoInt for second channel thanks to tja.
01d,11jul88,gae  updated to Terry Arden's frc21/9 version.
	   +tja  fixed 2nd MPCC handshake in tyCoInt.
01c,24jun88,gae  lint and cleanup.
01b,22feb88,dnw  changed to use m68561.h, misc clean-up.
01a,29may87,tja  written, by modifying v01d of the Force CPU2d 
*/

/*
DESCRIPTION
This is the driver for the Force SYS68K/CPU-21/29/32 on-board serial ports.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv () to
initialize the driver, and tyCoDevCreate to create devices.

TYCODRV
Before using the driver, it must be created by calling the routine:

.CS
    tyCoDrv ()
.CE

This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS

    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;     /* Name to use for this device *
    int channel;    /* Physical channel for this device (0 or 1) *
    int rdBufSize;  /* Read buffer size, in bytes *
    int wrtBufSize; /* Write buffer size, in bytes *

.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:

.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);

.CE

IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.
The baud rates available are 50, 110, 134, 200, 300, 600, 1200, 1050,
2400, 4800, 7200, 9600, 38400.

*/

#include "UniWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "tyLib.h"
#include "frc21.h"
#include "iv68k.h"
#include "r68561.h"

IMPORT int tyRead ();
IMPORT int tyWrite ();

typedef struct          /* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* TRUE = device already created */
    char *mpcc;		/* address of MPCC base */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv[N_MPCCS] = /* device descriptor */
    {
    {{{{NULL}}}, FALSE, MPCC_BASE_1},
    {{{{NULL}}}, FALSE, MPCC_BASE_2},
    };

LOCAL int tyCoDrvNum;      /* driver number assigned to this driver */

/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID tyCoInt ();


/*******************************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/

STATUS tyCoDrv ()

    {
    /* the Force board uses a Bus Interruptor Module to handle
     * on-board interrupts. A single vectored interrupt is generated
     * for the console MPCC and a single vectored interrupt is
     * generated for the slave MPCC unit by using a Parallel
     * Interface/Timer port. These vectors call a general 
     * interrupt handler to poll interrupt status bits and
     * call the appropriate handler. */

    (void) intConnect (INUM_TO_IVEC (INT_VEC_CO_MPCC),  tyCoInt, 0);
    (void) intConnect (INUM_TO_IVEC (INT_VEC_PIT1_PORT), tyCoInt, 0);

    tyCoHrdInit ();

    tyCoDrvNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen,
				(FUNCPTR) NULL, tyRead, tyWrite, tyCoIoctl);

    return (tyCoDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;     /* Name to use for this device */
    int channel;    /* Physical channel for this device (0 or 1) */
    int rdBufSize;  /* Read buffer size, in bytes */
    int wrtBufSize; /* Write buffer size, in bytes */

    {
    if (tyCoDrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}

    if (tyCoDv[channel].created)
	return (ERROR);

    /* initialize the ty descriptor */

    if (tyDevInit (&tyCoDv[channel].tyDev,
		   rdBufSize, wrtBufSize, tyCoStartup) != OK)
	return (ERROR);

    /* Mark the device as having been created, and add the device to
     * the I/O system */

    tyCoDv[channel].created = TRUE;
    return (iosDevAdd (&tyCoDv[channel].tyDev.devHdr, name, tyCoDrvNum));
    }
/*******************************************************************************
*
* tyCoHrdInit - initialize the MPCC
*
* This routine initializes the Force console MPCC and the slave MPCC for
* the UniWorks environment. 
*
* This routine must be called in supervisor mode, since it accesses I/O space.
*/

LOCAL VOID tyCoHrdInit ()

    {
    int i;
    int oldlevel;       /* current interrupt level mask */
    char zero = 0;
    TY_CO_DEV *ty;
    FAST char *mpcc;

    oldlevel = intLock ();   /* disable interrupts during init */

    for (i = 0; i < N_MPCCS; i++) 
	{
	/* setup pointer to ty device */

	ty = &tyCoDv[i];
	mpcc = ty->mpcc;

	/* Reset the MPCC, xmitter and rcvr. Then clear resets. */

	*MPCC_RCR (mpcc) = MPCC_RCR_RRES;
	*MPCC_TCR (mpcc) = MPCC_TCR_TRES;
	*MPCC_RCR (mpcc) = zero;		/* clear the reset */
	*MPCC_TCR (mpcc) = zero;		/* clear the reset */

	/* Set up the MPCC global regs. Set up protocol reg for
	 * 8 data bits, 1 stop bit. Clock control reg for local clock,
	 * divide by 3 prescaler. Error control reg for no parity.
	 * Initialize to 9600 baud. */

	*MPCC_PSR2 (mpcc) = MPCC_PSR2_1STOP | MPCC_PSR2_8BITS | MPCC_PSR2_ASYNC;
	*MPCC_CCR (mpcc)  = MPCC_CCR_PSCDIV | MPCC_CCR_TCLKO | MPCC_CCR_RCLKIN;

	tyCoSetBaudRate (ty, 9600);

	*MPCC_ECR (mpcc) = zero;		/* no parity */

	/* set up the receiver */

	*MPCC_RIER (mpcc)  = MPCC_RIER_RDA;	/* int only on data avail */
	*MPCC_RCR (mpcc)   = zero;		/* enable the receiver */

	/* set up the xmitter */

	*MPCC_TIER (mpcc)  = zero;		/* int disabled, until later */
	*MPCC_TCR (mpcc)   = MPCC_TCR_TEN;	/* enable the xmitter */

	/* set up the serial interface regs. */

	*MPCC_SIER (mpcc)  = zero;		/* disable interrupts */
	*MPCC_SICR (mpcc)  = MPCC_SICR_RTSLVL | 
			     MPCC_SICR_DTRLVL;	/* Assert DTR and RTS */
	}

    intUnlock (oldlevel);
    } 
/*******************************************************************************
*
* tyCoOpen - open file to MPCC
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/*******************************************************************************
*
* tyCoIoctl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to tyIoctl.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv; /* device to control */
    int request;    /* request code */
    int arg;        /* some argument */

    {
    STATUS status = OK;

    switch (request)
	{
	case FIOBAUDRATE:
	    tyCoSetBaudRate (pTyCoDv, arg);
	    break;

	default:
	    status = tyIoctl (&pTyCoDv->tyDev, request, arg);
	    break;
	}

    return (status);
    }
/*******************************************************************************
*
* tyCoInt - handle MPCC interrupt(s)
*
* This routine determines which source is generating an interrupt
* on the MPCC and subsequently passes control to the appropriate
* interrupt routine.
*/

LOCAL VOID tyCoInt ()

    {
    /* check console interrupts */

    if (*MPCC_RSR (MPCC_BASE_1) & MPCC_RSR_RDA)
	tyCoRxInt (0);		/* receiver interrupt */
    else if (*MPCC_TSR (MPCC_BASE_1) & MPCC_TSR_TDRA)
	tyCoTxInt (0);		/* xmitter interrupt */
    else
	{
	/* must be 2nd MPCC so reset h4 handshake */

	*PIT_PSR (PIT_BASE_1) &= ~H4_SENSE;
	*PIT_PSR (PIT_BASE_1) |=  H4_SENSE;

	if (*MPCC_RSR (MPCC_BASE_2) & MPCC_RSR_RDA)
	    tyCoRxInt (1);	/* receiver interrupt */
	else if (*MPCC_TSR (MPCC_BASE_2) & MPCC_TSR_TDRA)
	    tyCoTxInt (1);	/* xmitter interrupt */
	}
    }
/*******************************************************************************
*
* tyCoRxInt - handle a receiver interrupt
*
* This routine handles an interrupt from the on-board MPCC receiver.
*/

LOCAL VOID tyCoRxInt (channel)
    FAST int channel;

    {
    FAST TY_CO_DEV *ty = &tyCoDv[channel];

    /* get character from MPCC and give it to I/O system */

    tyIRd ((TY_DEV_ID) ty, *MPCC_RDR (ty->mpcc));

    *MPCC_RSR (ty->mpcc) = 0xff;	/* clear the interrupt bit */
    }
/*******************************************************************************
*
* tyCoTxInt - handle a transmitter interrupt
*
* This routine handles an interrupt from the on-board MPCC xmitter.
* If there is another character to be transmitted, it sends it.  If
* not, or if a device has never been created, we just
* disable the interrupt.
*/

LOCAL VOID tyCoTxInt (channel)
    FAST int channel;

    {
    FAST TY_CO_DEV *ty = &tyCoDv[channel];
    FAST char *mpcc = ty->mpcc;
    char outchar;
    char zero = 0;

    *MPCC_TIER (mpcc) = zero;		/* disable tx interrupts */

    /* get character to output, if any, from i/o system */

    if (tyITx (&ty->tyDev, &outchar) == OK)
	{
	*MPCC_TSR (mpcc)  = 0xff;		/* clear the interrupt bit */
	*MPCC_TDR (mpcc)  = outchar;		/* output char */
	*MPCC_TIER (mpcc) = MPCC_TIER_TDRA;	/* enable tx interrupt */
	}
    }
/*******************************************************************************
*
* tyCoSerialInt - serial interface interrupt level processing
*
* This routine handles an interrupt from the MPCC serial interface.
* Receiver and xmitter interrupts go ththrough different routines.
* This routine would catch DSR and CTS, except that we don't enable
* those anyway, so this routine should never be called.
*
* ARGSUSED
*/

LOCAL VOID tyCoSerialInt (channel)
    FAST int channel;

    {
    char zero = 0;

    *MPCC_SIER (tyCoDv[channel].mpcc) = zero;	/* disable serial interrupts */
    }
/*******************************************************************************
*
* tyCoStartup - transmitter startup routine
*
* Output a single character.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    FAST TY_CO_DEV *pTyCoDv;	/* ty device to start up */

    {
    char outchar;

    /* any character to send ? */

    if (tyITx (&pTyCoDv->tyDev, &outchar) == OK)
	{
	*MPCC_TDR (pTyCoDv->mpcc)  = outchar;		/* output char */
	*MPCC_TIER (pTyCoDv->mpcc) = MPCC_TIER_TDRA;	/* enable tx interrupt*/
	}
    }
/*******************************************************************************
*   
* tyCoSetBaudRate - set the MPCC's baud rate
*
* The baud rate is set up assuming that:
*     1. The local xtal oscillator is the clock.
*     2. The /3 prescaler is in place (PSCDIV == 1)
*     3. We're in ASYNC mode (/2).
*/

LOCAL VOID tyCoSetBaudRate (pTyCoDv, rate)
    TY_CO_DEV *pTyCoDv;		/* device to control */
    int rate;			/* desired rate */

    {
    FAST int brdiv = TY_XTAL_FREQ / (rate * 6);

    *MPCC_BRDR1 (pTyCoDv->mpcc) = LSB(brdiv);
    *MPCC_BRDR2 (pTyCoDv->mpcc) = MSB(brdiv);
    }
