/* taskLib.c - task management interface for the VRTX kernel */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library provides a generic interface to the VRTX kernel
task management.  Task management includes:

    - create, delete, suspend, resume, restart, prioritize tasks,
    - lock and unlock scheduling,
    - delay a task from running,
    - get and set tasks' registers,
    - get and set tasks' options,
    - find TCBs (task control blocks),
    - find TCBXs (task control block extensions),
    - find TDs (kernel independent task descriptors),
    - translate task ids to task names, and vice versa.

INCLUDE FILE: taskLib.h

SEE ALSO:
taskHookLib (1), taskVarLib (2), semLib (1), kernelLib (1), "Architecture"
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "taskLib.h"
#include "memLib.h"
#include "strLib.h"

#if ((CPU == MC68020) || (CPU == MC68030))
#include "vrtx32.h"
#else ((CPU != MC68020) && (CPU != MC68030))
#include "vrtx.h"
#endif (CPU)


IMPORT VOID vxTaskEntry ();
IMPORT FUNCPTR taskCreateTable[];
IMPORT FUNCPTR taskDeleteTable[];
IMPORT VRTX_CONFIG_TABLE usrCftbl;

/* globals */

/* taskIdCurrent is modified on each context switch to be the task id of
 * the running task;  it is referred to by taskIdSelf() to get the taskId
 * of the running task */

ULONG taskIdCurrent;		/* task id of current task */

/* locals */

LOCAL FUNCPTR taskBpHook;	/* rtn to install/remove bkpts for task */

/* forward declarations */

VX_TCB *taskTcb ();

/*******************************************************************************
*
* taskBpHookSet - set breakpoint hook for dbgLib
*
* This routine allows dbgLib to install its break-point install/remove routine
* used by taskOptionsSet.
* It should only be called by dbgInit (2).
*
* NOMANUAL
*/

VOID taskBpHookSet (bpHook)
    FUNCPTR bpHook;

    {
    taskBpHook = bpHook;
    }
/*******************************************************************************
*
* taskSpawn - spawn a task
*
* This routine creates and activates a new task with the specified parameters.
* If the task name is specified as NULL, the name will be an ASCII
* hexadecimal number of the task id.
*
* Bits in the options argument may be set to run with the following modes.
* .CS
*       VX_SUPERVISOR_MODE      -  execute in supervisor mode
*       VX_UNBREAKABLE          -  don't allow break point debugging
*       VX_DEALLOC_STACK        -  deallocate the stack on deletion
*       VX_FP_TASK              -  execute with coprocessor support
*       VX_STDIO                -  execute with standard I/O support
* .CE
* See definitions in taskLib.h.
*
* The entry address is the address of the main routine of the task.
* This routine will be called with the specified arguments (up to 10).
* A stack of the specified size will be allocated from the memory pool.
* Stack size should be even.  Every byte of this stack is initialized to 0xEE,
* as a debugging aid.  See checkStack (2) for stack size checking aids.
*
* See taskCreate (2) for creating tasks with preallocated stacks.
*
* BUGS:
* All tasks are spawned in supervisor mode, regardless of
* the specified options.  In effect, VX_SUPERVISOR_MODE is or'd into the
* user specified options.  This is done because other UniWorks facilities
* do not make a distinction of privileged instructions.
*
* RETURNS: task id, or ERROR if out of memory or unable to create task
*
* VARARGS5
*/

int taskSpawn (name, priority, options, stacksize, entryAddress, arg1,
	       arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10)
    char *name;			/* task name of new task     */
    int priority;		/* priority of new task      */
    int options;		/* options word for new task */
    int stacksize;		/* size (bytes) of stack     */
    FUNCPTR entryAddress;	/* entry point of task       */
    int arg1;			/* arguments passed to task  */
    int arg2;
    int arg3;
    int arg4;
    int arg5;
    int arg6;
    int arg7;
    int arg8;
    int arg9;
    int arg10;

    {
    char *topOfStack;
    char *bottomOfStack;
    char *newName = "1234567890";
    char *mallocName;
    int tid;

    if (name == NULL)
	name = newName;	/* nameless task gets id as name, eventually */

    /* allocate memory for stack and fill it for debugging */

    topOfStack = malloc ((unsigned) (stacksize + sizeof (TCBX) + 
				      		  strlen (name) + 1));

    if (topOfStack == NULL)
	return (ERROR);

    bottomOfStack = topOfStack + stacksize;

    mallocName = (char *)(((int) bottomOfStack) + sizeof (TCBX));


    bfill (topOfStack, stacksize, 0xee);

    tid = taskCreate (mallocName, priority, options | VX_DEALLOC_STACK,
		      bottomOfStack, stacksize, (TCBX *) bottomOfStack,
		      entryAddress, arg1, arg2, arg3, arg4,
		      arg5, arg6, arg7, arg8, arg9, arg10);

    if (tid == ERROR || taskActivate (tid) == ERROR)
	{
	free (topOfStack);
	return (ERROR);
	}

    /* we name it the task id after it's spawned */

    if (name == newName)	
	sprintf (mallocName, "%x", tid);	/* copy tid into name field */
    else
	strcpy (mallocName, name);		/* name was provided */

    return (tid);
    }
/*******************************************************************************
*
* taskCreate - create a task with stack at specified address
*
* This routine works like taskSpawn (2),
* but instead of allocating the stack and TCB extension automatically,
* it uses the stack and TCB extension passed as arguments.
* The stack will grow towards lower memory locations
* starting from the specified bottom of stack.
*
* Bits in the options argument may be set to run with the following modes.
* .CS
*       VX_SUPERVISOR_MODE      -  execute in supervisor mode
*       VX_UNBREAKABLE          -  don't allow debugging
*       VX_DEALLOC_STACK        -  deallocate the stack on deletion
*       VX_FP_TASK              -  execute with coprocessor support
*       VX_STDIO                -  execute with standard I/O support
* .CE
* See definitions in taskLib.h.
*
* The name of a task should be malloc'd along with the stack and TCB extension.
* taskSpawn will automatically name an unnamed task but taskCreate will not.
* Normally, tasks should be started by taskSpawn (2), not by taskCreate.
*
* BUGS:
* All tasks are spawned in supervisor mode, regardless of
* the specified options.  In effect, VX_SUPERVISOR_MODE is or'ed into the
* user specified options.  This is done because other UniWorks facilities
* do not yet make the distinction of privileged instructions.
*
* RETURNS: task id, or ERROR if unable to create task
*
* SEE ALSO: taskActivate (2), taskSpawn (2)
*
* VARARGS7
*/

int taskCreate (name, priority, options, botOfStack, stksize, pTcbx,
		entryAddress, arg1)
    char *name;			/* name of new task           */
    int priority;		/* priority of new task       */
    int options;		/* task option word           */
    FAST char *botOfStack;	/* bottom of new task's stack */
    int stksize;		/* size (bytes) of stack      */
    FAST TCBX *pTcbx;		/* address of new task's TCBX */
    FUNCPTR entryAddress;	/* initial pc of new task     */
    int arg1;			/* 1st of up to 10 arguments  */

    {
    static BOOL beenHereBefore;	/*XXX KLUDGE XXX */
    FAST int tid;
    FAST int *argptr;
    FAST int *sp;
    FAST int ix;
    STATUS status;

    if ((int) name == NULL)
	return (ERROR);

    options |= VX_SUPERVISOR_MODE;	/* TEMP! set SUP mode for all tasks */

    /* push args on the stack */

    argptr = &arg1 + MAX_TASK_ARGS;
    sp = (int *) botOfStack;

    for (ix = 0; ix < MAX_TASK_ARGS; ix++)
	*--sp = *--argptr;

    /* initialize tcb extension */

    bzero ((char *) pTcbx, sizeof (TCBX));

    pTcbx->entry       = entryAddress;
    pTcbx->errorStatus = OK;
    pTcbx->topOfStack  = botOfStack - stksize;
    pTcbx->botOfStack  = botOfStack;
    pTcbx->initialSP   = (char *) sp;
    pTcbx->options     = options;
    pTcbx->name        = name;

    for (ix = 0; ix < 3; ++ix)
	pTcbx->taskStd[ix] = ix;

    /* stop context switches so taskId won't get used by another task
     * in between the time that the task id is fetched and used,
     * and also, to keep the the new task from starting before we've had
     * a chance to run the create hooks.
     */
    vxLock ();			/* disable task switching */

    /* XXX MAJOR KLUDGE XXX */
    if (beenHereBefore)
	tid = vxGetFreeTaskId ();	/* next possible task id */
    else
	tid = 255;
    beenHereBefore = TRUE;
    /* XXX TO HERE XXX */

    pTcbx->taskId = tid;

    status = vxTXcreate (priority, tid, options & VX_SUPERVISOR_MODE,
			   vxTaskEntry, botOfStack, pTcbx->initialSP, pTcbx);

    if (status != OK)
	{
	vxUnlock ();
	return (ERROR);
	}

    /* the task has been created */

    vxTsuspend (tid);	/* suspend the task so we can run create hooks */

    /* Task switching must be enabled so create hooks can take semaphores
     * (ie. malloc) but this has a side effect that the task we have spawned
     * could be resumed before all of the create hooks are run. */

    vxUnlock ();

    return (tid);
    }
/*******************************************************************************
*
* taskActivate - activate an already created task
*
* A task is created in the suspended state and must be activated before it
* joins the ready list.  taskCreate and taskActivate are the lower level
* routines to taskSpawn (2).  A task may only be activated once.
*
* RETURNS: OK or ERROR if invalid task id or unable to activate task
*/

STATUS taskActivate (tid)
    int tid;			/* task id of new task */

    {
    FAST int ix;
    FAST TCBX *pTcbX = taskTcbX (tid);

    if (pTcbX == NULL)		/* task non-existent */
	return (ERROR);

    for (ix = 0; ix < VX_MAX_TASK_CREATE_RTNS; ++ix)	/* run create hooks */
	{
	if (taskCreateTable[ix] != NULL)
	    (*taskCreateTable[ix]) (pTcbX);
	}

    return (vxTresume (tid));	/* resume the task */
    }
/*******************************************************************************
*
* taskDelete - delete a task
*
* This routine suspends the task, calls any routines specified by
* taskDeleteHookAdd (2), deletes the task, and reclaims its stack.
*
* If a task attempts to delete itself, the deletion will be performed in
* the context of the exception task.
*
* RETURNS: OK or ERROR if invalid task id
*
* SEE ALSO: excLib (1), taskHookLib (1), taskDeleteHookAdd (2)
*
* VARARGS0
*/

IMPORT FUNCPTR taskVarDeleteHook();

STATUS taskDelete (tid)
    int tid;			/* task id of task to delete */

    {
    STATUS status;
    FAST int ix;
    TCBX *pTcbX = taskTcbX (tid);
    FUNCPTR tmp = NULL; 

    if (pTcbX == NULL)			/* task non-existent */
	return (ERROR);

    /* requests to kill self get sent to the exception task */

    if ((tid == 0) || (tid == taskIdCurrent))
	{
	/* if a task with id 0 tries to kill itself, we can't reclaim its
	 * stack or run delete hooks because there is no handle for the
	 * task for the exception task to use.
	 */

	if (taskIdSelf () == 0)
	    vxTdelete (0);
	else
	    excToDoAdd (taskDelete, taskIdCurrent);

	vxTsuspend (0);	/* wait for exception task to kill us */
	}


    /* the rest of this routine will only be executed by a task other than
     * the one being deleted */

    vxTsuspend (tid);

    /* run the delete hooks - make sure we call taskVarDeleteHook last */

    for (ix = 0; ix < VX_MAX_TASK_DELETE_RTNS; ++ix) {
	if (taskDeleteTable[ix] != NULL) {
	    if((*taskDeleteTable[ix]) == taskVarDeleteHook)
		tmp = taskVarDeleteHook;
	    else {
	    	(*taskDeleteTable[ix]) (pTcbX);
	    }
	}
    }

    if(tmp != NULL)
	(*tmp) (pTcbX);

    status = vxTdelete (tid);

    /* is the deallocate stack bit set in the task option word? */

    if (pTcbX->options & VX_DEALLOC_STACK)
	{
	/* topOfStack points to the stack and includes the TCBX */

        free (pTcbX->topOfStack);
	}

    return (status);
    }
/*******************************************************************************
*
* exit - exit a task
*
* A task may call this routine to exit (cease to exist as a task).
* A task can also exit simply by returning from the main routine.
*
* SEE ALSO: taskDelete (2)
*
* VARARGS0
* ARGSUSED
*/

VOID exit (code)
    int code;   /* just because standard C library exit takes an argument */
 
    {
    (void) taskDelete (0);
    }
/*******************************************************************************
*
* taskSuspend - suspend a task
*
* The specified routine will be suspended.  A task id of zero results in
* the suspension of the calling routine.
*
* RETURNS: OK or ERROR if unable to suspend task
*/

STATUS taskSuspend (tid)
    int tid;			/* task id of task to suspend */

    {
    return (vxTsuspend (tid));
    }
/*******************************************************************************
*
* taskResume - resume a task
*
* The specified task will be resumed.
*
* RETURNS: OK, or ERROR if invalid task id or unable to resume task
*/

STATUS taskResume (tid)
    int tid;			/* task id of task to resume */

    {
    return (vxTresume (tid));
    }
/*******************************************************************************
*
* taskRestart - restart a task
*
* This routine "restarts" a task.  The task is first deleted.
* Then it is recreated, with the same id, priority, options,
* original entry point, stack size, and (up to 10) parameters it had
* when it was deleted.
*
* NOTE
* If the task has modified any of its startup parameters, the
* restarted task will start with the changed values.
*
* RETURNS:
*  OK, or
*  ERROR if invalid task id, or couldn't be re-started
*/

int taskRestart (tid)
    int tid;			/* task id of task to restart */

    {
    int priority;
    int options;
    FUNCPTR entry;	/* entry point of task */
    int stacksize;
    int args [MAX_TASK_ARGS];
    TCBX *pTcbx;
    char *name;
    VX_TCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)
	return (ERROR);		/* specified task not found */

    pTcbx = pTcb->tcb_extension;

    /* save info from tcb that is about to be deleted */

    name      = pTcbx->name;
    priority  = pTcb->tcb_priority;
    options   = pTcbx->options;
    entry     = pTcbx->entry;
    stacksize = pTcbx->botOfStack - pTcbx->topOfStack;
    bcopy ((char *) pTcbx->initialSP, (char *) args, sizeof (args));

    (void) taskDelete (tid);

    return (taskSpawn (name, priority, options, stacksize, entry,
		       args[0], args[1], args[2], args[3], args[4],
		       args[5], args[6], args[7], args[8], args[9]));
    }
/*******************************************************************************
*
* taskOptionsSet - change task options
*
* Change the execution options of a task.
*
* Bits in the options argument may be set to run with the following modes.
* .CS
*       VX_SUPERVISOR_MODE      -  execute in supervisor mode
*       VX_UNBREAKABLE          -  don't allow debugging
*       VX_DEALLOC_STACK        -  deallocate the stack on deletion
*       VX_FP_TASK              -  execute with coprocessor support
*       VX_STDIO                -  execute with standard I/O support
* .CE
* See definitions in taskLib.h.
*
* RETURNS: OK, or ERROR if invalid task id
*
* SEE ALSO: taskOptionsSet (2)
*/

STATUS taskOptionsSet (tid, mask, value)
    int tid;		/* task id                   */
    int mask;		/* mask, 1 = change this bit */
    int value;		/* new options               */

    {
    FAST TCBX *pTcbX = taskTcbX (tid);

    if (pTcbX == NULL)
	return (ERROR);

    pTcbX->options = (pTcbX->options & ~mask) | value;

    /* if we are setting/resetting unbreakable option for current task,
     * then call breakpoint hook */

    if ((mask & VX_UNBREAKABLE) &&
	((tid == 0) || (tid == taskIdCurrent)) &&
	(taskBpHook != NULL))
	{
	(* taskBpHook) ((value & VX_UNBREAKABLE) == 0);
	}

    return (OK);
    }
/*******************************************************************************
*
* taskOptionsGet - examine task options
*
* This routine gets the specified tasks current execution options.
*
* Bits in the options argument may be set to run with the following modes.
* .CS
*       VX_SUPERVISOR_MODE      -  execute in supervisor mode
*       VX_UNBREAKABLE          -  don't allow debugging
*       VX_DEALLOC_STACK        -  deallocate the stack on deletion
*       VX_FP_TASK              -  execute with coprocessor support
*       VX_STDIO                -  execute with standard I/O support
* .CE
* See definitions in taskLib.h.
*
* RETURNS: OK or ERROR if invalid task id
*
* SEE ALSO: taskOptionsSet (2)
*/

STATUS taskOptionsGet (tid, pOptions)
    int tid;		/* task id        */
    int *pOptions;	/* task's options */

    {
    FAST TCBX *pTcbX = taskTcbX (tid);

    if (pTcbX == NULL)
	return (ERROR);

    *pOptions = pTcbX->options;
    return (OK);
    }
/*******************************************************************************
*
* taskPrioritySet - change priority of a task
*
* This routine changes the specified task's priority to the specified priority.
*
* RETURNS: OK, or ERROR if invalid task id
*
* SEE ALSO: taskPriorityGet (2)
*/

STATUS taskPrioritySet (tid, newPriority)
    int tid;			/* task id      */
    int newPriority;		/* new priority */

    {
    return (vxTpriority (tid, newPriority));
    }
/*******************************************************************************
*
* taskPriorityGet - examine priority of a task
*
* This routine is used to determine the current priority of a specified task.
*
* RETURNS:
*  OK and pPriority gets task priority, or
*  ERROR if invalid task id
*
* SEE ALSO: taskPrioritySet (2)
*/

STATUS taskPriorityGet (tid, pPriority)
    int tid;		/* task id                  */
    int *pPriority;	/* where to return priority */

    {
    FAST VX_TCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)
	return (ERROR);

    *pPriority = pTcb->tcb_priority;

    return (OK);
    }
/*******************************************************************************
*
* taskLock - disable kernel rescheduling
*
* This routine disables task context switching.
* The task that calls this routine
* must never preempt itself while task switching is disabled.
*
* For instance, semaphores should not be taken.
* This routine does not lock interrupts.
* To do so one should call intLock (2).
* taskLock (2) is preferable to intLock (2) because locking out interrupts
* adds interrupt latency to the system.
*
* RETURNS: OK
*
* SEE ALSO: taskUnlock(2), intLock(2)
*/

STATUS taskLock ()

    {
    vxLock ();
    return (OK);
    }
/*******************************************************************************
*
* taskUnlock - enable kernel rescheduling
*
* This routine is used to resume task context switching after a
* taskLock (2) has disabled it.
* Any tasks which were eligible to preempt the current task will now execute.
*
* RETURNS: OK
*
* SEE ALSO: taskLock (2)
*/

STATUS taskUnlock ()

    {
    vxUnlock ();
    return (OK);
    }
/*******************************************************************************
*
* taskDelay - delay a task from executing
*
* This routine caused the calling task to relinquish the CPU for the duration
* (in ticks) specified.  This is commonly referred to as manual rescheduling but
* it is also useful when waiting for some external condition that does not have
* an interrupt associated with it.
*
* RETURNS: OK, or ERROR if invalid task id
*/

STATUS taskDelay (ticks)
    int ticks;		/* number of ticks to suspend task */

    {
    return (vxTdelay (ticks));
    }
/*******************************************************************************
*
* taskTcb - get the task control block for a task id
*
* This routine returns a pointer to the TCB of the specified task.
*
* RETURNS: pointer to VRTX TCB, or NULL if invalid task id
*/

VX_TCB *taskTcb (tid)
    int tid;			/* task id */

    {
    VX_TCB *pTcb;
    int dummy;			/* bucket for stuff we don't need */

    if (vxTinquiry (tid, &dummy, &dummy, &dummy, &pTcb) == OK)
	return (pTcb);
    else
	{
	errnoSet (S_taskLib_TASK_ID_ERROR);
	return (NULL);
	}
    }
/*******************************************************************************
*
* taskTcbX - get the task control block extension for a task id
*
* This routine returns a pointer to the task control block extension.
* The contents of the TCBX is defined in taskLib.h.
* Users should not directly modify the contents of the TCB extension.
*
* RETURNS: pointer to TCB extension, or NULL if invalid task id
*/

TCBX *taskTcbX (tid)
    int tid;		/* task id */

    {
    VX_TCB *pTcb;	/* temp ptr to tcb */
    int dummy;		/* bucket for stuff we don't need */

    if (vxTinquiry (tid, &dummy, &dummy, &dummy, &pTcb) == OK)
	return (pTcb->tcb_extension);
    else
	{
	errnoSet (S_taskLib_TASK_ID_ERROR);
	return (NULL);
	}
    }
/*******************************************************************************
*
* taskTcbToTd - fill in task descriptor structure from VRTX TCB
*
* This routine assumes we have a legal pTcb.  It is a local routine that
* does all the work for the higher level call taskInfoGet (2).
*/

LOCAL VOID taskTcbToTd (pTcb, pTaskDesc)
    FAST VX_TCB *pTcb;	/* pointer to VRTX task control block */
    FAST TASK_DESC *pTaskDesc;	/* task descriptor to be filled in */

    {
    pTaskDesc->td_name	   = pTcb->tcb_extension->name;
    pTaskDesc->td_priority = pTcb->tcb_priority;
    pTaskDesc->td_id	   = pTcb->tcb_id;
    pTaskDesc->td_status   = pTcb->tcb_status;
    pTaskDesc->td_sp	   = pTcb->tcb_sp;
    pTaskDesc->td_usp	   = pTcb->tcb_usp;
    pTaskDesc->td_spbottom = pTcb->tcb_extension->botOfStack;
#if ((CPU == MC68020) || (CPU == MC68030))
    pTaskDesc->td_delay	   = (pTcb->tcb_status & VRTX_DELAY) ?
			     ((VX_TCB32 *)((char *)pTcb - 20))->t32_delay :
			     0;
#else ((CPU != MC68020) && (CPU != MC68030))
    pTaskDesc->td_delay    = pTcb->tcb_delay;
#endif (CPU)
    pTaskDesc->entry	   = pTcb->tcb_extension->entry;
    pTaskDesc->errorStatus = pTcb->tcb_extension->errorStatus;
    pTaskDesc->topOfStack  = pTcb->tcb_extension->topOfStack;
    pTaskDesc->initialSP   = pTcb->tcb_extension->initialSP;
    pTaskDesc->options	   = pTcb->tcb_extension->options;
    }
/*******************************************************************************
*
* taskInfoGet - get pointer to task's task descriptor
*
* This routine finds the TASK_DESC (task descriptor) of the specified task.
*
* NOTE
* Examination of TCBs should be restricted to debugging aids.
*
* RETURNS: OK or ERROR if invalid task id
*/

STATUS taskInfoGet (tid, pTaskDesc)
    int tid;			/* task id                        */
    TASK_DESC *pTaskDesc;	/* where to store task descriptor */

    {
    VX_TCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)
	return (ERROR);

    taskTcbToTd (pTcb, pTaskDesc);
    return (OK);
    }
/*******************************************************************************
*
* taskName - get name associated with a task id
*
* This routine returns the name of a task.
*
* RETURNS: pointer to task name, or NULL if invalid task id
*/

char *taskName (tid)
    int tid;		/* task id */

    {
    TCBX *pTcbX = taskTcbX (tid);

    if (pTcbX == NULL)
	return ((char *) NULL);

    return (pTcbX->name);
    }
/*******************************************************************************
*
* taskNameToId - lookup task id associated with a name
*
* This routine returns the task id.
*
* RETURNS: task id or ERROR if task by name `name' not found
*/

int taskNameToId (name)
    char *name;		/* name of task to look up */

    {
#if ((CPU == MC68020) || (CPU == MC68030))
    FAST VX_TCB32 *pT32;	/* pointer to VRTX32 super TCB */
#else ((CPU != MC68020) && (CPU != MC68030))
    FAST VX_TCB *pTcb;		/* pointer to VRTX TCB */
#endif (CPU)

#if ((CPU == MC68020) || (CPU == MC68030))
    /* The first 4 bytes of VRTX32 workspace is a pointer to the first super-TCB
     * in the active list. At an offset of 0x5a into that super-TCB is a pointer
     * to the next super-TCB, and so on. The last super-TCB in the list has a
     * pointer to 0x5a before the beginning of VRTX32 workspace, so that when
     * that pointer is followed we wind up at the head of the list. Thus the end
     * of the active list can only be detected by the fact that a super-TCB is
     * at a suspiciously low address. I hope we never get bitten by this. */

    for (pT32 = *(VX_TCB32**)usrCftbl.wrkspcAdrs;
	 (pT32 > (VX_TCB32*)usrCftbl.wrkspcAdrs);
	 pT32 = pT32->t32_next)
	{
	if (strcmp (pT32->t32_tcb.tcb_extension->name, name) == 0)
	    return (pT32->t32_tcb.tcb_id);	/* found it */
	}
#else ((CPU != MC68020) && (CPU != MC68030))
    for (pTcb = (*(VX_TCB**) usrCftbl.wrkspcAdrs);
	 (pTcb != NULL) && ((int)pTcb != -1);
	 pTcb = pTcb->tcb_next)
	{
	if (strcmp (pTcb->tcb_extension->name, name) == 0)
	    return (pTcb->tcb_id);		/* found it */
	}
#endif (CPU)

    errnoSet (S_taskLib_NAME_NOT_FOUND);
    return (ERROR);				/* found no match */
    }
/*******************************************************************************
*
* taskIsSuspended - check whether a task is suspended
*
* This routine tests the status field of the specified task to determine
* if it is suspended.
*
* RETURNS: TRUE if task is suspended, otherwise FALSE
*/

BOOL taskIsSuspended (tid)
    int tid;	/* task id */

    {
    FAST VX_TCB *pTcb = taskTcb (tid);

    if (pTcb != NULL && pTcb->tcb_status & VRTX_SUSPEND)
	return (TRUE);

    return (FALSE);
    }
/*******************************************************************************
*
* taskIsReady - check if a task is ready to run
*
* This routine tests the status field of the specified task to determine
* if it is ready.
*
* RETURNS: TRUE if task is ready, otherwise FALSE
*/

BOOL taskIsReady (tid)
    int tid;	/* task id */

    {
    FAST VX_TCB *pTcb = taskTcb (tid);

    if (pTcb != NULL && pTcb->tcb_status == VRTX_READY)
	return (TRUE);

    return (FALSE);
    }
/*******************************************************************************
*
* taskStatusString - return the task's status as a string
*
* This routine converts the VRTX task status word in the TCB
* and returns the appropriate string.
*
* EXAMPLE
* .CS
*  -> taskStatusString (taskNameToId ("shell"), xx=malloc(10));
*  new symbol "xx" added to symbol table.
*  value = 0 = 0x0
*  -> printf ("shell status = <%s>\n", xx)
*  shell status = <READY>
*  value = 2 = 0x2
*  ->
* .CE
*
* RETURNS:
*  OK, or ERROR if invalid task id, `pString' gets status string
*/

STATUS taskStatusString (tid, pString)
    int tid;		/* task to get string for            */
    char *pString;	/* where to return string (10 bytes) */

    {
    FAST VX_TCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)
	return (ERROR);

    switch (pTcb->tcb_status & 0x5e)
	{
	case 0x00:
	    switch (pTcb->tcb_status & (VRTX_DELAY | VRTX_SUSPEND))
		{
		case VRTX_READY:
		    strcpy (pString, "READY");
		    break;
		case VRTX_SUSPEND:
		    strcpy (pString, "SUSPEND");
		    break;
		case VRTX_DELAY:
		    strcpy (pString, "DELAY");
		    break;
		case (VRTX_DELAY | VRTX_SUSPEND):
		    strcpy (pString, "DELAY+S");
		    break;
		}
	    return (OK);

	case VRTX_PEND:		strcpy (pString, "PEND");  break;
	case VRTX_GETC:		strcpy (pString, "GETC");  break;
	case VRTX_PUTC:		strcpy (pString, "PUTC");  break;
	case VRTX_WAITC:	strcpy (pString, "WAITC"); break;
	case VRTX_QPEND:	strcpy (pString, "QPEND"); break;

	default:			/* unanticipated combination */
	    sprintf (pString, "0x%02x", pTcb->tcb_status);
	    return (OK);
	}

    if (pTcb->tcb_status & VRTX_SUSPEND)
	strcat (pString, "+S");		/* task also explicitly suspended */

    if (pTcb->tcb_status & VRTX_DELAY)
	strcat (pString, "+T");		/* timeout also in effect */

    return (OK);
    }
/*******************************************************************************
*
* taskIdDefault - default task id
*
* This routine maintains a global default task id.  This id is used by
* libraries that want to allow a task id argument to take on a default
* value if the user did not explicitly supply one.
* If the task id supplied to this routine is not zero (i.e. the user
* did specify a taskId) then the default id is set to that value,
* and that value is returned.  If the task id supplied to this routine
* is zero (i.e. the user did not specify a task id) then the default id
* is not changed and its value is returned.  Thus the value returned
* is always the last task id the user specified.
*
* RETURNS: most recently set non-zero task id
*
* SEE ALSO: dbgLib (1), "Debugging"
*/

int taskIdDefault (tid)
    int tid;    /* user supplied task id, if 0 then just return default */

    {
    static int defaultTaskId;	/* current default task id */

    if (tid != 0)
	defaultTaskId = tid;		/* update default */

    return (defaultTaskId);
    }
/*******************************************************************************
*
* taskIdSelf - get task id of running task
*
* Get task id of calling task. The task id will be invalid if called
* at interrupt level.  This routine simply returns global variable
* `taskIdCurrent' which is only valid during a task context.
*
* RETURNS: task id of calling task
*/

int taskIdSelf ()

    {
    return (taskIdCurrent);
    }
/*******************************************************************************
*
* taskIdVerify - verify the existence of a task
*
* This verifies the existence of the specified task by matching the task id
* with a encrypted version located within the task control block.
*
* RETURNS: OK or ERROR if invalid task id
*/

STATUS taskIdVerify (tid)
    int tid;	/* task id */

    {
    return ((taskTcb (tid) == NULL) ? ERROR : OK);
    }
/*******************************************************************************
*
* taskIdListGet - fill in array of active task ids
*
* This routine provides the calling task with a list of all of the active
* tasks.  The list consists of task ids, the last task id is a terminating 0.
* The list is unordered.
*
* WARNING:
* Kernel rescheduling is disabled with taskLock (2) while
* tasks are looked up.  There is no guarantee that all the
* tasks are valid and/or no new tasks have been created by
* the time this routine returns.
*
* RETURNS: number of tasks in list
*/

int taskIdListGet (idList, maxTasks)
    FAST int idList[];		/* array of task ids to be filled in */
    FAST int maxTasks;		/* max tasks idList can accommodate  */

    {
    FAST int *pId = idList;
#if ((CPU == MC68020) || (CPU == MC68030))
    FAST VX_TCB32 *pT32 = *(VX_TCB32**) usrCftbl.wrkspcAdrs;	/* tcb head */
#else ((CPU != MC68020) && (CPU != MC68030))
    FAST VX_TCB *pTcb = *(VX_TCB**) usrCftbl.wrkspcAdrs;	/* tcb head */
#endif (CPU)

    vxLock ();

#if ((CPU == MC68020) || (CPU == MC68030))
    /* The first 4 bytes of VRTX32 workspace is a pointer to the first
     * super-TCB in the active list. At an offset of 0x5a into that
     * super-TCB is a pointer to the next super-TCB, and so on. The last
     * super-TCB in the list has a pointer to 0x5a before the beginning
     * of VRTX32 workspace, so that when that pointer is followed we wind
     * up at the head of the list. Thus the end of the active list can
     * only be detected by the fact that a super-TCB is at a suspiciously
     * low address. I hope we never get bitten by this. */

    while ((pT32 > (VX_TCB32*) usrCftbl.wrkspcAdrs) && (--maxTasks >= 0))
	{
	*(pId++) = pT32->t32_tcb.tcb_id;
	pT32 = pT32->t32_next;
	}

#else ((CPU != MC68020) && (CPU != MC68030))

    while ((pTcb != NULL) && ((int)pTcb != -1) && (--maxTasks >= 0))
	{
	*(pId++) = pTcb->tcb_id;
	pTcb = pTcb->tcb_next;
	}

#endif (CPU)

    vxUnlock ();

    return (pId - idList);
    }
/*******************************************************************************
*
* taskRegsGet - get a task's registers from TCB and stack
*
* This routine gathers information about a task which is normally kept
* scattered about.  It returns, in the locations whose pointers are
* passed as parameters, the task's address and data registers, and its
* SP, SR and PC.  The address and data registers are returned in
* separate arrays, each containing 8 registers.
*
* NOTE
* This routine only works well if the specified task's TCB is not that
* of the calling task.  Results are unpredictable if a task tries to
* discover its own registers.
*
* RETURNS: OK, or ERROR if invalid task id
*
* SEE ALSO: taskRegsSet(2)
*/

STATUS taskRegsGet (tid, dregs, aregs, sp, sr, pc)
    int tid;		/* task id                            */
    FAST int dregs[];	/* buffer for data registers (0-7)    */
    FAST int aregs[];	/* buffer for address registers (0-6) */
    char **sp;		/* buffer for stack pointer           */
    USHORT *sr;		/* buffer for status register         */
    INSTR **pc;		/* buffer for program counter         */

    {
    FAST int ix;
    FAST VX_TCB *pTcb = taskTcb (tid);
    FAST TCB_STACK *pTcbStack;

    if (pTcb == NULL)
	return (ERROR);

    pTcbStack = (TCB_STACK *) pTcb->tcb_sp;

    /* is the task supervisor or user? */

    if ((pTcbStack->sr & 0x2000) == 0)
	*sp = pTcb->tcb_usp;		/* user mode */
    else
	{
	/* supervisor stack points to context stuff -
	 * 68020 - pop off 26 bytes, 680[01]0 - pop off 28 bytes
	 */

#if (CPU == MC68000)
	*sp = pTcb->tcb_sp + 26;
#else
	*sp = pTcb->tcb_sp + 28;
#endif
	}

    for(ix = 0; ix < 6; ix++)
	dregs[ix] = pTcb->tcb_data_regs[ix];

    dregs[6] = pTcbStack->d6;
    dregs[7] = pTcbStack->d7;

    for(ix = 0; ix < 4; ix++)
	aregs[ix] = pTcb->tcb_adrs_regs[ix];

    aregs[4] = pTcbStack->a4;
    aregs[5] = pTcbStack->a5;
    aregs[6] = pTcbStack->a6;

    *sr = pTcbStack->sr;

    *pc = pTcbStack->pc;

    return (OK);
    }
/*******************************************************************************
*
* taskRegsSet - set a task's registers
*
* This routine loads the given values into the given tasks TCB.
* The registers are contained in two arrays, which contain d0-d7
* and a0-a7, and three more fields are used for the SR, PC, SP.
*
* RETURNS: OK or ERROR if invalid task id
*
* SEE ALSO: taskRegsGet (2)
*/

STATUS taskRegsSet (tid, pDregs, pAregs, sp, sr, pc)
    int tid;		/* task id                 */
    FAST int pDregs[];	/* data registers (0-7)    */
    FAST int pAregs[];	/* address registers (0-6) */
    char *sp;		/* stack pointer           */
    unsigned short sr;	/* status register         */
    INSTR *pc;		/* program counter         */

    {
    FAST int ix;
    FAST VX_TCB *pTcb = taskTcb (tid);
    FAST TCB_STACK *pTcbStack;
    int d6, d7, a4, a5, a6;		/* temporaries for regs on stack */

    if (pTcb == NULL)
	return (ERROR);


    /* set sp or usp depending on whether task is supervisor or user */

    if ((sr & 0x2000) == 0)
	pTcb->tcb_usp = sp; 	/* user stack */
    else
	{
	/* adjust stack pointer to point after the context stuff that goes
	 * on the stack
	 */

#if (CPU == MC68000)
	sp -= 26;
#else
	sp -= 28;
#endif

	pTcb->tcb_sp = sp;	/* supervisor stack */
	}


    /* set the part of the context that resides in the tcb */

    for(ix = 0; ix < 6; ix++)
	pTcb->tcb_data_regs[ix] = pDregs[ix];

    for(ix = 0; ix < 4; ix++)
	pTcb->tcb_adrs_regs[ix] = pAregs[ix];


    /* set the part of the context that resides on the stack -
     * note that we first save the relevent registers in temporary
     * variables, because in some cases the register arrays that were
     * passed to us may have been on the stack, in the same vicinity
     * as we will be storing them (this happens, for example, in exception
     * and breakpoint handling).
     */

    d6 = pDregs[6];
    d7 = pDregs[7];

    a4 = pAregs[4];
    a5 = pAregs[5];
    a6 = pAregs[6];

    pTcbStack = (TCB_STACK *) pTcb->tcb_sp;

    pTcbStack->d6 = d6;
    pTcbStack->d7 = d7;

    pTcbStack->a4 = a4;
    pTcbStack->a5 = a5;
    pTcbStack->a6 = a6;

    pTcbStack->sr = sr;
    pTcbStack->pc = pc;

#if (CPU != MC68000)
    pTcbStack->format = 0;		/* repair the stack format */
#endif

    return (OK);
    }
/*******************************************************************************
*
* taskRegsShow - print contents of a task's registers
*
* This routine prints to standard out the contents of a task's registers.
*
* EXAMPLE
* .CS
*  -> taskRegsShow (taskNameToId ("shell"))
*
*  D0 =       0  D4 =       0  A0 =       0  A4 =       0
*  D1 =       0  D5 =       0  A1 =       0  A5 =  3f4a08  SR =   3004
*  D2 =       0  D6 =       0  A2 =       0  A6 =  3d6458  PC =  39ae2
*  D3 =       0  D7 =       a  A3 =       0  A7 =  3d6458
*  value = 1 = 0x1
* .CE
*/

VOID taskRegsShow (tid)
    int tid;		/* task id */

    {
    int dregs[8];	/* task's data registers */
    int aregs[7];	/* task's address registers */
    char *sp;		/* task's stack pointer */
    USHORT sr;		/* task's status register */
    INSTR *pc;		/* task's pc */

    if (taskRegsGet (tid, dregs, aregs, &sp, &sr, &pc) == ERROR)
        {
        printf ("taskRegsShow: invalid task id %#x\n", tid);
        return;
        }
 
    printf ("\n");
    printf ("D0 = %8x   D4 = %8x   A0 = %8x   A4 = %8x\n",
            dregs[0], dregs[4], aregs[0], aregs[4]);
    printf ("D1 = %8x   D5 = %8x   A1 = %8x   A5 = %8x   SR = %8x\n",
            dregs[1], dregs[5], aregs[1], aregs[5], sr & 0xffff);
    printf ("D2 = %8x   D6 = %8x   A2 = %8x   A6 = %8x   PC = %8x\n",
            dregs[2], dregs[6], aregs[2], aregs[6], pc);
    printf ("D3 = %8x   D7 = %8x   A3 = %8x   A7 = %8x\n",
            dregs[3], dregs[7], aregs[3], sp);
    }
/*******************************************************************************
*
* taskSRSet - set task status register
*
* This routine sets the status register of a task not running (i.e. the TCB
* must NOT be that of the calling task).  It is used by the debugging
* facilities to set the trace bit in the SR of a task being single-stepped.
*
* RETURNS: OK, or ERROR if invalid task id
*/

STATUS taskSRSet (tid, sr)
    int tid;		/* task id */
    unsigned short sr;	/* new SR  */

    {
    FAST VX_TCB *pTcb = taskTcb (tid);

    if (pTcb == NULL)		/* task non-existent */
	return (ERROR);

    ((TCB_STACK *) pTcb->tcb_sp)->sr = sr;

    return (OK);
    }
/****************************************************************************
*
* taskMyId() - return task id of calling task
*
*/
int taskMyId()
    {
    int id;
    int priority;
    int taskstatus;
    VX_TCB *pTcb;

    vxTinquiry (0, &id, &priority, &taskstatus, &pTcb);

    return (id);
    }

/*******************************************************************************
*
* taskPrintRegs - print contents of a task's registers
*
*/
VOID taskPrintRegs (tid)
    FAST int tid;		/* task id */

    {
    int dregs[8];		/* task's data registers */
    int aregs[7];		/* task's address registers */
    char *sp;			/* task's stack pointer */
    USHORT sr;			/* task's status register */
    INSTR *pc;			/* task's pc */
    FAST VX_TCB *pTcb = taskTcb (tid);

    taskRegsGet (pTcb->tcb_id, dregs, aregs, &sp, &sr, &pc);

    printf ("\n");
    printf ("D0 = %8x   D4 = %8x   A0 = %8x   A4 = %8x   Task = %8d\n",
	    dregs[0], dregs[4], aregs[0], aregs[4], pTcb->tcb_id);
    printf ("D1 = %8x   D5 = %8x   A1 = %8x   A5 = %8x   SR   = %8x\n",
	    dregs[1], dregs[5], aregs[1], aregs[5], sr & 0xffff);
    printf ("D2 = %8x   D6 = %8x   A2 = %8x   A6 = %8x   PC   = %8x\n",
	    dregs[2], dregs[6], aregs[2], aregs[6], pc);
    printf ("D3 = %8x   D7 = %8x   A3 = %8x   US = %8x   SS   = %8x\n",
	    dregs[3], dregs[7], aregs[3], pTcb->tcb_usp, sp);
    }
