/* pathLib.c - file/directory path library */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
This library provides functions for manipulating and parsing directory
and file path names for heirarchical file systems.
The path names are UNIX style with directory names separated by '/'s.
The directory '.' refers to the current directory.
The directory '..' refers to the parent directory.
Path names are handled in two forms:
as a null terminated string (such as "/abc/dir/file"), or
as an array of directory names
(such as a char** array with the entries "abc", "dir", "file", and NULL).

SEE ALSO
ioLib (1), iosDevFind (2)
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "iosLib.h"
#include "ioLib.h"
#include "strLib.h"
#include "memLib.h"

/*******************************************************************************
*
* pathParse - parse a full pathname into an array of directory/file names
*
* Parses a directory or file name which has directory names separated by '/'s.
* Copies the directory and file names into an array of strings.
* "." is returned for a null path name.
* The last entry in the returned array will be followed by a NULL entry.
*
* RETURNS: parsed directory name returned in `nameArray'
*
* NOMANUAL
*/ 

VOID pathParse (longName, nameArray)
    char *longName;	/* full path name */
    char **nameArray;	/* resulting array of pointers to directory names */

    {
    FAST char *pName;
    FAST char *p0;
    FAST char *p1;
    int nameCount = 0;
    int nChars;

    p0 = p1 = longName;

    while (*p0 != EOS)
	{
	/* use index () */

	while ((*p1 != '/') && (*p1 != EOS))
	    p1++;
	if (p0 == p1)	/* p1 hasn't moved forward, probably hit another '/' */
	    {
	    p0++; p1++;
	    continue;
	    }
	nChars = p1 - p0;
	nameArray [nameCount] = malloc ((unsigned) (nChars + 1));
	pName = nameArray [nameCount];
	bcopy (p0, pName, nChars);
	pName [nChars] = EOS;
	if (*p1 == '/')
	    p1++;
	p0 = p1;
	nameCount++;
	}

    /* return "." (current directory) if no directories were specified */

    if (nameCount == 0)
	{
	nameArray [0] = malloc (2);
	strcpy (nameArray [0], ".");
	nameCount++;
	}

    nameArray [nameCount] = NULL;
    }
/*******************************************************************************
*
* pathCondense - condense a file or directory path
*
* This routine condenses a path name by getting rid of any occurrences of
* "/../", "/./", and/or "//".
*/

VOID pathCondense (pathName)
    FAST char *pathName;	/* directory or file path */

    {
    char *nameArray [MAX_DIRNAMES];
    char newPathName [MAX_FILENAME_LENGTH];
    char *pTail;

    /* parse the stuff after the device name */

    (void) iosDevFind (pathName, &pTail);

    pathParse (pTail, nameArray);

    /* start with an empty path name, preserve initial '/' if necessary */

    if (*pTail == '/')
	strcpy (newPathName, "/");
    else
	*newPathName = EOS;

    pathAppend (newPathName, nameArray, 0);

    /* newPathName should be the same size or smaller than pathName */

    strcpy (pTail, newPathName);
    }
/*******************************************************************************
*
* pathAppend - generate path name by concatenating an array of directory names
*
* This is a recursive routine which takes an array of strings (directory names)
* and concatenates them together, separating them with '/'s and interpreting
* and eliminating occurrences of ".." and ".".
*
* RETURNS: new path name copied into `destString'
*/

VOID pathAppend (destString, nameArray, stringIndex)
    char *destString; /* destination string ( > total length of names) */
    char **nameArray; /* array of directory names, last entry is NULL  */
    int stringIndex;  /* index into nameArray of the current           */
                      /* directory name being appended to destString   */
    {
    char *p;

    /* point to current directory name */

    if ((p = nameArray [stringIndex]) == NULL)
	return;
    
    if (strcmp (p, ".") == 0)
	pathAppend (destString, nameArray, stringIndex + 1);

    else if (strcmp (p, "..") == 0)
	{
	pathRemoveTail (destString);
	pathAppend (destString, nameArray, stringIndex + 1);
	}
    else
	{
	if (stringIndex != 0)		/* this is wrong, other cases */
	    strcat (destString, "/");
	strcat (destString, nameArray [stringIndex]);
	pathAppend (destString, nameArray, stringIndex + 1);
	}
    }
/*******************************************************************************
*
* pathBuild - generate path name by concatenating a list of directory names
*
* Creates a path name ending in "/".
*
* RETURNS: new path name pointed to by `destString'
*/

VOID pathBuild (nameArray, stringIndex, destString)
    char **nameArray;  /* array of directory names, last entry is NULL */
    int stringIndex;   /* index into nameArray of the current          */
                       /* directory name being appended to destString  */
    char *destString;  /* destination string                           */

    {
    int i;

    *destString = EOS;

    if (stringIndex >= 0)
	for (i = 0; i <= stringIndex; i++)
	    {
	    strcat (destString, nameArray [i]);
	    strcat (destString, "/");
	    }
    }
/*******************************************************************************
*
* pathCat - concatenate directory path to file name
*
* This routine constructs a valid UNIX file name from a directory path
* and a file name.  The directory name may or may not start with a device name.
* The file name is a bit of a misnomer since it may or may not be prepended
* with directory names.  Usually, all that is necessary is to insert a
* slash ('/') between the directory and file name, however exceptional
* cases are also handled.
*
* If the fileName begins with a '/', '~', or '$', the result will be the
* file name, unless the directory name (dirName) begins with a device name.
* In the case that dirName begins with a device name, the file name is
* inserted after the device name part of dirName.
*
* If the directory path is just a device name, and the device name does not
* end with a '/' or ':', AND the file name does not start with a '/', '~',
* or '$', then a slash ('/') is inserted between the device name and the 
* file name.  This implementation originated for use with NFS devices in which
* full file names emulate UNIX style names.
*
* RETURNS: the resultant path name in `result'
*/

VOID pathCat (dirName, fileName, result)
    char *dirName;	/* directory path */
    char *fileName;	/* filename to be added to end of path */
    char *result;	/* where to form concatenated name */

    {
    char *pTail;
    FAST char *pResult = result;
    FAST int dirLen;
    FAST int devLen = 0;

    *pResult = EOS;

    /* does directory start with a device name? */

    (void) iosDevFind (dirName, &pTail);

    if (pTail != dirName)
	{
	/* directory starts with a device name, copy device name to result */

	devLen = (int) pTail - (int) dirName;
	bcopy (dirName, pResult, devLen);

	/*
	 * If dirName is only a device name, and the device name does not end
	 * with a '/' or ':', AND the file name does not start with a '~',
	 * '/', or '$',
	 * THEN, insert a slash ('/') between the device name and the 
	 * file name.
	 */

	pResult = &(result [devLen]);
	if ((*pTail == EOS) 		/* dirName is a device name */
	    && (*(pTail - 1) != '/') 	/* dirName ends with a '/' or ':' */
	    && (*(pTail - 1) != ':')
	    && (*fileName != '~')	/* fileName doesn't start with a '/' */
	    && (*fileName != '/')	/*    '~', or '$'. */
	    && (*fileName != '$'))
	    *pResult++ = '/';
	}

    if (fileName != NULL && ((*fileName == '~') || (*fileName == '/')
			       || (*fileName == '$')))
	{
	/*
	 * File name begins with a '~', '/', or '$'.
	 * Concatenate filename to result name.
	 */

	strcpy (pResult, fileName);
	}
    else if (dirName != NULL)
	{
	/* Concatenate directory name to result */

	strcpy (pResult, pTail);

	/* insert slash between directory name and file name if necessary */

	dirLen = strlen (pTail);
	if ((dirLen > 0) && (pTail [dirLen - 1] != '/'))
	    strcat (pResult, "/");

	/* Concatenate file name to result */

	strcat (pResult, fileName);
	}
    }
/*******************************************************************************
*
* pathLastName - point to the last name in a path name
*
* Point to character after the last '/' in the path name.
* Note that if the directory name ends with '/', the returned pointer
* will point to a null string.
*
* RETURNS: pointer to the last path name in `pLastName'
*/

VOID pathLastName (pathName, pLastName)
    char *pathName;
    char **pLastName;

    {
    char *p;

    if ((p = rindex (pathName, '/')) == NULL)
	*pLastName = pathName;
    else
	*pLastName = (char *) ((int) p + 1);
    }
/*******************************************************************************
*
* pathRemoveTail - remove the last directory name from a path name
*
* This routine replaces the last occurance of '/' with an EOS.
* If their is no '/', then the pathName becomes null.
*/

VOID pathRemoveTail (pathName)
    char *pathName;	/* path to have trailing '/' removed from */

    {
    char *p;

    if ((p = rindex (pathName, '/')) == NULL)
	*pathName = EOS;
    else
	*p = EOS;
    }
/*******************************************************************************
*
* pathSplit - split a path name into its directory and file parts
*
* This routine splits a valid UNIX style path name into its directory
* and file parts by finding the last '/'.  The results are copied
* into dirName and fileName.
*/

VOID pathSplit (fullFileName, dirName, fileName)
    char *fullFileName;	/* full file name being parsed */
    char *dirName;	/* result, directory name */
    char *fileName;	/* result, simple file name */

    {
    FAST int nChars;

    if (fullFileName != NULL)
	{
	char *p = rindex (fullFileName, '/');

	if (p == NULL)
	    {
	    strcpy (fileName, fullFileName);
	    strcpy (dirName, "");
	    }
	else
	    {
	    nChars = (int) p - (int) fullFileName;
	    strncpy (dirName, fullFileName, nChars);
	    dirName [nChars] = EOS;
	    strcpy (fileName, ++p);
	    }
	}
    else
	{
	strcpy (fileName, "");
	strcpy (dirName, "");
	}
    }
