/* sockLib.c - UNIX BSD 4.3 compatible socket library */

static char *copyright = "Copyright 1984-1988, Wind River Systems, Inc.";

/* 
modification history
--------------------
*/

/*
The library sockLib provides UNIX BSD 4.3 compatible socket calls.
These calls may be used to read, write, open,
and close sockets, either on the same CPU or across a network.
The calling sequences of these routines are identical to UNIX BSD 4.3.  

INCLUDE FILES: types.h, mbuf.h, socket.h, socketvar.h

SEE ALSO: UNIX BSD 4.3 documentation, "Network", netLib (1)
*/

#include "UniWorks.h"
#include "memLib.h"
#include "strLib.h"
#include "param.h"
#include "mbuf.h"
#include "protosw.h"
#include "socket.h"
#include "socketvar.h"
#include "uio.h"
#include "errno.h"
#include "ioLib.h"
#include "iosLib.h"
#include "vwModNum.h"
#include "utime.h"
#include "systm.h"

IMPORT STATUS soo_ioctl ();	/* sys_socket.c */
IMPORT ULONG tickGet ();


#define SOCKET_STRING	"(socket)"

typedef struct	/* SOCKET_DEV */
    {
    DEV_HDR devHdr;
    } SOCKET_DEV;

LOCAL SOCKET_DEV socketDev;	/* to please I/O system */
LOCAL int sockDrvNum;
LOCAL int selectDelayTime = 1;	/* set with selectDelayTicks() */

/* forward declarations */

LOCAL STATUS sockClose ();
LOCAL int sockRead ();
LOCAL int sockWrite ();


/*******************************************************************************
*
* sockInit - install the "socket driver" into the I/O system
*
* sockInit must be called once at configuration time before any socket
* operations are performed.  Called in netLib (1) by netLibInit (2).
*
* NOMANUAL
*/

STATUS sockInit ()

    {
    if (sockDrvNum > 0)
	return (OK);

    sockDrvNum = iosDrvInstall ((FUNCPTR) NULL, (FUNCPTR) NULL, (FUNCPTR) NULL,
		   (FUNCPTR) sockClose, (FUNCPTR) sockRead, 
		   (FUNCPTR) sockWrite, (FUNCPTR) soo_ioctl); 

    socketDev.devHdr.drvNum = sockDrvNum;	/* to please I/O system */

    return (sockDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* socket - open a socket
*
* Open a socket, and return a socket descriptor.  The socket descriptor
* is then passed to the other socket routines to identify the socket.
* Also, the socket descriptor is a standard I/O system ``file descriptor''
* (fd) and can be used in calls to open, close, read, write, and ioctl.
*
* RETURNS: socket descriptor or ERROR
*/

int socket (domain, type, protocol)
    int domain;		/* address family (AF_INET) */
    int type;		/* socket type (SOCK_STREAM) */
    int protocol;	/* socket protocol */

    {
    struct socket *so;
    int fd;
    char *pName;
    int status = socreate (domain, &so, type, protocol);

    if (status)
	{
	netErrnoSet (status);
        return (ERROR);
	}

    /* stick the socket into a file descriptor */

    pName = malloc ((unsigned)strlen (SOCKET_STRING) + 1);
    (void)strcpy (pName, SOCKET_STRING);

    if ((fd = iosFdNew (&socketDev, pName, (int) so)) == ERROR)
	{
	(void)sockClose (so);
	return (ERROR);
	}

    return (fd);
    }
/*******************************************************************************
*
* sockClose - close a socket
*
* sockClose is called by the I/O system to close a socket.
*/

LOCAL STATUS sockClose (s)
    struct socket *s;

    {
    return (soclose (s));
    }
/*******************************************************************************
*
* bind - bind a name to a socket
* 
* Bind is used to associate a network address with a given socket
* so that other processes may connect to it.  When a socket is
* created with socket (2) it belongs to an address family but
* has no assigned name.
* 
* RETURNS:
*    OK or
*    ERROR if invalid socket, or
*    address not available or in use, or
*    the socket is already bound
*/

STATUS bind (s, name, namelen)
    int s;		/* socket descriptor */
    SOCKADDR *name;	/* name to be bound */
    int namelen;	/* length of name */

    {
    struct mbuf *nam;
    int status;
    struct socket *so;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    status = sockargs (&nam, (caddr_t) name, namelen, MT_SONAME);

    if (status)
	{
	netErrnoSet (status);
        return (ERROR);
	}

    status = sobind (so, nam);

    if (status)
	{
	netErrnoSet (status);
        return (ERROR);
	}

    m_freem (nam);
    return (OK);
    }
/*******************************************************************************
*
* listen - define socket queue length
*
* Listen defines the number of connections which will be accepted 
* on a socket.  Up to `backlog' connections will be accepted on a
* socket;  each time an accept (2) is performed on the socket, a
* connection will be dequeued.
*
* RETURNS: OK or ERROR if invalid socket, or unable to listen
*/

STATUS listen (s, backlog)
    int s;		/* socket descriptor */
    int backlog;	/* number of connections to queue */
    
    {
    FAST int status;
    FAST struct socket *so;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    status = solisten (so, backlog);

    if (status)
	{
	netErrnoSet (status);
        return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* accept - accept a connection on a socket
*
* Accepts a connection to a socket, and returns a new socket created
* for the connection.  The socket must be bound to an address with
* bind (2), and is listening for connections after a listen (2).
* Accept dequeues the first connection and creates a new socket
* with the same properties of `s'.  Accept blocks the caller
* until a connection is present, unless the socket is marked as
* non-blocking.
*
* RETURNS
*    socket descriptor, or
*    ERROR if accept failed
*/

int accept (s, addr, addrlen)
    int s;		/* socket descriptor */
    SOCKADDR *addr;	/* peer address */
    int *addrlen;	/* peer address length  */

    {
    struct mbuf *nam;
    int namelen = *addrlen;
    int slev;
    FAST struct socket *so;
    FAST int fd;
    char *pName;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    slev = splnet ();

    if ((so->so_options & SO_ACCEPTCONN) == 0) 
        {
        netErrnoSet (EINVAL);
        splx (slev);
        return (ERROR);
        }

    if ((so->so_state & SS_NBIO) && so->so_qlen == 0) 
        {
        netErrnoSet (EWOULDBLOCK);
        splx(slev);
        return (ERROR);
        }

    while (so->so_qlen == 0 && so->so_error == 0) 
        {
        if (so->so_state & SS_CANTRCVMORE) 
            {
            so->so_error = ECONNABORTED;
            break;
            }
        sleep (so->so_timeoSemId);
        }

    if (so->so_error) 
        {
        netErrnoSet ((int) so->so_error);
        so->so_error = 0;
        splx (slev);
        return (ERROR);
        }

    { 
    struct socket *aso = so->so_q;
    if (soqremque(aso, 1) == 0)
	panic ("accept");
    so = aso;
    }

    nam = m_get (M_WAIT, MT_SONAME);
    (void) soaccept (so, nam);

    if (addr) 
        {
        if (namelen > nam->m_len)
            namelen = nam->m_len;
        /* XXX SHOULD COPY OUT A CHAIN HERE */
        (void) copyout (mtod(nam, caddr_t), (caddr_t)addr,
			namelen);
        (void) copyout ((caddr_t)&namelen, (caddr_t)addrlen,
			sizeof (*addrlen));
        }

    m_freem (nam);
    splx (slev);

    /* put the new socket into an fd */

    pName = malloc ((unsigned)strlen (SOCKET_STRING) + 1);
    (void)strcpy (pName, SOCKET_STRING);

    if ((fd = iosFdNew (&socketDev, pName, (int) so)) == ERROR)
	{
	(void)sockClose (so);
	return (ERROR);
	}

    return (fd);
    }
/*******************************************************************************
*
* connect - initiate a connection on a socket
*
* This routine establishes a virtual circuit between a local socket
* and another socket somewhere on the network if `s' is of type SOCK_STREAM.
* If `s' is of type SOCK_DGRAM then this call permanently specifies
* the peer to which datagrams are sent.  The `name' specifies the
* address of the other socket.
*
* RETURNS
*    OK if connection successful, or
*    ERROR if connection failed
*/

STATUS connect (s, name, namelen)
    int s;		/* socket descriptor */
    SOCKADDR *name;	/* address of the socket to connect */
    int namelen;	/* length of name, in bytes */

    {
    struct mbuf *nam;
    int slev;
    int status;
    FAST struct socket *so;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    if ((so->so_state & SS_NBIO) &&
	(so->so_state & SS_ISCONNECTING)) {
	netErrnoSet (EALREADY);
	return (ERROR);
	}

    status = sockargs (&nam, (caddr_t)name, namelen, MT_SONAME);
    if (status)
	{
	netErrnoSet (status);
        return (ERROR);
	}

    status = soconnect (so, nam);

    if (status)
	{
	netErrnoSet (status);
	so->so_state &= ~SS_ISCONNECTING;
	m_freem(nam);
	return (ERROR);
	}

    slev = splnet();

    if ((so->so_state & SS_NBIO) &&
        (so->so_state & SS_ISCONNECTING)) 
        {
        netErrnoSet (EINPROGRESS);
	splx (slev);
	so->so_state &= ~SS_ISCONNECTING;
	m_freem (nam);
        return (ERROR);
        }

    while ((so->so_state & SS_ISCONNECTING) && so->so_error == 0)
        sleep (so->so_timeoSemId);

    if (so->so_error)
	{
	netErrnoSet ((int) so->so_error);
	so->so_error = 0;
	splx (slev);
	so->so_state &= ~SS_ISCONNECTING;
	m_freem (nam);
	return (ERROR);
	}

    splx (slev);
    so->so_state &= ~SS_ISCONNECTING;
    m_freem (nam);
    return (OK);
    }
/*******************************************************************************
*
* sendto - send a message to a socket
*
* Sendto is used to send datagrams on a socket.
*
* RETURNS
*  number of bytes sent, or
*  ERROR if send failed
*/

int sendto (s, buf, len, flags, to, tolen)
    int s;			/* socket on which to send data */
    caddr_t buf;		/* pointer to data buffer */
    int len;			/* length of buffer */
    int flags;			/* flags to underlying protocols */
    struct sockaddr *to;	/* recipient's address */
    int	tolen;			/* length of to sockaddr */ 

    {
    struct msghdr msg;
    struct iovec aiov;
    struct socket *so;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    msg.msg_name         = (caddr_t) to;
    msg.msg_namelen      = tolen;
    msg.msg_iov          = &aiov;
    msg.msg_iovlen       = 1;
    aiov.iov_base        = buf;
    aiov.iov_len         = len;
    msg.msg_accrights    = 0;
    msg.msg_accrightslen = 0;

    return (sendit ((int) so, &msg, flags));
    }
/*******************************************************************************
*
* send - send data on a socket
*
* The routine send is used to transmit data on a previously
* established virtual circuit.
*
* RETURNS
*  number of bytes sent, or
*  ERROR if send failed
*/

int send (s, buf, len, flags)
    int s;		/* socket on which to send */
    caddr_t buf;	/* pointer to buffer to transmit */
    int len;		/* length of buffer */
    int flags;		/* flags to underlying protocols */

    {
    struct msghdr msg;
    struct iovec aiov;
    struct socket *so;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    msg.msg_name         = 0;
    msg.msg_namelen      = 0;
    msg.msg_iov          = &aiov;
    msg.msg_iovlen       = 1;
    aiov.iov_base        = buf;
    aiov.iov_len         = len;
    msg.msg_accrights    = 0;
    msg.msg_accrightslen = 0;

    return (sendit ((int) so, &msg, flags));
    }
/*******************************************************************************
*
* sockWrite - write to a socket
*
* This routine is called by the I/O system when a write is done on a socket
*/

LOCAL int sockWrite (so, buf, len)
    struct socket *so;
    caddr_t buf;
    int len;

    {
    struct msghdr msg;
    struct iovec aiov;

    msg.msg_name         = 0;
    msg.msg_namelen      = 0;
    msg.msg_iov          = &aiov;
    msg.msg_iovlen       = 1;
    aiov.iov_base        = buf;
    aiov.iov_len         = len;
    msg.msg_accrights    = 0;
    msg.msg_accrightslen = 0;

    return (sendit ((int) so, &msg, 0));
    }
/*******************************************************************************
*
* sendmsg - send a message on a socket
*
* Sendmsg is used to send datagrams on a socket.  It may
* be used in place of sendto to decrease the overhead of re-constructing
* the msghdr structure for each datagram.
*
* RETURNS
*  number of bytes sent, or
*  ERROR if send failed
*/

int sendmsg (s, msg, flags)
    int	s;		/* socket on which to transmit */
    struct msghdr *msg;	/* structure containing parameters to underlying
			   protocols */
    int	flags;		/* flags to underlying protocols */

    { 
    struct socket *so;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    return (sendit ((int) so, msg, flags));
    }
/*******************************************************************************
*
* sendit - send a msg
*
* sendit is called by the higher level send routines.
*/

LOCAL int sendit (s, mp, flags)
    int s;
    FAST struct msghdr *mp;
    int flags;

    {
    struct uio auio;
    FAST struct iovec *iov;
    FAST int i;
    struct mbuf *to, *rights;
    int len;
    int status;
    int retVal;
    FAST int slev = splnet ();

    auio.uio_iov = mp->msg_iov;
    auio.uio_iovcnt = mp->msg_iovlen;
    auio.uio_segflg = UIO_USERSPACE;
    auio.uio_offset = 0;			/* XXX */
    auio.uio_resid = 0;
    iov = mp->msg_iov;

    for (i = 0; i < mp->msg_iovlen; i++, iov++) 
        {
        if (iov->iov_len < 0) 
            {
            netErrnoSet (EINVAL);
	    splx (slev);
            return (ERROR);
            }
        if (iov->iov_len == 0)
            continue;

        auio.uio_resid += iov->iov_len;
        }

    if (mp->msg_name) 
        {
        status = sockargs (&to, mp->msg_name, mp->msg_namelen, MT_SONAME);
        if (status)
	    {
	    netErrnoSet (status);
	    splx (slev);
            return (ERROR);
	    }
        } 
    else
        to = 0;

    if (mp->msg_accrights) 
        {
        status = sockargs (&rights, mp->msg_accrights, mp->msg_accrightslen,
                          MT_RIGHTS);
        if (status)
	    {
	    if (to)
		m_freem(to);
	    netErrnoSet (status);
	    splx (slev);
	    return (ERROR);
	    }
        } 
    else
        rights = 0;

    len = auio.uio_resid;
    status = sosend((struct socket *) s, to, &auio, flags, rights);

    retVal = len - auio.uio_resid;

    if (rights)
        m_freem(rights);
    if (to)
        m_freem(to);

    if (status)
	{
	netErrnoSet (status);
	splx (slev);
	return (ERROR);
	}

    splx (slev);
    return (retVal);
    }
/*******************************************************************************
*
* recvfrom - receive a datagram
*
* recvfrom is used to receive data from a socket regardless of whether it
* is connected or not.  If `from' is non-zero, the source address of the
* message is filled in.  The parameter `fromlen' should be initialized
* to the size of the `from' buffer.  On return, it contains the actual
* size of the address stored.
* 
* RETURNS
*  number of bytes received, or
*  ERROR if receive failed
*/

int recvfrom (s, buf, len, flags, from, fromlen)
    int s;			/* socket on which to receive data */
    caddr_t buf;		/* pointer to data buffer */
    int len;			/* length of buffer */
    int flags;			/* flags to underlying protocols */
    struct sockaddr *from;	/* get's filled in with sender's address */
    int	*fromlen;		/* should contain length of from when called,*/
				/* get's filled in with actual length of */
				/* address on return */

    {
    struct msghdr msg;
    struct iovec aiov;
    struct socket *so;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    msg.msg_name         = (caddr_t) from;
    msg.msg_namelen      = *fromlen;
    msg.msg_iov          = &aiov;
    msg.msg_iovlen       = 1;
    aiov.iov_base        = buf;
    aiov.iov_len         = len;
    msg.msg_accrights    = 0;
    msg.msg_accrightslen = 0;

    return (recvit (so, &msg, flags, (caddr_t)fromlen, (caddr_t)0));
    }
/*******************************************************************************
*
* recv - receive on a socket
*
* Receive data from a connected socket.
*
* RETURNS
*  number of bytes received, or
*  ERROR if receive failed
*/

int recv (s, buf, len, flags)
    int s;		/* socket on which to receive data */
    caddr_t buf;	/* pointer to buffer where data will be written */
    int len;		/* length of buffer */
    int flags;		/* flags to underlying protocols */

    {
    struct msghdr msg;
    struct iovec aiov;
    struct socket *so;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    msg.msg_name         = 0;
    msg.msg_namelen      = 0;
    msg.msg_iov          = &aiov;
    msg.msg_iovlen       = 1;
    aiov.iov_base        = buf;
    aiov.iov_len         = len;
    msg.msg_accrights    = 0;
    msg.msg_accrightslen = 0;

    return (recvit (so, &msg, flags, (caddr_t)0, (caddr_t)0));
    }
/*******************************************************************************
*
* sockRead - read from a socket
*
* sockRead is called by the I/O system to do a read on a socket.
*/

LOCAL int sockRead (so, buf, len)
    struct socket *so;
    caddr_t buf;
    int len;

    {
    struct msghdr msg;
    struct iovec aiov;

    msg.msg_name         = 0;
    msg.msg_namelen      = 0;
    msg.msg_iov          = &aiov;
    msg.msg_iovlen       = 1;
    aiov.iov_base        = buf;
    aiov.iov_len         = len;
    msg.msg_accrights    = 0;
    msg.msg_accrightslen = 0;

    return (recvit (so, &msg, 0, (caddr_t)0, (caddr_t)0));
    }
/*******************************************************************************
*
* recvmsg - receive a message
* 
* Recvmsg is used to receive datagrams on a socket.  It may
* be used in place of recvfrom (2) to decrease the overhead of breaking down
* the msghdr structure for each datagram.
*
* RETURNS
*  number of bytes received, or
*  ERROR if receive failed
*/

int recvmsg (s, msg, flags)
    int	s;		/* socket on which to receive */
    struct msghdr *msg;	/* structure containing parameters to underlying
			   protocols */
    int	flags;		/* flags to underlying protocols */

    {
    struct socket *so;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    return (recvit (so, msg, flags, (caddr_t)&msg->msg_namelen,
		    (caddr_t)&msg->msg_accrightslen));
    }
/*******************************************************************************
*
* recvit - low level receive routine
*/

LOCAL int recvit (s, mp, flags, namelenp, rightslenp)
    struct socket *s;
    FAST struct msghdr *mp;
    int flags;
    caddr_t namelenp, rightslenp;

    {
    struct uio auio;
    FAST struct iovec *iov;
    FAST int i;
    struct mbuf *from, *rights;
    int len;
    int status;
    int retVal;
    FAST int slev = splnet ();

    auio.uio_iov = mp->msg_iov;
    auio.uio_iovcnt = mp->msg_iovlen;
    auio.uio_segflg = UIO_USERSPACE;
    auio.uio_offset = 0;			/* XXX */
    auio.uio_resid = 0;
    iov = mp->msg_iov;

    for (i = 0; i < mp->msg_iovlen; i++, iov++) 
        {
        if (iov->iov_len < 0) 
            {
            netErrnoSet (EINVAL);
	    splx (slev);
            return (ERROR);
            }
        if (iov->iov_len == 0)
            continue;

        auio.uio_resid += iov->iov_len;
        }

    len = auio.uio_resid;
    status = soreceive (s, &from, &auio, flags, &rights);

    retVal = len - auio.uio_resid;
    if (mp->msg_name) 
        {
        len = mp->msg_namelen;
        if (len <= 0 || from == 0)
            len = 0;
        else 
            {
            if (len > from->m_len)
                len = from->m_len;
            (void) copyout((caddr_t)mtod(from, caddr_t),
            (caddr_t)mp->msg_name, len);
            }
        (void) copyout ((caddr_t)&len, namelenp, sizeof (int));
        }
    if (mp->msg_accrights) 
        {
        len = mp->msg_accrightslen;
        if (len <= 0 || rights == 0)
            len = 0;
        else 
            {
            if (len > rights->m_len)
                len = rights->m_len;
            (void) copyout ((caddr_t)mtod(rights, caddr_t),
			    (caddr_t)mp->msg_accrights, len);
            }
        (void) copyout ((caddr_t)&len, rightslenp, sizeof (int));
        }
    if (rights)
        m_freem(rights);
    if (from)
        m_freem(from);

    if (status)
	{
	netErrnoSet (status);
	splx (slev);
	return (ERROR);
	}

    splx (slev);

    return (retVal);
    }
/*******************************************************************************
*
* setsockopt - set socket options
*
* Manipulate options associated with a socket.
* To manipulate options at the "socket" level, `level' is specified
* as SOL_SOCKET.  Any other levels use the appropriate protocol number.
*
* RETURNS:
*    OK, or ERROR if
*    invalid socket,
*    option unknown,
*    option length > MLEN,
*    out of mbufs, or
*    unable to set specified option.
*/

STATUS setsockopt (s, level, optname, optval, optlen)
    int	s;		/* target socket */
    int	level;		/* protocol level at which option resides */
    int	optname;	/* option name */
    char *optval;	/* pointer to option value */	
    int	optlen;		/* option length */

    {
    FAST struct mbuf *m = NULL;
    FAST struct socket *so;
    FAST int status;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    if (optlen > MLEN) 
        {
        netErrnoSet (EINVAL);
        return (ERROR);
        }

    if (optval) 
        {
        m = m_get(M_WAIT, MT_SOOPTS);
        if (m == NULL) 
            {
	    netErrnoSet (ENOBUFS);
	    return (ERROR);
            }
        
	copyin(optval, mtod(m, caddr_t), optlen);
        m->m_len = optlen;
        }
    
    if ((status = sosetopt (so, level, optname, m)))
	{
	netErrnoSet (status);
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* sockargs - encapsulate socket arguments into an mbuf
*/

LOCAL int sockargs (aname, name, namelen, type)
    struct mbuf **aname;
    caddr_t name;
    int namelen;
    int type;

    {
    FAST struct mbuf *m;

    if (namelen > MLEN)
        return (EINVAL);

    m = m_get (M_WAIT, type);
    if (m == NULL)
        return (ENOBUFS);

    m->m_len = namelen;
    copyin (name, mtod(m, caddr_t), namelen);
    *aname = m;

    return (OK);
    }
/*******************************************************************************
*
* getsockname - get socket name
*
* getsockname returns the current `name' for the specified socket `s'.
* The parameter `namelen' should be initialized to
* the amount of space in `name'.
* On return `namelen' contains the actual size of `name'.
*
* RETURNS: OK or ERROR if invalid socket or socket is not connected
*/

STATUS getsockname (s, name, namelen)
    int s;		/* socket descriptor */
    SOCKADDR *name;	/* where to return name */ 
    int *namelen;	/* amount of space available in name, */
			/* later filled in with actual size of name */
    {
    FAST struct mbuf *m;
    FAST int status;
    FAST struct socket *so;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    m = m_getclr (M_WAIT, MT_SONAME);

    if (m == NULL) 
	{
	netErrnoSet (ENOBUFS);
	return (ERROR);
	}

    status = (*so->so_proto->pr_usrreq)(so, PRU_SOCKADDR, 0, m, 0);

    if (status)
	{
	m_freem(m);
	netErrnoSet (status);
	return (ERROR);
	}

    if (*namelen > m->m_len)
	*namelen = m->m_len;

    copyout (mtod(m, caddr_t), (caddr_t) name, *namelen);
    m_freem(m);

    return (OK);
    }
/*******************************************************************************
*
* getpeername - get name of connected peer
*
* getpeername returns the name of the peer connected to
* socket `s' in `name'.  The parameter `namelen' should
* be initialized to the amout of space in `name'.
* On return `namelen' contains the actual size of `name'.
*
* RETURNS: OK or ERROR if invalid socket or socket is not connected
*/

STATUS getpeername (s, name, namelen)
    int s;		/* socket descriptor */
    SOCKADDR *name;	/* where to return name */ 
    int *namelen;	/* amount of space available in name, */
			/* later filled in with actual size of name */
    {
    FAST struct socket *so;
    FAST struct mbuf *m;
    FAST int status;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    if ((so->so_state & SS_ISCONNECTED) == 0) 
	{
	netErrnoSet (ENOTCONN);
	return (ERROR);
	}

    m = m_getclr(M_WAIT, MT_SONAME);
    if (m == NULL) 
	{
	netErrnoSet (ENOBUFS);
	return (ERROR);
	}

    status = (*so->so_proto->pr_usrreq)(so, PRU_PEERADDR, 0, m, 0);

    if (status)
	{
	netErrnoSet (status);
	m_freem (m);
	return (ERROR);
	}

    if (*namelen > m->m_len)
	*namelen = m->m_len;

    copyout (mtod(m, caddr_t), (caddr_t) name, *namelen);
    m_freem (m);

    return (OK);
    }
/*******************************************************************************
*
* selectDelaySet - set select delay
*
* Set the polling rate for select (2).
* The default polling rate is 1 tick.
*
* SEE ALSO: select (2)
*/

VOID selectDelaySet (nTicks)
    int nTicks;		/* number of clock ticks */

    {
    selectDelayTime = nTicks;
    }
/*******************************************************************************
*
* select - pend on a set of file descriptors
*
* Poll the set of fd's and then delay.
* The length of the delay can be specified with selectDelaySet (2).
* The parameter `nFds' specifies the first fd's to test in the sets.
* The 3 fd sets are bit arrays corresponding to the open file
* descriptors to test.
* If the `pTimeout' value is NULL then select blocks indefinitely.
*
* The I/O system currently does not contain the facilities
* needed to implement UNIX style select, which essentially pends a
* task until activity on a set of file descriptors is detected.  
* However, this routine provides a rough functional equivalent. 
*
* WARNING:
*   Currently, may only be used with fds of sockets.
*   Currently, only examines the `pReadFds'. 
*
* MACROS:
* The following macros are available for setting the appropriate bits
* in the fd_set structure:
* .CS
*  FD_SET(fd,&fdset)
*  FD_CLR(fd,&fdset)
*  FD_ISSET(fd,&fdset)
*  FD_ZERO(fd,&fdset)
* .CE
*
* RETURNS: number of file descriptors with activity, or 0 if timed out
*
* SEE ALSO: selectDelaySet (2)
*/

int select (nFds, pReadFds, pWriteFds, pExceptFds, pTimeOut)
    int nFds;		        /* number of fd's in sets to examine */
    fd_set *pReadFds;	        /* read file descriptors */
    fd_set *pWriteFds;	        /* write file descriptors */
    fd_set *pExceptFds;	        /* exception file descriptors */
    struct timeval *pTimeOut;	/* maximum time to wait for activity; */
				/* NULL to poll forever */
    {
    int quitTime;
    int nBytes;
    int fd;
    fd_set returnMask;
    struct socket *so;
    int numFound = 0;
    BOOL pollForever = pTimeOut == NULL;

    if (!pollForever)
	{
	/* get time in milliseconds (round to nearest millisecond) */

	quitTime = (pTimeOut->tv_sec * 1000) + (pTimeOut->tv_usec/1000);

	/* convert to ticks */

	quitTime = (quitTime * sysClkGetRate ()) / 1000;
	quitTime += tickGet ();
	}

    FD_ZERO (&returnMask);
  
    /* poll at least once, even if time is 0 */

    do
	{
	for (fd = 0; fd < nFds; fd++)
	    {
	    if (FD_ISSET(fd, pReadFds))
		{
		/* extract the socket from the fd */

		so = (struct socket *) iosFdValue (fd);

		if (so == (struct socket *)ERROR)
		    continue;

		/* see if anyone has connected, otherwise
		 * see if there's any data
		 */

		if ((so->so_options & SO_ACCEPTCONN) != 0 && so->so_qlen != 0) 
		    {
		    numFound++;
		    FD_SET(fd, &returnMask);
		    }
		else if (ioctl (fd, FIONREAD, &nBytes) != ERROR && nBytes > 0)
		    {
		    /* XXX if ioctl failed we should return! */
		    numFound++;
		    FD_SET(fd, &returnMask);
		    }
		}
	    }

	if (numFound > 0)
	    break;

	taskDelay (selectDelayTime);
	} while (pollForever || tickGet () < quitTime);

    *pReadFds = returnMask; 

    if (pWriteFds != NULL)
	FD_ZERO(pWriteFds);

    if (pExceptFds != NULL)
	FD_ZERO(pExceptFds);

    return (numFound);
    }
/******************************************************************************
*
* shutdown - shutdown a network connection
*
* The function shutdown causes all, or part, of a virtual circuit to be
* shut down.  If the value of `how' is 0, then receives will be disallowed.
* If how is 1, then sends will be disallowed.  If `how' is 2 then
* both are disallowed.
*
* RETURNS: OK or ERROR if invalid socket, or socket is not connected
*/

STATUS shutdown (s, how)
    int s;	/* the socket to shutdown */
    int how;	/* 0 = receives disallowed */
		/* 1 = sends disallowed */
		/* 2 = sends and receives disallowed */
    
    {
    struct socket *so;

    /* extract the socket from the fd */

    if ((so = (struct socket *)iosFdValue (s)) == (struct socket *)ERROR)
	return (ERROR);

    return (soshutdown (so, how));
    }
