/* ldLib.c - UNIX object module loader */

static char *copyright = "Copyright 1984-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
This library provides an object module loading facility.  UNIX a.out
format files may be loaded into memory, relocated properly, their
external references resolved, and their external definitions added to
the system symbol table for use by other modules, and from the shell.
Modules may be loaded from any I/O stream, anywhere in memory.

EXAMPLE
    fdX = open ("/devX/objFile", READ);
    ldLoad (fdX, ALL_SYMBOLS);
    close (fdX);

This code fragment would load the a.out file objFile located on device /devX/
into memory which would be allocated from the system memory pool.  All
external and static definitions from the file would be added to the
system symbol table.

This could also have been accomplished from the shell, by typing:
    ld (1) </devX/objFile

INCLUDE FILE: ldLib.h

SEE ALSO: "Architecture", usrLib (1), symLib (1), memLib (1)
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "a_out.h"
#include "ioLib.h"
#include "ldLib.h"
#include "memLib.h"
#include "strLib.h"
#include "symLib.h"
#include "sysSymTbl.h"

/* The different systems use different names for the same info in some
   structures.  Make them the same here */

#define A_OUT_HDR exec
#define TEXTSIZE a_text
#define DATASIZE a_data
#define BSSSIZE a_bss
#define TRSIZE a_trsize
#define DRSIZE a_drsize

#define U_SYM_STRUCT nlist
#define U_SYM_TYPE n_type
#define U_SYM_VALUE n_value

#define RTEXT	0
#define RDATA	1
#define RBSS	2

LOCAL char stringMemErrMsg [] =
    "ldLib error: insufficient memory for strings table (need %d bytes).\n";
LOCAL char extMemErrMsg [] =
    "ldLib error: insufficient memory for externals table (need %d bytes).\n";
LOCAL char cantAddSymErrMsg [] =
    "ldLib error: can't add '%s' to system symbol table - error = 0x%x.\n";
LOCAL char nonLongErrMsg [] =
    "ldLib error: attempt to relocate non-long address at 0x%x, code = %d.\n";
LOCAL char readStringsErrMsg [] =
    "ldLib error: can't read string table - status = 0x%x\n";

#ifdef	is68k
extern BOOL cdbFlag;			/* TRUE when cdb is connected */
#endif	is68k

/*******************************************************************************
*
* ldLoad - load object module into memory from pool
*
* This routine loads an a.out format object module from the specified
* stream, and loads the code, data and bss into memory allocated from
* the system memory pool.
*
* See ldLoadModule (2) for more detail.
*
* RETURNS
*      OK, or
*      ERROR if can't read file or
*               not enough memory or
*               illegal file format or
*               unresolved external references.
*
* SEE ALSO: ldLoadModule (2)
*/

STATUS ldLoad (fd, symFlag)
    int fd;		/* fd of file to load */
    int symFlag;	/* symbols to be added to table
			 * ([NO | GLOBAL | ALL]_SYMBOLS) */

    {
	return (ldLoadAt (fd, symFlag, (char *) NONE));
    }
/*******************************************************************************
*
* ldLoadAt - load object module into memory at address
*
* This routine loads an a.out format object module from the specified
* stream, and loads the code, data and bss into memory at the specified
* load address.  If loadAddress == NONE, it first allocates the necessary
* memory from the system memory pool, then loads there.
*
* See ldLoadModule (2) for more detail.
*
* RETURNS
*      OK, or
*      ERROR if can't read file or
*               not enough memory or
*               illegal file format.
*
* SEE ALSO: "Architecture", UNIX manual for a.out format
*/

STATUS ldLoadAt (fd, symFlag, loadAddress)
    int fd;			/* fd from which to read module */
    int symFlag;		/* symbols to be added to table 
				 *   ([NO | GLOBAL | ALL]_SYMBOLS) */
    char *loadAddress;		/* address where to load
				 *   (NONE = mem pool) */

    {
    char *pText = (loadAddress == (char *)NONE) ? LD_NO_ADDRESS : loadAddress;

    return (ldLoadModule (fd, symFlag, &pText, (char **)NULL, (char **)NULL));
    }
/*******************************************************************************
*
* ldLoadModule - load object module into memory
*
* This routine reads an a.out format object module from the specified
* fd, and loads the code, data and bss into memory at the specified
* load addresses, or into allocated memory as described below.
* The module is properly relocated as per the relocation commands in the file.
* Unresolved externals will be linked to symbols found in the system
* symbol table.  Symbols in the module being loaded can optionally
* be added to system symbol table.
*
* LINKING UNRESOLVED EXTERNALS
* As the module is loaded, any unresolved external references are resolved
* by looking up the missing symbols in the the system symbol table.
* If found, those references are correctly linked to the new module.
* If unresolved external references can not be found in the system symbol
* table, then an error message ("undefined symbol: ...") is printed for
* the symbol but the loading/linking continues.  In this case, ERROR will
* be returned after the module is loaded.
*
* ADDING SYMBOLS TO THE SYMBOL TABLE
* The symbols defined in the module being loaded may optionally be added
* to the system symbol table, depending on the value of symFlag:
*
*    symFlag		symbols that will be added to system symbol table
*    -------------- 	-------------------------------------------------
*    NO_SYMBOLS		none
*    GLOBAL_SYMBOLS	external symbols only
*    ALL_SYMBOLS	all symbols
*
* In addition, the following symbols are also added to the symbol table
* to indicate the the start of each segment:
* <filename>_text, <filename>_data, and <filename>_bss
* -- where "filename" is the name associated with the fd.
*
* RELOCATION
* The relocation commands in the object module are used to relocate
* the text, data, and bss segments of the module.  The location of each
* segment can be specified explicitly, or left unspecified in which
* case memory will be malloc'ed for the segment from the system memory pool.
* This is determined by the parameters ppText, ppData, and ppBss
* each of which can have the following values:
*
*   NULL                 = no load address is specified, none will be returned
*   ptr to LD_NO_ADDRESS = no load address is specified, return address via ptr
*   ptr to address       = load address is specified
*
* The ppText, ppData, and ppBss parameters are delightfully devious.
* For each one, if the pointer is NULL, the caller doesn't care where the
* segment gets loaded, and doesn't want to be told.  If the pointer is not
* NULL, then the pointer points to a second pointer, which points to where
* the segment should be loaded.  If that second pointer has the value
* LD_NO_ADDRESS, then the caller doesn't care where the segment gets loaded,
* but wishes to be told where it was put.  In that case, the LD_NO_ADDRESS
* gets replaced with a pointer to where the segment got loaded.
* (LD_NO_ADDRESS is used, rather than NULL, so that a segment may be loaded
* at the beginning of memory.)
*
* When either don't care method is used, the corresponding segment is placed
* following the preceding segment (where the ordering of segments is text,
* data, bss). Thus, if ppText is don't care, ppData indicates an actual
* address, and ppBss is don't care, then space will be allocated for text,
* and bss will be placed following data. The caller is responsible for
* ensuring that the area indicated by ppData is large enough to contain
* both data and bss.
*
* COMMON
* Some host compiler/linker combinations internally use another storage
* class called "common".  In "C", uninitialized global variables are
* eventually put in the BSS segment.  However, in partially linked object
* modules they are flagged internally as "common" and the Unix linker
* resolves these and places them in BSS as a final step in creating a
* fully linked object module.  However, the UniWorks loader is most often
* used to load partially linked object modules.  When the UniWorks loader
* encounters a variable labeled as "common", memory for the variable
* is allocated (with malloc) and the variable is entered in the system
* symbol table (if specified) at that address.  Note that most Unix
* loaders have an option that forces resolution of the "common" storage
* while leaving the module relocatable
* (e.g. with typical BSD Unix loaders, use options "-rd").
* 
* EXAMPLE
*
* Load a module into allocated memory; don't return segment addresses:
*
*   status = ldLoadModule (fd, GLOBAL_SYMBOLS, NULL, NULL, NULL);
*
* Load a module into allocated memory; return segment addresses:
*
*   pText = pData = pBss = LD_NO_ADDRESS;
*   status = ldLoadModule (fd, GLOBAL_SYMBOLS, &pText, &pData, &pBss);
*
* Load a module at a specified address:
*
*   pText = 0x800000;			/* address of text segment *
*   pData = pBss = LD_NO_ADDRESS	/* other segments follow by default *
*   status = ldLoadModule (fd, GLOBAL_SYMBOLS, &pText, &pData, &pBss);
*
* RETURNS:
*  OK, or
*  ERROR if can't read file or not enough memory or illegal file format
*
* SEE ALSO: "Architecture", UNIX manual for a.out format
*/

STATUS ldLoadModule (fd, symFlag, ppText, ppData, ppBss)
    FAST int fd;	/* fd from which to read module */
    int symFlag;	/* symbols to be added to table 
			 *   ([NO | GLOBAL | ALL]_SYMBOLS) */
    char **ppText;	/* load text segment at adress pointed to by this
			 * pointer, return load address via this pointer */
    char **ppData;	/* load data segment at adress pointed to by this
			 * pointer, return load address via this pointer */
    char **ppBss;	/* load bss segment at adress pointed to by this
			 * pointer, return load address via this pointer */

    {
    struct A_OUT_HDR hdr;	/* module's a.out header stored here */
    char *addresses [3];
    int  lengths [3];		/* lengths of text, data, and bss */
    char *pText = (ppText == NULL) ? LD_NO_ADDRESS : *ppText;
    char *pData = (ppData == NULL) ? LD_NO_ADDRESS : *ppData;
    char *pBss  = (ppBss  == NULL) ? LD_NO_ADDRESS : *ppBss;
    int status;
    int nbytes;
    FAST int numExternals;
    char **externalsBuf   = NULL;	/* buffer for reading externals */
    FAST char *stringsBuf = NULL;

    if (fioRead (fd, (char *) &hdr, sizeof (hdr)) != sizeof (hdr))
	goto error;

    /* if addresses not specified, allocate memory */

    if (pText == LD_NO_ADDRESS)
	{
	nbytes  = hdr.TEXTSIZE;

	if (pData == LD_NO_ADDRESS)
	    {
	    nbytes += hdr.DATASIZE;

	    if (pBss == LD_NO_ADDRESS)
		nbytes += hdr.BSSSIZE;
	    }

	if ((nbytes != 0) && ((pText = malloc ((unsigned) nbytes)) == NULL))
	    goto error;				/* malloc failed */
	}

    if (pData == LD_NO_ADDRESS)
	pData = pText + hdr.TEXTSIZE;

    if (pBss == LD_NO_ADDRESS)
	pBss = pData + hdr.DATASIZE;

    addresses [RTEXT] = pText;
    addresses [RDATA] = pData;
    addresses [RBSS]  = pBss;

    lengths [RTEXT]   = hdr.TEXTSIZE;
    lengths [RDATA]   = hdr.DATASIZE;
    lengths [RBSS]    = hdr.BSSSIZE;

    /* read in all of text and initialized data */

    if ((fioRead (fd, pText, (int) hdr.TEXTSIZE) != hdr.TEXTSIZE) ||
        (fioRead (fd, pData, (int) hdr.DATASIZE) != hdr.DATASIZE))
	goto error;


    /* read in symbol strings by seeking to string stuff,
     * reading the first long which tells how big the
     * string table is, and then reading the string table */

    if ((ioctl (fd, FIOSEEK, N_STROFF(hdr)) == ERROR) ||
        (fioRead (fd, (char *) &nbytes, 4) != 4))
	goto error;

    if (nbytes != 0)
	{
	/* allocate and read in the string table */

	if ((stringsBuf = malloc ((unsigned) nbytes)) == NULL)
	    {
	    printErr (stringMemErrMsg, nbytes);
	    goto error;
	    }

	nbytes -= 4;	/* subtract 4 byte length we just read */

	if (fioRead (fd, &stringsBuf [4], nbytes) != nbytes)
	    {
	    printErr (readStringsErrMsg, errnoGet ());
	    goto error;
	    }
	}

    /* allocate the externalsBuf */

    numExternals = hdr.a_syms / sizeof (struct U_SYM_STRUCT);

    if (numExternals != 0)
	{
	externalsBuf = (char **) malloc ((unsigned) numExternals * 
					sizeof (char *));
	if (externalsBuf == NULL)
	    {
	    printErr (extMemErrMsg, numExternals * sizeof (char *));
	    goto error;
	    }
	}

    /* add segment names to symbol table before other symbols */

    if (symFlag != NO_SYMBOLS)
	addSegNames (fd, pText, pData, pBss);

    /* read in symbol table */

    if (ioctl (fd, FIOSEEK, N_SYMOFF(hdr)) == ERROR)
	goto error;

    status = rdSymtab (fd, (int) hdr.a_syms, &externalsBuf, numExternals, 
		       symFlag, addresses, stringsBuf);

    if (stringsBuf != NULL)
	{
	free (stringsBuf);	/* finished with stringsBuf */
	stringsBuf = NULL;
	}


    /* relocate text and data segments;
     *   note: even if symbol table had missing symbols, continue with 
     *   relocation of those symbols that were found. */

    if (ioctl (fd, FIOSEEK, N_TXTOFF(hdr) + hdr.a_text + hdr.a_data) == ERROR)
	goto error;

    if ((status == OK) || (errnoGet () == S_symLib_SYMBOL_NOT_FOUND))
	{
        if ((relSegment (fd, (int) hdr.TRSIZE, RTEXT, addresses, lengths,
			 externalsBuf) != OK) ||
            (relSegment (fd, (int) hdr.DRSIZE, RDATA, addresses, lengths,
			 externalsBuf) != OK))
	     {
	     goto error;
	     }
	}

    if (externalsBuf != NULL)
	{
	free ((char *) externalsBuf);	/* finished with externalsBuf */
	externalsBuf = NULL;
	}

    /* return load addresses, where called for */

    if (ppText != NULL)
	*ppText = pText;

    if (ppData != NULL)
	*ppData = pData;

    if (ppBss != NULL)
	*ppBss = pBss;

    /* clear out bss */

    bzero (pBss, (int) hdr.BSSSIZE);

    return (status);


    /* error:
     * clean up dynamically allocated temporary buffers and return ERROR */

error:
    if (stringsBuf != NULL)
	free (stringsBuf);
    if (externalsBuf != NULL)
	free ((char *) externalsBuf);

    return (ERROR);
    }
/*******************************************************************************
*
* addSegNames - add segment names to symbol table
*/

LOCAL VOID addSegNames (fd, pText, pData, pBss)
    int fd;
    char *pText;
    char *pData;
    char *pBss;

    {
    char fullname [MAX_FILENAME_LENGTH];
    char symName[MAX_SYS_SYM_LEN + 1];
    char *pName;

    if (ioctl (fd, FIOGETNAME, fullname) == OK)
	pathLastName (fullname, &pName);	/* get last simple name */
    else
	pName = "???";


    if (pText != NULL)
	{
	sprintf (symName, "%s_text", pName);
	if (symAdd (sysSymTbl, symName, pText, (N_EXT | N_TEXT)) != OK)
	    printErr (cantAddSymErrMsg, symName, errnoGet());
	}

    if (pData != NULL)
	{
	sprintf (symName, "%s_data", pName);
	if (symAdd (sysSymTbl, symName, pData, (N_EXT | N_DATA)) != OK)
	    printErr (cantAddSymErrMsg, symName, errnoGet());
	}

    if (pBss != NULL)
	{
	sprintf (symName,  "%s_bss",  pName);
	if (symAdd (sysSymTbl, symName, pBss, (N_EXT | N_BSS)) != OK)
	    printErr (cantAddSymErrMsg, symName, errnoGet());
	}
    }
/*******************************************************************************
*
* rdSymtab - read and process a symbol table
*
* This routine reads in a symbol table from an object file and processes
* each of the external symbols defined therein.  This processing performs
* two functions:
*  1) defined symbols are entered in the system symbol table as
*     specified by the "symFlag" argument:
*	ALL_SYMBOLS	= all defined symbols (LOCAL and GLOBAL) are added,
*	GLOBAL_SYMBOLS	= only external (GLOBAL) symbols are added,
*	NO_SYMBOLS	= no symbols are added;
*  2) any undefined externals are looked up in the system table, and
*     if found, entered in the specified 'externals' array.  This array
*     is indexed by the symbol number (position in the symbol table).
*     Note that only undefined externals are filled in in 'externals' and
*     that values for all other types of symbols are unchanged (i.e. left
*     as garbage!).  This is ok because 'externals' is only used to 
*     perform relocations relative to previously undefined externals.
*     Note that "common" symbols have type undefined external - the value
*     field of the symbol will be non-zero for type common, indicating
*     the size of the object.
*
* If more than the specified maximum number of symbols are found in the
* symbol table, the externals array is extended to accomodate the additional
* symbols.
* If an undefined external cannot be found in the system symbol table,
* an error message is printed, but ERROR is not returned until the entire
* symbol table has been read, which allows all undefined externals to be
* looked up.
* Stabs and absolute symbols are automatically discarded.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS rdSymtab (fd, nbytes, externals, max_symbols, 
		       symFlag, addresses, symStrings)
    FAST int fd;		/* fd of file positioned to symbol table */
    FAST int nbytes;		/* # of bytes of symbol table */
    char ***externals;		/* pointer to pointer to array to fill in 
				   values of undef externals */
    int max_symbols;		/* max symbols that can be put in 'externals' */
    int symFlag;		/* symbols to be added to table 
				 *   ([NO|GLOBAL|ALL]_SYMBOLS) */
    char *addresses [3];	/* address of text, data, and bss segments */
    char *symStrings;		/* symbol string table (only meaningful in
				   BSD 4.2) */

    {
    struct U_SYM_STRUCT symbol;
    int sym_num = 0;
    int status  = OK;
    char *adrs;
    UTINY type;
    char *bias = addresses [RTEXT];
    FAST char *name;			/* symbol name (plus EOS) */

#ifdef  is68k
    if (cdbFlag)
        {
        cdbStartLoad(max_symbols + 1); /* tell the cdb server about this */
        cdbSymAdd(iosFdGetName(fd), bias);
        }
#endif  is68k

    for (; nbytes > 0; nbytes -= sizeof (symbol), ++sym_num)
	{
	/* read in next entry - symbol definition and name string */

	if (fioRead (fd, (char *) &symbol, sizeof (symbol)) != sizeof (symbol))
	    return (ERROR);

	/* we throw away stabs */

	if (symbol.n_type & N_STAB)
	    continue;

	/* symbol name string is in symStrings array indexed 
	 * by symbol.n_un.n_name */

	name = symStrings + symbol.n_un.n_strx;

#ifdef	NO_ABS_SYMS
	/* throw away absolute symbols */

	if ((symbol.U_SYM_TYPE & N_TYPE) == N_ABS) 
	    continue;
#endif	NO_ABS_SYMS


	/* add requested symbols to symbol table */

	if ((symbol.U_SYM_TYPE & N_TYPE) != N_UNDF)
	    {
	    if ((symFlag == ALL_SYMBOLS) ||
	        ((symFlag == GLOBAL_SYMBOLS) && (symbol.U_SYM_TYPE & N_EXT)))
		{
		if (symAdd (sysSymTbl, name, symbol.U_SYM_VALUE + bias,
			    symbol.U_SYM_TYPE) != OK)
		    printErr (cantAddSymErrMsg, name, errnoGet());
		}
#ifdef  is68k
                if (cdbFlag)
                    cdbSymAdd(name, symbol.U_SYM_VALUE + bias);
#endif  is68k
	    }
	else
	    {
	    /* undefined external symbol or "common" symbol -
	     *   common symbol type is denoted by undefined external with
	     *   the value set to non-zero.
	     */

	    /* if symbol is a common, then allocate space and add to
	     * symbol table as BSS
	     */

	    if (symbol.U_SYM_VALUE != 0)
		{
		/* common symbol - create new symbol */

		adrs = malloc ((unsigned) symbol.U_SYM_VALUE);
		if (adrs == NULL)
		    return (ERROR);

		/* zero the space */

		bzero (adrs, (int) symbol.U_SYM_VALUE);

		if (symAdd (sysSymTbl, name, adrs, (N_BSS | N_EXT)) != OK)
		    printErr (cantAddSymErrMsg, name, errnoGet());
#ifdef  is68k
                else if (cdbFlag)
                    cdbSymAdd(name, adrs);
#endif  is68k
		}


	    /* look up undefined external symbol in symbol table */

	    else if (symFindType (sysSymTbl, name, &adrs, &type,
					   N_EXT, N_EXT) != OK)
		{
		/* symbol not found in symbol table */

		printErr ("undefined symbol: %s\n", name);
		adrs = NULL;
		status = ERROR;
		}


	    /* add symbol address to externals table, but first
	     * check for enough room in 'externals' table */

	    if (sym_num == max_symbols)
		{
		/* no more room in symbol table -
		 * make externals table half as big again */

		max_symbols += max_symbols / 2;

		*externals = (char **) realloc ((char *) *externals, 
						(unsigned) max_symbols * 
						sizeof (char *));

		if (*externals == NULL)
		    return (ERROR);
		}

	    (*externals) [sym_num] = adrs;	/* enter address in table */
	    }
	}

#ifdef  is68k
    if (cdbFlag)
        cdbEndLoad();           /* load done, send stuff off */
#endif  is68k

    return (status);
    }
/*******************************************************************************
*
* relSegment - perform relocation commands for a segment
*
* This routine reads the specified relocation command segment and performs
* all the relocations specified therein.
* External relocations are looked up in the 'externals' table.
* All other relocations are relative to how much a particular segment has moved.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS relSegment (fd, nbytes, segment, addresses, lengths, externals)
    FAST int fd;		/* fd of file positioned to reloc commands */
    FAST int nbytes;		/* # of bytes of reloc commands for this seg */
    int  segment;		/* segment being relocated */
    char *addresses [3];	/* address of text, data, and bss segments */
    int  lengths [3];		/* lengths of text, data, and bss segments */
    char **externals;		/* addresses of external symbols */

    {
    FAST char **paddress;
    FAST char *segaddress  = addresses [segment];
    FAST char *textaddress = addresses [RTEXT];

    struct relocation_info rel_cmd;

    for (; nbytes > 0; nbytes -= sizeof (rel_cmd))
	{
	/* read relocation command */

	if (fioRead (fd, (char *) &rel_cmd, sizeof (rel_cmd)) != 
							sizeof (rel_cmd))
	    return (ERROR);


	/* calculate actual address in memory that needs relocation,
	 * and perform external or segment relative relocation */

	paddress = (char **) (segaddress + rel_cmd.r_address);

	if (rel_cmd.r_extern)
	    {
	    if (rel_cmd.r_length != 2)
		printErr (nonLongErrMsg, paddress, rel_cmd.r_length);
	    else
		{
		/* add external address to relocatable value */

		*paddress += (int) externals [rel_cmd.r_symbolnum];

		/* relocations marked pc-relative were done assuming
		 * text starts at 0.  Adjust for actual start of text. */

		if (rel_cmd.r_pcrel)
		    *paddress -= (int) textaddress;
		}
	    }
	else
	    {
	    /* pc-relative internal references are already ok.
	     * other internal references need to be relocated relative
	     * to start of the segment indicated by r_symbolnum. */

	    if (!rel_cmd.r_pcrel)
		{
		if (rel_cmd.r_length != 2)
		    printErr (nonLongErrMsg, paddress, rel_cmd.r_length);
		else
		    {
		    switch (rel_cmd.r_symbolnum & N_TYPE)
			{
                        /* Text, data, and bss may not be contiguous
                         * in UniWorks (unlike Unix).  Relocate by
                         * calculating the actual distances that a
                         * segment has moved.
                         */

			case N_TEXT:
			    *paddress += (int) textaddress;
			    break;

			case N_DATA:
			    *paddress += (int) addresses[RDATA] -
					 lengths [RTEXT];
			    break;

			case N_BSS:
			    *paddress += (int) addresses[RBSS] -
					 lengths [RTEXT] - lengths [RDATA];
			    break;

		        default:
			    printErr ("Unknown symbol number = %#x\n",
				        rel_cmd.r_symbolnum);
			}
		    }
		}
	    }
	}

    return (OK);
    }
