/* qdDrv.c - ISI/Liberator GP-SCSI disk driver */

static char *copyright = "Copyright 1988, Integrated Solutions, Inc.";

/* 
modification history
--------------------
*/

/*
DESCRIPTION
*/

#define	SCSI_LIB

/*#define 	DEBUG			/**/
#include "UniWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "rtLib.h"
#include "semLib.h"
#include "sysLib.h"
#include "types.h"

#include "buf.h"
#include "dkbad.h"
#include "qbvar.h"

#include "openchip.h"
#include "qd.h"
#include "qs.h"
#include "scsi.h"
#include "qsvar.h"
#include "qsreg.h"

/* Externs */

extern SEMAPHORE qs_sems[];

/* Globals */

int qdNum;				/* Number of this driver */
static char     devletters[] = {'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 0};
char qdbuf;

struct dkbad qdbad[Nqd];	/* a dkbad struct for each unit */
struct qb_diskst qdst[Nqd];	/* a disk struct for each unit */

/* Structs to hold info received by commands */

struct scsi_inq		qd_inq;		/* Info returned by INQUIRE cmd */
struct scsi_msen	qd_msen;	/* Info returned by MSENSE cmd */
struct scsi_msel	qd_msel;	/* Info returned by MSELECT cmd */
struct scsi_rcap	qd_rcap;	/* Info returned by RCAP cmd */
union scsi_sns		qd_sns;		/* for sense on normal r/w */


/* Struct for each logical dev created */
struct qb_device qddev[Nqdrt];	/* one for each possible partition */

/*
 * State of controller.
 */

struct qd_softc {
	char		qd_sense;	/* sense cmd ? */
	char		qd_retries;	/* number of retries */
	struct scsi_cmnd *qd_cmd;	/* ptr to scsi cmd struct */
	struct scsi_cmnd qd_cmdsav;	/* save for retry */
	struct buf	qd_buf;		/* buf struct */
	struct buf	qd_badbuf;	/* buf for badstrat */
	union scsi_cdb	qd_cdb;		/* command block */
} qdsoftc[NQD];


/*	        ________________________________________________________
 *  dev:        |15... |11 |10 | 9 | 8 | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |
 *	        --------------------------------------------------------
 *	                \___cont___/\__targ___/\___lun____/\__part____/
 *		         \________ctlr_______/
 *		          \_____________unit______________/
 */

/* Macros */

#define	QDD_PART(dev)		((dev) & 7)
#define	QDD_UNIT(dev)		((dev) >> 3)
#define	QDU_LUN(unit)		((unit)& 7)
#define	QDU_CTLR(unit)		(((unit) >> 3)&0xF)
#define	QDU_TARG(unit)		(QDU_CTLR(unit) & 7)
#define	QDU_CONT(unit)		(QDU_CTLR(unit) >> 3)
#define QDUP_PART(unit,part)	(((unit) << 3) | (part))
#define QDCL_UNIT(ctlr, lun)	(((ctlr) << 3) | (lun))
#define QDCT_CTLR(cont, targ)	(((cont) << 3) | (targ))


/* Forward declarations */

int qdCreate(), qdDelete(), qdOpen(), qdReset(), qdClose(), qdRead();
int qdWrite(), qdIoctl(), qdRdSec(), qdWrtSec(), qdstrat();

/************************************************************************
*
* qdDrv() - called from usrConfig to start up driver.
*
*/
qdDrv()
{
	register int cont, target, ctlr, lun;

	qdNum = iosDrvInstall(qdCreate, qdDelete, qdOpen, qdClose, qdRead,
		qdWrite, qdIoctl);

	/* Initialize each controller and create a device for each unit */
	for(cont=0;cont < NQD;cont++) {
		if(qsprobe(cont) != OK)
			continue;
		for(target=0;target < Nqd_targ_cont;target++) {
			ctlr = QDCT_CTLR(cont,target);
			for(lun=0;lun < Nqd_lun_targ;lun++) {
				qdDevCreate(ctlr,QDCL_UNIT(ctlr,lun));
			}
		}
	}
	return(OK);
}

/************************************************************************
*
* qdDevCreate - Set up qd_device struct, check that unit exists and is
* 		a drive, and set up partition info.
*
*/

qdDevCreate(ctlr,unit)
{
	char part;
	int tmpsize,status;
	register struct qd_softc *softc;
	register struct qb_diskst *st;
	register struct qb_device *dev;

	softc = &qdsoftc[QDU_CONT(unit)];
	st = &qdst[unit];

	/* Check to see if physical unit(drive) exists */
	if(qdinq(unit) == ERROR) {
		return(ERROR);
	}

	/* Get the geometry for the unit(drive) */
	if(qdgeom(unit) == ERROR) {
		printf("        qd%d at QS%d    ",unit,QDU_CONT(unit));
		printf("    *** can't get geometry\n");
		return(ERROR);
	}

	/* Create logical devices using partition info set up by diskpart */
	for(part=0;part < 8;part++) {

		dev = &qddev[QDUP_PART(unit,part)];
		dev->dev = QDUP_PART(unit,part); 

		if(st->st_size[part].nblocks == 0)
			continue;
		if(st->st_size[part].nblocks < 65536)
			tmpsize = st->st_size[part].nblocks;
		else
			tmpsize = 65535;
		status = rtDevInit( (RT_VOL_DESC *) dev,
			ST_spare1, ST_nspt, tmpsize, FALSE,
		   	RT_FILES_FOR_2_BLOCK_SEG, qdRdSec, qdWrtSec, NULL);
		if (status != OK) {
		    printf("        qd%d%c at QS%d    ",
			unit,devletters[part],QDU_CONT(unit));
		    printf("    *** qdDevCreate FAILED rtDevInit\n");
		    return (ERROR);
		}
		sprintf(dev->name,"%s%d%c", QD_DEV_NAME, 
			unit, devletters[part]);
		iosDevAdd((DEV_HDR *) dev, dev->name, qdNum);
	}
	dev = &qddev[QDUP_PART(unit,2)];

	badinit(dev, &(softc->qd_badbuf), &qdbad[unit], 
		qdstrat, st->st_nspt, st->st_ntpc,
		st->st_ncpd, st->st_size[part].cyloff);

	return(OK);	
}

/************************************************************************
*
* qdissue - issue a command
*
*/

qdissue(unit, cmd, timeout, loud)
	register struct scsi_cmnd *cmd;
	register int loud;
{
	char stat;
	int cont, targ;
	register struct qd_softc *softc;
#define SHHHH(x)		{ if(loud) x;}

	cont = QDU_CONT(unit);
	targ = QDU_TARG(unit);
	softc = &qdsoftc[unit];

	do {
		if (qscmdwait(cont, targ, cmd, 80000))  {
			SHHHH(logMsg("ERROR qd%d: scsi cmd 0x%x timed out\n",
				unit,softc->qd_cdb.cdb_0.cdb_0_cmd)); 
			return(ERROR);
		}
	} while ((stat = qddoerr(unit,cmd,loud)) == 1);
	return(stat);
}


/************************************************************************
*
* qddoerr - process errors
*
* RETURNS: -1 == fatal error; 1 == another command; 0 == OK
*
*/

qddoerr(unit, cmd, loud)
	register struct scsi_cmnd *cmd;
	register int loud;
{
	int cont, targ, lun;
	register struct qd_softc *softc;
	register union scsi_cdb *cdb;
	register union scsi_sns *sns;
#define SHHHH(x)		{ if(loud) x;}

	cont = QDU_CONT(unit);
	targ = QDU_TARG(unit);
	lun = QDU_LUN(unit);
	softc = &qdsoftc[unit];
	cdb = &softc->qd_cdb;
	sns = &qd_sns;

	/* Check for fatal errors */
	if(cmd->qs_finstat == SFN_SLCT_TIMEOUT) {
		SHHHH(logMsg("ERROR qd%d: could not select target\n",
		      unit)); 
		return(ERROR);
	}
	if(cmd->qs_finstat != SFN_NORMAL_END) {
		SHHHH(logMsg("ERROR qd%d: scsi cmd 0x%x did not finish\n",
		      unit, cdb->cdb_0.cdb_0_cmd)); 
		return(ERROR);
	}
	if(cmd->qs_flaqs & SCF_FATAL_COMB) {
		SHHHH(logMsg("ERROR qd%d: scsi cmd 0x%x failed\n",
		      unit, cdb->cdb_0.cdb_0_cmd)); 
		return(ERROR);
	}
	if((cmd->qs_status[0] & SCSTAT_CHK) && softc->qd_sense) {
		softc->qd_sense = FALSE;
		SHHHH(logMsg("ERROR qd%d: scsi SENSE cmd failed\n",
		      unit)); 
		return(ERROR);
	}

	/* Get sense data if it's there to be got */
	if(cmd->qs_status[0] & SCSTAT_CHK) {
		softc->qd_sense = TRUE;
		softc->qd_retries = 0;
		bcopy(cmd,&softc->qd_cmdsav,sizeof(struct scsi_cmnd));
		SET_CDB_0(cdb, CMD_SENSE, lun, 0, sizeof(union scsi_sns), 0, 0);
		SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
			SFN_NOT_FINISHED,LEN_STATUS,sns,
			sizeof(union scsi_sns),0,0);
		return(1);
	}

	/* Check out the sense data */
	if(softc->qd_sense) {
		softc->qd_sense = FALSE;
		if(sns->sns_7.sns_7_class != 7) {
			SHHHH(logMsg("ERROR qd%d: invalid sense class %d\n",
		              unit,sns->sns_7.sns_7_class));
			return(ERROR);
		}
		if(sns->sns_7.sns_7_key == SNS_7_KEY_UNIT_ATTN) {
			SHHHH(logMsg("qd%d: restarting unit\n",unit));
			SET_CDB_0(cdb, CMD_STARTSTOP, lun, 0, 1, 0, 0);
			SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
				SFN_NOT_FINISHED,LEN_STATUS,&qdbuf, 1,0,0);
			return(1);
		}
		if(sns->sns_7.sns_7_key == SNS_7_KEY_RECOVERED) {
			SHHHH(logMsg("qd%d: soft error\n",unit));
			return(0);
		}
		if(sns->sns_7.sns_7_key == SNS_7_KEY_ABORT_CMD) {
			if(softc->qd_retries++ < QD_MAXRETRY) {
				bcopy(&softc->qd_cmdsav,cmd,
					sizeof(struct scsi_cmnd));
				return(1);
			} else {
				SHHHH(logMsg("qd%d: hard error\n",unit));
				softc->qd_retries = 0;
				return(-1);
			}
		} else {
			SHHHH(logMsg("qd%d: hard error\n",unit));
			softc->qd_retries = 0;
			return(-1);
		}
	
	}
	return(0);
}
/************************************************************************
*
* qdinq - Check to see if unit exists
*
*/
qdinq(unit)
{
	register unsigned char cont,targ,ctlr,lun;
	register struct qd_softc *softc;
	register struct scsi_cmnd *cmd;
	register union scsi_cdb *cdb;

	cont = QDU_CONT(unit);
	targ = QDU_TARG(unit);
	ctlr = QDU_CTLR(unit);
	lun = QDU_LUN(unit);
	softc = &qdsoftc[cont];
	cdb = &softc->qd_cdb;

	softc->qd_cmd = (struct scsi_cmnd *) qsgetcmd(cont);
	cmd = softc->qd_cmd;

	/* Set up a scsi_cmnd struct with INQUIRY cmd */
	SET_CDB_0(cdb, CMD_INQUIRY, lun, 0, sizeof(qd_inq), 0, 0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&qd_inq,sizeof(qd_inq),0,0);
	cmd->qs_lun = lun;
	sprintf(cmd->dbgmsg,"inquire cmd, unit=%d ctlr=%d lun=%d\n",
		unit,ctlr,lun);
	/* Issue cmd and check results */
	qd_inq.inq_pdtype = 0xff;
	if(qdissue(unit, cmd, 80000,0))
		return(ERROR);
	if(qd_inq.inq_pdtype != INQ_PDT_DISK) {
		return(ERROR);
	}

	SET_CDB_0(cdb, CMD_STARTSTOP, lun, 0, 1, 0, 0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&qdbuf,1,0,0);
	sprintf(cmd->dbgmsg,"CMD_STARTSTOP unit=%d ctlr=%d lun=%d\n",
		unit,ctlr,lun);
	if(qdissue(unit, cmd, 8000000,1))
		return (ERROR);
	taskDelay(100);	/* wait for disk to spin up */

	SET_CDB_0(cdb, CMD_TESTREADY, lun, 0, 0, 0, 0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&qdbuf,1,0,0);
	sprintf(cmd->dbgmsg,"CMD_TESTREADY unit=%d ctlr=%d lun=%d\n",
		unit,ctlr,lun);
	if(qdissue(unit, cmd, 80000,1))
		return (ERROR);
	return(OK);
}

/************************************************************************
*
* qdgeom - get the unit's geometry
*
*/
qdgeom(unit)
{
	register unsigned char cont,targ,lun,ctlr;
	register struct qd_softc *softc;
	register struct qb_diskst *st;
	register union scsi_cdb *cdb;
	register struct scsi_cmnd *cmd;
	register struct pag_desc *pd;
	register int ncpd,ntpc,nspt,spd,spc;

	cont = QDU_CONT(unit);
	targ = QDU_TARG(unit);
	ctlr = QDU_CTLR(unit);
	lun = QDU_LUN(unit);
	softc = &qdsoftc[cont];
	st = &qdst[unit];
	cdb = &softc->qd_cdb;

	softc->qd_cmd = (struct scsi_cmnd *) qsgetcmd(cont);
	cmd = softc->qd_cmd;

	/* Get capacity information */
	SET_CDB_1(cdb,CMD_RCAPAC,lun,0,0,0,0,0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_1),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&qd_rcap,sizeof(qd_rcap),0,0);
	sprintf(cmd->dbgmsg,"CMD_RCAPAC , unit=%d ctlr=%d lun=%d\n",
		unit,ctlr,lun);
	if(qdissue(unit, cmd, 8000,1))
		return(ERROR);
	if(qd_rcap.rcap_bl != 512) {
		printf("        qd%d at QS%d    ",unit,QDU_CONT(unit));
		printf("    *** bad block size %d\n",qd_rcap.rcap_bl);
		return(ERROR);
	}
	spd = qd_rcap.rcap_lba + 1;	/* sectors per disk */

	/* Get cylinder information */
	SET_CDB_1(cdb,CMD_RCAPAC,lun,0,1,1,0,0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_1),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&qd_rcap,sizeof(qd_rcap),0,0);
	sprintf(cmd->dbgmsg,"rcap cmd - cylinder, unit=%d ctlr=%d lun=%d\n",
		unit,ctlr,lun);
	if(qdissue(unit, cmd, 8000,1))
		return(ERROR);
	spc = qd_rcap.rcap_lba + 1;	/* sectors per cylinder */

	/* Get current value geometry info */
	SET_CDB_0(cdb,CMD_MSENSE,lun,PD_GEOMETRY<<8,sizeof(qd_msen),0,0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&qd_msen,sizeof(qd_msen),0,0);
	sprintf(cmd->dbgmsg,"msense geo , unit=%d ctlr=%d lun=%d\n",
		unit,ctlr,lun);
	if(qdissue(unit, cmd, 8000,1))
		return(ERROR);
	pd = (struct pag_desc *)(((char *)&qd_msen.msen_bd)+qd_msen.msen_bdl);
	ncpd = SCSI_HML(pd->pd_pg.pg_geo.geo_cyl);
	ntpc = pd->pd_pg.pg_geo.geo_heads;

	/* Get current value format info */
	SET_CDB_0(cdb,CMD_MSENSE,lun,PD_FORMAT<<8,sizeof(qd_msen),0,0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&qd_msen,sizeof(qd_msen),0,0);
	sprintf(cmd->dbgmsg,"msense fmt , unit=%d ctlr=%d lun=%d\n",
		unit,ctlr,lun);
	if(qdissue(unit, cmd, 8000,1))
		return(ERROR);
	pd = (struct pag_desc *)(((char *)&qd_msen.msen_bd)+qd_msen.msen_bdl);
	nspt = pd->pd_pg.pg_fmt.fmt_spt;

	/* Test geometry for consistency */
	if((nspt * ntpc * ncpd) == 0) {
		printf("        qd%d at QS%d    ",unit,QDU_CONT(unit));
		printf("    *** nonsense geometry (%d x %d x %d)\n",
			nspt,ntpc,ncpd);
		return(ERROR);
	}

	/* If capacity is less than calculated size, try various asdjustments*/
	if ((nspt * ntpc * ncpd) != spd) {
	    /* if cylinder size too small, try subtracting alternate sectors */
	    if ((nspt * ntpc) > spc && pd->pd_pg.pg_fmt.fmt_tpz != 0)
		nspt -= pd->pd_pg.pg_fmt.fmt_alt_spz / pd->pd_pg.pg_fmt.fmt_tpz;
	    /* if cylsize still too small, try subtracting alt tracks */
	    if ((nspt * ntpc) > spc && ntpc == pd->pd_pg.pg_fmt.fmt_tpz)
		ntpc -= pd->pd_pg.pg_fmt.fmt_alt_tpz;
	    /* if spc still wrong */
	    if ((nspt * ntpc) != spc) {
		if (spc % ntpc) {	/* if spc not multiple of headcount */
		    if (spc % nspt)	/* and spc not multiple of tracksize */
			spc = nspt * ntpc;
		    else		/* spc multiple of tracksize  */
			ntpc = spc / nspt;
		} else			/* spc multiple of headcount */
		    nspt = spc / ntpc;
	    }
	    /* spc now ok; if still unequal, drop cylinders at end */
	    if ((nspt * ntpc * ncpd) != spd)
		ncpd = spd / spc;
	}

	printf("        qd%d  at QS%d  (%d x %d x %d)\n",
		unit,QDU_CONT(unit),nspt,ntpc,ncpd);
	diskpart(st->st_size, nspt, ntpc, ncpd);
	st->st_ncpd = ncpd;
	st->st_ntpc = ntpc;
	st->st_nspt = nspt;
	st->st_nspc = nspt * ntpc;
	st->st_spare1 = 512;
	return (OK);
}

/************************************************************************
*
* qdRdSec - read a 512 block sector
*
*/

qdRdSec(qddev,secNum,buffer)
	struct qb_device *qddev;
	int secNum;
	char *buffer;
{
	if(qdstrategy(qddev,secNum,buffer,B_READ) == B_ERROR)
		return(ERROR);
	return(OK);
}

/************************************************************************
*
* qdWrtSec - write a 512 block sector
*
*/

qdWrtSec(qddev,secNum,buffer)
	struct qb_device *qddev;
	int secNum;
	char *buffer;
{
	if(qdstrategy(qddev,secNum,buffer,B_WRITE) == B_ERROR)
		return(ERROR);
	return(OK);
}

/************************************************************************
*
* qdstrategy - set up io buf and call badstrat
*
*/

qdstrategy(qddev,secNum, buffer, io_flag)
	struct qb_device *qddev;
	int secNum;
	char *buffer;
{
	int unit,part;	
	register struct buf *bp;
	register struct qb_diskst *st;
	register struct qd_softc *softc;

	unit = QDD_UNIT(qddev->dev);
	softc = &qdsoftc[QDU_CONT(unit)];
	part = QDD_PART(qddev->dev);
	bp = &softc->qd_buf;

	semTake(&qs_sems[QDU_CONT(unit)]);
	st = &qdst[unit];
	bp->b_flags = io_flag;	
	bp->b_dev = qddev->dev;
	bp->b_bcount = 512;
	bp->b_addr = (UINT) buffer;
	bp->b_blkno = secNum + (ST_nspc * (st->st_size[part].cyloff + 1));

	badstrat(qddev, bp, &softc->qd_badbuf, &qdbad[unit], 
		qdstrat, st->st_nspt, st->st_ntpc,
		st->st_ncpd, st->st_size[part].cyloff);
	semGive(&qs_sems[QDU_CONT(unit)]);

	return (bp->b_flags & B_ERROR);
}

/*************************************************************************
*
* qdstrat - set up cdb and start command
*
*/

qdstrat(qddev,bp)
	struct qb_device *qddev;
	register struct buf *bp;
{
	register unsigned char cont,unit,lun;
	register struct qb_diskst *st;
	register struct qd_softc *softc;
	register struct scsi_cmnd *cmd;
	register union scsi_cdb *cdb;

	unit = QDD_UNIT(qddev->dev);
	cont = QDU_CONT(unit);
	lun = QDU_LUN(unit);
	st = &qdst[unit];
	softc = &qdsoftc[cont];
	softc->qd_cmd = (struct scsi_cmnd *) qsgetcmd(cont);
	cmd = softc->qd_cmd;
	cdb = &softc->qd_cdb;

	SET_CDB_1(cdb, (bp->b_flags & B_READ)?CMD_XREAD:CMD_XWRITE, lun, 
		0, bp->b_blkno, bp->b_bcount/512, 0, 0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_1),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,bp->b_addr,bp->b_bcount,0,0);
	sprintf(cmd->dbgmsg,"%s , dev=0x%x\n",
		(bp->b_flags & B_READ)?"CMD_XREAD":"CMD_XWRITE",qddev->dev);
	
	if(qdissue(unit, cmd, 8000,1))
		bp->b_flags |= B_ERROR;
	bp->b_flags |= B_DONE;
	return(OK);
}

/*************************************************************************
*
* qdCreate - create an RT-11 file
*
*/

int qdCreate(qddev,name,mode)
	register struct qb_device *qddev;
	char *name;
{
	RT_FILE_DESC	*pFd;
	
	pFd = rtCreate((RT_VOL_DESC *)qddev,name,mode);
	if(pFd == NULL)
		return(ERROR);
	return((int) pFd);
}

/*************************************************************************
*
* qdDelete - delete an RT-11 file
*
*/

qdDelete(qddev,name)
	register struct qb_device *qddev;
	char *name;
{
	return(rtDelete((RT_VOL_DESC *)qddev,name));
}

/*************************************************************************
*
* qdOpen - Open an RT-11 file
*
*/

qdOpen(qddev,name,mode)
	register struct qb_device *qddev;
	char *name;
{
	RT_FILE_DESC	*pFd;

	pFd = rtOpen((RT_VOL_DESC *)qddev,name,mode);
	if(pFd == NULL)
		return(ERROR);
	return((int) pFd);
}

/*************************************************************************
*
* qdClose - close an RT-11 file
*
*/

qdClose(pFd)
	RT_FILE_DESC *pFd;
{
	return(rtClose(pFd));
}

/*************************************************************************
*
* qdRead - read from an RT-11 file
*
*/

qdRead(pFd, buffer, maxbytes)
	RT_FILE_DESC	*pFd;
	char 		*buffer;
{
	return(rtRead(pFd, buffer, maxbytes));
}

/*************************************************************************
*
* qdWrite - write to an RT-11 file
*
*/

qdWrite(pFd, buffer, nbytes)
	RT_FILE_DESC	*pFd;
	char 		*buffer;
{
	return(rtWrite(pFd, buffer, nbytes));
}

/*************************************************************************
*
* qdReset
*
*/
int qdReset()
{
	return(OK);
}

/*************************************************************************
*
* qdMkfs - make an RT-11 file system
*
*/
qdMkfs(name)
	char *name;
{
	int             fd;

	fd = open(name, WRITE);
	if (fd == ERROR) {
		printf("qdMkfs: error opening qd device %s.\n", name);
		return (ERROR);
	}
	if (ioctl(fd, FIODISKINIT) == ERROR) {
		printf("smMkfs: error initializing qd device %s.\n", name);
		close(fd);
		return (ERROR);
	}
	close(fd);
	return (OK);
}
/*************************************************************************
*
* qdIoctl -
*
*/

qdIoctl(pFd, function, arg)
	RT_FILE_DESC	*pFd;
{
	return(rtIoctl(pFd, function, arg));
}

/************************************************************
*
* qdtestsec - For testing particular sectors. 
*
* NOTE: cmd is B_READ (1) or B_WRITE (0)
*/

qdtestsec(blkno, cmd, buffer, nblks, unit, part)
	int blkno, nblks;
	char cmd, *buffer;
	unsigned char unit, part;
{
	register struct buf *bp;
	register struct qb_diskst *st;
	register struct qd_softc *softc;

	st = &qdst[unit];
	softc = &qdsoftc[QDU_CONT(unit)];
	bp = &softc->qd_buf;

	semTake(&qs_sems[QDU_CONT(unit)]);
	bp->b_flags = cmd;
	bp->b_dev = (unit << 3) | part;
	bp->b_bcount = 512 * nblks;
	bp->b_addr = (UINT) buffer;
	bp->b_blkno = blkno;

	badstrat(&qddev[bp->b_dev], bp, &softc->qd_badbuf, &qdbad[unit], 
		qdstrat, st->st_nspt, st->st_ntpc,
		st->st_ncpd, st->st_size[part].cyloff);
	semGive(&qs_sems[QDU_CONT(unit)]);
	return (bp->b_flags & B_ERROR);

}

/***********************************************************************
*
* qdtestsec1 - add cylinder offset and call testsec
*
*/

qdtestsec1(secNum, cmd, buffer, nblks, unit, part)
{
	struct qb_diskst *st;
	st = &qdst[unit];
	secNum += secNum + (ST_nspt * ST_ntpc * 1);
	qdtestsec(secNum, cmd, buffer, nblks, unit, part);
}

