/* hex.c - convert an a.out format object file into Motorola hex records */

/*
modification history:
---------------------
*/

/*
SYNOPSIS
hex [-a <adrs>] [-l] [-v] [-p <PC>] [-s <SP>] file

DESCRIPTION
This program generates a Motorola hex format (S-record) file from an 
a.out format object module.
Normally, the entry address in the object module will be used as the starting
address of the module in the output S-records.
If the -a flag is specified, then that address is used as the starting address
of the module instead.
Normally S1 records will be generated for addresses less than 64K,
S2 records will be generated for address greater than 64K and less the 16M, and
S3 records will be generated for greater than that.

OPTIONS:
    -l	only S2 records will be generated.
    -a	specify entry address, rather than using address in the obj module.
    -v	output vector info at address 0.
    -p	use <PC> as the pc in the vector (meaningless without -v).
    -s	use <SP> as the sp in the vector (meaningless without -v).
*/


#include <stdio.h>

#include "UniWorks.h"
#include "a_out.h"

#ifndef VAX
#define htonl(x)	(x)
#define htons(x)	(x)
#else
#define htonl(x)	((((x) >> 24) & 0x000000ff) | \
			 (((x) >>  8) & 0x0000ff00) | \
			 (((x) <<  8) & 0x00ff0000) | \
			 (((x) << 24) & 0xff000000))
#define htons(x)	((((x) >> 8) & 0x00ff) | (((x) << 8) & 0xff00))
#endif

#define ntohl(x)	htonl(x)
#define ntohs(x)	htons(x)

LOCAL char usage [] =
    "usage: hex [-a <adrs>] [-l] [-v] [-p <PC>] [-s <SP>] file";


/*******************************************************************************
*
* main - convert an a.out format object file into Motorola hex records
*/

VOID main (argc, argv)
    int argc;
    char *argv [];

    {
    BOOL longFormat	= FALSE;	/* default = short format */
    BOOL useEntry	= TRUE;		/* default = use entry point address */
    BOOL outputVector	= FALSE;	/* default = no vector output */
    char *filename	= NULL;		/* default = standard in */
    struct exec hdr;
    int vecPC;
    BOOL vecPcSpecified = FALSE;
    int vecSP;
    BOOL vecSpSpecified = FALSE;
    unsigned int address;
    int i;
    int totalBytes;
    int bytesWritten = 0;
    unsigned int dataByte;
    unsigned int checkSum;
    int nBytes;

    /* crack the arguments */

    for (i = 1; i < argc; i++)
	{
	if (strcmp (argv [i], "-a") == 0)	/* ADDRESS */
	    {
	    if ((++i == argc) || (sscanf (argv [i], "%x", &address) == 0))
		error (usage);
	    useEntry = FALSE;
	    }

	else if (strcmp (argv [i], "-p") == 0)	/* VECTOR PC */
	    {
	    if ((++i == argc) || (sscanf (argv [i], "%x", &vecPC) == 0))
		error (usage);
	    vecPcSpecified = TRUE;
	    }

	else if (strcmp (argv [i], "-s") == 0)	/* VECTOR SP */
	    {
	    if ((++i == argc) || (sscanf (argv [i], "%x", &vecSP) == 0))
		error (usage);
	    vecSpSpecified = TRUE;
	    }

	else if (strcmp (argv [i], "-l") == 0)	/* LONG FORMAT */
	    longFormat = TRUE;

	else if (strcmp (argv [i], "-v") == 0)	/* OUTPUT VECTOR */
	    outputVector = TRUE;

	else					/* FILENAME */
	    {
	    if ((argv[i][0] == '-') || (filename != NULL))
		error (usage);
	    filename = argv [i];
	    }
	}

    if (filename != NULL)
	{
	if (freopen (filename, "r", stdin) == NULL)
	    error ("hex: can't open file %s", filename);
	}


    /* read object module header */

    if (fread ((char *)&hdr, sizeof (hdr), 1, stdin) != 1)
	error ("hex: error reading file header");

    if (useEntry)
	address = ntohl (hdr.a_entry);

    totalBytes = ntohl (hdr.a_text) + ntohl (hdr.a_data);

    /* write out vector, if requested */

    if (outputVector)
	{
	if (! vecPcSpecified)
	    vecPC = ntohl (hdr.a_entry);
	if (! vecSpSpecified)
	    vecSP = ntohl (hdr.a_entry);

	if (longFormat)
	    {
	    printf ("S20C000000");	/* 0C = byte count, 000000 = address */
	    checkSum = 0x0c;		/* length */
	    }
	else 
	    {
	    printf ("S10B0000");
	    checkSum = 0x0b;		/* length */
	    }

	/* Write out the vector.  SP first, then PC */

	writeBytes ((unsigned int)vecSP, &checkSum);
	writeBytes ((unsigned int)vecPC, &checkSum);

	/* print out the ones complement of the checksum */

	printf("%02X\n", (~checkSum) & 0xff);
	}

	
    /* write out lines of S-records */

    while (bytesWritten < totalBytes)
	{
	nBytes = min ((totalBytes - bytesWritten), 16);

	/* write out record type, length, & address fields */

	if (longFormat || ((address > 0xffff) && (address < 0x1000000)))
	    {
	    printf ("S2%02X%06X", 3 + nBytes + 1, address);
	    checkSum = 3 + nBytes + 1 +			/* length */
		       ((address >> 16) & 0xff) +	/* adrs msb */
		       ((address >> 8) & 0xff) +	/* adrs middle byte */
		       (address & 0xff);		/* adrs lsb */
	    }
	else if (address < 0x10000)
	    {
	    printf ("S1%02X%04X", 2 + nBytes + 1, address);
	    checkSum = 2 + nBytes + 1 +			/* length */
		       ((address >> 8) & 0xff) +	/* adrs msb */
		       (address & 0xff);		/* adrs lsb */
	    }
	else
	    {
	    printf ("S3%02X%08X", 4 + nBytes + 1, address);
	    checkSum = 4 + nBytes + 1 +			/* length */
		       ((address >> 24) & 0xff) +	/* address bits 24-31 */
		       ((address >> 16) & 0xff) +	/* address bits 16-23 */
		       ((address >> 8) & 0xff) +	/* address bits 8-15 */
		       (address & 0xff);		/* address bits 0-7 */
	    }

	/* read in a data byte, add it to the checksum, and print it out */

	for (i = 0; i < nBytes; i++)
	    {
	    dataByte = getchar ();

	    if ((int)dataByte == EOF)
		error ("hex: premature end of file");

	    checkSum += dataByte;
	    printf ("%02X", dataByte);
	    }

	/* print out the ones complement of the checksum */

	printf("%02X\n", (~checkSum) & 0xff);

	bytesWritten += nBytes;
	address      += nBytes;
	}
	
    printf("S904000000FB\n");
    exit (0);
    }
/*******************************************************************************
*
* writeBytes - write out the four bytes of an int, MSB first
*/

LOCAL writeBytes (data, cksum)
    unsigned int data;
    unsigned int *cksum;

    {
    unsigned int dataByte;
    
    dataByte = (data & 0xff000000) >> 24;
    *cksum += dataByte;
    printf ("%02X", dataByte);

    dataByte = (data & 0x00ff0000) >> 16;
    *cksum += dataByte;
    printf ("%02X", dataByte);

    dataByte = (data & 0x0000ff00) >> 8;
    *cksum += dataByte;
    printf ("%02X", dataByte);

    dataByte = (data & 0x000000ff);
    *cksum += dataByte;
    printf ("%02X", dataByte);
    }
/*******************************************************************************
*
* error - print error message and die
*
* This routine prints an error message on the standard error output and
* aborts the program with the error status ERROR.  The error message is
* output with the specified format with the single specified argument.
*
* VARARGS1
*/

VOID error (format, arg)
    char *format;	/* format of error message */
    char *arg;		/* argument supplied to format (arbitrary type!) */

    {
    fprintf (stderr, format, arg);
    fprintf (stderr, "\n");
    exit (ERROR);
    }
