/* condense.c - condense an object module, by removing superflous stuff */

/*
modification history
--------------------
*/

/*
SYNOPSIS

condense <objFile >condensedObjFile

DESCRIPTION

Condense reads an object module (a.out format) on standard in, and
writes an object module on standard out.  The output object module contains
all the same information, except that absolute symbols and pc relative
relocation commands are stripped out.  These have no effect, but can
consume large amounts of disk space.

Condense is mainly useful to reduce the size of the VxWorks symbol table,
and to increase the speed of loading across a network.
*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <a.out.h>		/* use "standard" */

#include "UniWorks.h"


/* The different systems use different names for the same info in some
   structures.  Make them the same here */

#define A_OUT_HDR exec
#define TEXTSIZE a_text
#define DATASIZE a_data
#define BSSSIZE a_bss
#define SYMSIZE a_syms
#define TRSIZE a_trsize
#define DRSIZE a_drsize
#define ENTRY a_entry

#define U_SYM_STRUCT nlist
#define U_SYM_TYPE n_type
#define U_SYM_VALUE n_value

char *malloc ();
char *calloc ();

LOCAL char usage [] = "usage: condense <objFile >condensedObjFile";


/*******************************************************************************
*
* main - condense an object module, by removing superflous stuff
*/

VOID main (argc)
    int argc;

    {
    char *symTbl;
    int strSize;
    char *strTbl;
    int nSyms;
    short *symIndex;
    struct nlist *symPtr;
    int newSymIndex;
    int delSym;
    char *condStrPtr;
    int nSym;
    int len;
    int relSize;
    int delSeg;
    int nSegs;
    struct relocation_info seg;
    struct A_OUT_HDR hdr;
    struct A_OUT_HDR outHdr;
    int nbytes;

    if (argc > 1)
	error (usage);

    /* Read the header of the input file, and copy it to the output header. */

    if (fread ((char *)&hdr, sizeof (hdr), 1, stdin) < 1)
	error ("Error reading header of input file");

    bcopy ((char *)&hdr, (char *)&outHdr, sizeof (hdr));

    /* Copy the text and data to the output file */

    fseek (stdin,  (long)N_TXTOFF (hdr), 0);
    fseek (stdout, (long)N_TXTOFF (hdr), 0);
    for (nbytes = hdr.TEXTSIZE + hdr.DATASIZE; nbytes > 0; --nbytes)
	putchar (getchar ());

    /* Allocate a symbol table array, and a string table.  Read the
       string table into memory. The size of the string table is the
       contained as the first four bytes in the string table. */

    symTbl = malloc ((unsigned int)hdr.SYMSIZE);

    if (symTbl == NULL)
	error ("Error symbol table too large");

    fseek (stdin, (long)N_STROFF (hdr), 0);
    fread ((char *)&strSize, sizeof (strSize), 1, stdin);

    strTbl = malloc ((unsigned int)strSize);

    if (strSize == NULL)
	error ("Error string table too large");

    if (fread ((char *)(strTbl + 4), strSize - 4, 1, stdin) != 1)
	error ("Error reading string table");

    /* Read the symbols into memory one at a time.  */

    nSyms = hdr.SYMSIZE / sizeof (*symPtr);
    symIndex = (short *) calloc (nSyms, sizeof (short));

    if (symIndex == NULL)
	error ("Error symbol index table too large");

    fseek (stdin, (long)N_SYMOFF (hdr), 0);

    symPtr = (struct nlist *) symTbl;
    newSymIndex = 0;
    delSym = 0;
    condStrPtr = strTbl + 4;

    for (nSym = 0; nSym < nSyms; nSym++)
	{
	fread ((char *)symPtr, sizeof (*symPtr), 1, stdin);

#ifdef	NO_ABS_SYMS
	if ((symPtr->n_type & N_TYPE) == N_ABS)
	    {
	    /* It's an absolute symbol, so we don't want to keep it. */

	    symIndex [nSym] = NONE;
	    delSym++;
	    }
	else
	    {
#endif	NO_ABS_SYMS
	    /* We do want to keep it.  Move its string down (plus EOS) past any
	       that we have deleted, set the new symbol number in the
	       index table, fix the string offset in the symbol,
	       and increment symPtr to read the next one. */

	    len = strlen (strTbl + symPtr->n_un.n_strx) + 1;
	    bcopy ((char *)strTbl + symPtr->n_un.n_strx, (char *)condStrPtr,
		    len);
	    symPtr->n_un.n_strx = condStrPtr - strTbl;
	    condStrPtr += len;
	    symIndex [nSym] = newSymIndex++;
	    symPtr++;
#ifdef	NO_ABS_SYMS
	    }
#endif	NO_ABS_SYMS
	}
    /* The symbol table and string table are now correct, in memory.  
       Update the symbol table size, in the new header. newSymIndex
       currently is the number of symbols. */

    outHdr.SYMSIZE = newSymIndex * sizeof (*symPtr);

    /* Next, we need to do the relocation segments.  We seek to
       the correct position on the input and output files, then read
       them one at a time.  If the segment is one we want to keep,
       we correct it (since its symbol number might have changed) and
       write it out.  Otherwise we ignore it. We do this separately
       for the text and data relocation segments. */

    relSize = 0;
    delSeg = 0;
    fseek (stdin,  (long)N_TXTOFF (hdr) + hdr.TEXTSIZE + hdr.DATASIZE, 0);
    fseek (stdout, (long)N_TXTOFF (hdr) + hdr.TEXTSIZE + hdr.DATASIZE, 0);

    for (nSegs = hdr.TRSIZE / sizeof (seg); nSegs > 0; --nSegs)
	{
	fread ((char *)&seg, sizeof (seg), 1, stdin);
	if (! seg.r_pcrel)
	    {
	    seg.r_symbolnum = symIndex [seg.r_symbolnum];
	    fwrite ((char *)&seg, sizeof (seg), 1, stdout);
	    relSize += sizeof (seg);
	    }
	else
	    delSeg++;
	}
    outHdr.TRSIZE = relSize;

    /* and the data relocation */

    relSize = 0;

    for (nSegs = hdr.DRSIZE / sizeof (seg); nSegs > 0; --nSegs)
	{
	fread ((char *)&seg, sizeof (seg), 1, stdin);
	if (! seg.r_pcrel)
	    {
	    seg.r_symbolnum = symIndex [seg.r_symbolnum];
	    fwrite ((char *)&seg, sizeof (seg), 1, stdout);
	    relSize += sizeof (seg);
	    }
	else
	    delSeg++;
	}
    outHdr.DRSIZE = relSize;

    /* Now the relocation info is all correctly output.  The header and
       the symbol table and the string table are ok in memory.  Write
       them to the output file. */

    strSize = condStrPtr - strTbl;
    fseek (stdout, (long)N_SYMOFF (outHdr), 0);
    fwrite ((char *)symTbl, (int)outHdr.SYMSIZE, 1, stdout);
    * (int *) strTbl = strSize;
    fwrite ((char *)strTbl, strSize, 1, stdout);

    fseek (stdout, (long)0, 0);
    fwrite ((char *)&outHdr, sizeof (outHdr), 1, stdout);

    fprintf (stderr, "\nDeleted %d symbols (out of %d) and %d rel segs.\n",
	    delSym, nSyms, delSeg);

    exit (0);
    }
/*******************************************************************************
*
* error - print error message and die
*
* This routine prints an error message on the standard error output and
* aborts the program with the error status ERROR.  The error message is
* output with the specified format with the single specified argument.
*
* VARARGS1
*/

VOID error (format, arg)
    char *format;	/* format of error message */
    char *arg;		/* argument supplied to format (arbitrary type!) */

    {
    fprintf (stderr, format, arg);
    fprintf (stderr, "\n");
    exit (ERROR);
    }
