/* rngLib.h - ring buffer subroutine library header */

/*
modification history
--------------------
01e,22aug88,gae  removed incorrect comment about using 'shorts'
		 for optimization.
01d,30may88,dnw  changed to v4 names.
01c,10nov87,dnw  changed ring pointers from shorts to ints.
		 rings can now be > 32K bytes.
01b,15aug84,dnw  removed several macros, now routines in rngLib.cs;
		   only rngGetC and rngPutC remain as macros.
01a,06jun84,dnw  culled from old drvLib.h
*/

#ifndef	INCrngLibh
#define	INCrngLibh


/* HIDDEN */

/* typedefs */

typedef struct		/* RING - ring buffer */
    {
    int pToBuf;		/* offset from start of buffer where to write next */
    int pFromBuf;	/* offset from start of buffer where to read next */
    int bufSize;	/* size of ring in bytes */
    char *buf;		/* pointer to start of buffer */
    } RING;

/* END_HIDDEN */

typedef RING *RING_ID;


/* function declarations */

RING_ID rngCreate ();


/****************************************************************
*
* RING macros:
*   The following macros are designed to do various operations on
*   the RING object.  By using them, users can avoid having to know
*   the structure of RING.  However they are intended to be very 
*   efficient and this is why they are macros in the first place.
*   In general the parameters to them should be register variables
*   for maximum efficiency.
*/

/****************************************************************
*
* RNG_ELEM_GET - get one character from a ring buffer
*
* This macro gets a single character from the specified ring buffer.
* Must supply temporary variable (register int) 'fromP'.
*
* RETURNS: 1 if there was a char in the buffer to return, 0 otherwise
*/

#define RNG_ELEM_GET(ringId, pCh, fromP)		\
    (						\
    fromP = (ringId)->pFromBuf,			\
    ((ringId)->pToBuf == fromP) ?		\
	0 					\
    :						\
	(					\
	*pCh = (ringId)->buf[fromP],		\
	(ringId)->pFromBuf = ((++fromP == (ringId)->bufSize) ? 0 : fromP), \
	1					\
	)					\
    )

/****************************************************************
*
* RNG_ELEM_PUT - put one character into a ring buffer
*
* This macro puts a single character into the specified ring buffer.
* Must supply temporary variable (register int) 'toP'.
*
* RETURNS: 1 if there was room in the buffer for the char, 0 otherwise
*/

#define RNG_ELEM_PUT(ringId, ch, toP)		\
    (						\
    toP = (ringId)->pToBuf,			\
    (toP == (ringId)->pFromBuf - 1) ?		\
	0 					\
    :						\
	(					\
    	(toP == (ringId)->bufSize - 1) ?	\
	    (					\
	    ((ringId)->pFromBuf == 0) ?		\
		0				\
	    :					\
		(				\
		(ringId)->buf[toP] = ch,	\
		(ringId)->pToBuf = 0,		\
		1				\
		)				\
	    )					\
	:					\
	    (					\
	    (ringId)->buf[toP] = ch,		\
	    (ringId)->pToBuf++,			\
	    1					\
	    )					\
	)					\
    )

#endif	INCrngLibh
