/* openchip.h - ISI Liberator OpenChip general structures and defines */

/* 
 * This file includes definitions for the following:
 *
 * openchip, scratch area, tdduscc dma regs, tdlink and sclink structs
 * and general purpose output ports
 */
 
/* OpenChip DMA Bus Addresses */

#define IO_ADRS_TYA		0x1000100	/* SCN68562 A reg address */
#define IO_ADRS_TYB		0x1000120	/* SCN68562 B reg address */
#define IO_ADRS_PALETTE		0x1000140	/* Video */
#define IO_ADRS_PORT1	(*(char *)0x1000148)	/* GP output port 1 */
#define IO_ADRS_PORT2	(*(char *)0x1000150)	/* GP output port 2 */
#define IO_ADRS_QXDMA		0x1000160	/* SEEQ8003 DMA reg */
#define IO_ADRS_QX	((char *) 0x1000168)	/* SEEQ8003 reg address */
#define IO_ADRS_SCSIDMA		0x1000170	/* SCSI DMA reg */
#define IO_ADRS_SCSI		0x1000178	/* WD33C93 reg address */

/* OpenChip general interrupts */

#define VEC_GENBASE	0x80
#define VEC_VIDFRAME	(VEC_GENBASE + 0)
#define VEC_CLOCK	(VEC_GENBASE + 1)
#define VEC_AUDIO	(VEC_GENBASE + 2)
#define VEC_KEYBOARD	(VEC_GENBASE + 3)
#define VEC_SCSI	(VEC_GENBASE + 4)
#define VEC_ETHERNET	(VEC_GENBASE + 5)
#define VEC_ACFAIL	(VEC_GENBASE + 6)
#define VEC_SYSFAIL	(VEC_GENBASE + 7)

/* DUSCC channel 0 interrupts */
#define VEC_DU0_BASE	0x88
#define VEC_DU0_RX	(VEC_DU0_BASE + 0)	/* Receiver Ready */
#define VEC_DU0_TX	(VEC_DU0_BASE + 1)	/* Transmitter Ready */
#define VEC_DU0_SP	(VEC_DU0_BASE + 2)	/* RX/TX Status */
#define VEC_DU0_EX	(VEC_DU0_BASE + 3)	/* External Status */
#define VEC_DU0_DR	(VEC_DU0_BASE + 4)	/* RX DMA */
#define VEC_DU0_DW	(VEC_DU0_BASE + 5)	/* TX DMA */

/* DUSCC channel 1 interrupts */
#define VEC_DU1_BASE	0x90
#define VEC_DU1_RX	(VEC_DU1_BASE + 0)	/* Receiver Ready */
#define VEC_DU1_TX	(VEC_DU1_BASE + 1)	/* Transmitter Ready */
#define VEC_DU1_SP	(VEC_DU1_BASE + 2)	/* RX/TX Status */
#define VEC_DU1_EX	(VEC_DU1_BASE + 3)	/* External Status */
#define VEC_DU1_DR	(VEC_DU1_BASE + 4)	/* RX DMA */
#define VEC_DU1_DW	(VEC_DU1_BASE + 5)	/* TX DMA */


/*
 * linked chain structure for DUSCC DMA. 10 bytes.
 */
struct tdlink {
	char		*dl_cptr;
	struct tdlink	*dl_lnptr;
	unsigned char	dl_cnt;
};

/*
 * per DUSCC channel DMA registers. 32 bytes.
 */
struct tdduscc {
	struct tdlink	du_rxchn;
	unsigned char	du_rxhalt;
	unsigned char	du_rxbuf;	/* read only */
	struct tdlink	du_txchn;
	unsigned char	du_rxstat;
/*
 * bit positions in the du_rxstat and du_txstat register in op_chan0 and
 * op_chan1. Notice that COC and ERR are resettable whereas ACT and INEA are
 * writable.
 */
#define	OP_DU_ACT	0x08
#define OP_DU_COC	0x04
#define	OP_DU_ERR	0x02
#define	OP_DU_INEA	0x01

	unsigned char	du_txstat;
	unsigned char	du_vector;
	short		du_pad[3];	/* alignment padding */
};

#ifndef	SC_LINK_DEFINED
struct qs_link {
        caddr_t sl_ptr;
        unsigned long sl_cnt;
        struct qs_link *sl_lnptr;
};
#define	SC_LINK_DEFINED
#endif	SC_LINK_DEFINED

/*
 * A structure definition giving the arrangement of the OpenChip
 * programmer usable registers. Notice that all 32 bit registers
 * must be long aligned.
 */

struct openchip {

	/* offset 0x00 */

	struct oc_scratch *op_scratch;
	char		op_reset;	/* a write here resets the OpenChip */
	unsigned char	op_system;	/* read only configuration registers */

#define	OP_SYS_MSIZE		1
#define	OP_SYS_CODE_MASK	0x1e
#define	OP_SYS_CODE_SHIFT	1	/* no> of right shifts to align code */

	unsigned short	op_version;
	unsigned long	op_sysclk;
	unsigned short	op_sysperiod;
	unsigned char	op_genvec;	/* general purpose interrupt vector */

#define	OP_INT_MASK	0x07
#define	OP_INT_VIDEO	0
#define	OP_INT_CLOCK	1
#define	OP_INT_AUDIO	2
#define	OP_INT_KEYBOARD	3
#define	OP_INT_SCSI	4
#define	OP_INT_ETHERNET	5
#define OP_INT_ACFAIL	6
#define	OP_INT_SYSFAIL	7

	char		op_pad1;	/* alignment padding */

	/* offset 0x10 */

	unsigned short	op_clkpreset;
	unsigned char	op_clkstat;

#define OP_CLK_INEA	1
#define OP_CLK_TICK	2

	unsigned short	op_bustimeout;	/* access to internal BUSTP register */

#define DEFBUSTIMEOUT	1600		/* 50 uSec */

	short		op_pad2;	/* alignment padding */
	unsigned char	op_failstat;	/* ACFAIL/SYSFAIL register */

#define	OP_ACSY_ACFL	0x20
#define	OP_ACSY_SYSFL	0x10
#define	OP_ACSY_2OFF	0x08
#define	OP_ACSY_1OFF	0x04
#define	OP_ACSY_AINEA	0x02
#define	OP_ACSY_SINEA	0x01

	short		op_pad2_0[3];	/* alignment padding */

	/* offset 0x20 */

	long		*op_vidstart;	/* start of video list */
	long		*op_vidpos;	/* current posn. in video list, R/O */
	unsigned char	op_viewsyn;
	unsigned char	op_vidcnt;	/* direct access to the internal */
	unsigned char	op_vidblk;	/* VIDCNT, VIDBLK and VIDSYN */
	unsigned char	op_vidsyn;	/* registers. */
	long		op_pad3;		/* alignment padding */

	/* offset 0x30 */

	struct tdduscc	op_duchan[2];

	/* offset 0x70 */

	long		*op_audptr;	/* waveform table/tone volume */
	unsigned short	op_audlen;	/* waveform/tone period */
	unsigned short	op_audrptcnt;
	unsigned char	op_audstat;

#define OP_AUD_INEA	0x01
#define OP_AUD_ERR	0x02
#define OP_AUD_FIN	0x04
#define OP_AUD_ACT	0x08
#define OP_AUD_CONT	0x10
#define OP_AUD_TONE	0x20

	short		op_pad6[3];	/* alignment padding */

	/* offset 0x80 */

	unsigned char	op_kbdata;
	unsigned char	op_kbdstat;

#define OP_KB_INEA	1
#define OP_KB_FRM	2
#define OP_KB_OVFL	4
#define OP_KB_RDY	8

	short		op_pad7[7];	/* 4 byte alignment padding */

	/* offset 0x90 */

	unsigned char	op_scsistatus;	/* copy of WD33C93 status register */
	unsigned char	op_scstat;	/* SCSI state disconn, reselect, etc.*/

#define	OP_ST_STATEMASK	0xe0
#define	OP_ST_MST_MASK	0x18
#define	OP_ST_PHSMASK	0x07

#define	SC_DISCONNECT	0x00
#define	SC_RESELECTED	0x20
#define	SC_RSLMSGIN	0x40
#define	SC_SELECTING	0x60
#define	SC_SELMSGOUT	0x80
#define	SC_CONNECTED	0xa0

#define MST_NULL	0x00
#define	MST_DISC	0x08
#define	MST_CMPLT	0x10
#define	MST_CMPLT2	0x18

#define	OP_ST_PHS_MSG_IN	7
#define	OP_ST_PHS_MSG_OUT	6
#define OP_ST_PHS_UNS_IN	5
#define OP_ST_PHS_UNS_OUT	4
#define	OP_ST_PHS_STATUS	3
#define	OP_ST_PHS_COMMAND	2
#define	OP_ST_PHS_DATA_IN	1
#define	OP_ST_PHS_DATA_OUT	0

#define	OP_ST_PHS_IO	1
#define	OP_ST_PHS_CD	2
#define	OP_ST_PHS_MSG	4

	unsigned char	op_scsum;	/* pointer + count at start of DMA */
	unsigned char	op_scsintr;	/* SCSI interrupt reason */

#define	SCNT_REASON_MASK	7

#define	SCNT_NULL	0
#define	SCNT_MANUAL	1
#define	SCNT_MANCMND	2
#define	SCNT_MANRESEL	3
#define	SCNT_PROTOCOL	4
#define	SCNT_BUSERROR	5
#define	SCNT_DMA_BUSERROR	6

	unsigned char	op_sccntrl;	/* SCSI control register */

#define	OP_SC_INEA	0x01
#define	OP_SC_AUTO	0x02
#define OP_SC_DACT	0x04
#define OP_SC_SELATN	0x08

	unsigned short	op_scsiwrite;	/* write SCSI chip address and data */
	char		*op_scdmaptr;	/* DMA pointer */
	struct lunelem	*op_lunbase;	/* base of the lunelem array */

	/* offset 0xa0 */

	struct eth_desc	*op_ethrx;
	struct eth_desc	*op_ethtx;
	unsigned char	op_ethrstat;	/* Ethernet status register */
	unsigned char	op_ethtstat;	/* Ethernet status register */

#define	OP_ETH_INEA	1
#define	OP_ETH_ERR	2
#define	OP_ETH_EVNT	4
#define	OP_ETH_ACT	8
};

#define	OPENCHIP	((struct openchip *)0x1000000)



/* Names for the bits in the sync patterns in the video list */

#define OP_VID_VSYNC           0x01
#define OP_VID_HSYNC           0x02
#define OP_VID_PSYNC           0x04
#define OP_VID_BLANK           0x08
#define OP_VID_DATA            0x10

/* Flags for use with a repeat count of 0 */

#define OP_VIDL_FRM             0x08
#define OP_VIDL_RSTRT           0x10
#define OP_VIDL_PL0             0x20
#define OP_VIDL_PL1             0x40
#define OP_VIDL_PCHNG           0x80

/* poll delay between OpenChip accesses in cached loops */

#define	OC_POLL_DELAY()		{register i; for (i = 15; --i > 0;);}

/*
 * A structure definition giving the layout of variables in the OpenChip
 * scratch area.
 */

struct oc_scratch {
	struct qs_link sr_datlnk;
	char *sr_pcmnd;	/* pointer to command bytes */
	char *sr_pstat;	/* pointer to status bytes */
	char sr_ncmnd;	/* number of command bytes */
	char sr_nstat;	/* number of status bytes */
	unsigned char sr_msginbuf;
	char sr_pad;
		/*
		 * head and tail pointers to the
		 * ready to start list.
		 */
	struct scsi_cmd *sr_hdstrt;
	struct scsi_cmd *sr_tlstrt;
		/*
		 * head and tail pointers to the
		 * completed command list.
		 */
	struct scsi_cmd *sr_hdcmplt;
	struct scsi_cmd *sr_tlcmplt;
		/*
		 * pointers to the current lunelem
		 * and command structures.
		 */
	struct lunelem *sr_ptrlun;
	struct scsi_cmd *sr_ptrcmnd;
};


/* Liberator general purpose output ports */

#define Q_OP_PORT1	(*(char *)0x1000148)	/* addr of output port 1 */
#define Q_OP_PORT2	(*(char *)0x1000150)	/* addr of output port 2 */

/*
 * The DATIN signal is read through the DUSCC chip GPI2B input.
 */
#define	DATIN	(((struct dusccchan *)0x1000100)[1].du_ictsr & DU_ICTS_GPI2)

/* Values that may be written to output ports 1 and 2 */

#define OP1_RES_VIDSEL0 0
#define OP1_SET_VIDSEL0 1
#define OP1_RES_VIDSEL1 2
#define OP1_SET_VIDSEL1 3
#define OP1_RES_PALSEL0 4
#define OP1_SET_PALSEL0 5
#define OP1_RES_PALSEL1 6
#define OP1_SET_PALSEL1 7
#define OP1_RES_TURNOFF 8
#define OP1_SET_TURNOFF 9
#define OP1_RES_VIDSTOP 10
#define OP1_SET_VIDSTOP 11
#define OP1_RES_RESFIFO 12		/* reset fifo */
#define OP1_SET_RESFIFO 13		/* enable fifo */
#define OP1_RES_RESSCSI 14		/* enable SCSI */
#define OP1_SET_RESSCSI 15		/* reset SCSI */

#define OP2_RES_DATOUT 0		/* serial data */
#define OP2_SET_DATOUT 1
#define OP2_RES_STROBE 2		/* serial clock */
#define OP2_SET_STROBE 3
#define OP2_RES_TIMSEL 4		/* disable clock */
#define OP2_SET_TIMSEL 5		/* enable clock */
#define OP2_RES_RAMSEL 6		/* disable nv-RAM */
#define OP2_SET_RAMSEL 7		/* enable nv-RAM */
#define OP2_RES_KEYRESET 8		/* enable kb */
#define OP2_SET_KEYRESET 9		/* reset kb */
#define OP2_RES_ETHER	12		/* reset Ethernet */
#define OP2_SET_ETHER	13		/* enable Ethernet */
