/* tyCoDrv - serial driver for SCN68562 */

/*#define	DEBUG		/**/

#include "UniWorks.h"
#include "types.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "wdLib.h"
#include "tyLib.h"

#include "config.h"
#include "scn68562.h"
#include "openchip.h"
#include "liberator.h"

#define	NUM_DU_CHANS		2
#define CHANNEL0_VECTOR		VEC_DU0_DR
#define CHANNEL1_VECTOR		VEC_DU1_DR
#define TX_FIFO_SIZE		4
#define RX_FIFO_SIZE		4
#define DFLT_BAUD		9600

short btbl[] = {
    6, 
    1152, 	/* 50		BAUD*/
    768, 	/* 75		BAUD*/
    524, 	/* 110		BAUD*/
    430, 	/* 134		BAUD*/
    384, 	/* 150		BAUD*/
    288, 	/* 200		BAUD*/
    192, 	/* 300		BAUD*/
    96, 	/* 600		BAUD*/
    48, 	/* 1200		BAUD*/
    32, 	/* 1800		BAUD*/
    24, 	/* 2400		BAUD*/
    12, 	/* 4800		BAUD*/
    6, 		/* 9600		BAUD*/
    3, 		/* 19200	BAUD*/
    2 		/* 38400	BAUD*/
};

/*
 * The DUSCC chip is clocked at 14.7456MHz. Unfortunately the internal baud
 * rate generator does not generate exactly the rates required by Unix so the
 * clock/timer will have to be used. The clock is divided by 4 before being
 * fed to the clock/timer and the output of the clock/timer should be 32 times
 * the required bit rate if the times 16 receive mode is to be used.
 * In addition, before all this is done, it is divided by 2 for good measure !
 */

#define	BD(sp)	(14745600/(2 * 4 * sp * 32))

struct tydev { 	
	TY_DEV tyDev;
	BOOL created;	/* true if this device has really been created */
	char channel;
	};

/* Globals */

int tyCoNum;
char icr;
struct tydev tyCoDev[NUM_DU_CHANS];

struct dusccchan *duaddr[NUM_DU_CHANS] = {
	(struct dusccchan *) IO_ADRS_TYA,
	(struct dusccchan *) IO_ADRS_TYB,
};

/* forward declarations */

int tyCoOpen ();
int tyCoRead ();
int tyCoWrite ();
STATUS tyCoIoctrl ();
int tyCoStartup();
VOID tyICTSR();
VOID tyTransmit();
VOID tyReceive();
VOID tyRx0Intr();
VOID tyTx0Intr();
VOID tySp0Intr();
VOID tyEx0Intr();
VOID tyDr0Intr();
VOID tyDw0Intr();
VOID tyRx1Intr();
VOID tyTx1Intr();
VOID tySp1Intr();
VOID tyEx1Intr();
VOID tyDr1Intr();
VOID tyDw1Intr();

/***********************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*
*/

STATUS tyCoDrv ()
{
	tyCoNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen, 
		(FUNCPTR) NULL, tyCoRead, tyCoWrite, tyCoIoctrl);

	intConnect(INUM_TO_IVEC(VEC_DU0_RX),tyRx0Intr,NONE);
	intConnect(INUM_TO_IVEC(VEC_DU0_TX),tyTx0Intr,NONE);
	intConnect(INUM_TO_IVEC(VEC_DU0_SP),tySp0Intr,NONE);
	intConnect(INUM_TO_IVEC(VEC_DU0_EX),tyEx0Intr,NONE);
	intConnect(INUM_TO_IVEC(VEC_DU0_DR),tyDr0Intr,NONE);
	intConnect(INUM_TO_IVEC(VEC_DU0_DW),tyDw0Intr,NONE);

	intConnect(INUM_TO_IVEC(VEC_DU1_RX),tyRx1Intr,NONE);
	intConnect(INUM_TO_IVEC(VEC_DU1_TX),tyTx1Intr,NONE);
	intConnect(INUM_TO_IVEC(VEC_DU1_SP),tySp1Intr,NONE);
	intConnect(INUM_TO_IVEC(VEC_DU1_EX),tyEx1Intr,NONE);
	intConnect(INUM_TO_IVEC(VEC_DU1_DR),tyDr1Intr,NONE);
	intConnect(INUM_TO_IVEC(VEC_DU1_DW),tyDw1Intr,NONE);

	if(tyCoHrdInit (0,CHANNEL0_VECTOR) == ERROR) 
		return(ERROR);
	if(tyCoHrdInit (1,CHANNEL1_VECTOR) == ERROR)
		return(ERROR);

	return (OK);
}

/******************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;		/* Name to use for this device */
	int channel;	/* Physical channel for this device (0 or 1) */
	int rdBufSize;	/* Read buffer size, in bytes */
	int wrtBufSize;	/* Write buffer size, in bytes */

{
	/* if there is a device already on this channel, don't do it */

	if (tyCoDev [channel].created)
		return (ERROR);
	tyCoDev[channel].channel = channel;

	/* 
	 * initialize the ty descriptor, and turn on the bit for this
	 * receiver in the interrupt mask 
	 */

	if (tyDevInit ((TY_DEV_ID) &tyCoDev [channel], 
		rdBufSize, wrtBufSize, tyCoStartup) != OK)
		return (ERROR);

	/* 
	 * Mark the device as having been created, and add the device to
	 * the io sys 
	 */

	tyCoDev [channel].created = TRUE;


	return (iosDevAdd ((DEV_HDR *) &tyCoDev [channel], name, tyCoNum));
}

/******************************************************************
*
* tyCoHrdInit - initialize the DUSCC.
*
*/

tyCoHrdInit(dev,vector)
	short dev;
	int vector;
{
	short i;
	unsigned int s;
	register int speed;
	struct dusccchan *addr = duaddr[dev];	
	struct nv_ram nvram;

	s = intLock();

	/* Reset */
	addr->du_ccr = DU_CCR_DISABL_TX;
	addr->du_ccr = DU_CCR_DISABL_RX;
	addr->du_ccr = DU_CCR_CT_STOP;

	addr->du_pcr = DU_PCR_X2_CRYS | DU_PCR_GPO2 | DU_PCR_SY_RTS | 
			DU_PCR_RTxC_INPUT | DU_PCR_TRxC_INPUT;
	addr->du_cmr1 = DU_CMR1_NRZ | DU_CMR1_PRTY_NONE | DU_CMR1_ASYNC;

	/*
	 * transmit clock is times 32 clock/timer. The clock timer is divided by
	 * 2 before delivery to the transmitter section where the times 16 mode
	 * is used.
	 */

	addr->du_ttr = DU_TTR_CLK_32xOWN_CT;
	addr->du_rtr = DU_RTR_CLK_CT;

	/*
	 * TxRDY goes active if the Tx FIFO is empty and RxRDY goes active if
	 * the Rx FIFO is not empty. ALso RTS  = high and DTR = high (GPO1)
	 */

	addr->du_omr = DU_OMR_TxRDY_EMPTY | DU_OMR_RxRDY_NEMPT |
		DU_OMR_GPOUT1 | DU_OMR_RTS; 

	/*
	 * select the crystal as the source for the clock timer and do not
	 * prescale it.
	 */

	addr->du_ctcr = DU_CTCR_0_DTCT_PRESET | DU_CTCR_1_SCALE | DU_CTCR_4CLK;

	/*
	 * interrupt enable register.
	 */

	addr->du_ier = DU_IER_TSR7_3 | DU_IER_RXRDY;

	/* 8 data bits tx, 1 stop bit, enable tx control by CTS */
	addr->du_tpr = DU_TPR_TX_8BITS | DU_TPR_1STOP | DU_TPR_TX_CTS;

	/* 8 data bits rx, strip rx top bit, enable receiver */
	addr->du_rpr = DU_RPR_RX_8BITS | DU_RPR_STRIP_PRTY | DU_RPR_DCD_ENBL_RX;

	/* set the baudrate to the setting in nv_ram
	 * the lower four bits of the port config word determine
	 * the baud rate, as defined in ttydev.h
	 * if the config word contains zero, use 9600 baud
	 */
	if (sysGetNvRam (&nvram) != OK)   /* parity error in nv ram */
	    speed = 13;
	else 
	    speed = nvram.sio[dev] & 0xf;
	if (speed != 15) {
	    addr->du_ctprh = btbl[speed] >> 8;
	    addr->du_ctprl = btbl[speed];
	} 
	else {
	    addr->du_ttr = DU_TTR_CLK_BRG | 0xf;
	    addr->du_rtr = DU_RTR_CLK_BRG | 0xf;
	}

	/*
	 * turn on the master interrupt enable for this channel and set the
	 * DUSCC up to modify vector bits 0:2 so that it will work properly
	 * with the OpenChip.
	 */

	OPENCHIP->op_duchan[dev].du_vector = vector;
	if (dev == 0) {
		icr = DU_ICR_A_INT_ENBL | DU_ICR_INTLV_A | 
			DU_ICR_VCTR0 | DU_ICR_VIS | DU_ICR_MDFY_2_0;
	}
	else
		icr |= DU_ICR_B_INT_ENBL | DU_ICR_INTLV_A | 
			DU_ICR_VCTR0 | DU_ICR_VIS | DU_ICR_MDFY_2_0;

	duaddr[0]->du_icr = icr;
	duaddr[0]->du_ivr = VEC_GENBASE;

	/* Finally we enable rx, tx and set the counter going */
	addr->du_ccr = DU_CCR_CT_PRST;
	addr->du_ccr = DU_CCR_CT_START;

	/* 
	 * There is a bug in the DUSCC which can cause a permanent transmit
	 * break condition when the transmitter is enabled. The following 
	 * code gets round this problem.
	 */

	addr->du_cmr2 = DU_CMR2_CONN_LOOP | DU_CMR2_POLLED | DU_CMR2_NO_FCS;
	addr->du_ccr = DU_CCR_RST_TX;
	addr->du_ccr = DU_CCR_ENBL_TX;
	for (i = 100; --i > 0;)
		{
	    	/* delay slightly for the condition to be cleared */
		}
	addr->du_ccr = DU_CCR_DISABL_TX;

	/* end of bug fix */

	addr->du_cmr2 = DU_CMR2_CONN_NORM | DU_CMR2_POLLED | DU_CMR2_NO_FCS;

	addr->du_ccr = DU_CCR_RST_TX;
	addr->du_ccr = DU_CCR_ENBL_TX;

	addr->du_ccr = DU_CCR_RST_RX;
	addr->du_ccr = DU_CCR_ENBL_RX;

	intUnlock(s);
	return(OK);
}

/**********************************************************************
*
* tyCoOpen - open file to DUSCC
*
* ARGSUSED
*/

int tyCoOpen (pTyCoDv, name, mode)
	struct tydev *pTyCoDv;
	char *name;
	int mode;
{
	return ((int) pTyCoDv);
}

/**********************************************************************
*
* tyCoRead - Task level read routine for DUSCC.
*
*/

int tyCoRead (pTyCoDv, buffer, maxbytes)
	struct tydev *pTyCoDv;
	char *buffer;
	int maxbytes;

{
	return (tyRead ((TY_DEV_ID) pTyCoDv, buffer, maxbytes)); 
}

/**********************************************************************
*
* tyCoWrite - Task level write routine for DUSCC.
*
*/

int tyCoWrite (pTyCoDv, buffer, nbytes)
	struct tydev *pTyCoDv;
	char *buffer;
	int nbytes;

{
	return (tyWrite ((TY_DEV_ID) pTyCoDv, buffer, nbytes)); 
}

/***********************************************************************
*
* tyCoIoctrl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to tyIoctl.
*/

STATUS tyCoIoctrl (pTyCoDv, request, arg)
	struct tydev *pTyCoDv;	/* device to control */
	int request;	/* request code */
	int arg;		/* some argument */

{
	unsigned int s;
	STATUS status;
	struct dusccchan *addr;

	switch (request) {
		case FIOBAUDRATE:

			s = intLock();
			addr = duaddr[pTyCoDv->channel];
			addr->du_ccr = DU_CCR_DISABL_TX;
			addr->du_ccr = DU_CCR_DISABL_RX;
			addr->du_ccr = DU_CCR_CT_STOP;
			if (arg != 38400) {
			    addr->du_ttr = DU_TTR_CLK_32xOWN_CT;
			    addr->du_rtr = DU_RTR_CLK_CT;
			    addr->du_ctprh = BD (arg) >> 8;
			    addr->du_ctprl = BD (arg);
			} 
			else {
			    addr->du_ttr = DU_TTR_CLK_BRG | 0xf;
			    addr->du_rtr = DU_RTR_CLK_BRG | 0xf;
			}
			addr->du_ccr = DU_CCR_CT_PRST;
			addr->du_ccr = DU_CCR_CT_START;
			addr->du_ccr = DU_CCR_ENBL_TX;
			addr->du_ccr = DU_CCR_ENBL_RX;
			intUnlock(s);

			status = OK;
			break;

		default:
			status = tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg);
			break;
	}
	return (status);
}

VOID tyCoStartup(pTyCoDv)
	struct tydev *pTyCoDv;
{
	char outchar;

	if(pTyCoDv->channel == 0)
		tyTransmit(pTyCoDv->channel,DU_GSR_A_TXRDY);
	else
		tyTransmit(pTyCoDv->channel,DU_GSR_B_TXRDY);
}

/************************************************************************
*
* tyRx0Intr - Receiver Interrupt routines
* tyRx1Intr 
*
*/

VOID tyRx0Intr()
{
unsigned char c;

	if(tyCoDev[0].created)
		tyReceive(duaddr[0],DU_GSR_A_RXRDY,0);
	else
		c = duaddr[0]->du_rxfifo;
}


VOID tyRx1Intr() 
{ 
unsigned char c;

	if(tyCoDev[1].created)
		tyReceive(duaddr[1],DU_GSR_B_RXRDY,1);
	else
		c = duaddr[1]->du_rxfifo;
}

/************************************************************************
*
* tyReceive - called by receiver interrupt routines to get the character
*
*/

VOID tyReceive(addr,rdychar,channel)
	struct dusccchan *addr;
	char rdychar, channel;
{
	char stat,c;

	while(addr->du_gsr & rdychar) {

		c = addr->du_rxfifo;
		stat = addr->du_rsr & (DU_RSR_FRM_ERR | DU_RSR_PRTY_ERR);
		addr->du_rsr = stat;
		tyIRd((TY_DEV_ID) &tyCoDev[channel], c);
		addr->du_gsr |= rdychar;
	}
}

/***********************************************************************
*
* tyTx0Intr - transmitter interrupt routines
* tyTx1Intr
* 
*/

VOID tyTx0Intr()
{

	if(tyCoDev[0].created)
		tyTransmit(0,DU_GSR_A_TXRDY);
}
VOID tyTx1Intr()
{

	if(tyCoDev[1].created)
		tyTransmit(1,DU_GSR_B_TXRDY);
}

/***************************************************************************
*
* tyTransmit - called by the transmitter interrupt routines to put
*	       a character
*
*/

VOID tyTransmit(dev,rdychar)
	char dev,rdychar;
{
	char n;
	char outchar;
	struct dusccchan *addr;
	struct tydev *tydevptr;

	addr = duaddr[dev];
	tydevptr = &tyCoDev[dev];

	if(tyITx((TY_DEV_ID) tydevptr, &outchar) == OK)
		addr->du_txfifo = outchar;
}	

/* tySp0Intr,tySp1Intr */

VOID tySp0Intr()
{

	duaddr[0]->du_trsr |= DU_TRSR_TX_EMPTY | DU_TRSR_CTS_UNDRN;
	tyTransmit(0,DU_GSR_A_TXRDY);
}
VOID tySp1Intr()
{

	duaddr[1]->du_trsr |= DU_TRSR_TX_EMPTY | DU_TRSR_CTS_UNDRN;
	tyTransmit(1,DU_GSR_B_TXRDY);
}

/* tyEx0Intr, tyEx1Intr */

VOID tyEx0Intr() 
{ 

	duaddr[0]->du_ictsr = DU_ICTS_CT_ZERO | DU_ICTS_DLTA_DCD |
		DU_ICTS_DLTA_CTS_LC;
}

VOID tyEx1Intr() 
{ 

	duaddr[1]->du_ictsr = DU_ICTS_CT_ZERO | DU_ICTS_DLTA_DCD |
		DU_ICTS_DLTA_CTS_LC;
}


/* Rx DMA interrupts - should not occur */
VOID tyDr0Intr() 
{ 
}

VOID tyDr1Intr() 
{ 
}

/* Tx DMA interrupts - should not occur */
VOID tyDw0Intr() 
{ 
}

VOID tyDw1Intr() 
{ 
}

