/* pipeDrv.c - pipe I/O driver */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
This is the pipe driver.  It allows I/O to be performed between tasks
similarly to the way it is performed to devices, via constructs
known as pipes.  Pipes are created and given names (with pipeDevCreate) and
can then be read and written with normal read and write calls.

The main difference between pipes and device I/O is that pipe I/O is done in
discrete units, known as messages.  All the data written as a single write
call is kept together, and will all be read by a single read call.  In
the same way, a read call will read only the number of bytes which were
written to the message being read, even though there may be more messages
in the pipe waiting to be read.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, ptyDrv () to
initialize the driver, and ptyDevCreate () to create devices.
 
PIPEDRV
Before using the driver, it must be initialized by calling the routine:
.CS
    pipeDrv ()
.CE
This routine must be called before any reads, writes, or
pipeDevCreates.  Normally, it is called from usrRoot (2) in usrConfig (1).
 
CREATING PIPES
Before a pipe can be used, it must be created.  This is done
with the pipeDevCreate call:
.CS
STATUS pipeDevCreate (name, nMsgs, nBytes)
    char *name;  /* name of pipe to be created                 *
    int nMsgs;   /* number of msgs in ring buffer to allocate  *
    int nBytes;  /* number of bytes in ring buffer to allocate *
.CE
For instance, to create the device pipe "/pipe/demo",
with up to 10 messages of size 100 bytes, the proper call would be:
.CS
   pipeDevCreate ("/pipe/demo", 10, 100);
.CE
This device can now be opened, read, and written to;
creating and closing are not applicable.
 
IOCTL
The pipe driver only responds to the FIONREAD code.
*/

#include "UniWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "rngLib.h"
#include "semLib.h"

#define LOCK	oldLevel = intLock ();
#define UNLOCK	intUnlock (oldLevel);

#define MSG_OVERHEAD	(sizeof (short))


typedef struct			/* PIPE_DEV */
    {
    DEV_HDR devHdr;			/* pipe device header */
    RING_ID rngBuf;			/* ring buffer for this pipe */
    int maxBytes;			/* max bytes in pipe */
    int nbytes;				/* number of bytes in pipe */
    short moreBytes;
    SEM_ID readSemId;
    BOOL rdSemNeeded;
    } PIPE_DEV;


int pipeLostMsgs;		/* to inspect # of dropped messages */

LOCAL int pipeDrvNum;		/* driver number of pipe driver */

/* forward declarations */

LOCAL int pipeOpen ();
LOCAL int pipeRead ();
LOCAL int pipeWrite ();
LOCAL STATUS pipeIoctl ();

/*******************************************************************************
*
* pipeDrv - pipe driver main routine
*
* This routine should be called before any pipes are created.  It
* initializes and installs the driver.
*
* RETURNS: OK or ERROR
*/

STATUS pipeDrv ()

    {
    /* check if driver already installed */

    if (pipeDrvNum == 0)
	{
	pipeDrvNum = iosDrvInstall ((FUNCPTR) NULL, (FUNCPTR) NULL, pipeOpen,
			     (FUNCPTR) NULL, pipeRead, pipeWrite, pipeIoctl);
	}

    return (pipeDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* pipeDevCreate - create a pipe device
*
* This routine is called once for each pipe to be created.  It allocates
* the various structures necessary, and initializes the device.  The pipe
* will have a maximum of maxmsgs messages in the pipe at once.  Once that
* many are in the pipe, a task attempting to write to the pipe will be
* suspended until a message has been read.
*
* RETURNS: OK or ERROR
*/

STATUS pipeDevCreate (name, nMsgs, nBytes)
    char *name;		/* name of pipe to be created                 */
    int nMsgs;		/* number of msgs in ring buffer to allocate  */
    int nBytes;		/* number of bytes in ring buffer to allocate */

    {
    FAST PIPE_DEV *pPipeDev;
    int bufSize = nMsgs * (nBytes + MSG_OVERHEAD);

    if (pipeDrvNum < 1)
        {
        errnoSet (S_ioLib_NO_DRIVER);
        return (ERROR);
        }

    if (((pPipeDev = (PIPE_DEV *) malloc (sizeof (PIPE_DEV))) == NULL) ||
        ((pPipeDev->rngBuf = rngCreate (bufSize)) == NULL))
	{
	return (ERROR);
	}


    /* initialize pipe descriptor and add i/o device to system */

    pPipeDev->nbytes    = 0;
    pPipeDev->moreBytes = 0;
    pPipeDev->maxBytes  = bufSize - MSG_OVERHEAD;

    pPipeDev->readSemId = semCreate ();
    pPipeDev->rdSemNeeded = TRUE;

    return (iosDevAdd ((DEV_HDR *) pPipeDev, name, pipeDrvNum));
    }

/* routines supplied to I/O system */

/*******************************************************************************
*
* pipeOpen - open a pipe file
*
* This routine is called to open a pipe.  It returns a pointer to the
* device.  This routine is normally reached only via the I/O system.
*
* ARGSUSED
*/

LOCAL int pipeOpen (pPipeDev, name, mode)
    PIPE_DEV *pPipeDev;
    char *name;
    int mode;

    {
    return ((int) pPipeDev);
    }
/*******************************************************************************
*
* pipeRead - read bytes from a pipe
*
* This routine reads up to maxbytes bytes of the next message in the pipe.
* If the message is too long, the additional bytes are just discarded.
* 
* RETURNS:
*  number of bytes actually read;
*  will be between 1 and maxbytes, or zero if end of file, or ERROR
*/

LOCAL int pipeRead (pPipeDev, buffer, maxbytes)
    FAST PIPE_DEV *pPipeDev;	/* pointer to pipe descriptor */
    char *buffer;		/* pointer to buffer to receive bytes */
    unsigned int maxbytes;	/* max number of bytes to read into buffer */

    {
    FAST RING_ID ringId = pPipeDev->rngBuf;
    int nbytes;

    if (maxbytes == 0)
	return (0);

    /* wait for something to be in pipe */

    semTake (pPipeDev->readSemId);

    if (pPipeDev->moreBytes == 0)
	{
	/* get length of next message (first two bytes) */

	rngGetBuf (ringId, (char *) &pPipeDev->moreBytes, sizeof (short));
	}

    nbytes = min (pPipeDev->moreBytes, maxbytes);

    rngGetBuf (ringId, buffer, nbytes);

    pPipeDev->moreBytes -= nbytes;
    pPipeDev->nbytes    -= nbytes;	/* decrement number of bytes in pipe */


    /* if more stuff in buffer, enable next reader */

    if ((pPipeDev->moreBytes == 0) && rngEmpty (ringId))
	pPipeDev->rdSemNeeded = TRUE;
    else
	semGive (pPipeDev->readSemId);

    return (nbytes);
    }
/*******************************************************************************
*
* pipeWrite - write bytes to a pipe
*
* This routine writes a message of `nbytes' to the pipe.
*
* RETURNS: number of bytes written or ERROR
*/

LOCAL int pipeWrite (pPipeDev, buffer, nbytes)
    FAST PIPE_DEV *pPipeDev;	/* pointer to pipe descriptor */
    char *buffer;		/* pointer to buffer from which to write bytes*/
    FAST int nbytes;		/* number of bytes to write */

    {
    FAST RING_ID ringId = pPipeDev->rngBuf;
    int oldLevel;
    FAST int bytesToWrite  = nbytes + MSG_OVERHEAD;
    short shortNbytes = nbytes;

    /* check for valid number of bytes to write */

    if (nbytes == 0)
	return (0);

    if (bytesToWrite > pPipeDev->maxBytes)
	{
	logMsg ("pipeDrv: msg write too long:\n  pipe = \"%s\", length = %d, \
max = %d.\n",
	        pPipeDev->devHdr.name, nbytes,
		pPipeDev->maxBytes - MSG_OVERHEAD);
	errnoSet (S_ioLib_DEVICE_ERROR);
	return (ERROR);
	}


    /* wait for there to be room to put entire msg; no partial msgs */

    LOCK;

    while (bytesToWrite > rngFreeBytes (ringId))
	{
	if (intContext ())
	    {
	    ++pipeLostMsgs;
	    UNLOCK;
	    return (ERROR);
	    }

	UNLOCK;
	taskDelay (1);
	LOCK;
	}


    /* put message length followed by message */

    rngPutBuf (ringId, (char *) &shortNbytes, sizeof (short));
    rngPutBuf (ringId, buffer, nbytes);

    pPipeDev->nbytes += nbytes;		/* increment number of bytes in pipe */


    /* if more room in pipe, enable next writer;
     * if pipe was empty, enable next reader */

    if (pPipeDev->rdSemNeeded)
	{
	pPipeDev->rdSemNeeded = FALSE;
	semGive (pPipeDev->readSemId);
	}

    UNLOCK;

    return (nbytes);
    }
/*******************************************************************************
*
* pipeIoctl - do device specific control function
*
* The only ioctl request recognized is FIONREAD.
*
* RETURNS:
*  OK and `arptr' gets number of bytes in pipe, or
*  ERROR if request is not FIONREAD
*/

LOCAL STATUS pipeIoctl (pPipeDev, request, argptr)
    PIPE_DEV *pPipeDev;		/* pointer to pipe descriptor */
    int request;		/* ioctl code */
    int	*argptr;		/* where to send answer */

    {
    FAST STATUS status = OK;

    switch (request)
	{
	case FIONREAD:
	    *argptr = pPipeDev->nbytes;
	    break;

	default:
	    status = ERROR;
	    errnoSet (S_ioLib_UNKNOWN_REQUEST);
	}

    return (status);
    }
