/* bootConfig.c - system configuration module for boot roms */

static char *copyright = "Copyright 1984-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This is the WRS-supplied configuration module for the VxWorks boot rom.
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "a_out.h"
#include "ctype.h"
#include "ftpLib.h"
#include "ioLib.h"
#include "ldLib.h"
#include "strLib.h"
#include "sysLib.h"
#include "taskLib.h"
/*
#include "socket.h"
*/
#include "sysSymTbl.h"
#include "version.h"

#include "config.h"
#include "iv68k.h"

IMPORT VOID sysVwTrapRtn ();	/* UniWorks trap handler entry */
IMPORT VOID sysMemParityError();/* memory parity err handling routine */
IMPORT char *bootCrackParams();	/* interpret boot line */
IMPORT char *sysModel ();	/* model name of system CPU */
IMPORT ULONG tickGet ();	/* get current time in ticks */

IMPORT char *sysExcMsg;		/* catastrophic exception message for display */
IMPORT char edata;		/* automatically defined by the loader */
IMPORT char end;		/* automatically defined by the loader */
IMPORT int  remLastResvPort;	/* XXX */


#define TIMEOUT		7	/* number of seconds before auto-boot */
#define MAX_LINE        160	/* max line length for input to 'm' routine */
/* **
#define MAX_FIELD_LEN	44	/* max boot line field length */

#define RSHD		514	/* rshd service */

SYMTAB_ID sysSymTbl;

LOCAL int sysStartType;		/* i.e. ROM_INIT or ROM_RE_INIT */

LOCAL int consoleFd;
LOCAL char *netDevList [] =
    {
#ifdef INCLUDE_EX
    "ex",
#endif
#ifdef INCLUDE_ENP
    "enp",
#endif
#ifdef INCLUDE_NW
    "nw",
#endif
#ifdef INCLUDE_PN
    "pn",
#endif
#ifdef INCLUDE_BP
    "bp",
#endif
#ifdef INCLUDE_LN
    "ln",
#endif
    };


/* forward declarations */

VOID usrClock ();
VOID usrRoot ();
LOCAL VOID bootCmdLoop ();
LOCAL VOID bootExcHandler ();


/*******************************************************************************
*
* usrInit - user defined system initialization routine
*
* THIS IS THE FIRST "C" CODE EXECUTED AFTER BOOT UP. This routine is
* called by the assembly language start-up code in vxInit.  It is
* called before kernel multi-tasking is enabled, in system mode, with
* the interrupts locked out.
*
* It starts by clearing BSS, so all variables are initialized to 0 as
* per the C specification, then copies the DATA segment from ROM into
* RAM, if this is a ROM-based system.
* Then it initializes the hardware by calling sysHwInit, sets up some
* vectors, spawns usrRoot to do the bulk of the initialization, and
* starts the kernel.  The root task, usrRoot, will start up AFTER
* the kernelGo call.
*
* NOMANUAL
*/

VOID usrInit (startType)
    int startType;		/* BOOT_COLD, BOOT_WARM, BOOT_WARM_NO_AUTO */

    {

#if (CPU==MC68020)
    sysCacheEnable (TRUE);		/* enable cache memory */
#endif

    bzero (&edata, &end - &edata);	/* clear bss */

    sysStartType = startType;

    /* set the vector base register (except on 68000);
     * initialize exception/interrupt vectors */

#if ((CPU==MC68010) || (CPU==MC68020))
    intSetVecBase ((FUNCPTR *) VEC_BASE_ADRS);
#endif

    excVecInit ();
    intSetVec (TRAPNUM_TO_IVEC (TRAP_VXWORKS), sysVwTrapRtn);  /* vxWks trap */


    /* do system dependent hardware initialization */

    sysHwInit ();

    /* start the kernel specifying usrRoot as the root task */

    kernelInit (TRAP_KERNEL, usrRoot, (ULONG) ROOT_STACK_SIZE,
		(ULONG) FREE_RAM_ADRS, (ULONG) sysMemTop (),
		(ULONG) ISR_STACK_SIZE, (ULONG) INT_LOCK_LEVEL);
    }
/*******************************************************************************
*
* usrRoot - Root task
*
* The root task must perform any initialization that cannot be
* done before a kernelGo.
*
* It initializes the I/O system, install drivers, create devices,
* sets up the network, etc., as necessary for the particular configuration.
* It may also create the system symbol table if one is to be included.
* It should then load and/or spawn additional tasks as needed.
* In a debug configuration, this is usually just a taskSpawn of the shell.
*
* NOMANUAL
*/

VOID usrRoot ()

    {
    /* set up parity error interrupt routine, if interrupt vector is defined */

#ifdef INT_VEC_PARITY_ERROR
    intConnect (INUM_TO_IVEC (INT_VEC_PARITY_ERROR), sysMemParityError, 0);
#endif

    /* set up system timer */

    wdLibInit ();			/* init watchdog lib */
    sysClkConnect (usrClock, 0);	/* connect clock interrupt routine */
    sysClkSetRate (60);			/* set sys clock rate */
    sysClkEnable ();			/* start it */

    tickSet ((ULONG) 0);		/* set initial time to zero */


    /* initialize i/o and file system */

    iosInit (NUM_DRIVERS, NUM_FILES, "/null");	/* init i/o system */

    /* install driver for on-board ports and make devices */

    tyCoDrv ();

    /* create console device */
#ifdef TARGET_UBAR_68K2
    tyCoDevCreate (CONSOLE_DEVICE, 1, 512, 512);
#else
    tyCoDevCreate (CONSOLE_DEVICE, 0, 512, 512);
#endif

    /* make specified console device be standard in/out */

    consoleFd = open (CONSOLE_DEVICE, UPDATE);	/* open console device */
    ioctl (consoleFd, FIOBAUDRATE, 9600);	/* set to 9600 baud */
    ioctl (consoleFd, FIOOPTIONS,
	   OPT_ECHO | OPT_CRMOD | OPT_TANDEM | OPT_7_BIT); /* set raw mode */
    ioGlobalStdSet (STD_IN, consoleFd);		/* set as std input */
    ioGlobalStdSet (STD_OUT, consoleFd);	/* set as std output */
    ioGlobalStdSet (STD_ERR, consoleFd);	/* set as std error */


    taskSpawn ("idle", 254, VX_SUPERVISOR_MODE | VX_UNBREAKABLE,
	       1000, idle);

    /* install pipe driver and 
     * initialize exception reporting, debugging, and logging */

    pipeDrv ();				/* init pipe driver */
    excInit ();				/* init exception reporting/handling */
    excHookAdd (bootExcHandler);	/* set bootrom exception handler */
    logInit (consoleFd, 5);		/* init logging */
    sigInit ();				/* initialize signals */


    taskSpawn ("boot", 1, VX_SUPERVISOR_MODE, 5000, bootCmdLoop);
    }
/*******************************************************************************
*
* usrClock - user defined system clock interrupt routine
*
* This routine is called at interrupt level on each clock interrupt.
* It is installed in sysInit by a sysClkConnect call.
* It calls all the other packages that need to know about clock ticks,
* including the kernel itself.
*
* If the application needs anything to happen at clock interrupt level,
* it should be added to this routine.
*
* NOMANUAL
*/

VOID usrClock ()

    {
    wdTick ();		/* check watchdog timers */
    tickAnnounce ();	/* announce system tick to kernel */
    }
/*******************************************************************************
*
* bootCmdLoop - read and execute user commands forever (until boot)
*/

LOCAL VOID bootCmdLoop ()

    {
    char line [MAX_LINE];
    char *pLine;
    char *tmp;
    int nwords;
    int nbytes;
    int value;
    int adr;
    int bytesRead = 0;
    int autoBootTime;
    int timeLeft;
    int timeMarker;
    FUNCPTR entry;


    /* flush std in to get rid of any garbage;
     * (i.e. Heurikon HKV2F gets junk in usart if no terminal connected). */

    ioctl (STD_IN, FIOFLUSH);


    if (sysStartType == BOOT_COLD)
	{
	/* print out id message */

	printf ("\n\n\n\n\n\n\n\n\n\n\n\n");
	printf ("                         VxWorks System Boot \n\n\n");
	printf ("                 copyright Wind River Systems, Inc., 1988\n");
	printf ("\n\n\n\n\n");
	printf ("CPU: %s\n", sysModel ());
	printf ("Version: %s\n", UniWorksVersion);
	printf ("Creation date: %s\n\n", creationDate);

	/* this is a cold boot so get the default boot line */


#ifdef	NV_RAM_SIZE
	if (sysNvRamGet (BOOT_LINE_ADRS, NV_RAM_SIZE, 0) == ERROR)
	    {
	    strcpy (BOOT_LINE_ADRS, DEFAULT_BOOT_LINE);
	    *(BOOT_LINE_ADRS + strlen (BOOT_LINE_ADRS) + 1) = EOS; /* 2nd EOS */
	    }
#else	NV_RAM_SIZE
	strcpy (BOOT_LINE_ADRS, DEFAULT_BOOT_LINE);
	*(BOOT_LINE_ADRS + strlen (BOOT_LINE_ADRS) + 1) = EOS; /* 2nd EOS */
#endif	NV_RAM_SIZE
	}

    if (sysStartType != BOOT_WARM_NO_AUTOBOOT)
	{
	printf ("\nPress any key to stop auto-boot...\n");

	/* Loop looking for a char, or timeout after TIMEOUT seconds */

	autoBootTime = (int) tickGet () + sysClkGetRate () * TIMEOUT;
	timeMarker = (int) tickGet ();
	timeLeft = TIMEOUT;
	printf ("%2d\r", timeLeft);
	while (((int) tickGet () < autoBootTime) && (bytesRead == 0))
	    {
	    ioctl (consoleFd, FIONREAD, &bytesRead);
	    if ((int) tickGet () == timeMarker + sysClkGetRate ())
		{
		timeMarker = (int) tickGet ();
		printf ("%2d\r", --timeLeft);
		}
	    }
	/* put the console back in line mode so it echoes (so's you can bang
	   on it to see if it's still alive) */

	ioctl (consoleFd, FIOOPTIONS, OPT_TERMINAL);

	if (bytesRead == 0)    /* nothing typed so auto-boot */
	    {
	    printf ("\nauto-booting...\n\n");
	    if (bootLoad (BOOT_LINE_ADRS, &entry) == OK)
		go (entry);		/* ... and never return */
	    else
		{
		printf ("Can't load boot file!!\n");
		taskDelay (sysClkGetRate ());	/* pause a second */
		reboot (BOOT_WARM_NO_AUTOBOOT);	/* something is awry */
		}
	    }
	}


    /* If we're here, either we aren't auto-booting, or we got an error
       autobooting, or someone stopped the auto-booting. */

    /* read and execute the ROM command */

    ioctl (consoleFd, FIOOPTIONS, OPT_TERMINAL);	/* put console 
							 * in line mode */
    printf ("\n");

    FOREVER
	{
        printErr ("[VxWorks Boot]: ");

        fioRdString (STD_IN, line, sizeof (line));

	nwords = 0;
	adr = 0;

	pLine = line;
	skipSpace (&pLine);

	if (*pLine == EOS)	/* skip blank line */
	    continue;

	if (*pLine == '$') 	/* explicit boot line?*/
	    {
	    if (bootLoad (&pLine [1], &entry) == OK)
		go (entry);
	    else
		taskDelay (sysClkGetRate ());	/* pause a second */
		reboot (BOOT_WARM_NO_AUTOBOOT);	/* something is awry */
	    }

	tmp = pLine + 1;
	switch (*pLine)
	    {
	    case 'd':		/* display */
		skipSpace (&tmp);
		skipHexSpecifier (&tmp);
		sscanf (&tmp [0], "%x,%d", &adr, &nwords);
		d ((char *) adr, nwords);
		break;

	    case 'e':		/* exception */
		printExcMsg (sysExcMsg);
		break;

	    case 'f':		/* fill */
		skipSpace (&tmp);
		skipHexSpecifier (&tmp);
		sscanf (&tmp [0], "%x,%x,%x", &adr, &nbytes, &value);
		bfill ((char *) adr, nbytes, value);
		break;

	    case 'm':		/* modify */
		skipSpace (&tmp);
		skipHexSpecifier (&tmp);
		sscanf (&tmp [0], "%x", &adr);
		m ((char *) adr);
		break;

#ifdef TARGET_HK_V2F
	    case 's':		/* system controller */
		skipSpace (&tmp);
		skipHexSpecifier (&tmp);
		sscanf (&tmp [0], "%d", &value);
		if (value != 0)
		    {
		    sysBCLSet ((ULONG) HK_BCL_SYS_CONTROLLER,
			       (ULONG) HK_BCL_SYS_CONTROLLER);
		    printf ("System controller on.\n");
		    }
		else
		    {
		    sysBCLSet ((ULONG) HK_BCL_SYS_CONTROLLER,
			       (ULONG) ~HK_BCL_SYS_CONTROLLER);
		    printf ("System controller off.\n");
		    }
		break;
#endif TARGET_HK_V2F

	    case 'p':		/* print boot params */
		bootPrintParams (BOOT_LINE_ADRS);
		break;

	    case 'c':		/* change boot params */
		bootPromptForParams (BOOT_LINE_ADRS);
		printf ("\n");
#ifdef	NV_RAM_SIZE
		sysNvRamSet (BOOT_LINE_ADRS, strlen (BOOT_LINE_ADRS) + 1, 0);
#endif	NV_RAM_SIZE
		break;

	    case 'l':		/* load */
		if (bootLoad (BOOT_LINE_ADRS, &entry) == OK)
		    printf ("entry = 0x%x\n", entry);
		else
		    {
		    taskDelay (sysClkGetRate ());	/* pause a second */
		    reboot (BOOT_WARM_NO_AUTOBOOT);	/* something is awry */
		    }
		break;
		    
	    case 'g':		/* go */
		skipSpace (&tmp);
		skipHexSpecifier (&tmp);
		if (sscanf (&tmp [0], "%x", &entry) == 0)
		    printf ("invalid entry point\n");
		else
		    go (entry);
		break;
			
            case '?':			/* help */
            case 'h':			/* help */
		bootHelp ();
		break;

            case '@':			/* autoboot */
		if (bootLoad (BOOT_LINE_ADRS, &entry) == OK)
		    go (entry);
		else
		    {
		    taskDelay (sysClkGetRate ());	/* pause a second */
		    reboot (BOOT_WARM_NO_AUTOBOOT);	/* something is awry */
		    }
		break;

	    default:
		printf ("Unrecognized command. Type '?' for help.\n");
		break;

            } /* switch */
        } /* FOREVER */
    }
/*******************************************************************************
*
* bootHelp - print brief help list
*/

LOCAL VOID bootHelp ()
    {
    static char *helpMsg[] =
	{
	" ?                      - print this list",
	" @                      - boot (load and go)",
	" p                      - print boot params",
	" c                      - change boot params",
	"                           <.> period clears field",
	" l                      - load boot file",
	" g adrs                 - go to adrs",
	" d adrs[,n]             - display memory",
	" m adrs                 - modify memory",
	" f adrs, nbytes, value  - fill memory",
	" e                      - print catastrophic VxWorks exception",
#ifdef TARGET_HK_V2F
	" s [0/1]                - system controller 0 = off, 1 = on",
#endif TARGET_HK_V2F
	" $<dd>(n,m)host:/file h=# e=# b=# g=# u=usr pw=passwd",
	"",
	};

    FAST int i;

    printf ("\n\n");

    for (i = 0; i < NELEMENTS (helpMsg); ++i)
	printf ("%s\n", helpMsg [i]);

    printf ("available boot devices <dd>:");
    for (i = 0; i < NELEMENTS (netDevList); ++i)
	printf (" %s", netDevList[i]);

    printf ("\n");
    }
/*******************************************************************************
*
* bootLoad - load a module into memory
*/

LOCAL STATUS bootLoad (paramString, pEntry)
    char *paramString;
    FUNCPTR *pEntry;

    {
    char bootDev [BOOT_FIELD_LEN];	/* boot device code */
    char hostName [BOOT_FIELD_LEN];	/* name of host */
    char bootFile [BOOT_FIELD_LEN];	/* name of file to boot */
    char ead [BOOT_FIELD_LEN];		/* ethernet internet addr */
    char bad [BOOT_FIELD_LEN];		/* backplane internet addr */
    char had [BOOT_FIELD_LEN];		/* host internet addr */
    char gad [BOOT_FIELD_LEN];		/* gateway internet addr */
    char usr [BOOT_FIELD_LEN];		/* user id */
    char passwd [BOOT_FIELD_LEN];	/* password */
    int procNum;			/* where to return processor number */
    char *pS;
    char ifname [20];
    char *inetAdrs;
    int netmask;
    FAST int ix;
    STATUS status;
    char nad [BOOT_FIELD_LEN];		/* host's network internet addr */

    /* interpret boot command */

    pS = bootCrackParams (paramString, bootDev, hostName, bootFile, ead, bad,
			     had, gad, usr, passwd, &procNum, &sysFlags);
    if (*pS != EOS)
	{
	/* print error msg with '^' where parse failed */

	printf ("Error in boot command:\n%s\n%*c\n", paramString,
		pS - paramString + 1, '^');
	return (ERROR);
	}

    bootPrintParams (paramString);


    /* set our processor number: may establish vme access, etc. */

    sysSetProcNum (procNum);


    /* put boot command at address expected by usrConfig */

    strcpy (BOOT_LINE_ADRS, paramString);
    *(BOOT_LINE_ADRS + strlen (paramString) + 1) = EOS;	/* 2nd EOS for ISI */


    hostTblInit ();		/* initialize remote command libary */

    /* start the network */
    
    remLastResvPort = 1010;	/* XXX kludge to shorten opening delay */
    netStart ();


    /* attach boot device */

    inetAdrs = ead;	/* assume we will use enet inet adrs */

    if (FALSE)		/* so INCLUDEs will work with 'else if's */
	;
#ifdef INCLUDE_EX
    else if (strcmp (bootDev, "ex") == 0)
	status = exattach (0);
#endif
#ifdef INCLUDE_ENP
    else if (strcmp (bootDev, "enp") == 0)
	status = enpattach (0, IO_ADRS_ENP, INT_VEC_ENP, INT_LVL_ENP);
#endif
#ifdef INCLUDE_IE
    else if (strcmp (bootDev, "ie") == 0)
	status = ieattach (0, IO_ADRS_IE, INT_VEC_IE, INT_LVL_IE);
#endif
#ifdef INCLUDE_NW
    else if (strcmp (bootDev, "nw") == 0)
	status = nwattach (0);
#endif
#ifdef INCLUDE_LN
	else if (strcmp (bootDev, "ln") == 0)
	    status = lnattach (0, IO_ADRS_LN, INT_VEC_LN, INT_LVL_LN);
#endif
#ifdef INCLUDE_PN
    else if (strcmp (bootDev, "pn") == 0)
	status = pnattach (0);
#endif
#ifdef INCLUDE_BP
    else if ((strcmp (bootDev, "bp") == 0) ||
	     (strncmp (bootDev, "bp=", 3) == 0))
	{
	char *bpAnchor = BP_ANCHOR_ADRS;

	sscanf (bootDev, "bp=%x", &bpAnchor);	/* pick off any anchor adrs */
	bootDev[2] = EOS;			/* get rid of any "=..." */
	status = bpattach (0, bpAnchor, sysProcNum, BP_INT_NONE, 0, 0, 0);
	inetAdrs = bad;		/* use bp inet adrs instead */
	}
#endif

    else
	{
	printf ("Boot device \"%s\" unknown; known devices:", bootDev);
	for (ix = 0; ix < NELEMENTS (netDevList); ++ix)
	    printf (" %s", netDevList[ix]);
	printf ("\n");

	return (ERROR);
	}

    if (status != OK)
	{
	printf ("Error attaching \"%s\" device: status = 0x%x.\n", bootDev,
		errnoGet ());
	return (ERROR);
	}


    /* configure the boot device with the specified inet address and net mask */

    if (inetAdrs[0] == EOS)
	{
	printf ("Inet address not specified for boot device \"%s\".\n",bootDev);
	return (ERROR);
	}

    strcpy (ifname, bootDev);
    strcat (ifname, "0");

    netmask = bootNetmaskExtract (inetAdrs);

    if (netmask != 0)
	ifMaskSet (ifname, netmask);

    ifconfig (ifname, "inet", inetAdrs, 0, 0, 0, 0);


    /* if a gateway was specified, extract the network part of the host's
     * address and add a route to this network */

    if (gad[0] != EOS)
        {
	inet_netof_string (had, nad);
	routeAdd (nad, gad);
        }

    /* associate hostName with the specified host address */

    remAddHost (hostName, had);


    /* load specified file */

    if (netLoad (hostName, bootFile, usr, passwd, pEntry) != OK)
	{
	printf ("Error loading file: status = 0x%x.\n", errnoGet ());
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* netLoad - downLoad a file from a remote machine via the network.
*
* The remote shell daemon on the machine 'host' is used to download
* the given file to the specified previously opened network file descriptor.
* The remote userId should have been set previously by a call to remIam.
* If the file does not exist, the error message from the Unix 'host' 
* is printed to the VxWorks standard error fd and ERROR is returned.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS netLoad (hostName, fileName, usr, passwd, pEntry)
    char *hostName;
    char *fileName;
    char *usr;
    char *passwd;
    FUNCPTR *pEntry;

    {
    int fd;
    int errFd;		/* for receiving standard error messages from Unix */
    char command[100];
    struct exec header;
    FAST int nBytes;
    BOOL bootFtp = (passwd[0] != EOS);

    if (bootFtp)
	{
	if (ftpXfer (hostName, usr, passwd, "", "RETR %s", "", fileName,
		     &errFd, &fd) == ERROR)
	    return (ERROR);
	}
    else
	{
	sprintf (command, "cat %s", fileName);

	fd = rcmd (hostName, RSHD, usr, usr, command, &errFd);
	if (fd == ERROR)
	    return (ERROR);
	}

    nBytes = fioRead (fd, (char *) &header, sizeof (struct exec));
    if (nBytes < sizeof (struct exec))
	goto readErr;

    printf ("%d", header.a_text);
    nBytes = fioRead (fd, (char *) header.a_entry, (int) header.a_text);
    if (nBytes < header.a_text)
	goto readErr;

    printf (" + %d", header.a_data);
    nBytes = fioRead (fd, (char *) (header.a_entry + header.a_text),
		      (int) header.a_data);
    if (nBytes < header.a_data)
	goto readErr;

    printf (" + %d\n", header.a_bss);

    close (fd);
    close (errFd);

    *pEntry = (FUNCPTR) header.a_entry;
    return (OK);

readErr:
    /* check standard error on Unix */
    if (bootFtp)
	(void) ftpGetReply (errFd, FALSE); /* error message on std. err */
    else
	{
	char buf [100];
	int errBytesRecv = fioRead (errFd, buf, sizeof (buf));

	if (errBytesRecv > 0)
	    {
	    /* print error message on standard error fd */

	    buf [errBytesRecv] = EOS;
	    printErr ("%s:%s: %s", hostName, fileName, buf);
	    }
	}

    close (fd);
    close (errFd);
    return (ERROR);
    }
/*******************************************************************************
*
* go - start at specified address
*/

LOCAL VOID go (entry)
    FUNCPTR entry;

    {
    printErr ("starting from 0x%x...\n", entry);

    taskDelay (sysClkGetRate () / 4); /* give the network a moment to close */

    ifreset ();		/* reset any network board so it doesn't interrupt */

    (entry) ();		/* go to entry point - never to return */
    }

/*******************************************************************************
*
* m - modify memory
*
* This routine prompts the user for modifications to memory, starting at the
* specified address.  It prints each address, and the current contents of
* that address, in turn.  The user can respond in one of several ways:
* .CS
*	RETURN   - No change to that address, but continue
*		   prompting at next address.
*	<number> - Set the contents to <number>.
*	. (dot)	 - No change to that address, and quit.
*	<EOF>	 - No change to that address, and quit.
* .CE
* All numbers entered and displayed are in hexadecimal.
* Memory is treated as 16-bit words.
*
* SEE ALSO: mRegs(2)
*/

LOCAL VOID m (adrs)
    char *adrs;		/* address to change */

    {
    char line[MAX_LINE + 1];	/* leave room for EOS */
    char *pLine;		/* ptr to current position in line */
    int value;			/* value found in line */
    char excess;

    /* round down to word boundary */

    for (adrs = (char *) ((int) adrs & 0xfffffffe);	/* start on even addr */
         ;						/* FOREVER */
	 adrs = (char *) (((short *) adrs) + 1))	/* bump as short ptr */
	{
	/* prompt for substitution */

	printf ("%06x:  %04x-", adrs, (*(short *)adrs) & 0x0000ffff);

	/* get substitution value:
	 *   skip empty lines (CR only);
	 *   quit on end of file or invalid input;
	 *   otherwise put specified value at address */

	if (fioRdString (STD_IN, line, MAX_LINE) == EOF)
	    break;

	line[MAX_LINE] = EOS;	/* make sure input line has EOS */

	for (pLine = line; isspace (*pLine); ++pLine)	/* skip leading spaces*/
	    ;

	if (*pLine == EOS)			/* skip field if just CR */
	    continue;

	if (sscanf (pLine, "%x%1s", &value, &excess) != 1)
	    break;				/* quit if not number */

	* (short *) adrs = value;		/* assign new value */
	}

    printf ("\n");
    }
/*******************************************************************************
*
* d - display memory
*
* Display contents of memory, starting at adrs.  Memory is displayed in
* words.  The number of words displayed defaults to 64.  If
* nwords is non-zero, that number of words is printed, rounded up to
* the nearest number of full lines.  That number then becomes the default.
*/

LOCAL VOID d (adrs, nwords)
    FAST char *adrs;	/* address to display */
    int nwords;		/* number of words to print. */
			/* If 0, print 64 or last specified. */

    {
    static char *last_adrs;
    static int dNbytes = 128;

    FAST int nbytes;
    FAST int byte;
    char ascii [17];

    ascii [16] = EOS;			/* put an EOS on the string */

    nbytes = 2 * nwords;

    if (nbytes == 0)
	nbytes = dNbytes;	/* no count specified: use current byte count */
    else
	dNbytes = nbytes;	/* change current byte count */

    if (adrs == 0)
	adrs = last_adrs;	/* no address specified: use last address */

    adrs = (char *) ((int) adrs & ~1);	/* round adrs down to word boundary */


    /* print leading spaces on first line */

    bfill ((char *) ascii, 16, '.');

    printf ("%06x:  ", (int) adrs & ~0xf);

    for (byte = 0; byte < ((int) adrs & 0xf); byte++)
	{
	printf ("  ");
	if (byte & 1)
	    printf (" ");	/* space between words */
	if (byte == 7)
	    printf (" ");	/* extra space between words 3 and 4 */

	ascii[byte] = ' ';
	}


    /* print out all the words */

    while (nbytes-- > 0)
	{
	if (byte == 16)
	    {
	    /* end of line:
	     *   print out ascii format values and address of next line */

	    printf ("  *%16s*\n%06x:  ", ascii, (int) adrs);

	    bfill ((char *) ascii, 16, '.');	/* clear out ascii buffer */
	    byte = 0;				/* reset word count */
	    }

	printf ("%02x", *adrs & 0x000000ff);
	if (byte & 1)
	    printf (" ");	/* space between words */
	if (byte == 7)
	    printf (" ");	/* extra space between words 3 and 4 */

	if (*adrs == ' ' || (isascii (*adrs) && isprint (*adrs)))
	    ascii[byte] = *adrs;

	adrs++;
	byte++;
	}


    /* print remainder of last line */

    for (; byte < 16; byte++)
	{
	printf ("  ");
	if (byte & 1)
	    printf (" ");	/* space between words */
	if (byte == 7)
	    printf (" ");	/* extra space between words 3 and 4 */

	ascii[byte] = ' ';
	}

    printf ("  *%16s*\n", ascii);	/* print out ascii format values */

    last_adrs = adrs;
    }
/*******************************************************************************
*
* bootExcHandler - bootrom exception handling routine
*/

LOCAL VOID bootExcHandler (tid)
    int tid;		/* task id */

    {
    int dregs[8];	/* task's data registers */
    int aregs[7];	/* task's address registers */
    char *sp;		/* task's stack pointer */
    USHORT sr;		/* task's status register */
    INSTR *pc;		/* task's pc */

    /* get tcb of task to be traced */

    if (taskRegsGet (tid, dregs, aregs, &sp, &sr, &pc) != ERROR)
	{
	trcStack ((int *) aregs[6], (int *) sp, pc, (FUNCPTR) NULL);
	taskRegsShow (tid);

	taskDelay (sysClkGetRate ());	/* pause a second */
	}

    reboot (BOOT_WARM_NO_AUTOBOOT);
    }
/*******************************************************************************
*
* skipHexSpecifier - advance pointer past 0x, 0X, $
*
* Increments the string pointer passed as a parameter to the next
* character in the string after a hex specifier.
*/

LOCAL VOID skipHexSpecifier (strptr)
    char **strptr;	/* pointer to pointer to string */

    {
    if ((**strptr == '0') && (((*strptr) [1] == 'X') || ((*strptr) [1] == 'x')))
	*strptr += 2;
    else
	if (**strptr == '$')
	    *strptr += 1;
    }
/*******************************************************************************
*
* skipSpace - advance pointer past white space
*
* Increments the string pointer passed as a parameter to the next
* non-white-space character in the string.
*/

LOCAL VOID skipSpace (strptr)
    char **strptr;	/* pointer to pointer to string */

    {
    while (isspace (**strptr))
	++*strptr;
    }
/*******************************************************************************
*
* printExcMsg - print exception message
*
* Avoid printing control characters in exception message area.
*/

LOCAL VOID printExcMsg (string)
    char *string;

    {
    printf ("\n");
    while (*string != EOS && isprint (*string))
	printf ("%c", *string++);
    printf ("\n");
    }
