/****************************************************************************
 File: cntrapp.c

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 You may use this Sample Code any way you please provided you 
 do not resell the code and that this notice (including the above 
 copyright notice) is reproduced on all copies.  THIS SAMPLE CODE 
 IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, AND GO CORPORATION 
 EXPRESSLY DISCLAIMS ALL IMPLIED WARRANTIES, INCLUDING BUT NOT 
 LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE. IN NO EVENT WILL GO CORPORATION BE LIABLE TO YOU 
 FOR ANY CONSEQUENTIAL,INCIDENTAL,OR INDIRECT DAMAGES ARISING OUT OF 
 THE USE OR INABILITY TO USE THIS SAMPLE CODE.

 $Revision:   1.6  $
   $Author:   kcatlin  $
     $Date:   18 Mar 1992 08:49:32  $

 This file contains the implementation of the counter application class.

****************************************************************************/

#ifndef APP_INCLUDED
#include <app.h>
#endif

#ifndef APPMGR_INCLUDED
#include <appmgr.h>
#endif

#ifndef OS_INCLUDED
#include <os.h>
#endif

#ifndef RESFILE_INCLUDED
#include <resfile.h>
#endif

#ifndef FRAME_INCLUDED
#include <frame.h>
#endif

#ifndef DEBUG_INCLUDED
#include <debug.h>
#endif

#ifndef TKTABLE_INCLUDED
#include <tktable.h>
#endif

#ifndef MENU_INCLUDED
#include <menu.h>
#endif

#ifndef CNTR_INCLUDED
#include <cntr.h>
#endif

#ifndef CNTRAPP_INCLUDED
#include <cntrapp.h>
#endif

#include <methods.h>

#include <string.h>
#include <stdio.h>

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                           Defines, Types, Globals, Etc 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef enum CNTRAPP_DISPLAY_FORMAT {

	dec, oct, hex

}  CNTRAPP_DISPLAY_FORMAT,
   *P_CNTRAPP_DISPLAY_FORMAT;

typedef struct CNTRAPP_INST {

	P_CNTRAPP_DISPLAY_FORMAT  pFormat;
	OBJECT                    fileHandle;
	OBJECT                    counter;

}  CNTRAPP_INST,
   *P_CNTRAPP_INST;

static const TK_TABLE_ENTRY CntrAppMenuBar[] = {
	{"Representation", 0, 0, 0, tkMenuPullDown, clsMenuButton},
		{"Dec", msgCntrAppChangeFormat, dec},
		{"Oct", msgCntrAppChangeFormat, oct},
		{"Hex", msgCntrAppChangeFormat, hex},
		{pNull},
	{pNull}
};


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                           Local Functions
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 BuildString

 Local function to build a label string
****************************************************************************/

STATUS LOCAL BuildString(
	P_STRING        p,
	P_CNTRAPP_INST pData)
{
	CNTR_INFO ci;
	STATUS    s;

	ObjCallRet(msgCntrGetValue, pData->counter, &ci ,s);

	switch (*(pData->pFormat)) {

		case dec:
			sprintf(p, " Counter value is %d ", ci.value);
			break;

		case oct:
			sprintf(p, " Counter value is %o ", ci.value);
			break;

		case hex:
			sprintf(p, " Counter value is %x ", ci.value);
			break;

		default:
			sprintf(p, " Unknown Representation Requested ");
			break;
	}

	return stsOK;

} /* BuildString */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                           Message Handlers 							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
	CntrAppInit

	Respond to msgInit.

****************************************************************************/

MsgHandler(CntrAppInit)
{
	CNTRAPP_INST inst;

	Dbg(Debugf("CntrApp:CntrAppInit");)

	inst.counter = pNull;
	inst.fileHandle = pNull;
	inst.pFormat = pNull;

	// Update instance data.
	ObjectWrite(self, ctx, &inst);

	return stsOK;
	MsgHandlerParametersNoWarning;

} /* CntrAppInit */


/****************************************************************************
	CntrAppSave

	Respond to msgSave.

	Save the counter object using ResPutObject. The counter will
	receive msgSave and save any data it needs.

	The application doesn't need to save any data, the format data is
	memory mapped.

****************************************************************************/

MsgHandlerWithTypes(CntrAppSave, P_OBJ_SAVE, P_CNTRAPP_INST)
{
	STATUS s;

	Dbg(Debugf("CntrApp:CntrAppSave");)

	// Save the counter object.
	ObjCallRet(msgResPutObject, pArgs->file, pData->counter, s);

	return stsOK;
	MsgHandlerParametersNoWarning;

} /* CntrAppSave */


/****************************************************************************
	CntrAppRestore

	Respond to msgRestore.

	Open the file holding the application data, memory map the file.

	Restore the counter object by sending msgResGetObject -- the counter
	will receive msgRestore.
****************************************************************************/

MsgHandlerWithTypes(CntrAppRestore, P_OBJ_RESTORE, P_CNTRAPP_INST)
{
	FS_NEW        fsn;
	CNTRAPP_INST inst;
	STATUS        s;

	Dbg(Debugf("CntrApp:CntrAppRestore");)

	// Get handle for format file, save the handle
	ObjCallRet(msgNewDefaults, clsFileHandle, &fsn, s);
	fsn.fs.locator.pPath = "formatfile";
	fsn.fs.locator.uid = theWorkingDir;
	ObjCallRet(msgNew, clsFileHandle, &fsn, s);

	inst.fileHandle = fsn.object.uid; 

	// Map the file to memory
	ObjCallRet(msgFSMemoryMap, fsn.object.uid, &inst.pFormat, s );

	// Restore the counter object.
	ObjCallJmp(msgResGetObject, pArgs->file, &inst.counter, s, Error);

	// Update instance data.
	ObjectWrite(self, ctx, &inst);

	return stsOK;
	MsgHandlerParametersNoWarning;

Error:

	return s;

} /* CntrAppRestore */


/****************************************************************************
	CntrAppFree

	Respond to msgFree.
****************************************************************************/

MsgHandlerWithTypes(CntrAppFree, P_ARGS, P_CNTRAPP_INST)
{
	STATUS s;

	Dbg(Debugf("CntrApp:CntrAppFree");)

	ObjCallRet(msgDestroy, pData->counter, Nil(P_ARGS), s);

	// Unmap the file
	ObjCallRet(msgFSMemoryMapFree, pData->fileHandle, Nil(P_ARGS), s);

	// Free the file handle
	ObjCallRet(msgDestroy, pData->fileHandle, Nil(P_ARGS), s );

	return stsOK;
	MsgHandlerParametersNoWarning;

} /* CntrAppFree */


/****************************************************************************
	CntrAppAppInit

	Respond to msgAppInit.
	Create the file to hold the memory mapped data.
****************************************************************************/

MsgHandler(CntrAppAppInit)
{
	CNTR_NEW                cn;
	FS_NEW                  fsn;
	STREAM_READ_WRITE       fsWrite;
	CNTRAPP_DISPLAY_FORMAT  format;
	CNTRAPP_INST           inst;
	STATUS                  s;

	Dbg(Debugf("CntrApp:CntrAppAppInit");)

	inst = IDataDeref(pData, CNTRAPP_INST);

	//
	// Create the counter object.
	//
	ObjCallRet(msgNewDefaults, clsCntr, &cn, s);
	cn.cntr.initialValue = 42;
	ObjCallRet(msgNew, clsCntr, &cn, s);

	inst.counter = cn.object.uid;

	//
	// Create a file, fill it with a default value
	//
	ObjCallRet(msgNewDefaults, clsFileHandle, &fsn, s);
	fsn.fs.locator.pPath = "formatfile";
	fsn.fs.locator.uid = theWorkingDir;
	ObjCallRet(msgNew, clsFileHandle, &fsn, s);

	format = dec;
	fsWrite.numBytes = SizeOf(CNTRAPP_DISPLAY_FORMAT);
	fsWrite.pBuf = &format;
	ObjCallRet(msgStreamWrite, fsn.object.uid, &fsWrite, s);

	inst.fileHandle = fsn.object.uid;

	//
	// Map the file to memory
	//
	ObjCallRet(msgFSMemoryMap, fsn.object.uid, &inst.pFormat, s);

	// Update instance data.
	ObjectWrite(self, ctx, &inst);

	return stsOK;
	MsgHandlerParametersNoWarning;

} /* CntrAppAppInit */


/****************************************************************************
	CntrAppOpen

	Respond to msgAppOpen.

	It's important that the ancestor be called AFTER all the frame
	manipulations in this routine because the ancestor takes care of any
	layout that is necessary.
****************************************************************************/

MsgHandlerWithTypes(CntrAppOpen, P_ARGS, P_CNTRAPP_INST)
{
	APP_METRICS am;
	MENU_NEW    mn;
	LABEL_NEW   ln;
	STATUS      s;
	char        buf[30];

	Dbg(Debugf("CntrApp:CntrAppOpen");)

	// Increment the counter.
	ObjCallRet(msgCntrIncr, pData->counter, Nil(P_ARGS), s);

	// Build the string for the label.
	StsRet(BuildString(buf, pData), s);

	// Create the label.
	ObjCallRet(msgNewDefaults, clsLabel, &ln, s);
	ln.label.pString = buf;
	ln.label.style.scaleUnits = bsUnitsFitWindowProper;
	ln.label.style.xAlignment = lsAlignCenter;
	ln.label.style.yAlignment = lsAlignCenter;
	ObjCallRet(msgNew, clsLabel, &ln, s); 

	// Get app metrics.
	ObjCallJmp(msgAppGetMetrics, self, &am, s, Error);

	// Set the label as the clientWin.
	ObjCallJmp(msgFrameSetClientWin, am.mainWin, ln.object.uid, s, Error);

	// Create and add menu bar.
	ObjCallJmp(msgNewDefaults, clsMenu, &mn, s, Error);
	mn.tkTable.client = self;
	mn.tkTable.pEntries = CntrAppMenuBar;
	ObjCallJmp(msgNew, clsMenu, &mn, s, Error);

	ObjCallJmp(msgAppCreateMenuBar, self, &mn.object.uid, s, Error);
	ObjCallJmp(msgFrameSetMenuBar, am.mainWin, mn.object.uid, s, Error);

	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	return s;

} /* CntrAppOpen */


/****************************************************************************
	CntrAppClose

	Respond to msgAppClose.
	Be sure that the ancestor is called FIRST.  The ancestor extracts the
	frame, and we want the frame extracted before performing surgery on it.

****************************************************************************/

MsgHandler(CntrAppClose)
{
	APP_METRICS am;
	STATUS      s;

	Dbg(Debugf("CntrApp:CntrAppClose");)

	// Free the menu bar.
	ObjCallJmp(msgAppGetMetrics, self, &am, s, Error);
	ObjCallJmp(msgFrameDestroyMenuBar, am.mainWin, pNull, s, Error);

	return stsOK;
	MsgHandlerParametersNoWarning;

Error:
	return s;
} /* CntrAppClose */


/****************************************************************************
	CntrAppChangeFormat

	Respond to msgCntrAppChangeFormat.
	Update the memory mapped data.
****************************************************************************/

MsgHandlerWithTypes(CntrAppChangeFormat, P_ARGS, P_CNTRAPP_INST)
{
	APP_METRICS  am;
	WIN          thelabel;
	STATUS       s;
	char         buf[30];

	Dbg(Debugf("CntrApp:CntrAppChangeFormat");)

	//
	// Update mmap data
	//
	*(pData->pFormat) = (CNTRAPP_DISPLAY_FORMAT)(U32)pArgs;

	// Build the string for the label.
	StsRet(BuildString(buf, pData), s);

	// Get app metrics.
	ObjCallRet(msgAppGetMetrics, self, &am, s);

	// Get the clientWin.
	ObjCallRet(msgFrameGetClientWin, am.mainWin, &thelabel, s);

	// Set the label string.
	ObjCallRet(msgLabelSetString, thelabel, buf, s);

	return stsOK;
	MsgHandlerParametersNoWarning;
} /* CntrAppChangeFormat */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                           Installation 								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
	ClsCntrAppInit

	Create the application class.
****************************************************************************/


STATUS GLOBAL
ClsCntrAppInit (void)
{
	APP_MGR_NEW new;
	STATUS      s;

	ObjCallJmp(msgNewDefaults, clsAppMgr, &new, s, Error);

	new.object.uid         = clsCntrApp;
	new.cls.pMsg         = clsCntrAppTable;
	new.cls.ancestor     = clsApp;
	new.cls.size         = SizeOf(CNTRAPP_INST);
	new.cls.newArgsSize  = SizeOf(APP_NEW);

	strcpy(new.appMgr.company, "GO Corporation");
	strcpy(new.appMgr.defaultDocName, "Counter Application");

	ObjCallJmp(msgNew, clsAppMgr, &new, s, Error);

	return stsOK;

Error:
	return s;
} /* ClsCntrAppInit */


/****************************************************************************
	main

	Main application entry point.
****************************************************************************/

void CDECL
main(
	int			argc,
	char *		argv[],
	U16			processCount)
{
	if (processCount == 0) {

		StsWarn(ClsCntrAppInit());
		AppMonitorMain(clsCntrApp, objNull);

	} else {

		StsWarn(ClsCntrInit());
		AppMain();
	}
	Unused(argc); Unused(argv); // Suppress compiler's "unused parameter" warnings
} /* main */



