/***************************************************************************
 File: clockapp.c

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.
			  
 You may use this Sample Code any way you please provided you 
 do not resell the code and that this notice (including the above 
 copyright notice) is reproduced on all copies.  THIS SAMPLE CODE 
 IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, AND GO CORPORATION 
 EXPRESSLY DISCLAIMS ALL IMPLIED WARRANTIES, INCLUDING BUT NOT 
 LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE. IN NO EVENT WILL GO CORPORATION BE LIABLE TO YOU 
 FOR ANY CONSEQUENTIAL,INCIDENTAL,OR INDIRECT DAMAGES ARISING OUT OF 
 THE USE OR INABILITY TO USE THIS SAMPLE CODE.

 $Revision:   1.45.1.0  $
   $Author:   pstahl  $
     $Date:   14 Apr 1992 12:08:20  $

 This file contains the class definition and methods for clsClockApp.

 The clock application is an extensive exploration of much of the API 
 diguised as a simple accessory: a digital alarm clock.  Option card 
 management, system preferences, timers, gesture processing, simple layout,
 customized notes, embedding, the power button, various types of controls 
 and fields, quick help and the busy clock are all exercised by this 
 application.

 todo:
 	Up and dn arrows should go beyond font size lists and fill in write-in.
	State of alarm off/on should be filed and alarm rescheduled on app restore.
	Instance 0 should manage alarms.
	Add midnight and noon to am/pm lists (everywhere)
	Local control to override prefs such as time zone, format, etc.
****************************************************************************/

#ifndef POWER_INCLUDED
#include <power.h>
#endif

#ifndef SYSGRAF_INCLUDED
#include <sysgraf.h>
#endif

#ifndef TIMER_INCLUDED
#include <timer.h>
#endif

#ifndef XGESTURE_INCLUDED
#include <xgesture.h>
#endif

#ifndef WIN_INCLUDED
#include <win.h>
#endif

#ifndef APPWIN_INCLUDED
#include <appwin.h>
#endif

#ifndef APP_INCLUDED
#include <app.h>
#endif

#ifndef APPMGR_INCLUDED
#include <appmgr.h>
#endif

#ifndef APPTAG_INCLUDED
#include <apptag.h>
#endif

#ifndef APPDIR_INCLUDED
#include <appdir.h>
#endif

#ifndef CMDBAR_INCLUDED
#include <cmdbar.h>
#endif

#ifndef FS_INCLUDED
#include <fs.h>
#endif

#ifndef TLAYOUT_INCLUDED
#include <tlayout.h>
#endif

#ifndef TK_INCLUDED
#include <tk.h>
#endif

#ifndef TKFIELD_INCLUDED
#include <tkfield.h>
#endif

#ifndef PREFS_INCLUDED
#include <prefs.h>
#endif

#ifndef CLOCKAPP_INCLUDED
#include <clockapp.h>
#endif

#ifndef OPTION_INCLUDED
#include <option.h>
#endif

#ifndef OPTTABLE_INCLUDED
#include <opttable.h>
#endif

#ifndef NOTE_INCLUDED
#include <note.h>
#endif

#ifndef BUSY_INCLUDED
#include <busy.h>
#endif

#ifndef FONTMGR_INCLUDED
#include <fontmgr.h>
#endif

#ifndef STDMSG_INCLUDED
#include <stdmsg.h>
#endif

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef UID_INCLUDED
#include <uid.h>
#endif

#ifndef OS_INCLUDED
#include <os.h>
#endif

#ifndef DEBUG_INCLUDED
#include <debug.h>
#endif

#ifndef _CTYPE_H_INCLUDED
#include <ctype.h>
#endif

#ifndef _STRING_H_INCLUDED
#include <string.h>
#endif

#ifndef _STDIO_H_INCLUDED
#include <stdio.h>
#endif

#ifndef _TIME_H_INCLUDED
#include <time.h>
#endif

#ifndef SEL_INCLUDED
#include <sel.h>
#endif

// private header files that need no guardians
#include <methods.h>
#include <ncbwin.h>
#include <clabel.h>
#include <cwin.h>

/**************************** Function Macros ***********************/

#define StsRetNoWarn(se,s) StsRet(se, s)

#define ZapWin(pWin) \
	(void)ObjectCall(msgDestroy, *pWin, Nil(P_ARGS)); \
	*pWin	= Nil(WIN)

//
// helper macro for ClockPrefsChanged
//
#define PrefChanged(name) ( \
	(tagPr##name == pArgs->prefId) \
)

/**************************** Constants ***********************/

#define SPACE ' '

#define corkNotInserted 0
#define corkInCard 1
#define corkInNote 2

#define stsDirtyControl stsOK
#define stsCleanControl MakeWarning(clsClockApp, 1)

#define noDate 0  // flags an empty date field value

// Must be greater than max returned by 
// PrefsDateToString and PrefsTimeToString.
#define MAX_DT_STR 60

// Limits to contents of alarm note
#define MAX_MESSAGE 60
#define MAX_MESS_LINE 13  // length of line on option card in em's

#define defaultSnooze 5   // minutes that "postpone" button delays by default 
#define defaultSnoozeStr " 5"

// keep these two in sync
#define alarmNoteTimeout  6000   // milliseconds before note auto-clears
#define alarmNoteTimeoutStr "Dismiss Note After 6 sec."

#define defaultFontSize 12

#define alarmIndicator 20
#define alarmIndicatorFont "GO55"

#define minFontSize 8
#define maxFontSize 255

// codes for various clock formats

#define timeUp 1
#define dateUp 2
#define bothUp	(timeUp|dateUp)
#define oneRow 4
#define oneCol 8
#define timeFirst 16
#define sideBySide 32
#define alarmOnTime 64

#define timeOnly 		(timeUp | oneRow | oneCol | alarmOnTime)
#define dateOnly		(dateUp | oneRow | oneCol)
#define dateAndTime	(bothUp | oneRow | sideBySide | alarmOnTime)
#define timeAndDate 	(bothUp | oneRow | sideBySide | timeFirst)
#define dateOverTime	(bothUp | oneCol | alarmOnTime)
#define timeOverDate	(bothUp | oneCol | timeFirst | alarmOnTime)

#define defaultFmt timeAndDate

#define fontPrune (P_UNKNOWN)(U32)(fimPruneDupFamilies | fimPruneSymbolFonts)

Enum16(OS_DAY_OF_WEEK) {
  osSunday			= 0,
  osMonday			= 1,
  osTuesday			= 2,
  osWednesday		= 3,
  osThursday		= 4,
  osFriday			= 5,
  osSaturday		= 6
};

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *									Option Card Constants 			                  *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define tagFormat 				MakeTag(clsClockApp, 1)
#define tagFont 				MakeTag(clsClockApp, 2)
#define tagTimeSize				MakeTag(clsClockApp, 3)
#define tagDateSize				MakeTag(clsClockApp, 4)
#define tagAlarmDay				MakeTag(clsClockApp, 5)
#define tagAlarmHour			MakeTag(clsClockApp, 6)
#define tagAlarmMinutes			MakeTag(clsClockApp, 7)
#define tagAlarmAmPm			MakeTag(clsClockApp, 8)
#define tagAlarmBeep			MakeTag(clsClockApp, 9)
#define tagAlarmTimeout			MakeTag(clsClockApp, 10)
#define tagAlarmOn				MakeTag(clsClockApp, 11)
#define tagAlarmMultiple 		MakeTag(clsClockApp, 12)
#define tagAlarmMessage			MakeTag(clsClockApp, 13)
#define tagAlarmSnooze			MakeTag(clsClockApp, 14)
#define tagAlarmOptions			MakeTag(clsClockApp, 15)
#define tagAlarmEveryDay 		MakeTag(clsClockApp, 16)
#define tagAlarmDateWriteIn 	MakeTag(clsClockApp, 17)
#define tagAlarmDateFld			MakeTag(clsClockApp, 18)
#define tagAlarmNoteButtons 	MakeTag(clsClockApp, 19)
#define tagAlarmSnoozeEnable	MakeTag(clsClockApp, 20)

#define tagDateSizeWriteIn		MakeTag(clsClockApp, 201)
#define tagDateSizeIntFld		MakeTag(clsClockApp, 202)
#define tagTimeSizeWriteIn		MakeTag(clsClockApp, 203)
#define tagTimeSizeIntFld		MakeTag(clsClockApp, 204)


#define maxFormat 5
#define nextFormat 1
// prevFormat is the number of formats less one, for a zero based array this
// is maxFormat.  We use this to go backward by going forward with wrap-around
#define prevFormat maxFormat


// Used by up and down arrow gestures, -1 and 256 are sentinels for search.
// Any size in this table must appear in both dateFontSize and timeFontSize.
#define maxRatchet 12
static const S16 fontRatchets[] = { -1, 8, 10, 12, 14, 16, 18, 20, 24, 36, 48, 60, 256 };

/***************************************************************************
 The choice list of clock formats
***************************************************************************/

static const TK_TABLE_ENTRY	clockFormatChoices[] = {
	{"Time Only", 0, 0, timeOnly},
	{"Date Only", 0, 0, dateOnly},
	{"Time Followed by Date", 0, 0, timeAndDate},
	{"Date Followed by Time", 0, 0, dateAndTime},
	{"Time Over Date", 0, 0, timeOverDate},
	{"Date Over Time", 0, 0, dateOverTime},
	{pNull}
};

/***************************************************************************
 The choice list of date font sizes. Identical to timeFontSize except for
 the tags on the write-in field.
***************************************************************************/

static const TK_TABLE_ENTRY dateFontSize[] = {
	{"8" ,	0, 0, 8},
	{"10",	0, 0, 10},
	{"12",	0, 0, 12},
	{"14",	0, 0, 14},
	{"16",	0, 0, 16},
	{"18",	0, 0, 18},
	{"20",	0, 0, 20},
	{"24",	0, 0, 24},
	{"36",	0, 0, 36},
	{"48",	0, 0, 48},
	{"60",	0, 0, 60},
	{0,		0, 0, tagDateSizeWriteIn, tkLabelEntry},
	{0,		3, 3, tagDateSizeIntFld, tkBorderMarginNone, clsIntegerField},
	{pNull}
};

/***************************************************************************
 The choice list of time font sizes.
***************************************************************************/

static const TK_TABLE_ENTRY timeFontSize[] = {
	{"8" ,	0, 0, 8},
	{"10",	0, 0, 10},
	{"12",	0, 0, 12},
	{"14",	0, 0, 14},
	{"16",	0, 0, 16},
	{"18",	0, 0, 18},
	{"20",	0, 0, 20},
	{"24",	0, 0, 24},
	{"36",	0, 0, 36},
	{"48",	0, 0, 48},
	{"60",	0, 0, 60},
	{0,		0, 0, tagTimeSizeWriteIn, tkLabelEntry},
	{0,		3, 3, tagTimeSizeIntFld, tkBorderMarginNone, clsIntegerField},
	{pNull}
};

/***************************************************************************
 The choice list for alarm period ("Day")
***************************************************************************/

static const TK_TABLE_ENTRY alarmDay[] = {
	{"Day",		0, 0, tagAlarmEveryDay},
	{"Monday",		0, 0, osMonday},
	{"Tuesday",		0, 0, osTuesday},
	{"Wednesday",	0, 0, osWednesday},
	{"Thursday",	0, 0, osThursday},
	{"Friday",		0, 0, osFriday},
	{"Saturday",	0, 0, osSaturday},
	{"Sunday",		0, 0, osSunday},
	{0,				0, 0, tagAlarmDateWriteIn, tkLabelEntry},
	{"  /  /  ",	8, 8, tagAlarmDateFld, tkBorderMarginNone, clsDateField},
	{pNull}
};


/***************************************************************************
 Alarm card components
***************************************************************************/

/***************************************************************************
 Alarm time AM/PM control choices.
***************************************************************************/

static const TK_TABLE_ENTRY alarmAmPmChoices[] = {
	{"AM", 0, 0, 0},
	{"PM", 0, 0, 1},
	{pNull}
};

/***************************************************************************
 Alarm time choices - HH:MM AM/PM
***************************************************************************/

static const TK_TABLE_ENTRY alarmTime[] = {
	{0, 2, 2, tagAlarmHour, 0, clsIntegerField, hlpClkAppAlarmTime},
	{":", 0, 0, 0, 0, clsLabel, hlpClkAppAlarmTime},
	{0, 2, 2, tagAlarmMinutes, 0, clsIntegerField, hlpClkAppAlarmTime},
	{alarmAmPmChoices, 0, 0, tagAlarmAmPm, 0, clsPopupChoice,
        hlpClkAppAlarmTime},
	{pNull}
};

/***************************************************************************
 This control sets whether happens "This" <day> or "Every" <day>.
***************************************************************************/

static const TK_TABLE_ENTRY alarmThisEveryChoices[] = {
	{"This", 0, 0, 0},
	{"Every", 0, 0, 1},
	{pNull}
};

/***************************************************************************
 This table combines the "this"/"every" and <day> controls into one line.
***************************************************************************/

static const TK_TABLE_ENTRY alarmDayLine[] = {

	{alarmThisEveryChoices, 0, 0, tagAlarmMultiple, 
		0, clsPopupChoice, hlpClkAppAlarmDay},

	{alarmDay, 0, 0, tagAlarmDay, 
		0, clsPopupChoice, hlpClkAppAlarmDay},

	{pNull}
};

/***************************************************************************
 Alarm enable choices.
***************************************************************************/

static const TK_TABLE_ENTRY alarmOffOnChoices[] = {
	{"Off", 0, 0, 0},
	{"On",  0, 0, 1},
	{pNull}
};

/***************************************************************************
 The clock display option card.
***************************************************************************/

static const TK_TABLE_ENTRY clockDisplayCardEntries[] = {

	{"Show:", 0, 0, 0, 0, 0, hlpClkAppDisplayShow},
	{clockFormatChoices, 1, 0, tagFormat, 0, clsPopupChoice, 
		hlpClkAppDisplayShow},

	{"Font:", 0, 0, 0, 0, 0, hlpClkAppDisplayFont},
	{fontPrune, 1, 0, tagFont, tkNoClient | tkPopupChoiceFont, clsPopupChoice,
		hlpClkAppDisplayFont},

	{"Date Size:", 0, 0, 0, 0, 0, hlpClkAppDisplayDSize},
	{dateFontSize, 4, 0, tagDateSize, tkTableVertical, clsPopupChoice, 
		hlpClkAppDisplayDSize},

	{"Time Size:", 0, 0, 0, 0, 0, hlpClkAppDisplayTSize},
	{timeFontSize, 4, 0, tagTimeSize, tkTableVertical, clsPopupChoice, 
		hlpClkAppDisplayTSize},

	{pNull}
};

/***************************************************************************
 The clock alarm option card toggle choices.
***************************************************************************/

static const TK_TABLE_ENTRY clockAlarmOptions[] = {
	{0, 0, 0, tagAlarmSnoozeEnable, tkLabelEntry, 0, hlpClkAppAlarmOptions},
	{0, 0, 0, 1, tkTableHorizontal | tkBorderMarginNone | 
        tkTableYAlignBaseline, clsTkTable, hlpClkAppAlarmOptions},
		{"Repeat in", 0, 0, 0, tkBorderMarginNone, clsLabel,
            hlpClkAppAlarmOptions},
		{defaultSnoozeStr, 2, 2, tagAlarmSnooze, tkBorderMarginNone, 
			clsIntegerField, hlpClkAppAlarmOptions},
		{"min.", 0, 0, 0, tkBorderMarginNone, clsLabel, hlpClkAppAlarmOptions},
		{pNull},
	{alarmNoteTimeoutStr, 0, 0, tagAlarmTimeout, 0, 0, hlpClkAppAlarmOptions},
	{"Tone", 0, 0, tagAlarmBeep, 0, 0, hlpClkAppAlarmOptions},
	{pNull}
};

/***************************************************************************
 The clockalarm option card.
***************************************************************************/

static const TK_TABLE_ENTRY clockAlarmCardEntries[] = {
	{"Alarm:", 0, 0, 0, 0, 0, hlpClkAppAlarmOn},
	{alarmOffOnChoices, 1, 0, tagAlarmOn, 0, clsPopupChoice,
        hlpClkAppAlarmOn},

	{"Day:", 0, 0, 0, 0, 0, hlpClkAppAlarmDay},
	{alarmDayLine, 0, 0, 0, tkTableHorizontal | tkTableYAlignBaseline,
        clsTkTable, hlpClkAppAlarmDay},

	{"Time:", 0, 0, 0, 0, 0, hlpClkAppAlarmTime},
	{alarmTime, 0, 0, 0, tkTableHorizontal | tkTableYAlignBaseline,
        clsTkTable, hlpClkAppAlarmTime},

	{"Message:", 0, 0, 0, 0, 0, hlpClkAppAlarmMessage},
	{0, MAX_MESS_LINE, 0, tagAlarmMessage, 0, clsField,
        hlpClkAppAlarmMessage},

	{"Options:", 0, 0, 0, 0, 0, hlpClkAppAlarmOptions},
	{clockAlarmOptions, 0, 0, tagAlarmOptions, tkTableYAlignBaseline,
        clsToggleTable, hlpClkAppAlarmOptions},

	{"Alarm Note\nCork Board:", 0, 0, 0, 
		tkLabelWordWrap, 0, hlpClkAppAlarmCork},

	{pNull}
};


/**************************** TYPES ***************************/

#define filedData \
	U8					fmt;							\
	U8					timeSize;						\
	U8					dateSize;						\
	U16					fontId;							\
	TIMER_ALARM_MODE	alarmType;						\
	U32					alarmDate;  					\
	OS_DAY_OF_WEEK		alarmDayOfWeek;					\
	U8					alarmHours;						\
	U8					alarmMinutes;					\
	BOOLEAN				alarmIsPM;						\
	BOOLEAN				alarmBeep;						\
	U8					alarmTimeout;					\
	BOOLEAN				alarmMultiple;					\
	CHAR				alarmMessage[MAX_MESSAGE];		\
	U8					alarmSnooze;					\
	BOOLEAN				alarmSnoozeEnable;

typedef struct CLOCK_APP_FILED_DATA {
	filedData
} CLOCK_APP_FILED_DATA, *P_CLOCK_APP_FILED_DATA;

typedef struct CLOCK_APP_DATA {

	filedData	// the filed portion of the instance data must come first

	// the windows
	OBJECT		appWin;
	OBJECT		clockWin;
	OBJECT		dateAllWin;
	OBJECT		dateWin;
	OBJECT		timeAllWin;
	OBJECT		timeWin;
	OBJECT		amPmWin;
	OBJECT		alarmWin;

	OBJECT		corkboard;
	U8			corkIn;			// set to corkInCard or corkInNote
	OBJECT		alarmCard;
	OBJECT		alarmNote;

	OS_HANDLE	tickHandle;		// once-per-second or on-next-minute
	BOOLEAN		oneShotTick;	// controls type of tick.
	OS_HANDLE	alarmHandle;

	// these two hold the time displayed in an alarm note.
	U8			noteHours;
	U8			noteMinutes;
	
	BOOLEAN		alarmOn;		 // true if the alarm is enabled

	OBJECT		self;

} CLOCK_APP_DATA, *P_CLOCK_APP_DATA;

typedef P_CLOCK_APP_DATA	*PP_CLOCK_APP_DATA;

typedef char DATE_TIME_STR[MAX_DT_STR];

typedef char AM_PM_STR[5];

typedef char ALARM_STR[3];


/***************************************************************************
                  MISC. HELPERS           
***************************************************************************/

/***************************************************************************
 LayoutUnitsToPixels is a utility that converts system layout units into
 screen pixels.
***************************************************************************/

S32 LayoutUnitsToPixels (
	S32	layoutUnits)
{
	BORDER_UNITS	bu;

	bu.win			= theRootWindow;
	bu.fromUnits	= bsUnitsLayout;
	bu.toUnits		= bsUnitsDevice;
	bu.size.w		= layoutUnits;
	bu.size.h		= 0;
	ObjCallWarn(msgBorderConvertUnits, clsBorder, &bu);

	return bu.size.w;
}

STATUS CloseOpenCorkboardDocs (
	P_CLOCK_APP_DATA	pInst)
{
	STATUS				s;
	WIN_ENUM			winEnum;
	WIN					winArray[10];
	U16					i;
	APP_WIN_METRICS		awm;
	APP_DIR_UPDATE_UID	adUID;
	FS_NEW				fsn;

	winEnum.max		=
	winEnum.count	= 10;
	winEnum.pWin	= winArray;
	winEnum.pFlags	= pNull;
	winEnum.next	= 0;
	winEnum.flags	= wsEnumChildren;

	while (true) {
		s = ObjCallWarn(msgWinEnum, pInst->corkboard, &winEnum);

		if (s == stsEndOfData || winEnum.count == 0)
			break;

		if (s < stsOK)
			return s;

		// For every child window...
		for (i = 0; i < winEnum.count; i++) {

			// If the window is an app icon window (ie: subclass of clsAppWin)
			// see if the document is currently open. If so, close it.
			if (ObjectCall(msgIsA, winArray[i], clsAppWin) == stsOK) {

				fsn.object.uid	= objNull;

				ObjCallJmp(msgAppWinGetMetrics, winArray[i], &awm, s,
                    loopError);

				// If the document is open, close it.
				if (awm.state != awClosed) {

					// Create an app dir handle to the document.
					ObjectCall(msgNewDefaults, clsAppDir, &fsn);
					fsn.fs.dirIndex		= awm.appUUID;
					fsn.fs.mode			= fsUseDirIndex;
					fsn.fs.exist		= fsExistOpen | fsNoExistGenError;
					ObjCallJmp(msgNew, clsAppDir, &fsn, s, loopError);

					// Get the app UID from the dir handle.
					adUID.pPath	= pNull;
					ObjCallJmp(msgAppDirGetUID, fsn.object.uid, &adUID, s,
                        loopError);

					// Close the document.
					ObjSendWarn(msgAppCloseTo, adUID.uid,
                                (P_ARGS)appCloseToNormal, 0);
				}
loopError:
				// If we create an app dir handle, destroy it.
				if (fsn.object.uid != objNull)
					ObjectCall(msgDestroy, fsn.object.uid, Nil(OBJ_KEY));

				continue;
			}
		}
	}

	return stsOK;
}

/***************************************************************************
 InitNonFiled initializes the portion of the application's instance data
 which is not filed or restored.
***************************************************************************/

void InitNonFiled(P_CLOCK_APP_DATA pInst) {

	pInst->clockWin		= Nil(WIN);
	pInst->dateAllWin	= Nil(WIN);
	pInst->dateWin		= Nil(WIN);
	pInst->timeAllWin	= Nil(WIN);
	pInst->timeWin		= Nil(WIN);
	pInst->amPmWin		= Nil(WIN);
	pInst->alarmWin		= Nil(WIN);

	pInst->tickHandle	= NULL;
	pInst->oneShotTick	= false;
	pInst->alarmHandle	= NULL;
	pInst->noteHours	= 0;
	pInst->noteMinutes	= 0;

	pInst->corkboard	= objNull;
	pInst->corkIn		= corkNotInserted;
	pInst->alarmCard	= objNull;
	pInst->alarmNote	= objNull;

	pInst->alarmOn		= false;

}  // InitNonFiled


/****************************************************************************
 DateTimeAdd is used by the 'snooze" function to postpone an alarm by some
 number of seconds.
***************************************************************************/

void DateTimeAdd(P_OS_DATE_TIME pDateTime, time_t seconds) {

	time_t timeAsSeconds;
	struct tm aTM;  // standard C time structure

	aTM.tm_sec 		= pDateTime->seconds;
	aTM.tm_min 		= pDateTime->minutes;
	aTM.tm_hour		= pDateTime->hours;
	aTM.tm_mday		= pDateTime->day;
	aTM.tm_mon		= pDateTime->month;
	aTM.tm_year		= (int) pDateTime->year;
	aTM.tm_isdst	= -1;				   

	timeAsSeconds = mktime(&aTM);

	timeAsSeconds += seconds;

	_localtime(&timeAsSeconds, &aTM);
					   				   	  
	pDateTime->seconds		= aTM.tm_sec; 	
	pDateTime->minutes		= aTM.tm_min; 	
	pDateTime->hours		= aTM.tm_hour;	
	pDateTime->day			= aTM.tm_mday;	
	pDateTime->month		= aTM.tm_mon;	
	pDateTime->year			= aTM.tm_year;	
	pDateTime->dayOfWeek	= aTM.tm_wday;	   
							
}  // DateTimeAdd


/***************************************************************************
 ChildCount returns the number of child windows of a given window.
***************************************************************************/

U16 ChildCount(WIN win) {

#define enumLimit 10

	WIN_ENUM enumWin;
	WIN wins[enumLimit];
	WIN_FLAGS winFlags[enumLimit];
	STATUS s;

	enumWin.max = enumWin.count = enumLimit;
	enumWin.count  = maxU16;
	enumWin.pWin = wins;
	enumWin.pFlags = winFlags;
	enumWin.next = 0;		// start at the begining
	enumWin.flags = wsEnumChildren;

	s = ObjectCall(msgWinEnum, win, &enumWin);

	if (enumWin.pWin != (P_WIN) wins)
		OSHeapBlockFree(enumWin.pWin);

	Dbg(Debugf("ClockApp: ChildCount() returning %d", enumWin.count);)

	return enumWin.count;

}  // ChildCount


/***************************************************************************
 TwelveHour returns a boolean based on the system time format preference.
 True is returned for a 12-hour clock and false for a 24-hour clock
***************************************************************************/

BOOLEAN TwelveHour(void) {

	RES_READ_DATA     rrd;
	U8						timeFormat;
	STATUS            s;


	rrd.resId = tagPrTimeFormat;
	rrd.heap = Nil(OS_HEAP_ID);
	rrd.pData = &timeFormat;
	rrd.length = SizeOf (timeFormat);
	s = ObjCallWarn (msgResReadData, theSystemPreferences, &rrd);
	if (s < stsOK)
		return true;  // default to 12 hour format
	else
		return timeFormat == prTime12Hour;

} // TwelveHour


/***************************************************************************
 DisplaySeconds returns a boolean based on the system time seconds format 
 preference.  True is returned if seconds are to be displayed (hh:mm:ss) and
 false if they are not displayed (hh:mm).
***************************************************************************/

BOOLEAN DisplaySeconds(void) {

	RES_READ_DATA     rrd;
	U8					timeSeconds;
	STATUS            s;

	rrd.resId = tagPrTimeSeconds;
	rrd.heap = Nil(OS_HEAP_ID);
	rrd.pData = &timeSeconds;
	rrd.length = SizeOf (timeSeconds);
	s = ObjCallWarn (msgResReadData, theSystemPreferences, &rrd);
	if (s < stsOK)
		return false;  // default to HH:MM format
	else
		return timeSeconds == prTimeSecondsDisplay;

} // DisplaySeconds


/***************************************************************************
 Deblank removes leading and trailing blanks from a string.
***************************************************************************/

void Deblank(char s[])
{
  char *tmp;
  U16 ix;

  tmp = s;
  while ( *tmp == SPACE ) {
    tmp++;
  }

  ix = strlen(s);
  if (ix>0) {
    ix--;
    while ( s[ix] == SPACE ) {
      s[ix--] = NULL;
    } // while
  }

  strcpy(s, tmp);

} // Deblank

  
/***************************************************************************
 CancelAlarm kills an OS one-shot timer alarm if it is defined.
***************************************************************************/

void CancelAlarm(P_OS_HANDLE pHandle) {

	if (*pHandle != NULL) {
		ObjCallWarn(msgTimerAlarmStop, clsTimer, (P_ARGS) *pHandle);
		*pHandle = NULL;
	}

} // CancelAlarm


/***************************************************************************
 CancelTick kills an OS periodic timer alarm if it is defined.
***************************************************************************/

void CancelTick(P_OS_HANDLE pHandle) {

	if (*pHandle != NULL) {
		Dbg(Debugf("ClockApp: in CancelTick handle is %ld", *pHandle);)
		ObjCallWarn(msgTimerStop, clsTimer, (P_ARGS) *pHandle);
		*pHandle = NULL;
	}

}  // CancelTick


/***************************************************************************
 ScheduleTick cancels the base-line tick registered with the timer and starts
 a new one.  If the clock is displaying seconds then this is a periodic tick,
 once per second.  If seconds are not displayed then the system time is
 read and a one-shot tick is scheduled to occur on the next minute boundary.
 We can't just sync up at the start and then use a periodic one-minute tick
 because sync will be lost when the machine is powered off and on again.
***************************************************************************/

STATUS ScheduleTick(P_CLOCK_APP_DATA pInst, OBJECT self) {

	TIMER_INTERVAL_INFO intervalInfo;
	TIMER_REGISTER_INFO registerInfo;
	OS_DATE_TIME dateTime;
	STATUS s;

	CancelTick(&pInst->tickHandle);

	//
	// Once a second (periodic) or minute (one-shot) depending on options
	//
	if (DisplaySeconds()) {

		//
		// schedule a tick-per-second
		//
		intervalInfo.client 	= self;
		intervalInfo.interval	= 1000;
		intervalInfo.clientData	= pNull;

		ObjCallRet(msgTimerRegisterInterval, clsTimer, &intervalInfo, s);
		pInst->tickHandle = intervalInfo.transactionHandle;
		pInst->oneShotTick = false;

	} else {
		//
		// delay to next minute start
		//

		OSGetTime (SizeOf(dateTime), &dateTime);
				   
		registerInfo.client 		= self;
		// delay varies from 1 to 60 seconds
		registerInfo.time			= 1000L * (60 - dateTime.seconds);
		registerInfo.clientData	= pNull;

		Dbg(Debugf(
			"ClockApp: in ScheduleTick time is %02d:%02d:%02d, delay is %ld",
			dateTime.hours,
			dateTime.minutes,
			dateTime.seconds,
			registerInfo.time
		););

		ObjCallRet(msgTimerRegister, clsTimer, &registerInfo, s);
		pInst->tickHandle = registerInfo.transactionHandle;
		pInst->oneShotTick = true;
	} //  delay to minute

	Dbg(Debugf("ClockApp: in ScheduleTick new handle is %ld",
        pInst->tickHandle);)

	return stsOK;

}  // ScheduleTick


/***************************************************************************
 ValidField checks a numeric field against upper and lower bounds and
 displays a note if the field is out of spec.
***************************************************************************/

STATUS ValidField(
	U32 value, 
	U32 low, 
	U32 high, 
	P_STRING pFieldName
) {

	if ( (low <= value ) && (value <= high) ) {
		return stsOK;
	} else {
		StdMsg(stsClockFieldRangeError, pFieldName, low, high);
		return stsFailed;
	}  // else field out of range

}  // ValidField

/***************************************************************************
 GetDateTimeStr reads the system date-and-time and builds the strings for
 the seperate labels which make up the clock's display.
***************************************************************************/

void GetDateTimeStr(
	DATE_TIME_STR dateStr, 
	DATE_TIME_STR timeStr,
	AM_PM_STR amPmStr,
	ALARM_STR alarmStr,
    P_OS_DATE_TIME pDateTime,
	P_CLOCK_APP_DATA pInst
)

{
	P_STRING tmpDate, tmpTime;
	P_STRING locOfAPM;


	//
	// If date and time are on the same line then add two spaces between them.
	// One space per label so as to be in the font size of that label
	//

	//
	// add the leading space
	//
	tmpDate = dateStr;
	tmpTime = timeStr;
	if (pInst->fmt & sideBySide) {
		if (pInst->fmt & timeFirst) {
			*tmpDate++ = ' ';
		} else {
			*tmpTime++ = ' ';
		}
	}

	OSGetTime (SizeOf(OS_DATE_TIME), pDateTime);
	PrefsDateToString (pDateTime, tmpDate);
	PrefsTimeToString (pDateTime, tmpTime);
	Deblank(tmpTime);

	//
	// add the trailing space
	//
	if (pInst->fmt & sideBySide) {
		if (pInst->fmt & timeFirst) {
			strcat(timeStr, " ");
		} else {
			strcat(dateStr, " ");
		}
	}


	//
	// extract AM/PM (and trailing space) if present
	//
	if	( (locOfAPM = strchr(tmpTime, 'A')) ||	
          (locOfAPM = strchr(tmpTime, 'P')) ) {
		strcpy(amPmStr, locOfAPM);  // grab "A.M." or "P.M."
		*(locOfAPM) = NULL;         // chop off "A.M." or "P.M.", 
                                    //   leave trailing space
	} else
		amPmStr[0] = NULL;

	//
	// return simple-minded alarm indicator for now
	//
	strcpy(alarmStr, "  ");
	if (pInst->alarmOn)
		alarmStr[1] = alarmIndicator;

	Dbg(Debugf("ClockApp: GetDateTimeStr - <%s> <%s> <%s> <%s>", 
		timeStr, amPmStr, dateStr, alarmStr););

} // GetDateTimeStr


/***************************************************************************
 SetLabelFont (re)sets the font of a label if the label is defined.
***************************************************************************/

STATUS SetLabelFont(OBJECT win, U16 fontId) {

	SYSDC_FONT_SPEC spec;
	STATUS s;

	if (win) {
		ObjCallRet(msgLabelGetFontSpec, win, &spec, s);
		spec.id = fontId;
		ObjCallRet(msgLabelSetFontSpec, win, &spec, s);

		// should be able to remove this when msgLabelSetFontSpec does dirtying
		ObjCallRet(msgWinSetLayoutDirty, win, (P_ARGS) (U32) true, s);
	}
}  // SetLabelFont

/***************************************************************************
 SetLabelScale (re)sets the font size of a label if the label is defined.
***************************************************************************/

STATUS SetLabelScale(OBJECT win, U8 newScale) {

	STATUS s;

	if (win) {
		ObjCallRet(msgLabelSetScale, win, (P_ARGS) (U32) newScale, s);

		// should be able to remove this when msgLabelSetScale does dirtying
		ObjCallRet(msgWinSetLayoutDirty, win, (P_ARGS) (U32) true, s);
	}

}  // SetLabelScale


/***************************************************************************
						CLOCK DISPLAY PROCEDURES
****************************************************************************/

/***************************************************************************
 DisplayTime builds strings based on the current date-and-time and stuffs
 them into the correct labels.
***************************************************************************/

void DisplayTime (P_CLOCK_APP_DATA pInst)
{
	DATE_TIME_STR dateStr, timeStr;
    OS_DATE_TIME dateTime;
	AM_PM_STR amPmStr;
	ALARM_STR alarmStr;
    WIN_METRICS wm;

	GetDateTimeStr(dateStr, timeStr, amPmStr, alarmStr, &dateTime, pInst);

	if (pInst->timeWin) {
		(void)ObjCallWarn(msgLabelSetString, pInst->timeWin, timeStr);
	}

	if (pInst->amPmWin) {
		(void)ObjCallWarn(msgLabelSetString, pInst->amPmWin, amPmStr);
	}

	if (pInst->alarmWin) {
		(void)ObjCallWarn(msgLabelSetString, pInst->alarmWin, alarmStr);
	}

	if (pInst->dateWin) {
		(void)ObjCallWarn(msgLabelSetString, pInst->dateWin, dateStr);
	}

    // we gotta lay out in case some string has gotten longer.
    // if we don't do this here, we risk displaying "10:0..." instead
    // of "10:00" when the time turns over from 9:00 to 10:00.
    // Optimization: do this only when the hour turns over.

    if (dateTime.seconds == 0 && dateTime.minutes == 0) {

        Dbg(Debugf("ClockApp: laying out app window");)
        wm.options = wsLayoutDefault | wsLayoutMinPaint;
        (void)ObjectCall(msgWinLayout, pInst->appWin, &wm);
    }

} /* DisplayTime */


/***************************************************************************
						CLOCK WINDOW PROCEDURES
****************************************************************************/


/***************************************************************************
 CreateLabel is used to create whatever labels are used to constuct the
 main clock window.
***************************************************************************/

STATUS CreateLabel (
	OBJECT self,
	P_OBJECT child,
	P_STRING initStr,
	U8 size,
	U16 font
)
{
	LABEL_NEW		ln;
	STATUS			s;

	ObjCallRet(msgNewDefaults, clsClockLabel, &ln, s);

	//
	//	Set the appearance of the label.
	//
	ln.label.pString			= initStr;

	// use Helvetica font
	ln.label.style.fontType		= lsFontCustom;
	ln.label.font.id			= font;

  	ln.label.style.scaleUnits	= bsUnitsPoints;
  	ln.label.scale				= size;  // font size in points

	//
	// set alignment in case we are zoomed and this label is the main window
	//
	ln.label.style.xAlignment	= lsAlignCenter;
	ln.label.style.yAlignment	= lsAlignCenter;

	ln.control.client			= self; 	// have gestures sent to us

	ln.border.style.edge		= bsEdgeNone; // bsEdgeAll
	ln.border.style.drag		= bsDragHoldDown;
	ln.border.style.getDeltaWin = true;
	ln.border.style.top			= bsTopUp;

	ln.gWin.helpId				= hlpClkAppMainWin;

	//
	//	Create the label.
	//
	ObjCallRet(msgNew, clsClockLabel, &ln, s);

	*child = ln.object.uid;

	return stsOK;

} /* CreateLabel */


/***************************************************************************
 CreateClockWindow builds the main display window and any needed children.
 There can be as many as 4 sub-windows: time digits, am/pm, alarm indicator
 and date. Labels only repaint the right-most characters that change. For this
 reason the time digits are kept in a seperate window from the am/pm indicator
 to minimize flash as the time ticks. The alarm indicator lives in its own 
 to allow contol over placement and font size.

 The code proceeds by examining the styles encoded within the format code 
 (pInst->fmt), creating the individual labels as needed. If the time display
 contains more than one label then the labels are combined into a window 
 using table layout. The same is done for the date display but this is only 
 possible when the time is not displayed, causing the alarm indicator label 
 to be combined with the date label. Finally if both the date and time are 
 displayed then their top-level windows are combined in a final table-layout
 window.

 So in the simplest case the clock window can be a single label. This is 
 true when only the date or a 24-hour format time is displayed and the alarm
 is not set. The worst case is with the alarm enabled, a 12-hour clock, and
 both date and time displayed. 
***************************************************************************/

STATUS CreateClockWindow (P_CLOCK_APP_DATA pInst, OBJECT self)
{
	STATUS			s;
	TBL_LAYOUT_NEW	layout;
	WIN_METRICS		wm;
	DATE_TIME_STR	dateStr, timeStr;
    OS_DATE_TIME    dateTime;
	AM_PM_STR		amPmStr;
	ALARM_STR		alarmStr;
	OBJECT			child;
	U16				alarmFontId;

	// destroy clockWin if it exists
	if (pInst->clockWin != Nil(WIN)) {

		// let the frame know we have taken the client window
		ObjCallRet(msgFrameSetClientWin, pInst->appWin, Nil(WIN), s);

		// Make sure the clockWin is not selected.
		s = ObjectCall(msgSelIsSelected, pInst->appWin, pNull);

		if (s == stsOK) {

			ObjCallRet(msgSelSetOwner, theSelectionManager, pNull, s);
		}

		ZapWin(&pInst->clockWin);

		pInst->dateAllWin		= Nil(WIN);
		pInst->dateWin			= Nil(WIN);
		pInst->timeAllWin		= Nil(WIN);
		pInst->timeWin			= Nil(WIN);
		pInst->amPmWin			= Nil(WIN);
		pInst->alarmWin		= Nil(WIN);

	}

	//
	// Create the date window if needed.
	//

	GetDateTimeStr(dateStr, timeStr, amPmStr, alarmStr, &dateTime, pInst);

	if (pInst->fmt & dateUp) {
		StsJmp(
			CreateLabel(
				self, 
				&pInst->dateWin, 
				dateStr, 
				pInst->dateSize, 
				pInst->fontId),
			s, error);
		Dbg(Debugf("clockApp: created date window");)
	}

	//
	// create alarm indicator window
	//
	if (pInst->alarmOn) {

		alarmFontId = SysDcFontId(alarmIndicatorFont);

		StsJmp(
			CreateLabel(
				self, 
				&pInst->alarmWin, 
				alarmStr, 
				(U8)((pInst->fmt & timeUp)
                        ? pInst->timeSize
                        : pInst->dateSize),
				alarmFontId),
			s, error);
		Dbg(Debugf("clockApp: created alarm window with size %d",
				(pInst->fmt & timeUp) ? pInst->timeSize : pInst->dateSize);)
	}

	if (pInst->fmt & timeUp) {
		//
		// create time window(s)
		//
		StsJmp(
			CreateLabel(
				self, 
				&pInst->timeWin, 
				timeStr, 
				pInst->timeSize, 
				pInst->fontId),
			s, error);
		Dbg(Debugf("clockApp: created time window");)

		if (TwelveHour() || 
				((pInst->fmt & alarmOnTime) && pInst->alarmWin)) {

			// combine time, optional am/pm and alarm windows

			ObjCallRet(msgNewDefaults, clsTableLayout, &layout, s);

			layout.tableLayout.numCols.value = 1;

			if (TwelveHour() ) {
				StsJmp(
					CreateLabel(
						self, 
						&pInst->amPmWin, 
						amPmStr, 
						pInst->timeSize, 
						pInst->fontId),
					s, error);
				Dbg(Debugf("clockApp: created AM/PM window");)

				layout.tableLayout.numCols.value++;  // another col for AM/PM
			} // if twelve hour clock

			if ( pInst->alarmWin && (pInst->fmt & alarmOnTime))
				layout.tableLayout.numCols.value++;

			layout.tableLayout.numCols.constraint = tlAbsolute;

			layout.tableLayout.numRows.value = 1;
			layout.tableLayout.numRows.constraint = tlAbsolute;

			layout.tableLayout.rowHeight.constraint = tlGroupMax;

			layout.border.style.drag = bsDragHoldDown;
			layout.border.style.getDeltaWin = true;

			layout.gWin.helpId = hlpClkAppMainWin;

			//
			// cause timeAllWin to be centered when zoomed or page level
			// not relevent when shrink-wrapped or top level window
			//
			layout.tableLayout.style.tblXAlignment = tlAlignCenter;
			layout.tableLayout.style.tblYAlignment = tlAlignCenter;

			ObjCallJmp(msgNew, clsTableLayout, &layout, s, error);

			pInst->timeAllWin	= layout.object.uid;


			//
			//	Insert one, two or three labels into the window.
			//

			wm.parent	= pInst->timeAllWin;
			wm.options	= wsPosTop;

			ObjCallRet(msgWinInsert, pInst->timeWin, &wm, s);

			if (TwelveHour()) {
				ObjCallRet(msgWinInsert, pInst->amPmWin, &wm, s);
			}

			if ( (pInst->fmt & alarmOnTime) && pInst->alarmWin ) {
				ObjCallRet(msgWinInsert, pInst->alarmWin, &wm, s);
			}

			Dbg(Debugf("clockApp: created timeAll window");)


		}  // if am/pm or alarmOnTime
		else
		{  // timeAllWin is just a label
			pInst->timeAllWin	= pInst->timeWin;
		}  // simple timeAllWin
	}  // if timeUp

	if ( pInst->fmt & dateUp ) {
	
		if (pInst->alarmWin && !(pInst->fmt & alarmOnTime)) {

			//
			// alarm indicator lives in date window
			//

			ObjCallRet(msgNewDefaults, clsTableLayout, &layout, s);

			layout.tableLayout.numRows.value = 1;
			layout.tableLayout.numRows.constraint = tlAbsolute;

			layout.tableLayout.numCols.value = 2;
			layout.tableLayout.numCols.constraint = tlAbsolute;

			layout.tableLayout.rowHeight.constraint = tlGroupMax;
	
			layout.border.style.drag = bsDragHoldDown;
			layout.border.style.getDeltaWin = true;

			layout.gWin.helpId = hlpClkAppMainWin;

			//
			// cause dateAllWin to be centered when zoomed or page level
			// not relevent when shrink-wrapped or not top level window
			//
			layout.tableLayout.style.tblXAlignment = tlAlignCenter;
			layout.tableLayout.style.tblYAlignment = tlAlignCenter;

			ObjCallJmp(msgNew, clsTableLayout, &layout, s, error);

			pInst->dateAllWin	= layout.object.uid;

			//
			//	Insert both labels into the window.
			//

			wm.parent	= pInst->dateAllWin;
			wm.options	= wsPosTop;

			ObjCallRet(msgWinInsert, pInst->dateWin, &wm, s);

			ObjCallRet(msgWinInsert, pInst->alarmWin, &wm, s);

			Dbg(Debugf("clockApp: created dateAll window");)

		}  // if alarm in date
		else
		{
			// As the alarm indicator is note with the date
			// the dateAll window is just dateWin.
			pInst->dateAllWin	= pInst->dateWin;
		}
	} // if dateUp


	//
	// Now layout overall combination of date and time windows
	//

	if ((pInst->fmt & bothUp) == bothUp) {
		ObjCallRet(msgNewDefaults, clsTableLayout, &layout, s);

		// don't file 'cause we (re)create all windows at open
		layout.win.flags.style &= ~wsSendFile;  

		layout.tableLayout.numRows.value = (pInst->fmt & oneRow) ? 1 : 2;
		layout.tableLayout.numRows.constraint = tlAbsolute;

		layout.tableLayout.numCols.value = (pInst->fmt & oneCol) ? 1 : 2;
		layout.tableLayout.numCols.constraint = tlAbsolute;

		if (pInst->fmt & sideBySide) {
			layout.tableLayout.rowHeight.constraint = tlChildrenMax;
			layout.tableLayout.colWidth.constraint = tlGroupMax;
		} 
		else 
		{
			layout.tableLayout.rowHeight.constraint = tlGroupMax;
			layout.tableLayout.colWidth.constraint = tlChildrenMax;
		}
		
		layout.border.style.drag = bsDragHoldDown;
		layout.border.style.getDeltaWin = true;

		layout.gWin.helpId = hlpClkAppMainWin;

		//
		// cause clockWin to be centered when zoomed or page level
		// not relevent when shrink-wrapped
		//
		layout.tableLayout.style.tblXAlignment = tlAlignCenter;
		layout.tableLayout.style.tblYAlignment = tlAlignCenter;

		ObjCallJmp(msgNew, clsTableLayout, &layout, s, error);

		pInst->clockWin = layout.object.uid;

		Dbg(Debugf("clockApp: created main window");)

		//
		//	Insert both labels into the window.
		//

		wm.parent	= pInst->clockWin;
		wm.options	= wsPosTop;

		ObjCallRet(msgWinInsert, pInst->dateAllWin, &wm, s);

		// optionally place time at back so it appears at top or left
		wm.options	= (pInst->fmt & timeFirst) ? wsPosBottom : wsPosTop;

		ObjCallRet(msgWinInsert, pInst->timeAllWin, &wm, s);

	} // if both up
	else // date or time only
		pInst->clockWin = 
			(pInst->fmt & timeUp) ? pInst->timeAllWin : pInst->dateAllWin;
	
	child = pInst->clockWin;

	// Create a clock win - it will wrap the current child.
	ObjectCall(msgNewDefaults, clsClockWin, &layout);
	layout.win.flags.style |= wsShrinkWrapHeight | wsShrinkWrapWidth;
	ObjCallRet(msgNew, clsClockWin, &layout, s);
	pInst->clockWin = layout.object.uid;

	// Insert the proper child into the clock win.
	wm.options	= wsPosTop;
	wm.parent	= pInst->clockWin;
	ObjCallRet(msgWinInsert, child, &wm, s);

	//
	// Insert the clock window into the app window.
	//
	ObjCallRet(msgFrameSetClientWin, pInst->appWin, pInst->clockWin, s);

	//
	// Tell app window to wrap around label(s)
	//
	ObjCallRet(msgWinGetFlags, pInst->appWin, &wm, s);
	wm.flags.style |= wsShrinkWrapHeight | wsShrinkWrapWidth;
	ObjCallRet(msgWinSetFlags, pInst->appWin, &wm, s);

	wm.options = wsLayoutDefault;
	ObjCallRet(msgWinLayout, pInst->appWin, &wm, s);

	return stsOK;

error:

	Dbg(Debugf("ClockApp: error creating subwindow");)
	ZapWin(&pInst->clockWin);
	ZapWin(&pInst->dateAllWin);
	ZapWin(&pInst->dateWin);
	ZapWin(&pInst->timeAllWin);
	ZapWin(&pInst->timeWin);
	ZapWin(&pInst->amPmWin);
	ZapWin(&pInst->alarmWin);

	return s;

} /* CreateClockWindow */

/***************************************************************************
				MESSAGE HANDLING PROCEDURES (METHODS)
 ***************************************************************************/


/***************************************************************************
	ClockFree
	
	Destroy the clock instance data and clock object.
****************************************************************************/

MSG_HANDLER ClockFree(
	const MESSAGE	msg,
	const OBJECT	self,
	const P_ARGS	pArgs,
	const CONTEXT	ctx,
	const PP_CLOCK_APP_DATA pData)

{

	P_CLOCK_APP_DATA	pInst;

	Dbg(Debugf("ClockApp: entering ClockAppFree");)

	pInst = *pData;

	//
	// Free the clock window instance data
	//
	StsWarn(OSHeapBlockFree(pInst));
	pInst = pNull;

	// clsmgr calls ancestor after

	return stsOK;

	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} /* ClockFree */


/***************************************************************************
	ClockInit
	
	Respond to msgInit. Allocate the clock instance data.
****************************************************************************/

MSG_HANDLER ClockInit(
	const MESSAGE	msg,
	const OBJECT	self,
	const P_ARGS	pArgs,
	const CONTEXT	ctx,
	const PP_CLOCK_APP_DATA pData)

{
	RES_READ_DATA       rrd;
	PREF_SYSTEM_FONT	prefFont;
	STATUS s;

	P_CLOCK_APP_DATA	pInst;

	pInst = *pData;

	Dbg(Debugf("ClockApp: entering ClockInit");)

	//
	// Allocate the instance data
	//
	OSHeapBlockAlloc(osProcessHeapId, SizeOf(CLOCK_APP_DATA), &pInst);
	
	StsJmp( ObjectWrite(self, ctx, &pInst), s, error);

	InitNonFiled(pInst);

	//
	// set default configuration
	//
	pInst->fmt			= defaultFmt;

	rrd.resId = tagPrUserFont;
	rrd.heap = Nil(OS_HEAP_ID);
	rrd.pData = &prefFont;
	rrd.length = SizeOf (prefFont);
	s = ObjCallWarn (msgResReadData, theSystemPreferences, &rrd);
	if (s < stsOK)
		pInst->fontId		= SysDcFontId("HE55");
	else
		pInst->fontId		= prefFont.spec.id;

	pInst->timeSize	= defaultFontSize;
	pInst->dateSize	= defaultFontSize;

	pInst->alarmDate = noDate;
	pInst->alarmType = timerEveryDay;
	pInst->alarmDayOfWeek = osMonday;
	pInst->alarmHours = 10;
	pInst->alarmMinutes = 30;
	pInst->alarmIsPM = false;
	pInst->alarmBeep	= false;
	pInst->alarmTimeout = false;
	pInst->alarmMultiple = false;
	pInst->alarmMessage[0] = NULL;
	pInst->alarmSnooze = defaultSnooze;
	pInst->alarmSnoozeEnable = false;
	pInst->self = self;

	return stsOK;

error:
	(void)OSHeapBlockFree(pInst);
	pInst = pNull;

	return s;

	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} /* ClockInit */

/***************************************************************************
	ClockSave
	
	Respond to msgSave. Write instance data to file.
****************************************************************************/

MSG_HANDLER ClockSave(
	const MESSAGE	msg,
	const OBJECT	self,
	const P_OBJ_SAVE	pArgs,
	const CONTEXT	ctx,
	const PP_CLOCK_APP_DATA pData)

{
	STATUS s;

	STREAM_READ_WRITE	write;

	P_CLOCK_APP_DATA	pInst;

	pInst = *pData;

	Dbg(Debugf("ClockApp: entering ClockSave");)

	write.numBytes = SizeOf(CLOCK_APP_FILED_DATA);
	write.pBuf = pInst;
	ObjCallRet(msgStreamWrite, pArgs->file, &write, s);
	
	ObjCallRet(msgResPutObject, pArgs->file, pInst->corkboard, s);

	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} /* ClockSave */


/***************************************************************************
	ClockRestore
	
	Respond to msgRestore. Read filed data.
****************************************************************************/

MSG_HANDLER ClockRestore(
	const MESSAGE	msg,
	const OBJECT	self,
	const P_OBJ_RESTORE	pArgs,
	const CONTEXT	ctx,
	const PP_CLOCK_APP_DATA pData)

{
	STATUS s;

	STREAM_READ_WRITE	read;

	P_CLOCK_APP_DATA	pInst;

	pInst = *pData;

	Dbg(Debugf("ClockApp: entering ClockRestore");)

	//
	// Allocate the instance data
	//
	OSHeapBlockAlloc(osProcessHeapId, SizeOf(CLOCK_APP_DATA), &pInst);
	
	StsJmp( ObjectWrite(self, ctx, &pInst), s, error);

	InitNonFiled(pInst);

	read.numBytes = SizeOf(CLOCK_APP_FILED_DATA);
	read.pBuf = pInst;
	ObjCallWarn(msgStreamRead, pArgs->file, &read);

	ObjCallRet(msgResGetObject, pArgs->file, &pInst->corkboard, s);

	pInst->self = self;

	return stsOK;

error:
	(void)OSHeapBlockFree(pInst);
	pInst = pNull;

	return s;

	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} /* ClockRestore */

/***************************************************************************
	ClockAppOpen
	
	Respond to msgAppOpen.
****************************************************************************/

MSG_HANDLER ClockAppOpen(
	const MESSAGE			msg,
	const OBJECT			self,
	const P_APP_OPEN		pArgs,
	const CONTEXT			ctx,
	const PP_CLOCK_APP_DATA pData)

{
	APP_METRICS			am;
	STATUS				s;
	CORKBOARD_WIN_NEW	cbWinNew;
	P_CLOCK_APP_DATA	pInst;

	Dbg(Debugf("ClockApp: entering ClockAppOpen");)

	pInst = *pData;

	//
	// Get the main application window.
	//
	ObjCallRet(msgAppGetMetrics, self, &am, s);
	pInst->appWin = am.mainWin;  // remember this for re-layout

	ObjCallRet(msgFrameResizeEnable, am.mainWin, (P_ARGS) (U32) false, s);

	//
	// Create a clock window.
	//
	StsRet(CreateClockWindow(pInst, self), s);

	//
	// Create the corkboard if not already created.
	//
	if (pInst->corkboard == pNull) {

		ObjCallRet(msgNewDefaults, clsNoteCorkBoardWin, (P_ARGS)&cbWinNew, s);
		cbWinNew.border.style.edge				= bsEdgeAll;
		cbWinNew.border.style.lineStyle			= bsLineDouble;
//		cbWinNew.win.flags.style				&= ~wsShrinkWrapWidth;
//		cbWinNew.win.bounds.size.w				= LayoutUnitsToPixels(110);
//		cbWinNew.win.bounds.size.h				= LayoutUnitsToPixels(15);
		cbWinNew.gWin.helpId					= hlpClkAppAlarmCork;
		cbWinNew.embeddedWin.style.selection	= ewSelect;
		ObjCallRet(msgNew, clsNoteCorkBoardWin, (P_ARGS) &cbWinNew, s);

		pInst->corkboard	= cbWinNew.object.uid;
		pInst->corkIn		= corkNotInserted;
		Dbg(Debugf("ClockApp: new corkboard uid is %ld", cbWinNew.object.uid);)
	}

	//
	// Schedule tick message
	//
	StsRet(ScheduleTick(pInst, self), s);

	//
	// observe the system preferences for changes to date/time formats
	//
	ObjCallRet(msgAddObserver, theSystemPreferences, (P_ARGS) self, s);

	//
	// observe the power switch to refresh on power-up
	//
	ObjCallRet(msgAddObserver, thePowerButton, (P_ARGS) self, s);

	pArgs->childAppParentWin = pInst->clockWin;

	return stsOK;

	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} /* ClockAppOpen */


/***************************************************************************
	ClockAppClose
	
	Respond to msgAppClose.
****************************************************************************/

MSG_HANDLER ClockAppClose(
	const MESSAGE	msg,
	const OBJECT	self,
	const P_ARGS	pArgs,
	const CONTEXT	ctx,
	const PP_CLOCK_APP_DATA pData)

{
	FRAME_METRICS	fm;
	APP_METRICS   	am;
	STATUS			s;

	P_CLOCK_APP_DATA	pInst;

	Dbg(Debugf("ClockApp: entering ClockAppClose");)

	pInst = *pData;

	//
	// Drop the observation of the preferences
	//
	ObjCallWarn(msgRemoveObserver, clsPreferences, pNull);

	//
	// Drop the observation of the power button
	//
	ObjCallWarn(msgRemoveObserver, thePowerButton, pNull);

	//
	// Drop the note if it is still up
	//
	if (pInst->alarmNote) {
		ObjCallWarn(msgNoteCancel, pInst->alarmNote, (P_ARGS) msgClockAlarmOK);
		pInst->alarmNote = objNull;
	}

	//
	// Cancel tick if periodic
	//
	if (!pInst->oneShotTick)
		CancelTick(&pInst->tickHandle);

	CancelAlarm(&pInst->alarmHandle);

	pInst->alarmOn = false;

	//
	// Get the application metrics
	//
	ObjCallRet(msgAppGetMetrics, self, &am, s);

	//
	// Deinstall the client window (view)
	//
	ObjCallRet(msgFrameGetMetrics, am.mainWin, &fm, s);

	fm.clientWin = Nil(WIN);

	ObjCallRet(msgFrameSetMetrics, am.mainWin, &fm, s);

	//
	// Destroy the clock window
	//
	ObjCallRet(msgDestroy, pInst->clockWin, Nil(P_ARGS), s);
	pInst->clockWin = Nil(WIN);

	return stsOK;


	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} /* ClockAppClose */	

/***************************************************************************
	ClockAppGetDisplayString
	
	Respond to msgClockAppGetDisplayString.
****************************************************************************/
MSG_HANDLER ClockAppGetDisplayString (
	const MESSAGE	msg,
	const OBJECT	self,
	const P_ARGS	pArgs,
	const CONTEXT	ctx,
	const PP_CLOCK_APP_DATA pData)
{
	DATE_TIME_STR		dateStr;
	DATE_TIME_STR		timeStr;
    OS_DATE_TIME        dateTime;
	AM_PM_STR			amPmStr;
	ALARM_STR			alarmStr;
	STATUS				s;
	P_STRING			pBuf;
	P_CLOCK_APP_DATA	pInst;


	pInst = *pData;

	// Allocate a string buffer.
	StsRet(OSHeapBlockAlloc(osProcessSharedHeapId, (SIZEOF)(MAX_DT_STR),
            &pBuf), s);

	// Get the date and time strings.
	GetDateTimeStr(dateStr, timeStr, amPmStr, alarmStr, &dateTime, pInst);

	if (amPmStr) {

		strcat(timeStr, amPmStr);
	}

	// Process the current display format.
	switch (pInst->fmt) {

	case timeOnly:

		strcpy(pBuf, timeStr);

		break;

	case dateOnly:

		strcpy(pBuf, dateStr);

		break;

	case dateAndTime:
	case dateOverTime:

		strcpy(pBuf, dateStr);
		strcat(pBuf, " ");
		strcat(pBuf, timeStr);

		break;

	case timeAndDate:
	case timeOverDate:
	
		strcpy(pBuf, timeStr);
		strcat(pBuf, " ");
		strcat(pBuf, dateStr);

		break;

	default:

		strcpy(pBuf, "unknown format");

		Dbg(Debugf("ClockApp: %s", pBuf);)
	}

	Dbg(Debugf("ClockApp: msgClockAppGetDisplayString = [%s]", pBuf);)

	// Give buffer to caller.
	*((PP_STRING)pArgs) = pBuf;

	return stsOK;

	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

}	// ClockAppGetDisplayString

/****************************************************************************

 DestroyControl is used by ClockRefreshCard to remove unneeded controls 
 from class App's option cards.

***************************************************************************/

STATUS DestroyControl(OBJECT win, TAG controlTag) {

	OBJECT			control;
	STATUS			s;

	control = (OBJECT)ObjectCall(msgWinFindTag, win, (P_ARGS) controlTag);

	if (control != objNull) {

		// Destroy the control.
		s = ObjCallWarn(msgDestroy, control, (P_ARGS) (U32) false);
	} else {
		Dbg(Debugf("ClockApp: DestroyControl could not find tag of control %lx",
                    controlTag);)
	}

	return stsOK;

}	// DestroyControl


/**************************************************************************
 IsDirty returns true if the value of a control has been altered. The 
 control will be returned in the variable pointed to by pC if pC is not null.
***************************************************************************/

BOOLEAN IsDirty(WIN win, TAG controlTag, P_OBJECT pC) {

	BOOLEAN	dirty;
	OBJECT	control;
	P_OBJECT	pControl;

	pControl = (pC == pNull ? &control : pC);

	*pControl = (OBJECT)ObjectCall(msgWinFindTag, win, (P_ARGS) controlTag);

	if (*pControl != objNull) {
		ObjCallWarn(msgControlGetDirty, *pControl, &dirty);
	} else {
		dirty = false;
		Dbg(Debugf("ClockApp: IsDirty could not find tag of control %lx",
                    controlTag);)
	}

	return dirty;

} // IsDirty


/**************************************************************************
 CleanControl will clear the 'dirty' status of a control.
***************************************************************************/

STATUS CleanControl(WIN win, TAG controlTag) {

	OBJECT			control;
	STATUS			s;

	control = (OBJECT)ObjectCall(msgWinFindTag, win, (P_ARGS) controlTag);

	if (control != objNull) {
		ObjCallRet(msgControlSetDirty, control, (P_ARGS) (U32) false, s);
	} else {
		Dbg(Debugf("ClockApp: CleanControl could not find tag of control %lx",
                    controlTag);)
	}

	return stsOK;

}	// CleanControl


/**************************************************************************
 ReadControl will validate and pass back the value of a numeric control.
 Three return values are possible:
	stsFailed -			the control contained an invalid number or none at all
	stsDirtyControl -	the control had been set but not cleaned
	stsCleanControl -	the control had already been cleaned
***************************************************************************/

STATUS ReadControl(
	WIN cardWin, 
	TAG controlTag, 
	P_U32 pValue, 
	U32 low, 
	U32 high, 
	P_STRING pFieldName,
	BOOLEAN alwaysRead
) {

	WIN control;
	U32 suspectValue;
	STATUS s;

	if ( IsDirty(cardWin, controlTag, &control) || alwaysRead ) {

		s = ObjCallWarn(msgControlGetValue, control, &suspectValue);

		if ((s == stsIntegerFieldEmpty)||(s == stsIntegerFieldInvalid)) {

			Dbg(Debugf("ClockApp: empty or invalid integer field %s",
                        pFieldName);)
			StdMsg(stsClockIntFieldInvalid, pFieldName);
			return stsFailed;

		} else if ((s == stsDateFieldEmpty)||(s == stsDateFieldInvalid) ) {

			Dbg(Debugf("ClockApp: empty or invalid date field %s",
                        pFieldName);)
			StdMsg(stsClockDateFieldInvalid, pFieldName);
			return stsFailed;

		} else if ( (low != high) || (low != 0) ) {

			StsRetNoWarn(ValidField(suspectValue, low, high, pFieldName), s);
		}

		*pValue = suspectValue;

		return stsDirtyControl;

	} else
		return stsCleanControl;

} // ReadControl


/***************************************************************************
 ReadStrControl will validate and pass back the value of a string control.
 Two return values are possible:
	stsDirtyControl -	the control had been set but not cleaned
	stsCleanControl -	the control had already been cleaned
***************************************************************************/

STATUS ReadStrControl(
	WIN cardWin, 
	TAG controlTag, 
	P_STRING str,
 	U32 strSize,
	OBJECT self
) {

	WIN control;
	CONTROL_STRING	cs;
	char tempStr[100];
	STATUS s;

	Unused(self);

	if ( IsDirty(cardWin, controlTag, &control) ) {

		// Get the value of the label.
		tempStr[0] = NULL;
		cs.len		= SizeOf(tempStr);
		cs.pString	= tempStr;
		ObjCallRet(msgLabelGetString, control, &cs, s);

		Deblank(tempStr);

		strncpy(str, tempStr, (size_t) strSize-1);
		str[strSize-1] = NULL;

		return stsDirtyControl;

	} else
		return stsCleanControl;

} // ReadStrControl


/***************************************************************************
 FontSizeReadControl will validate and pass back the value of a font size
 control. These controls contain a choice list whose last entry is a 
 integer write-in field.
 Three return values are possible:
	stsFailed -			the control contained an invalid number or none at all
	stsDirtyControl -	the control had been set but not cleaned
	stsCleanControl -	the control had already been cleaned
***************************************************************************/

STATUS FontSizeReadControl(
	WIN cardWin, 
	TAG controlTag, 
	TAG writeInTag,
	TAG intFieldTag,
	P_U32 pValue, 
	P_STRING pFieldName
) {

	U32 suspectValue;
	STATUS s;

	if (IsDirty(cardWin, controlTag, pNull) ||
		IsDirty(cardWin, intFieldTag, pNull) ) {

		StsRetNoWarn(
			ReadControl(
				cardWin, 
				controlTag, 
				&suspectValue, 
				0, 
				0, 
				pNull, 
				true
			),
			s
		);

		if ( suspectValue == (U32) writeInTag ) {

			StsRetNoWarn(
				ReadControl(
					cardWin, 
					intFieldTag, 
					&suspectValue, 
					minFontSize, 
					maxFontSize, 
					pFieldName, 
					true
				),
				s
			);

			Dbg(Debugf("ClockApp: FontSizeReadControl, wrote %ld",
                        suspectValue);)

		} else {
			Dbg(Debugf("ClockApp: FontSizeReadControl, selected %ld",
                        suspectValue););
		}

		*pValue = suspectValue;

		return stsDirtyControl;

	}  // if either dirty
	else
		return stsCleanControl;

} // FontSizeReadControl


/***************************************************************************
 RefreshControl sets a numeric control's value.
***************************************************************************/

STATUS RefreshControl(WIN cardWin, TAG controlTag, U32 value) {

	OBJECT control;
	STATUS s;

	Dbg(Debugf("ClockApp: in RefreshControl tag = %d, value = %ld",
		Tag(controlTag), value););

	control = (WIN) ObjectCall(msgWinFindTag, cardWin, (P_ARGS) controlTag);
	if (control != objNull) {
		ObjCallRet(msgControlSetValue, control, (P_ARGS) value, s);
		ObjCallRet(msgControlSetDirty, control, (P_ARGS) (U32) false, s);
	} else {
		Dbg(Debugf("ClockApp: could not find tag in RefreshControl()"););
	}

	return stsOK;

}   // RefreshControl


/***************************************************************************
 RefreshStrControl sets a string control's value.
***************************************************************************/

STATUS RefreshStrControl(WIN cardWin, TAG controlTag, P_STRING str) {

	OBJECT control;
	STATUS s;

	Dbg(Debugf("ClockApp: in RefreshStrControl tag = %d, value = %s",
		Tag(controlTag), str););

	control = (WIN) ObjectCall(msgWinFindTag, cardWin, (P_ARGS) controlTag);
	if (control != objNull) {
		ObjCallRet(msgLabelSetString, control, (P_ARGS) str, s);
		ObjCallRet(msgControlSetDirty, control, (P_ARGS) (U32) false, s);
	} else {
		Dbg(Debugf("ClockApp: could not find tag in RefreshStrControl()"););
	}

	return stsOK;

}   // RefreshStrControl


/***************************************************************************
 ControlAlarmTimeFormat controls the visibility of the AM/PM control based
 on the system preference for a 12 or 24-hour time format.
***************************************************************************/

void ControlAlarmTimeFormat(P_CLOCK_APP_DATA pInst, OBJECT self) {

	WIN amPmControl;

	if (pInst->alarmCard != objNull) {

		amPmControl = (OBJECT)ObjectCall(msgWinFindTag, pInst->alarmCard,
                                         (P_ARGS) tagAlarmAmPm);

		if (amPmControl != objNull && 
             ((STATUS)amPmControl) != stsBadObject) {

			// make visible based on 12/24 hour preference.
			ObjCallWarn(
				msgWinSetVisible, 
				amPmControl, 
				(P_ARGS) (U32) TwelveHour()
			);
		} else {
			Dbg(Debugf("ClockApp: ControlAlarmTimeFormat could not find "
                       "tag of tagAlarmAmPm");)
		}


		// cause card to reflect new time format
  		ObjCallWarn(msgOptionRefresh, self, pNull);

	}  // if alarm card exists

} // ControlAlarmTimeFormat


/***************************************************************************
	ApplyDisplayCard
	
	scan display option card
****************************************************************************/

STATUS ApplyDisplayCard (
	P_OPTION_CARD		pArgs,
	P_CLOCK_APP_DATA pInst,
	P_BOOLEAN pAppLayout)

{
	STATUS			s;
	U32				value;
	char fontName[5];

	StsRetNoWarn(
		ReadControl(
			pArgs->win, 
			tagFont, 
			&value, 
			0, 
			0, 
			pNull, 
			false
		), 
		s
	);

	if (s == stsDirtyControl) {
		pInst->fontId = (U16) value;
		SysDcFontString( (U16) value, fontName);
		Dbg(Debugf("ClockApp: new font id is 0x%lx, \"%s\"", value, fontName);)
		SetLabelFont(pInst->timeWin, pInst->fontId);
		SetLabelFont(pInst->amPmWin, pInst->fontId);
		SetLabelFont(pInst->dateWin, pInst->fontId);
		SetLabelFont(pInst->alarmWin, pInst->fontId);
		*pAppLayout = true;
	}

	StsRetNoWarn(
		FontSizeReadControl(
			pArgs->win, 
			tagTimeSize, 
			tagTimeSizeWriteIn, 
			tagTimeSizeIntFld,
			&value,
			"Time Size"
		), 
		s
	);

	if (s == stsDirtyControl) {
		pInst->timeSize = (U8) value;
		Dbg(Debugf("ClockApp: new time size is %ld", value);)

		StsRet(SetLabelScale(pInst->timeWin, (U8) value), s);
		StsRet(SetLabelScale(pInst->amPmWin, (U8) value), s);

		if (pInst->alarmWin && (pInst->fmt & alarmOnTime)) {
			StsRet(SetLabelScale(pInst->alarmWin, (U8) value), s);
			Dbg(Debugf("ClockApp: new alarm window size is %ld", value);)
		}
		*pAppLayout = true;
	}

	StsRetNoWarn(
		FontSizeReadControl(
			pArgs->win, 
			tagDateSize, 
			tagDateSizeWriteIn, 
			tagDateSizeIntFld,
			&value, 
			"Date Size"
		),
		s
	);

	if (s == stsDirtyControl) {
		pInst->dateSize = (U8) value;
		Dbg(Debugf("ClockApp: new date size is %ld", value);)

		StsRet(SetLabelScale(pInst->dateWin, (U8) value), s);
		if (!(pInst->fmt & alarmOnTime)) {
			StsRet(SetLabelScale(pInst->alarmWin, (U8) value), s);
			Dbg(Debugf("ClockApp: new alarm window size is %ld", value);)
		}
		*pAppLayout = true;
	}

	StsRetNoWarn(
		ReadControl(
			pArgs->win, 
			tagFormat, 
			&value, 
			0,
			0,
			pNull,
			false
		),
		s
	);
	
	if (s == stsDirtyControl) {
		pInst->fmt = (U8) value;
		Dbg(Debugf("ClockApp: new format is 0x%lx", value);)
		*pAppLayout = true;
	}

	CleanControl(pArgs->win, tagFormat);
	CleanControl(pArgs->win, tagFont);
	CleanControl(pArgs->win, tagTimeSize);
	CleanControl(pArgs->win, tagTimeSizeIntFld);
	CleanControl(pArgs->win, tagDateSize);
	CleanControl(pArgs->win, tagDateSizeIntFld);

	return stsOK;

}  // ApplyDisplayCard


/****************************************************************************
	ApplyAlarmCard
	
	scan alarm option card
****************************************************************************/

STATUS ApplyAlarmCard (
	OBJECT				self,
	P_OPTION_CARD		pArgs,
	P_CLOCK_APP_DATA pInst,
	P_BOOLEAN pAppLayout)
{
	STATUS			s;
	U32				value;
	U32				tempDate;
	U32				hourLowLimit, hourHighLimit;
	BOOLEAN			reschedule;
	TIMER_ALARM_INFO alarmInfo;
	Dbg(OS_DATE_TIME dateTime;)

	reschedule = false;

	if	(IsDirty( *(P_WIN)pArgs, tagAlarmDay, pNull) || 
			IsDirty( *(P_WIN)pArgs, tagAlarmDateFld, pNull) ) {

		reschedule = true;
		StsRetNoWarn(
			ReadControl(
				pArgs->win, 
				tagAlarmDay, 
				&value, 
				0, 
				0, 
				pNull, 
				true
			),
			s
		);

		if (value == tagAlarmDateWriteIn) {
			StsRetNoWarn(
				ReadControl(
					pArgs->win, 
					tagAlarmDateFld, 
					&value, 
					0, 
					0, 
					"alarm date", 
					true
				),
				s
			);

			pInst->alarmDate     = value;
			pInst->alarmType     = timerAbsoluteDate;
		    pInst->alarmMultiple = false;
			StsRet(RefreshControl(pArgs->win, tagAlarmMultiple,
                (U32) pInst->alarmMultiple), s);
			Dbg(Debugf("ClockApp: new alarm date set to %ld", value);)
			reschedule = true;

		}  // if date write-in
		else if (value == tagAlarmEveryDay) {
			pInst->alarmType = timerEveryDay;
			reschedule = true;
			Dbg(Debugf("ClockApp: new alarm is every day");)
		}  // if every day
		else if 
			((osSunday <= (OS_DAY_OF_WEEK) value) && 
				((OS_DAY_OF_WEEK) value <= osSaturday) ){
			pInst->alarmType = timerEveryWeek;
			pInst->alarmDayOfWeek = (U8) value;
			reschedule = true;
			Dbg(Debugf("ClockApp: new alarm is on day-of-week %ld", value);)
		} // else weekly
		else {
			Dbg(Debugf("ClockApp: unknown tag in alarm day"););
		}
	}

    if (value != tagAlarmDateWriteIn) {
	    StsRetNoWarn(
		    ReadControl(
			    pArgs->win, 
			    tagAlarmMultiple, 
			    &value, 
			    0, 
			    0, 
			    pNull, 
			    false
		    ), 
		    s
	    );
	
	    if (s == stsDirtyControl) {
		    pInst->alarmMultiple = (U8) value;
		    Dbg(Debugf("ClockApp: new alarm one shot option is %ld", value);)
	    }
    }

	StsRetNoWarn(
		ReadControl(
			pArgs->win, 
			tagAlarmAmPm, 
			&value, 
			0, 
			0, 
			pNull, 
			false
		), 
		s
	);
	
	if (s == stsDirtyControl) {
		pInst->alarmIsPM = (BOOLEAN) value;
		Dbg(Debugf("ClockApp: new alarmIsPM is %ld", value);)
	}

	if (TwelveHour()) {
		hourLowLimit = 1;
		hourHighLimit = 12;
	} else {
		hourLowLimit = 0;
		hourHighLimit = 23;
	}

	StsRetNoWarn(
		ReadControl(
			pArgs->win, 
			tagAlarmHour, 
			&value, 
			hourLowLimit, 
			hourHighLimit, 
			"Hour", 
			true
		), 
	s);

	if (s == stsDirtyControl) {

		pInst->alarmHours = (U8) value;

		if (!TwelveHour()) {
			// convert to internal 12 hour format
			pInst->alarmIsPM = (pInst->alarmHours > 11);

			// maps 0 hours into 12 hours, 1..23 into 1..11, 1..11 
			pInst->alarmHours = (U8)
				((pInst->alarmHours ? pInst->alarmHours-1 : (U8) 23) % 12 + 1);
		}

		reschedule = true;
		Dbg(Debugf("ClockApp: new alarm hour is %d, IsPM =%d",
			pInst->alarmHours, pInst->alarmIsPM);)
	}

	StsRetNoWarn(
		ReadControl(
			pArgs->win, 
			tagAlarmMinutes, 
			&value, 
			0, 
			59, 
			"Minutes", 
			false
		), 
		s
	);
	
	if (s == stsDirtyControl) {
		pInst->alarmMinutes = (U8) value;
		reschedule = true;
		Dbg(Debugf("ClockApp: new alarm minute is %ld", value);)
	}

	StsRetNoWarn(
		ReadControl(
			pArgs->win, 
			tagAlarmBeep, 
			&value, 
			0, 
			0, 
			pNull, 
			false
		), 
		s
	);
	
	if (s == stsDirtyControl) {
		pInst->alarmBeep = (BOOLEAN) value;
		Dbg(Debugf("ClockApp: new alarm beep is %ld", value);)
	}

	StsRetNoWarn(
		ReadControl(
			pArgs->win, 
			tagAlarmTimeout, 
			&value, 
			0, 
			0, 
			pNull, 
			false
		), 
		s
	);
	
	if (s == stsDirtyControl) {
		pInst->alarmTimeout = (U8) value;
		Dbg(Debugf("ClockApp: new alarm timeout option is %ld", value);)
	}


	StsRetNoWarn(
		ReadStrControl(
			pArgs->win, 
			tagAlarmMessage, 
			pInst->alarmMessage, 
			SizeOf(pInst->alarmMessage), 
			self
		), 
		s
	);
#ifdef DEBUG
	if (s == stsDirtyControl) {
		Debugf("ClockApp: new alarm message is \"%s\"", 
			pInst->alarmMessage);
	}
#endif

	StsRetNoWarn(
		ReadControl(
			pArgs->win, 
			tagAlarmSnoozeEnable, 
			&value, 
			0, 
			0, 
			pNull, 
			false
		), 
		s
	);
	
	if (s == stsDirtyControl) {
		pInst->alarmSnoozeEnable = (U8) value;
		Dbg(Debugf("ClockApp: new alarm snooze enable value is %ld", value);)
	}

	StsRetNoWarn(
		ReadControl(
			pArgs->win, 
			tagAlarmSnooze, 
			&value, 
			2, 
			99, 
			"Repeat delay", 
			false
		), 
		s
	);
	
	if (s == stsDirtyControl) {
		pInst->alarmSnooze = (U8) value;
		Dbg(Debugf("ClockApp: new alarm snooze value is %ld", value);)
	}

	StsRetNoWarn(
		ReadControl(
			pArgs->win, 
			tagAlarmOn, 
			&value, 
			0, 
			0, 
			pNull, 
			false
		), 
		s
	);
	
	if (s == stsDirtyControl) {
		pInst->alarmOn = (BOOLEAN) value;
		reschedule = true;
        // set *pAppLayout here because: if user has turned the control
        // from on to off then the pInst->alarmOn branch (which sets
        // *pAppLayout) won't be taken below.  But we want to repaint
        // the clock anyhow to remove the alarm icon.
		*pAppLayout = true;
		Dbg(Debugf("ClockApp: new alarm enable is %ld", value);)
	}

	if (reschedule) {
		
		CancelAlarm(&pInst->alarmHandle);  // cancel any pending alarms

		if (pInst->alarmOn) {

#ifdef DEBUG
			OSGetTime(SizeOf(OS_DATE_TIME), &dateTime);
					  
			Debugf("ClockApp: Time is %02d:%02d:%02d",
				dateTime.hours,
				dateTime.minutes,
				dateTime.seconds
			);
#endif

			alarmInfo.alarmMode = pInst->alarmType;

            switch (pInst->alarmType) {

                case timerEveryDay:
    				Dbg(Debugf("ClockApp: alarm day is every day");)
                    break;

                case timerAbsoluteDate:
				    tempDate = pInst->alarmDate;
				    Dbg(Debugf("ClockApp: alarm day is %ld", tempDate);)
                    // day is [1,31], actual day of month
				    alarmInfo.alarmTime.day   = tempDate % 100;
				    tempDate /= 100;
                    // month is [0,11], offset from January
				    alarmInfo.alarmTime.month = tempDate % 100 - 1;
				    tempDate /= 100;
                    // year is [0,maxU32], offset from 1900 A.D.
                    // so we must strip off the century
                    // (clsDateField assumes 20th century for PenPoint 1.0)
				    alarmInfo.alarmTime.year  = tempDate % 100;
                    break;

                case timerEveryWeek:
    				Dbg(Debugf("ClockApp: alarm day-of-week is %d", \
                                pInst->alarmDayOfWeek);)
	    			alarmInfo.alarmTime.dayOfWeek = pInst->alarmDayOfWeek;
                    break;

                default:
    				Dbg(Debugf( \
                        "ClockApp: unknown alarm type in ApplyAlarmCard");)
                    break;
			}

			alarmInfo.alarmTime.hours = pInst->alarmHours;

			// alarm requires 24 hour clock
			if (alarmInfo.alarmTime.hours == 12)
				alarmInfo.alarmTime.hours = 0;
			if (pInst->alarmIsPM) 
				alarmInfo.alarmTime.hours += 12;

			alarmInfo.alarmTime.minutes = pInst->alarmMinutes;
			alarmInfo.alarmTime.seconds = 0;

			alarmInfo.client = self;

			Dbg(Debugf("ClockApp: setting alarm for %02d:%02d:%02d",
				alarmInfo.alarmTime.hours,
				alarmInfo.alarmTime.minutes,
				alarmInfo.alarmTime.seconds
			);)

			s = ObjectCall(msgTimerAlarmRegister, clsTimer, &alarmInfo);

			if (s >= stsOK) {
				pInst->alarmHandle = alarmInfo.transactionHandle;
				pInst->noteHours = alarmInfo.alarmTime.hours;
				pInst->noteMinutes = pInst->alarmMinutes;
			} else {
				StdMsg(stsClockAlarmInvalid);
				return stsFailed;
			} // else alarm register failed

			*pAppLayout = true;

		} // if alarm now on

	}  // if reschedule

	Dbg(Debugf("ClockApp: cleaning controls in ApplyAlarmCard");)
	CleanControl(pArgs->win, tagAlarmDay);
	CleanControl(pArgs->win, tagAlarmDateFld);
	CleanControl(pArgs->win, tagAlarmHour);
	CleanControl(pArgs->win, tagAlarmMinutes);
	CleanControl(pArgs->win, tagAlarmAmPm);
	CleanControl(pArgs->win, tagAlarmBeep);
	CleanControl(pArgs->win, tagAlarmTimeout);
	CleanControl(pArgs->win, tagAlarmOn);
	CleanControl(pArgs->win, tagAlarmMultiple);
	CleanControl(pArgs->win, tagAlarmMessage);
	CleanControl(pArgs->win, tagAlarmSnooze);
	CleanControl(pArgs->win, tagAlarmSnoozeEnable);
	Dbg(Debugf("ClockApp: done cleaning controls in ApplyAlarmCard");)

	return stsOK;

}  // ApplyAlarmCard


/****************************************************************************
	ClockOptionApplyCard
	
	Respond to msgOptionApplyCard.
****************************************************************************/

MSG_HANDLER ClockOptionApplyCard (
	const MESSAGE				msg,
	const OBJECT				self,
	const P_OPTION_CARD		pArgs,
	const CONTEXT				ctx,
	const PP_CLOCK_APP_DATA pData)
{
	STATUS			s;
	BOOLEAN			appLayout;

	P_CLOCK_APP_DATA	pInst;

	Dbg(Debugf("ClockApp: entering ClockOptionApplyCard");)

	pInst = *pData;

	appLayout = false;

	switch (pArgs->tag) {
	
		case tagClkAppDisplayCard:
			StsRet(ApplyDisplayCard(pArgs, pInst, &appLayout), s);
			break;

		case tagClkAppAlarmCard:
			StsRet(ApplyAlarmCard(self, pArgs, pInst, &appLayout), s);
			break;

		default:
			return ObjectCallAncestor(msg, self, pArgs, ctx);

	} // switch on card tag

	if (appLayout) {

		// rebuild clock window(s)

		Dbg(Debugf("ClockApp: rebuilding clock window(s)");)
		CreateClockWindow(pInst, self);

	}

	return stsOK;

	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

}  //  ClockOptionApplyCard


/****************************************************************************
	ClockAddCards
	
	Respond to msgOptionAddCards.
****************************************************************************/

MSG_HANDLER ClockAddCards (
	const MESSAGE				msg,
	const OBJECT				self,
	const P_OPTION_TAG		pArgs,
	const CONTEXT				ctx,
	const PP_CLOCK_APP_DATA pData)
{
	STATUS				s;
	OPTION_CARD			card;
	OPTION_STYLE		optionStyle;

	Dbg(Debugf("ClockApp: entering ClockAddCards");)

	if (pArgs->tag == tagAppDocOptSheet) {

		//
		// disable select sensing of sheet
		//
		ObjCallRet(msgOptionGetStyle, pArgs->option, &optionStyle, s);
		optionStyle.senseSelection = false;
		ObjCallRet(msgOptionSetStyle, pArgs->option, &optionStyle, s);

		// Add the display card to the option sheet.
		memset(&card, 0, sizeof(OPTION_CARD));
		card.tag = tagClkAppDisplayCard;
		card.win = objNull;
		card.pName = "Display";
		card.client = self;
		ObjCallRet(msgOptionAddCard, pArgs->option, &card, s);

		// Add the alarm card to the option sheet.
		memset(&card, 0, sizeof(OPTION_CARD));
		card.tag = tagClkAppAlarmCard;
		card.win = objNull;
		card.pName = "Alarm";
		card.client = self;
		ObjCallRet(msgOptionAddLastCard, pArgs->option, &card, s);

	}  // if app option sheet

	return ObjCallAncestorWarn(msg, self, pArgs, ctx);

	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

}  // ClockAddCards

/****************************************************************************
	ClockProvideCard
	
	Respond to msgOptionProvideCard.
****************************************************************************/

MSG_HANDLER ClockProvideCard (
	const MESSAGE				msg,
	const OBJECT				self,
	const P_OPTION_CARD			pArgs,
	const CONTEXT				ctx,
	const PP_CLOCK_APP_DATA pData)
{
	STATUS				s;
	OPTION_TABLE_NEW	otn;
	BORDER_STYLE		bs;
	WIN					label, msgField;
	LABEL_STYLE			labelStyle;
	MENU_BUTTON_STYLE	mButtonStyle;
	GWIN_STYLE			gws;

	P_CLOCK_APP_DATA	pInst;

	Dbg(Debugf("ClockApp: entering ClockProvideCard");)

	pInst = *pData;

	if(pArgs->tag == tagClkAppDisplayCard) {

		// Create the display card.
		ObjCallWarn(msgNewDefaults, clsOptionTable, &otn);
		otn.tkTable.pEntries = clockDisplayCardEntries;
		otn.gWin.helpId = tagClkAppDisplayCard;
		otn.win.tag = tagClkAppDisplayCard;
		ObjCallRet(msgNew, clsOptionTable, &otn, s);

		// track down the date font size popup
		label = (OBJECT)ObjectCall(msgWinFindTag, otn.object.uid, 
                                   (P_ARGS) tagDateSize);

		if (label != objNull) {

			// modify the label so that the displayed write-in does not clip
			ObjCallRet(msgLabelGetStyle, label, (P_ARGS) &labelStyle, s);
			labelStyle.numCols = lsNumAbsolute;
			ObjCallRet(msgLabelSetStyle, label, (P_ARGS) &labelStyle, s);
			ObjCallRet(msgLabelSetCols, label, (P_ARGS) (U32) 6, s);  //in ems

		} else {
			Dbg(Debugf("ClockApp: could not find tagDateSize");)
		}

		// track down the time font popup
		label = (OBJECT)ObjectCall(msgWinFindTag, otn.object.uid,
                                   (P_ARGS) tagTimeSize);

		if (label != objNull) {

			// modify the label so that the displayed write-in does not clip
			ObjCallRet(msgLabelGetStyle, label, (P_ARGS) &labelStyle, s);
			labelStyle.numCols = lsNumAbsolute;
			ObjCallRet(msgLabelSetStyle, label, (P_ARGS) &labelStyle, s);
			ObjCallRet(msgLabelSetCols, label, (P_ARGS) (U32) 6, s);  //in ems

		} else {
			Dbg(Debugf("ClockApp: could not find tagTimeSize");)
		}

		pArgs->win = otn.object.uid;

	} else if(pArgs->tag == tagClkAppAlarmCard) {

		// Create the alarm card.
		ObjCallWarn(msgNewDefaults, clsOptionTable, &otn);
		otn.tkTable.pEntries = clockAlarmCardEntries;
		otn.gWin.helpId = tagClkAppAlarmCard;
		otn.win.tag = tagClkAppAlarmCard;
		ObjCallRet(msgNew, clsOptionTable, &otn, s);
		pInst->alarmCard = otn.object.uid;

		// track down the alarm day popup
		label = (OBJECT)ObjectCall(msgWinFindTag, otn.object.uid,
                                   (P_ARGS) tagAlarmDay);

		if (label != objNull) {

			// force label to size of widest value : Wednesday (9)
			// otherwise the size of the date field AS CHAR BOXES is used

			ObjCallRet(msgLabelGetStyle, label, (P_ARGS) &labelStyle, s);
			labelStyle.numCols = lsNumAbsolute;
			ObjCallRet(msgLabelSetStyle, label, (P_ARGS) &labelStyle, s);
			ObjCallRet(msgLabelSetCols, label, (P_ARGS) (U32) 9, s); //in ems

			ObjCallRet(msgMenuButtonGetStyle, label, (P_ARGS)&mButtonStyle, s);
			mButtonStyle.getWidth = false;
			ObjCallRet(msgMenuButtonSetStyle, label, (P_ARGS)&mButtonStyle, s);

		} else {
			Dbg(Debugf("ClockApp: could not find tagAlarmDay");)
		}

		// Reset the field/gwin metrics for the "Message:" field to get
		// better handwriting recognition.
		msgField = (OBJECT)ObjectCall(msgWinFindTag, otn.object.uid,
                                      (P_ARGS)tagAlarmMessage);
		ObjCallWarn(msgGWinGetStyle, msgField, &gws);
		gws.firstEnter	= true;
		ObjCallWarn(msgGWinSetStyle, msgField, &gws);

		// add the cork board to the option card
		ObjCallRet(msgBorderGetStyle, pInst->corkboard, &bs, s);
		bs.edge = bsEdgeAll;
		ObjCallRet(msgBorderSetStyle, pInst->corkboard, &bs, s);

		ObjCallRet(msgTkTableAddAsLast, pInst->alarmCard, 
                   (P_ARGS) pInst->corkboard, s);
		pInst->corkIn = corkInCard;

		pArgs->win = pInst->alarmCard;

        // control visibility of am/pm write in:
		ControlAlarmTimeFormat(pInst, self);
	}  // if app option sheet
	else
		ObjCallAncestorCtxWarn(ctx);

	return stsOK;

	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

}  // ClockAddCard

/****************************************************************************
	ClockRefreshCard
	
	Respond to msgOptionRefreshCard.
****************************************************************************/

MSG_HANDLER ClockRefreshCard (
	const MESSAGE				msg,
	const OBJECT				self,
	const P_OPTION_CARD		pArgs,
	const CONTEXT				ctx,
	const PP_CLOCK_APP_DATA pData)
{
	STATUS 	s;
	U32		alarmTypeValue;
	U8			alarmHours;

	P_CLOCK_APP_DATA	pInst;

	Dbg(Debugf("ClockApp: ClockRefreshCard called with tag %ld", pArgs->tag);)

	pInst = *pData;

	switch (pArgs->tag) {
	
		case tagClkAppDisplayCard:

			StsRet(RefreshControl(pArgs->win, tagFormat, (U32) pInst->fmt), s);

			StsRet(RefreshControl(pArgs->win, tagFont, (U32) pInst->fontId), s);

			StsRet(RefreshControl(pArgs->win, tagTimeSize,
                                  (U32) pInst->timeSize), s);
			StsRet(RefreshControl(pArgs->win, tagDateSize,
                                  (U32) pInst->dateSize), s);
			return stsOK;

		case tagAppOptAccessCard:  // intercept the Access card.

			Dbg(Debugf("ClockApp: Destroying unneeded controls on Access card");)

			// Get the EDITING label.
			DestroyControl(pArgs->win, tagAppOptReadOnlyLabel);

			// Get the EDITING control.
			DestroyControl(pArgs->win, tagAppOptReadOnly);

			// Get the HOT MODE label.
			DestroyControl(pArgs->win, tagAppOptHotModeLabel);

			// Get the HOT MODE control.
			DestroyControl(pArgs->win, tagAppOptHotMode);

			break;

		case tagAppOptControlsCard:  // intercept the Controls card.

			Dbg(Debugf("ClockApp: Destroying unneeded controls on Controls card");)

			// Destroy the MENU BAR control.
			DestroyControl(pArgs->win, tagAppOptCtrlMenuBar);
		
			// Destroy the SCROLL MARGINS control.
			DestroyControl(pArgs->win, tagAppOptCtrlScrollBars);
		
			break;

		case tagClkAppAlarmCard:

			if (pInst->alarmType == timerEveryDay)
				alarmTypeValue = tagAlarmEveryDay;
			else if (pInst->alarmType == timerEveryWeek)
				alarmTypeValue = pInst->alarmDayOfWeek;
			else if (pInst->alarmType == timerAbsoluteDate)
				alarmTypeValue = tagAlarmDateWriteIn;
			else {
				Dbg(Debugf("ClockApp: bad alarm type in ClockRefreshCard"););
			}

			StsRet(RefreshControl(pArgs->win, tagAlarmDay,
                                  (U32) alarmTypeValue), s);

			if (pInst->alarmDate != noDate) {
				StsRet(RefreshControl(pArgs->win, tagAlarmDateFld,
                                      (U32) pInst->alarmDate), s);
			}

			alarmHours = pInst->alarmHours;
			
			if (!TwelveHour()) {
				// convert to 24 hour clock
				if (alarmHours == 12)
					alarmHours = 0;
				if (pInst->alarmIsPM) 
					alarmHours += 12;
			}

			StsRet(RefreshControl(pArgs->win, tagAlarmHour,
                                  (U32) alarmHours), s);

			StsRet(RefreshControl(pArgs->win, tagAlarmMinutes,
                                  (U32) pInst->alarmMinutes), s);

			StsRet(RefreshControl(pArgs->win, tagAlarmAmPm,
                                  (U32) pInst->alarmIsPM), s);

			StsRet(RefreshControl(pArgs->win, tagAlarmBeep,
                                  (U32) pInst->alarmBeep), s);

			StsRet(RefreshControl(pArgs->win, tagAlarmTimeout,
                                  (U32) pInst->alarmTimeout), s);

			StsRet(RefreshControl(pArgs->win, tagAlarmMultiple,
                                  (U32) pInst->alarmMultiple), s);

			StsRet(RefreshControl(pArgs->win, tagAlarmSnooze,
                                  (U32) pInst->alarmSnooze), s);

			StsRet(RefreshControl(pArgs->win, tagAlarmSnoozeEnable,
                                  (U32) pInst->alarmSnoozeEnable), s);

			StsRet(RefreshControl(pArgs->win, tagAlarmOn,
                                  (U32) pInst->alarmOn), s);

			StsRet(RefreshStrControl(pArgs->win, tagAlarmMessage,
                                  pInst->alarmMessage), s);
			break;

	}  // switch on card tag

	// Let clsApp prepare the card.
	return ObjectCallAncestor(msg, self, pArgs, ctx);

	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

}  // ClockRefreshCard


/****************************************************************************
	ClockGetAppWin
	
	Handle msgAppGetAppWin. This allows a move out of the corkboard to find
	the app's window.
****************************************************************************/

MSG_HANDLER ClockGetAppWin (
	const MESSAGE				msg,
	const OBJECT				self,
	const P_APP_GET_APP_WIN	pArgs,
	const CONTEXT				ctx,
	const PP_CLOCK_APP_DATA pData)

{
	APP_WIN_METRICS	awm;
	OBJECT				win[10];
	WIN_ENUM   			we;
	BOOLEAN				match;
	U16					i;
	STATUS				s;

	P_CLOCK_APP_DATA	pInst;

	Dbg(Debugf("ClockApp: ClockGetAppWin entered");)

	pInst = *pData;

	// pArgs->win = objNull;		

	// if ancestor found a match or we have no cork the return stsOK
	if ( (pArgs->win != objNull) || (pInst->corkboard == objNull) ) {

		return stsOK;
	}

	// Recursively enum children of the main win.
	we.max		= 10;
	we.count	= maxU16;
	we.pWin		= win;
	we.pFlags	= pNull;
	we.flags	= wsEnumChildren | wsEnumRecursive | wsEnumSelf;
	we.next		= 0;
	s = ObjectCall(msgWinEnum, pInst->corkboard, &we);

	// Errors?
	if (s != stsOK AND s != stsEndOfData) {

		StsPrint(s);
		return s;
	}

	Dbg(Debugf("ClockApp: num child windows = %d", we.count);)

	// Search the app wins for a matching uuid.
	for (i = 0, match = false; i < we.count; i++) {

		// Is this window of clsAppWin?
		s = ObjectCall(msgIsA, we.pWin[i], (P_ARGS)clsAppWin);

		if (s == stsOK) {

			// Get this embedded win's uuid.
			s = ObjectCall(msgAppWinGetMetrics, we.pWin[i], &awm);
			if (s != stsOK) {

				continue;
			}

			// Matching uuid?
			if (SameUUIDs(awm.appUUID, pArgs->uuid)) {

				match = true;

				break;
			}
		}
	}

	// Return the app win.
	pArgs->win = match ? we.pWin[i] : objNull;

	// Free the enum buffer.
	if (we.pWin != (P_OBJECT)win) {

		StsWarn(OSHeapBlockFree(we.pWin));
	}

	return stsOK;


	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

}	// ClockGetAppWin

/****************************************************************************
	ClockUPrefsChanged
	
	Respond to msgPrefsPreferenceChanged.  Causes relayout of clock
	if the date or time format has changed.
****************************************************************************/

MSG_HANDLER ClockPrefsChanged(
	const MESSAGE	msg,
	const OBJECT	self,
	const P_PREF_CHANGED	pArgs,
	const CONTEXT	ctx,
	const PP_CLOCK_APP_DATA pData)

{
	BOOLEAN reschedule, changeAlarmAmPm;
	STATUS s;

	P_CLOCK_APP_DATA	pInst;

	Dbg(Debugf("ClockApp: ClockPrefsChanged, prefId = %lx", pArgs->prefId));

	pInst = *pData;

	if (
		PrefChanged(Time) ||
		(changeAlarmAmPm = PrefChanged(TimeFormat)) ||
		(reschedule = PrefChanged(TimeSeconds)) ||
		PrefChanged(DateFormat)
	) {
		
		if (reschedule) {
			//
			// Reschedule tick message
			//

			Dbg(Debugf("ClockApp: system pref on Seconds changed, rescheduling");)
			StsRet(ScheduleTick(pInst, self), s);
		}

		if (changeAlarmAmPm) {
			Dbg(Debugf("ClockApp: time format pref changed, changing alarm format");)
			ControlAlarmTimeFormat(pInst, self);
		}

		Dbg(Debugf("ClockApp: system prefs changed, re-laying out clock");)
		StsRet(CreateClockWindow(pInst, self), s);

	}

	return stsOK;


	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} /* ClockPrefsChanged */


/****************************************************************************
	ClockPowerUp
	
	Respond to msgPBMachinePoweringUp.  Causes refresh of clock display.
****************************************************************************/

MSG_HANDLER ClockPowerUp(
	const MESSAGE	msg,
	const OBJECT	self,
	const P_UNKNOWN pArgs,
	const CONTEXT	ctx,
	const PP_CLOCK_APP_DATA pData)

{
	P_CLOCK_APP_DATA	pInst;

	Dbg(Debugf("ClockApp: ClockPowerUp");)

	pInst = *pData;

	DisplayTime(pInst);

	return stsOK;


	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} // ClockPowerUp


/***************************************************************************
 RemoveCorkFromNote extracts the corkboard from an alarm note and intserts
 it back into the alarm option card.
***************************************************************************/

STATUS RemoveCorkFromNote(P_CLOCK_APP_DATA pInst) {

	BORDER_STYLE bs;
	WIN_METRICS wm;
	STATUS s;

	if (pInst->corkIn == corkInNote) {

		ObjCallRet(msgWinExtract, pInst->corkboard, &wm, s);
		pInst->corkIn = corkNotInserted;
		Dbg(Debugf("ClockApp: extracted corkboard from note");)

//		wm.flags.style &= ~wsShrinkWrapWidth;
//		ObjCallRet(msgWinSetFlags, pInst->corkboard, &wm, s);

//		wm.bounds.size.w = LayoutUnitsToPixels(110);  // pixels
//		ObjCallRet(msgWinDelta, pInst->corkboard, &wm, s);

		ObjCallRet(msgBorderGetStyle, pInst->corkboard, &bs, s);
		bs.edge = bsEdgeAll;
		ObjCallRet(msgBorderSetStyle, pInst->corkboard, &bs, s);

		ObjCallRet(msgTkTableAddAsLast, pInst->alarmCard,
                    (P_ARGS) pInst->corkboard, s);
		pInst->corkIn = corkInCard;

		ObjCallRet(msgWinSetLayoutDirty, pInst->corkboard,
                    (P_ARGS) (U32) true, s);

		wm.options = wsLayoutDefault;
		ObjCallRet(msgWinLayout, pInst->corkboard, &wm, s);

		Dbg(Debugf("ClockApp: returned corkboard to alarm card");)
	}

}  // RemoveCorkFromNote


/****************************************************************************
	ClockNoteDone
	
	Respond to msgNoteDone.
****************************************************************************/

MSG_HANDLER ClockNoteDone(
	const MESSAGE	msg,
	const OBJECT	self,
	const P_UNKNOWN pArgs,  // MESSAGE
	const CONTEXT	ctx,
	const PP_CLOCK_APP_DATA pData)

{

	STATUS s;
	TIMER_ALARM_INFO alarmInfo;

	P_CLOCK_APP_DATA	pInst;

	Dbg(Debugf("ClockApp: msgNoteDone");)

	pInst = *pData;

	switch ( (MESSAGE) pArgs ) {

		case msgClockNoteAccept:

			Dbg(Debugf("ClockApp: accept of misc. note");)

			break;

		case msgClockAlarmOK:

			Dbg(Debugf("ClockApp: OK from alarm note");)

			pInst->alarmNote = objNull;

			RemoveCorkFromNote(pInst);

			if (!pInst->alarmMultiple) {
				pInst->alarmHandle = NULL;
				pInst->alarmOn = false;

				// cause posibly visible card to update "alarm on" control
				ObjCallWarn(msgOptionRefresh, self, pNull);

				// regenerate clock windows to remove alarm window
				StsRet(CreateClockWindow(pInst, self), s);

				Dbg(Debugf("ClockApp: completed non-multiple alarm");)
			}
			break;

		case msgClockAlarmPostpone:

			pInst->alarmNote = objNull;

			RemoveCorkFromNote(pInst);

			OSGetTime (SizeOf(OS_DATE_TIME), &alarmInfo.alarmTime);
					   
			Dbg(Debugf(
                "ClockApp: postpone alarm from %02d:%02d:%02d by %d minutes",
				alarmInfo.alarmTime.hours,
				alarmInfo.alarmTime.minutes,
				alarmInfo.alarmTime.seconds,
				pInst->alarmSnooze
			););

			alarmInfo.alarmMode = timerEveryDay;
			alarmInfo.client = self;
			alarmInfo.clientData = pNull;

			DateTimeAdd(&alarmInfo.alarmTime, (time_t)pInst->alarmSnooze * 60L);
			pInst->noteMinutes = alarmInfo.alarmTime.minutes;
			pInst->noteHours = alarmInfo.alarmTime.hours;

			ObjCallRet(msgTimerAlarmRegister, clsTimer, &alarmInfo, s);
			pInst->alarmHandle = alarmInfo.transactionHandle;

			Dbg(Debugf("ClockApp:   postpone alarm to %02d:%02d:%02d",
				alarmInfo.alarmTime.hours,
				alarmInfo.alarmTime.minutes,
				alarmInfo.alarmTime.seconds
			););

	}  // switch on message

	CloseOpenCorkboardDocs(pInst);

	return stsOK;


	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} /* ClockNoteDone */


/****************************************************************************
	ClockUpdateDisplay
	
	Respond to msgTimerNotify.  Refresh display.
****************************************************************************/

MSG_HANDLER ClockUpdateDisplay(
	const MESSAGE	msg,
	const OBJECT	self,
	const P_TIMER_NOTIFY	pArgs,
	const CONTEXT	ctx,
	const PP_CLOCK_APP_DATA pData)

{
	STATUS s;

	P_CLOCK_APP_DATA	pInst;

	pInst = *pData;

	if (pInst->clockWin) {
		if (pInst->tickHandle == pArgs->transactionHandle) {
			DisplayTime(pInst);
			if ( pInst->oneShotTick ) {
				pInst->tickHandle = NULL;  // clsTimer has forgotten this handle
				// ask for a tick in a minute
				StsRet(ScheduleTick(pInst, self), s);
			}  // if oneShotTick
		}
		else {
			Dbg(Debugf(
                "ClockUpdateDisplay: Timer notify with unknown handle %ld",
				pArgs->transactionHandle););
		}
	}
	else {
		Dbg(Debugf("ClockUpdateDisplay: Ignoring latent tick after app. close");)
	}

	return stsOK;


	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} /* ClockUpdateDisplay */

/****************************************************************************
	Beep

	This routine beeps like a wristwatch alarm.
****************************************************************************/
STATUS Beep (
	OBJECT				self,
	P_CLOCK_APP_DATA	pInst)
{
    U16                 i;
    OS_TASK_ID          thisTask;
	OS_PRIORITY_CLASS	currentPriorityClass;
	U8					currentPriority;

    thisTask = OSThisTask();
    Dbg(Debugf("ClockApp Beep %4x begining", thisTask);)

	// Stop the timer notifications since they affect the timing of the beeps.
	CancelTick(&pInst->tickHandle);

    // bump up this task's priority temporarily so it won't be interrupted
	// while beeping.
	OSTaskPriorityGet(thisTask, &currentPriorityClass, &currentPriority);
	OSTaskPrioritySet(thisTask, osThisTaskOnly, osHighPriority, 3);

    // the following loop will give three spaced, high-pitched 
	// "beep-beep" tones, similar to what a digital wristwatch does.
    for (i = 0; i < 3; i++) {
        OSTone(2000, 100, 1);	// Beep for 100 msecs
		OSTaskDelay(100);		// Wait 100 msecs
        OSTone(2000, 100, 1);	// Beep for 100 msecs
		if (i < 2)
			OSTaskDelay(500);	// Wait 500 msecs
    }

	// Start up the timer notifications.
	ScheduleTick(pInst, self);

	// Reset the task priority to normal.
	OSTaskPrioritySet(thisTask, osThisTaskOnly, 
        currentPriorityClass, currentPriority);

	return stsOK;
}


/****************************************************************************
	ClockAlarmNotify
	
	Respond to msgTimerAlarmNotify.  Display Alarm Note.
****************************************************************************/

MSG_HANDLER ClockAlarmNotify(
	const MESSAGE			msg,
	const OBJECT			self,
	const P_TIMER_NOTIFY	pArgs,
	const CONTEXT			ctx,
	const PP_CLOCK_APP_DATA pData)

{
	STATUS				s;
	NOTE_NEW			nn;
	WIN_METRICS			wm;
    BORDER_STYLE        bordStyle;
	TK_TABLE_ENTRY 		messageTable[2];
	TK_TABLE_ENTRY 		controlTable[7];
	U8 					nextControl;
	U16 				corkPop;
	CHAR				postponeStr[30];  // "Repeat in N minutes"
	CHAR				noteStr[100];
    OS_DATE_TIME        dateTime;
	P_CLOCK_APP_DATA	pInst;

	pInst = *pData;

	if (pInst->alarmHandle == pArgs->transactionHandle) {

		Dbg(Debugf("ClockApp: Alarm notify");)

		ObjCallRet(msgNewDefaults, clsNote, &nn, s);

        dateTime.hours   = (U32)pInst->noteHours;
		dateTime.minutes = (U32)pInst->noteMinutes;
		dateTime.seconds = 0;
    	PrefsTimeToString (&dateTime, noteStr);

		if (pInst->alarmMessage[0]) {
			strcat(noteStr, " - ");
			strcat(noteStr, pInst->alarmMessage);
		} else {
			strcat(noteStr, " - Reminder.");
		}

		//
		// Set up a tkTable for the reminder note and corkboard window
		//
		memset(messageTable, 0, sizeof(messageTable));
		messageTable[0].arg1		= noteStr;
		messageTable[0].helpId		= hlpClkAppNoteMess;
		nn.note.pContentEntries		= messageTable;

		//
		// Set up a tkTable for the command bar controls (Repeat and
		// Dismiss).
		//
		nextControl = 0;
		memset(controlTable, 0, sizeof(controlTable));

		controlTable[nextControl].childClass	= clsCommandBar;
		controlTable[nextControl].tag			= tagAlarmNoteButtons;
		controlTable[nextControl++].helpId		= hlpClkAppNoteButtons;

		if (pInst->alarmSnoozeEnable) {
			sprintf(postponeStr, "Repeat: %d %s", pInst->alarmSnooze,
				(pInst->alarmSnooze == 1) ? "minute" : "minutes");

			controlTable[nextControl].arg1		= postponeStr;
			controlTable[nextControl].arg2		= msgClockAlarmPostpone;
			controlTable[nextControl++].helpId	= hlpClkAppNoteSnooze;
		}

		controlTable[nextControl].arg1		= "Dismiss";
		controlTable[nextControl].arg2		= msgClockAlarmOK;
		controlTable[nextControl++].helpId	= hlpClkAppNoteDismiss;

		nn.note.pCmdBarEntries			= controlTable;
		nn.note.metrics.client			= self;
		nn.note.metrics.flags			= nfAppTitle;
		nn.note.metrics.autoDismissMsg	= msgClockAlarmOK;	// treat like OK button
		nn.note.pTitle					= pNull;
		if (pInst->alarmTimeout) {
			nn.note.metrics.flags		|= nfTimeout;
			nn.note.metrics.timeout		= alarmNoteTimeout;	// in milliseconds
		} 

		//
		// Create the note window.
		//
		ObjCallRet(msgNew, clsNote, &nn, s);
		pInst->alarmNote = nn.object.uid;

		//
		// Add the corkboard window into the message table if there are
		// any documents in it.
		//
		corkPop = ChildCount(pInst->corkboard);

		if (corkPop) {
			WIN	msgTableWin;

			if (pInst->corkIn == corkInCard) {
				ObjCallRet(msgWinExtract, pInst->corkboard, pNull, s);
				Dbg(Debugf("ClockApp: extracted corkboard");)
			}

			msgTableWin = (OBJECT)ObjectCall(msgWinFindTag, nn.object.uid, 
                                             (P_ARGS)tagNoteTkTable);
			ObjCallRet(msgTkTableAddAsLast, msgTableWin, 
                        (P_ARGS)pInst->corkboard, s);

            // turn off corkboard's border for display inside note
            ObjCallWarn(msgBorderGetStyle, pInst->corkboard, &bordStyle);
            bordStyle.edge = bsEdgeNone;
            ObjCallWarn(msgBorderSetStyle, pInst->corkboard, &bordStyle);

			ObjCallWarn(msgWinSetLayoutDirtyRecursive, pInst->alarmNote, 
                        (P_ARGS)true);
			pInst->corkIn = corkInNote;

		 	Dbg(Debugf("ClockApp: inserted corkboard into note");)
		} else {
			Dbg(Debugf("ClockApp: corkboard is empty"););
        }

		//
		// Layout the note window.
		//
		wm.options	= wsLayoutResize;
		ObjCallRet(msgWinLayout, pInst->alarmNote, &wm, s);
			
		//
		// Pop up the alarm note.
		//
		ObjCallWarn(msgClockDisplayNote, self, (P_ARGS) pInst->alarmNote);

		//
		// If we're set up to beep, do so.
		//
		if (pInst->alarmBeep)
			StsWarn(Beep(self, pInst));
	}
	else
		Dbg(Debugf("ClockApp: Timer alarm notify with unknown handle");)

	return stsOK;


	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} /* ClockAlarmNotify */


/****************************************************************************
	ClockDisplayNote
	
	Respond to msgClockDisplayNote.  Display Alarm Note.
****************************************************************************/

MSG_HANDLER ClockDisplayNote(
	const MESSAGE	msg,
	const OBJECT	self,
	const P_UNKNOWN pArgs,  // OBJECT, the note's uid
	const CONTEXT	ctx,
	const PP_CLOCK_APP_DATA pData)

{
	STATUS s;
	P_CLOCK_APP_DATA	pInst;

	Dbg(Debugf("ClockApp: display clock note");)

	pInst = *pData;

	ObjCallRet(msgNoteShow, pArgs, pNull, s);

	return stsOK;

	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} /* ClockDisplayNote */

/****************************************************************************
	ClockNewFormat
	
	Respond to msgClockNewFormat.  Layout clock in new format.
****************************************************************************/

MSG_HANDLER ClockNewFormat(
	const MESSAGE	msg,
	const OBJECT	self,
	const P_UNKNOWN pArgs,  // U8 - the new format
	const CONTEXT	ctx,
	const PP_CLOCK_APP_DATA pData)

{
	P_CLOCK_APP_DATA	pInst;

	Dbg(Debugf("ClockApp: msgClockNewFormat");)

	pInst = *pData;

	pInst->fmt = (U8) (U32) pArgs;

	CreateClockWindow(pInst, self);

	// cause posibly visible card to update font size controls
	ObjCallWarn(msgOptionRefresh, self, pNull);

	return stsOK;

	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} /* ClockNewFormat */

/****************************************************************************
	RatchetFontSize
	
	Common code for date and time font sizes

	todo:
		If one size is pegged the other will not move.
		Use one table for both font size controls and for ratcheting
****************************************************************************/

void RatchetFontSize(P_U8 pSize, BOOLEAN up) {

	U16 index;

	index = 0;

	if (up) {
		index = 0;
		while (fontRatchets[index] <= (S16) *pSize)
			index++;
		if (index == maxRatchet)
			index--;
	} else {  // down
		index = maxRatchet;
		while (fontRatchets[index] >= (S16) *pSize)
			index--;
		if (index == 0)
			index++;
	}

	*pSize = (U8) fontRatchets[index];

}  // RatchetFontSize

/****************************************************************************
	BumpFontSize
	
	Common code for up and down arrows
****************************************************************************/

void BumpFontSize(P_CLOCK_APP_DATA pInst, OBJECT self, BOOLEAN up) {

	WIN_METRICS wm;

	RatchetFontSize(&pInst->timeSize, up);
	StsWarn(SetLabelScale(pInst->timeWin, pInst->timeSize));
	StsWarn(SetLabelScale(pInst->amPmWin, pInst->timeSize));

	RatchetFontSize(&pInst->dateSize, up);
	StsWarn(SetLabelScale(pInst->dateWin, pInst->dateSize));

	StsWarn(
		SetLabelScale(
			pInst->alarmWin, 
			(U8) ((pInst->fmt & alarmOnTime) ?
                   pInst->timeSize :
                   pInst->dateSize)
		)
	);

	Dbg(Debugf(
		"ClockApp: new date & time font sizes are %d and %d",
		pInst->dateSize,
		pInst->timeSize
	);)

	wm.options = wsLayoutDefault;
	ObjCallWarn(msgWinLayout, pInst->appWin, &wm);

	// cause posibly visible card to update font size controls
	ObjCallWarn(msgOptionRefresh, self, pNull);

}  // BumpFontSize


/****************************************************************************
	BumpFormat
	
	Common code for left and right flicks
****************************************************************************/

void BumpFormat(P_CLOCK_APP_DATA pInst, OBJECT self, U16 direction) {

	U16 tmpIndex;
	U16 curIndex;
	U8 newFormat;

	tmpIndex = 0;
	curIndex = 0xffff;

	while ( clockFormatChoices[tmpIndex].arg1 && (tmpIndex <= maxFormat) ) {
		if (clockFormatChoices[tmpIndex].tag == pInst->fmt)
			curIndex = tmpIndex;
		tmpIndex++;
	}

	if (curIndex == 0xffff) {
		Dbg(Debugf("ClockApp: internal error - no match for format");)
		newFormat = defaultFmt;  // force to known state
	} else {
		newFormat = (U8)
			clockFormatChoices[(curIndex + direction) % (maxFormat + 1)].tag;
		Dbg(Debugf("ClockApp: new format is %d", newFormat);)
	}

	// post message so as to not destroy the label that called us
	// during the re-layout
	ObjectPost(msgClockNewFormat, self, (P_ARGS) (U32)  newFormat, 0);

}  // BumpFormat

/****************************************************************************
	ClockGesture
	
	Respond to msgGWinForwardedGesture.
****************************************************************************/

MSG_HANDLER ClockGesture(
	const MESSAGE	msg,
	const OBJECT	self,
	const P_GWIN_GESTURE	pArgs,
	const CONTEXT	ctx,
	const PP_CLOCK_APP_DATA pData)

{
	Dbg(char str[100];)
	Dbg(TIMER_NOTIFY tn;)
	WIN_METRICS wm;

	P_CLOCK_APP_DATA	pInst;

	pInst = *pData;

	wm.options = wsLayoutDefault;  // set up wm in case we need to relayout

	switch (pArgs->msg) {
		
		case xgsUpArrow:
			Dbg(Debugf("ClockApp: up-arrow gesture");)
			BumpFontSize(pInst, self, true);  // up = true
			return stsOK;

		case xgsDownArrow:
			Dbg(Debugf("ClockApp: down-arrow gesture");)
			BumpFontSize(pInst, self, false);  // up = false, ie. down
			return stsOK;

		case xgsFlickRight:
			Dbg(Debugf("ClockApp: flick right gesture");)
			BumpFormat(pInst, self, nextFormat);
			return stsOK;

		case xgsFlickLeft:
			Dbg(Debugf("ClockApp: flick left gesture");)
			BumpFormat(pInst, self, prevFormat);
			return stsOK;

		case xgsFlickDown:
			Dbg(Debugf("ClockApp: flick down gesture");)
			ObjectPostU32(msgAppCloseTo, self, (P_ARGS) appCloseToNextState);
			return stsOK;

		case xgsPGesture:
			return stsRequestDenied;

		case xgsPressHold:
		case xgs1Tap:
		case xgsTapHold:
		case xgsPlus:
		case xgsTGesture:
			return stsRequestForward;

#ifdef DEBUG
		case xgs3Tap:
			Dbg(Debugf("ClockApp: triple tap gesture");)
			ObjCallWarn(msgWinLayout, pInst->appWin, &wm);
			Dbg(ObjCallWarn(msgDump, self, pNull);)	 
			return stsOK;

		case xgsTrplFlickUp:
			Dbg(Debugf("ClockApp: triple flick gesture");)
			ObjCallWarn(msgWinDumpTree, pInst->appWin, pNull);
			return stsOK;								   

		case xgsTrplFlickDown:
			Dbg(Debugf("ClockApp: triple flick down, fake an alarm");)
			tn.transactionHandle = pInst->alarmHandle;
			tn.clientData = pNull;
			ObjCallWarn(msgTimerAlarmNotify, self, &tn);
			return stsOK;							  
#endif

	} // switch msg

	Dbg(Debugf("ClockApp: let app handle gesture %s",ClsMsgToString(msg, str)));

	return ObjectCall(msgAppExecuteGesture, self, pArgs);

	// suppress compiler warnings
	MsgHandlerParametersNoWarning;

} /* ClockGesture */


/****************************************************************************
	ClsClockAppInit
	
	Install the application.
****************************************************************************/

STATUS ClsClockAppInit (void)
{
	APP_MGR_NEW	new;
	STATUS		s;

	Dbg(Debugf("ClockApp: entering ClsClockAppInit");)

	//
	// Install the class.
	//
	ObjCallRet(msgNewDefaults, clsAppMgr, &new, s);

	new.object.uid	 				= clsClockApp;
	new.object.key 	  				= (OBJ_KEY)clsClockAppTable;
	new.object.cap					|= objCapCall;

	new.cls.pMsg					= clsClockAppTable;
	new.cls.ancestor	  			= clsApp;
	new.cls.size					= SizeOf(CLOCK_APP_DATA);
	new.cls.newArgsSize				= SizeOf(APP_NEW);

	new.appMgr.flags.hotMode		= true;  // FIXME when inst 0 files alarms
	new.appMgr.flags.allowEmbedding = false;
	new.appMgr.flags.stationery		= false;
	new.appMgr.flags.accessory		= true;

	strcpy(new.appMgr.defaultDocName, "Clock");
	strcpy(new.appMgr.version, "1.0");
	strcpy(new.appMgr.company, "GO Corporation");

	// 213 (octal) is the "circle-c" copyright symbol
	new.appMgr.copyright =
		"\213 1992 GO Corporation, All Rights Reserved.";

	ObjCallRet(msgNew, clsAppMgr, &new, s);

	return stsOK;

} /* ClsClockAppInit */


/****************************************************************************
	GOMain
	
	Main application entry point.
****************************************************************************/

void CDECL main (int argc, char *argv[], U16 instance)
{

	Unused(argc);
	Unused(argv);

	if (instance == 0) {

		ClsNoteCorkBoardWinInit();
		ClsClockWinInit();
		ClsClockLabelInit();

		//
		// Initialize self.
		//
		StsWarn(ClsClockAppInit());

		AppMonitorMain(clsClockApp, objNull);

	} else {
		//
		// Start the application.
		//
		AppMain();
	}

} /* main */
