/****************************************************************************

 xtemplt.h
 Copyright 1992 by GO Corporation, All Rights Reserved

 Translation Template Specifications for input fields

 $Revision:   1.9  $
   $Author:   ghullend  $
     $Date:   28 Feb 1992 14:31:46  $



****************************************************************************/
#ifndef XTEMPLT_INCLUDED
#define XTEMPLT_INCLUDED

/*
	Debug Flags:
	
		/DS0010	Compilation: print ASCII input and hex-address of result.
		/DS0020	Choices: print Hex address and ASCII list of choices plus count.

*/

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef OS_INCLUDED
#include <os.h>
#endif

#ifndef XLATE_INCLUDED
#include <xlate.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				   Definitions											   *	
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
	maxXTemplateXlateChoices is the number of different symbols that
	can be in a CharList template.  maxXtmPictureLength is the longest a
	Picture template may be.
*/

#define maxXTemplateXlateChoices	128	// Alphabet Size
#define maxXtmPictureLength			128	// Picture string length limit

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				   Common Typedefs										   *	
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****  Template Types  ****/

/*
	Templates are used to constrain handwritten input in order to
	increase translation accuracy.  For example, if a field can only
	contain digits, constraining the input for that field to only digits
	guarantees that the letters 'O', 'l', and 'Z', are never seen for the
	digits '0', '1', and '2'.  There are several different ways to
	constrain input, each of which corresponds to a different template
	type
*/

Enum16(XTEMPLATE_TYPE) {
   	xtmTypeNone,		// no constraints
   	xtmTypeGesture,		// limited to known gestures
   	xtmTypeShape,		// limited to known shapes (NOT IMPLEMENTED)
   	xtmTypeCharList,	// limited to a set of characters
   	xtmTypeWordList,	// limited to a set of words
   	xtmTypePicture,		// described by a picture language
   	xtmTypeRegEx,		// described by a regular expresssion (NOT IMPLEMENTED)
	xtmTypeTrie,		// precompiled
};


/****  Template Modes  ****/

/*
	A template may be interpreted in a variety of special modes.  In general,
	these modes describe circumstances under which incomplete input will be
	considered the same as complete input.
*/

Enum16(XTEMPLATE_MODE) {
	xtmModeDefault	  = 0,		// No special modes
	xtmModePrefixOK   = flag0,	// input matching a prefix of the template is
								// considered to match the template.
	xtmModeLoopBackOK = flag1,	// the template is considered to repeat over
								// and over
	xtmModeCoerced	  = flag2,	// Input should be coerced to match the
								// template, even if it doesn't match exactly
								// Only meaningful for xtmTypeWordList templates.
};

/****  Template Header  ****/

/*
	Every template is a single allocated block of memory containing no
	internal pointers.  The template header contains information about
	the template, including what's needed to file a template.
*/

typedef struct XTEMPLATE_TRIE_HEADER {
	U16				xtmTrieLength;
	U16				xtmTrieRevision;
	XTEMPLATE_TYPE	xtmTrieType;
	XTEMPLATE_MODE	xtmTrieMode;
} XTEMPLATE_TRIE_HEADER, * P_XTEMPLATE_TRIE_HEADER,
	* * PP_XTEMPLATE_TRIE_HEADER;

/*
	xtmTrieLength:	the length in bytes of the entire template.  If you want
					to file a template, this is the number of bytes to write out.  Templates
					contain no internal pointers, so they file and unfile quite easily.
	xtmTrieRevision:	the revision level of the data structure.
	xtmTrieType:	is the type of the template.
	xtmTrieMode:	is the mode of the template.
*/


/****  Template Metrics Structure  ****/

/*
	This structure is returned via the XTemplateGetMetrics subroutine,
	described below.	The major uses of this structure are to get a
	pointer to the template header in order to get the template length so
	it can be filed, and to get access to the original template string.
*/

typedef struct XTEMPLATE_METRICS {
	P_XTEMPLATE_TRIE_HEADER	pXtmHeader;	// Template len, rev, etc.
	P_CHAR		pXtmString;		 		// Original string, NULL for word
										// list or gesture
	P_UNKNOWN	pXtmTrieBase;			// Base of compressed TRIE structure
	U16			xtmTrieBaseLen;			// Size of the compressed region
} XTEMPLATE_METRICS, * P_XTEMPLATE_METRICS;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				   Functions											   *	
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 XTemplateCompile	returns STATUS
	Given a type and an ASCII template representation, build a template
	structure.

	Space is allocated as required.

*/

/* Basic Xtemplate Arguments */

typedef struct XTM_ARGS {
     XTEMPLATE_TYPE	xtmType;		// What kind of template?
	 XTEMPLATE_MODE	xtmMode;		// What special modes?
     P_UNKNOWN		pXtmData;		// ascii template
} XTM_ARGS, *P_XTM_ARGS;

STATUS EXPORTED XTemplateCompile(
	P_XTM_ARGS		pXtmArgs,		// Xtemplate Arguments
	OS_HEAP_ID     	heap,			// heap to use
	PP_UNKNOWN     	ppXtmDigested	// Out: compiled template
);

/*

 The currently implemented types have the following meanings:

	xtmTypeNone:	pXtmData is unused.  This is the same as having no
					template at all.

	xtmTypeGesture:		pXtmData points to an XTEMPLATE_GESTURE_LIST.

	xtmTypeCharList:	pXtmData contains a list of valid characters.

	xtmTypeWordList:	pXtmData contains a list of all the different
                 	words that are legal in this field.  This should be
				 	a PP_STRING pointing to a list of pointers to the
				 	words.  Each word is a normal null-terminated string
				 	and the pointer list must be terminated with a Nil(P_STRING).

	xtmTypePicture:		pXtmData contains a list of all the picture
                 	strings that are valid in this field.  A picture
                 	string contains any of the following characters:

                 	9:	 input must be a digit (0-9)
                 	a:	 input must be alphabetic
                 	A:	 input must be upper-case alphabetic
                 	n:	 input must be alphanumeric
                 	N:	 input must be upper-case alphanumeric
                 	x:	 input may be anything
					[:	 introduces a list of characters, Unix-style.
					    	[abc] is a single character position which
						 	must contain 'a', 'b', or 'c'.  [a-m] matches
						 	any letter 'a' through 'm'.  [a\-m] matches any
					    	of 'a', '-', or 'm'.  
                 	\:	 literal escape.  Input must match next
                   	character.  (Only needed to escape the above
                   	special characters).
                 
                 	For example, a modern California licence plate
                 	looks like this:
                 
                    	#AAA###
                 
                 	To include older forms of California plates, we
                 	might use:
                 
                    	#AAA###
                    	###AAA
                    	AAA###
                 
                 	either \n or tab separated.  N.B. Multiple
                 	picture strings will not be supported in the
                 	first release.
                 
                 	A Social Security Number (with mandatory
                 	hyphens) would be coded like this:
                 
                    	###-##-####
                 
                 	Pictures currently can't be used for variable
                 	length data.
*/

/* This special structure is used for xtmTypeGesture templates. */

typedef struct XTEMPLATE_GESTURE_LIST {
   U32         count;      // number of gestures in the list
   P_MESSAGE   pGestures;  // pointer to array of allowed gestures
} XTEMPLATE_GESTURE_LIST, * P_XTEMPLATE_GESTURE_LIST;


/****************************************************************************
 XTemplateGetMetrics	returns STATUS
	Given a pointer to a translation template, extract various salient facts
	about it and return them.

	Can fail if the template version is too far out of date.
*/

STATUS EXPORTED XTemplateGetMetrics(
	P_UNKNOWN			pXTemplate,	// Template to extract the metrics of
	P_XTEMPLATE_METRICS	pXtmMetrics	// Out: metrics of the template
);

/****************************************************************************
 XTemplateSetMode	returns STATUS
	Change the mode in an already-created XTemplate.

	Changing xtmPrefixOK or xtmLoopBackOK may have no effect.
*/

STATUS EXPORTED XTemplateSetMode(
	P_UNKNOWN	   pXTemplate,	// Compressed Template 
	XTEMPLATE_MODE xtmMode		// New mode
);

/****************************************************************************
 XTemplateFree	returns STATUS
	Free an existing Template.

	Checks for pNull and just returns stsOK
*/

STATUS EXPORTED XTemplateFree(
     P_UNKNOWN      pXtmDigested    // compiled template ptr.
);

/****************************************************************************
 XTemplateWordListSort	returns void
	Given a pointer to a list of pointers to strings, sort the list of
	pointers so the strings appear in alphabatical order.
		
	Last pointer in list must be Nil(P_STRING)
*/

void EXPORTED XTemplateWordListSort(
	PP_CHAR	ppStringBase	// compiled template
);

/****************************************************************************
 XTemplateCheckWord  returns BOOLEAN
	Check if a word is in a template.

*/

BOOLEAN EXPORTED XTemplateCheckWord(
   	P_UNKNOWN     pXtmData,			// compiled template
	P_CHAR	 pWord				// Word to check
);

/****************************************************************************
 XTemplateCheckGesture  returns BOOLEAN
	Check if a gesture is in a template.

*/

BOOLEAN EXPORTED XTemplateCheckGesture(
	P_UNKNOWN     pXtmData,			// compiled template
	MESSAGE		  gesture			// gesture to test
);

/****************************************************************************
 XTemplateAddWord returns STATUS
	Add a word to a wordlist template.

*/

STATUS EXPORTED XTemplateAddWord(
	PP_UNKNOWN     	ppXtmData,		// In/Out: compiled template
	P_CHAR			pWord,			// Word to add
	OS_HEAP_ID     	heap	         // heap to use
);

/****************************************************************************
 XTemplateDeleteWord returns STATUS
	Delete a word from a wordlist template.

*/

STATUS EXPORTED XTemplateDeleteWord(
	PP_UNKNOWN     ppXtmData,		// In/Out: compiled template
	P_CHAR			pWord,			// Word to add
	OS_HEAP_ID     heap	         // heap to use
);

/****************************************************************************
 XTempltInit	returns STATUS
	DLL Initialization routine.
   
	Included for compatibility; not to be called by developers.

*/

STATUS EXPORTED XTempltInit(void);

#endif	// XTEMPLT_INCLUDED

