/****************************************************************************
 xlist.h

 Copyright 1992, GO Corporation, All Rights Reserved

 $Revision:   1.11  $
	 $Date:   26 Feb 1992 11:53:30  $

 This file contains the API definition for xlist.
 Xlists provide a set of dynamic list routines used by translators.

 The functions described in this file are contained in XLIST.LIB.

 An xlist is a set of routines for manipulating a list of items of
 data type P_XLIST_ELEMENT.  These items are allocated from a
 heap passed into the xlist when it is created.  Elements have
 some flag settings, a data type, and a pointer.  The pointer
 points to data defined by the data type, whose allocation is
 dependent on the flag settings.

 Elements in the list are indexed from 0 to entries-1. A series
 of functions are provide to create and destroy lists, traverse
 lists, access and set list elements, insert new elements, and
 delete elements.

 In addition, functions are provided to "filter" data from the xlist.
 These filters either extract useful data from the xlist in the form
 of a data structure, or actually "mutates" the xlist into an xlist
 of a different format. These filters are defined in this file and in
 xlfilter.h.

 Xlists of various types are used throughout the system.  Primarily, they
 are used to pass translation information between the hwx system and the
 client.  See xlate.h for example uses in the hwx engine; and gwin.h,
 spaper.h, or insert.h for example uses inside the UI toolkit.

 Typical users create xlists (XListNew), add and delete items
 (XListInsert, XListDelete), access the value of items (via filters
 or XListGet), traverse (XListTraverse) and free them (XListFree). Other
 functions, while useful, are rarely used.

 Xlists have associated with them a heap with which use to allocate
 the memory needed to store the elements (P_XLIST__ELEMENT). They can also
 use this heap to allocate space for the data pointer field of an element,
 when the corresponding elements flag setting is xfHeapAlloc.
 In this situation, the element data pointer will be freed when the
 xlist is freed, or when XListFreeData is called.  Allocating other memory
 off the xlist heap, although not recommended, is possible.  It would be
 the clients responsibility to free this data.  However, typically the
 user of an xlist will allocate space for the data pointer off of the
 heap using XListAlloc, insert an element into the xlist with the data,
 and allow the xlist to manage and free the memory.
****************************************************************************/
#ifndef XLIST_INCLUDED
#define XLIST_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef GEO_INCLUDED
#include <geo.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****  Xlist Data Structure  ****/
/*
 * A pointer to an xlist is a pointer to a private data structure.
 * This pointer is passed to the xlist function to create, destroy,
 * and manipulate xlists.
 */
typedef P_UNKNOWN P_XLIST;

/****  Xlist Flags  ****/
/*
 * These flags are stored in the xlist.  They are useful to store
 * xlist specific data.  flag0 through flag15 are reserved for GO
 * internal use, while flag16 through flag31 is for client use.
 * The only flag currently used indicates that XList2Text has been
 * run on the xlist.  This optimizes successive calls to this
 * xlist filter, allowing it to return without running the filter.
 * Running the filter a second time, because the flag is clear,
 * is harmless.
 */
#define xflXList2Text	flag0

/****  Element Types  ****/
/*
 * These are the data types for elements of an xlist. An element contains
 * a type, a data pointer and flags.  For each data type,
 * the data pointer varies.
 */
Enum16(XTYPE) {
	xtNull,			// pData = null = 0
	xtBounds,		// pData = P_BDATA (clsXGesture, clsXText)
	xtGesture,		// pData = P_GDATA (clsXGesture)
	xtText,			// pData = P_STRING (clsXText, XList2Text)
	xtObject,		// pData = OBJECT
	xtBoundsX,		// pData = P_BDATA (screen relative)
	xtCharAttrs,	// pData = P_XLIST_CHAR_ATTRS (txtxlist.h)
	xtParaAttrs,	// pData = P_XLIST_PARA_ATTRS (txtxlist.h)
	xtTabs,			// pData = P_XLIST_TABS (txtxlist.h)
	xtCharPos,		// pData = TEXT_INDEX
	xtTextList,		// pData = P_WORD_LIST (hwx)
	xtSpare1,		// pData =
	xtSpare2,		// pData =
	xtSpare3,		// pData =
	xtSpare4,		// pData =
	xtGeometric,	// pData = P_XGEO_DATA unused
	xtTextListEnd,	// pData = NULL (sPaper)
	xtTextWord,		// pData = P_XTEXT_WORD (xtext) (clsXtext, sPaper)
	xtStroke16,		// pData = P_SPAPER_STROKE_DATA (spaper)
	xtSpace,		// pData = U32 unused
	xtTeachData,	// pData = P_XTEACH_DATA (xteach)
	xtUID,			// pData = UID of the gesture object
	xtEmbedObject,	// pData = P_TEXT_EMBED_OBJECT (txtdata.h)
	xtExtended,		// pData = UID,client data
	xtLastEntry		// last entry in the xtList
};

/****  Xlist Element  ****/
/*
 * This data structure defines an element in an xlist.  An xlist element
 * contains some flags, a data type, and a pointer to some data.
 * The allocation and type of data depend on both the flags and the data
 * type of the element.
 */
typedef struct XLIST_ELEMENT {
	U16 flags;				// Element flags. Mostly allocation information.
	XTYPE type;				// Type of data in pData
	P_UNKNOWN pData;		// Pointer to data of element
} XLIST_ELEMENT,  *P_XLIST_ELEMENT;

/****  Element Flags  ****/
/*
 * These flags are stored in the XLIST_ELEMENT flags, and indicate
 * information about the elements.  They can be changed dynamically
 * simply by accessing the xlist element. Other flags not used are
 * reserved for future use.
 */

/****  Allocation flags  ****/
/*
 * These flags indicate how to treat memory for the element. They indicate
 * how the element will be freed when the xlist is freed,
 * and how to allocate space for the element when duplicated via XListDup
 * (cannot duplicate xfObject). Setting more than one of these flags
 * will have unpredictable results, as these are mutually exclusive flags.
 */
#define xfHeapAlloc		flag0		// Allocated from the xlist heap
#define xfObject		flag1		// Element is an object. Cannot duplicate
#define xfXList			flag14		// Element is a P_XLIST

/*
 * This flag indicates that the elements data is used elsewhere, and
 * should not be freed when freeing the xlist.  It will be the clients
 * responsibility to free the data if he sets this flag.
 */
#define xfExtracted		flag15		// Set if the data is used elsewhere

/****  Traversal function  ****/
/*
 * This callback function is used to as a function template
 * called on elements of the xlist when traversing the xlist.  See
 * XListTraverse for more details. This function takes an
 * xlist, an xlist element, and a user defined data pointer.
 */
typedef STATUS FunctionPtr(P_XPROC) (
	P_XLIST pXlist,
	P_XLIST_ELEMENT pElement,
	P_UNKNOWN pUserData);

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Public Functions								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 XListNew	returns STATUS
	Creates a new xlist.

 Creates and allocates an xlist from the specified heap, using the
 heap to allocate space for the P_XLIST_ELEMENT entries in the list, and
 for XListAlloc.
*/
STATUS  PASCAL XListNew(
	OS_HEAP_ID heap,			// In: heap to allocate the xlist
	P_XLIST *ppXList);			// Out: Pointer to the P_XLIST

/****************************************************************************
 XListFree	returns STATUS
	Frees an xlist and all its data.

 Traverses the xlist elements and frees the data (unless the element
 has xfExtracted set).  For each element, frees the memory appropriately
 by traversing the xlist with function XListFreeData.

 See Also
	XListFreeData
*/
STATUS  PASCAL XListFree(
	P_XLIST pXList);		// In: xlist to free

/****************************************************************************
 XListGetFlags 			returns STATUS
	Passes back the XList flags for the xlist.

 flag0 through flag15 are reserved for GO internal use.  flag16 through
 flag31 are for client use.
*/
STATUS  PASCAL XListGetFlags(
	P_XLIST pXList,			// In: xlist to get the flags from
	P_U32   pFlags);		// Out: pointer to the flags

/****************************************************************************
 XListSetFlags					returns STATUS
	Sets the XList Flags.

 Sets the flags associated with the xlist.  flag0 through flag15 are
 reserved for GO internal use.  flag16 through flag31 are for client use.
*/
STATUS  PASCAL XListSetFlags(
	P_XLIST pXList,		 	// In: xlist to set the flags from
	U32 flags);				// In: new flags to set

/****************************************************************************
 XListMetrics					returns STATUS
	Passes back the number of entries and heap Id.

 Passes back the number of entries in the xlist, and the heap used
 to allocate xlist memory.  Note that there is no corresponding
 'set' metrics function, as dynamically changing the heap or
 count would have drastic side affects.
*/
typedef struct XLIST_METRICS {
	OS_HEAP_ID heap;
	U16 entries;
} XLIST_METRICS,  *P_XLIST_METRICS;

STATUS  PASCAL XListMetrics(
	P_XLIST pXList,				// In: xlist to get the metrics from
	P_XLIST_METRICS pMetrics);	// Out: metrics of the xlist

/****************************************************************************
 XListInsert					returns STATUS
	Creates a new element at the index'th location.

 Allocates space for and creates a P_XLIST_ELEMENT in the xlist at
 the specified location.  If index >= entries, the element
 is appended to the end of the list.  The element data pointer allocation
 and storage depends on the type of the element. The following example
 shows a client inserting a 7 character string into an xlist. The element
 type is xtText and the insertion at the beginning of the xlist:

//{
 	XLIST_ELEMENT elem;
	elem.type = xtText;
	elem.flags = xfHeapAlloc;
	XListAlloc(pXList, 7, &elem.pData);
	strcpy(elem.pData, "String");
	XListInsert(pXList, 0, &elem);
//}
*/
STATUS  PASCAL XListInsert(
	P_XLIST pXList,			// In: xlist to insert item into
	U16 index,				// In: index of location to insert at
	P_XLIST_ELEMENT pElem);	// In: element to insert

/****************************************************************************
 XListDelete					returns STATUS
	Delete the element at the index'th location.

 Delete the element at the specified location.  This calls XListFreeData
 to free any memory taken by the element data pointer.  Frees memory
 associated with storing the P_XLIST_ELEMENT in the xlist.

 See Also
	XListFreeData
*/
STATUS  PASCAL XListDelete(
	P_XLIST pXList,		// In: xlist to delete item from
	U16 index);			// In: index of item to delete

/****************************************************************************
 XListTraverse		returns STATUS
	Iterates across the list of elements.

 Iterates across the elements in the xlist.  A callback
 function (pProc) is handed to this function, and is called for each
 element passing in the element and a client pointer as defined in
 P_XPROC.  If any call to pProc returns anything but stsOK, the traversal
 is terminated and the status code returned.  Nested traversals are
 allowed and supported.

 See Also
	XListIndex
*/

STATUS  PASCAL XListTraverse(
	P_XLIST pXList,			// In: xlist to traverse
	P_XPROC pProc,			// In: call back function to call for each element
	P_UNKNOWN pUserData);	// In/Out: User defined data pointer

/****************************************************************************
 XListIndex						returns STATUS
	Passes back the current traversal index.

 Passes back the index for the current traversal.  If no traversal is
 taking place, returns 0.  Note that if nested traversals are taking
 place, the index of the current traversal will be returned.  Once the
 sub-traversal is completed, the parent traversals index is restored
 and returned appropriately via calls to XListIndex.

 See Also
	XListTraverse
*/
STATUS  PASCAL XListIndex(
	P_XLIST pXList,			// In: pointer to xlist
	P_U16 pIndex);			// Out: current index


/****************************************************************************
 XListSet						returns STATUS
	Stores the copy of the index'th element.

 Stores the passed in element as the element in the specified location.
 If index is > number of entries, will store in the last item in the list.
 Care should be taken, as the old item stored in that location is not
 freed and is the clients responsibility.  Useful only if changing an
 entire item in the xlist.  Rarely used.

*/
STATUS  PASCAL XListSet(
	P_XLIST pXList,			// In: xlist pointer
	U16 index,				// In: index of element
	P_XLIST_ELEMENT pPtr);	// In: new element to store at location

/****************************************************************************
 XListGet						returns STATUS
	Passes back a copy of the index'th element.

 Passes back a copy of the index'th element. The element, data type,
 and data pointer will be copied. Hence the data pointer is a direct
 pointer to the data.
*/
STATUS  PASCAL XListGet(
	P_XLIST pXList,			// In: xlist pointer
	U16 index,				// In: index of element
	P_XLIST_ELEMENT pPtr);	// Out: Copy of element data.

/****************************************************************************
 XListGetPtr					returns STATUS
	Passes back a pointer to the index'th element.

 Passes back  a pointer to the index'th element in the xlist.  Extreme
 care should be taken when accessing this pointer, as it is the pointer
 stored in the xlist. Useful only if the client wishes to change some
 information about an existing item in the xlist. Rarely used. Note
 that the data pointer field is the same returned by XListGet.

 See Also
	XListGet
*/
STATUS  PASCAL XListGetPtr(
	P_XLIST pXList,
	U16 index,
	P_XLIST_ELEMENT *ppPtr);

/****************************************************************************
 XListAlloc						returns STATUS
	Allocate some memory from the XList heap.

 Allocates memory off of the xlist heap.  Typically used
 to allocate space for the data pointer of an element that
 has xfHeapAlloc set.  Space for such an element data pointer
 will be freed in XListFreeData, called when the xlist is freed
 via XListFree, or when the item is deleted via XListDelete. Other
 memory can be allocated using this function, although it is the
 clients responsibility to ensure that it is freed.

 See Also
	XListFreeData
	XListFree
*/
STATUS  PASCAL XListAlloc(
	P_XLIST pXList,			// In: xlist pointer
	SIZEOF size,			// In: size of the requested allocation
	P_UNKNOWN pMem);		// Out: pointer to the allocated memory


/****************************************************************************
 XListFreeData					returns STATUS
	Releases the data with the given entry.

 Frees data associated with the passed in element.  Returns stsOK
 if xfExtracted is set on the element.  Frees the memory appropriately
 if xfHeapAlloc is set.  Sends msgFree to the object if xfObject is set.
 Calls XListFree if xfXList is set. Called from XListFree for each element
 in the xlist, and called from XListDelete when an item is deleted from
 the xlist.

 See Also
	XListFree
	XListDelete
*/
STATUS  PASCAL XListFreeData(
	P_XLIST pXList,			// In: xlist pointer
	P_XLIST_ELEMENT pElem,	// In: element to free
	P_UNKNOWN pUserData);	// In: User defined data structure

/****************************************************************************
 XListDup						returns STATUS
	Duplicates the contents of one xlist into another.

 Traverses the source xlist and calls XListDupElement for each
 item in the source xlist with the destination xlist. If XListDupElement
 returns a non-stsOK return code for an element in the xlist, the xlist
 to the point of the return code is copied and the duplication
 terminated at that point.

 See Also
	XListDupElement

 Return Value
	stsBadParam:		the xlist duplication terminated before completion
*/
STATUS  PASCAL XListDup(
	P_XLIST pSrcXList,		 // In: source xlist
	P_XLIST pDestXList);	 // In/Out: destination xlist

/****************************************************************************
 XListDupElement					returns STATUS
	Duplicate the source element, append to the destination.

 Duplicates the element and appends it to the end of the destination
 xlist. When the element is xfHeapAlloc, allocates space for the element
 from the destination xlist heap, and memcpy's the contents.
 When the element is xfXList, creates a new xlist using the passed in
 xlist's heap, and duplicates all elements in the xlist.  Any other
 element data type (xfObject) is not copy-able and will return stsBadParam.

 Return Value
	stsBadParam:		The element type could not be duplicated.

 See Also
	XListDup
*/
STATUS  PASCAL XListDupElement(
	P_XLIST pXList,
	P_XLIST_ELEMENT pElem,
	P_XLIST pDestXList);

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Xlist Filters									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 XList2Gesture					returns STATUS
	Extracts the gestural information from an xlist.

 Given an xlist containing xtBounds followed by xtGesture, (the xlist
 typically returned by the clsXGesture translator after completed
 translation), this function extracts the useful information and stores
 in a standard c data structure.  This function is used internally in
 gWin to convert the gesture translator data structure into a more
 useful form.

 See Also
	gwin.h
	xlate.h
	xgesture.h
*/
typedef struct X2GESTURE {
	U32	 msg;				// gesture type
	RECT32  bounds;			// gesture bounding box
	XY32	hotPoint;		// gesture hot point
} X2GESTURE,  *P_X2GESTURE;

STATUS  PASCAL XList2Gesture(
	P_XLIST pXList, 		// In: xlist to run filter on
	P_X2GESTURE pData);		// Out: converted data structure

/****************************************************************************
 XList2StringLength				returns STATUS
	Passes back the length of the string that XList2String will need.

 Computes the necessary length of a string that XList2String will
 need to copy a string.  Includes space for the terminating
 null character.
*/
STATUS  PASCAL XList2StringLength(
	P_XLIST pXList,
	P_U16 pLength);

/****************************************************************************
 XList2String					returns STATUS
	Extracts the text information from an xlist.

 Converts an xtBounds/xtText xlist into a string.  Clips the returned
 string at the passed in count.  This string includes a null terminating
 character.  The function takes an xlist of the form:

 	[xtBounds [xtText]]

 and converts it into a string.  As an example, suppose the xlist contains:

 	xtBounds1 xtText1 xtText2 xtText3 xtBounds2 xtText4 xtText5.

 This is converted into:

 	xtText1xtText2xtText3\nxtText4xtText5

 More typically, this function called on an xlist that has had
 adjacent xtText entries merged by XList2Text.  Typical usage is
 during processing of an xlist returned from msgXlateGetData.
 Here the client simply wants to know the string returned, so he
 will call XList2Text, XList2StringLength (unless he knows how big the
 string will be), and XList2String to get the string.

 See Also
	XList2Text
	XList2StringLength
	xlate.h
*/
typedef struct X2STRING {
	U16	 count;				// In: buffer size
	P_CHAR pString;		// In: pointer to the buffer
} X2STRING,  *P_X2STRING;

STATUS  PASCAL XList2String(
	P_XLIST pXList,			// In: xlist to process
	P_X2STRING pData);		// In: X2String data structure pointer

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Debugging Functions								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 XListDump							returns STATUS
	Debugging interface for displaying an xlist in the debug log.

 When called on an xlist, traverses the elements and displays useful
 information about the xlist in the debug log. It displays
 this information by calling a display routine that is dependent on the
 type of the element.  A display routine can be registered for an element
 type using XListDumpSetup. If no display routine has been provided for
 an element type, it will display the generic information for the element
 consisting of the type, the flags, and the element data pointer.

 See Also
	XListDumpSetup
	XListTraverse
*/
STATUS EXPORTED XListDump(
	P_XLIST pXList);		// In: array header

/****************************************************************************
 XListDumpSetup						returns STATUS
	Sets the xlist debug log display routine by type.

 Called to register display routines for xlist element types with the
 xlist. This display routine will be called when the particular element
 type traversed when calling XListDump.

 See Also
	XListDump
	XListTraverse
*/
STATUS EXPORTED XListDumpSetup(
	XTYPE   type,		// In: xtype to bind this procedure to
	P_XPROC pProc,		// In: function to be called when dumping
	U32	 data);			// In: type specific data passed to pProc in traversal

#endif
