/****************************************************************************
 spaper.h

 Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.8  $
   $Author:   kbentley  $
     $Date:   12 Mar 1992 13:18:22  $

 This file contains the API definition for clsSPaper.

 clsSPaper inherits from clsView.

 An instance of clsSPaper (or sPaper, for short) provides stroke redisplay,
 very simple stroke editing, and translation.
****************************************************************************/

/**** Road Map ****/
/*
 A typical sPaper client simply creates an sPaper with self as the
 listener.  The client than handles the msgSPaperXlateCompleted
 notification and uses msgSPaperGetXlateData to extract the resulting data.

 Clients or subclasses who wish to get involved in the sPaper's management 
 of strokes might use:
	-:	msgSPaperClear
	-:	msgSPaperAddStroke
	-:	msgSPaperDeleteStrokes

 Clients or subclasses who wish to be involved in controlling when
 translation is triggered might use:
	-:	msgSPaperComplete
	-:	msgSPaperAbort

 If a client only needs translation, the client may not need to use
 sPaper at all.  The client may be able to use translators (xlate.h)
 and scribbles (scribble.h) directly.
*/


/**** SPaper Components ****/
/*
 An sPaper manages a translator and a scribble.  The sPaper observes the 
 the translator and the translator observes the scribble.

 The sPaper has a listener which is specified when the sPaper is
 created.  An sPaper makes the listener an observer of the sPaper.  As a
 result, the listener receives sPaper notifications.

*/


/**** Typical Scenario ****/
/*
 The usual scenario for an spaper follows.  The spaper is created and
 inserted onto the screen.  The spaper receives pen strokes which it
 passes on to its scribble which in turn passes them on to a translator.
 At some point, the spaper is "complete" either via an external
 notification or optionally when an out of proximity event is received.
 The spaper notifies the scribble and the scribble notifies the translator.
 When the translator is complete, it notifies the spaper which in turn
 notifies its listener. The listener then asks the spaper for the
 translated data and the spaper gets the data from the translator.

 Here's a typical flow of messages between the sPaper, its scribble, its
 translator and the sPaper's listener.  (This scenario uses messages
 defined in input.h, pen.h, xlate.h and scribble.h)

 When the sPaper receives a msgInputEvent that contains a stroke (see pen.h)
 it self sends msgSPaperAddStroke, which sends msgScrAddStroke to the
 scribble.
//{
	input system	-->	msgInputEvent		--> sPaper
 	sPaper			--> msgScrAddStroke 	--> scribble
//}

 The scribble then sends msgScrAddStroke to its observers.  One of the
 scribble's observers is the sPaper's translator.
//{	
	scribble	 	--> msgScrAddStroke 	--> translator
//}

 Eventually sPaper receives msgSPaperComplete.  (A client may send
 msgSPaperComplete to the sPaper.  Alternatively, depending on the sPaper's
 flags, the sPaper may self send msgSPaperComplete.  For example, see the
 description of the spProx flag elsewhere in this file.) In response to
 msgSPaperComplete, the sPaper sends msgScrComplete to the scribble.  In
 turn, the scribble notifies its observers (including the translator) with
 msgScrCompleted.
//{
	sPaper 			--> msgScrComplete 		--> scribble
	scribble 		--> msgScrCompleted		--> translator
//}

 The translator responds to msgScrCompleted by sending msgXlateCompleted to
 its observers, which include the sPaper.  The sPaper responds to 
 msgXlateCompleted by sending msgSPaperXlateCompleted to its observers,
 which include the listener.
//{
	translator 		--> msgXlateCompleted 		--> sPaper
	sPaper 			--> msgSPaperXlateCompleted --> listener
//}
 
 The listener typically sends msgSPaperGetXlateData to the sPaper to
 retrieve the translated data.  In response to msgSPaperGetXlateData, the
 sPaper sends msgXlateData to the translator.
//{
	listener 		--> msgSPaperGetXlateData 	--> sPaper
	sPaper 			--> msgXlateData 			--> translator
//}
*/


/**** Debugging Flags ****/
/*
 clsSPaper uses the debugging flag set 'h'.  Defined flags are:
	0010:	General sPaper debugging information
	0020:	sPaper translation debugging information
	0080:	sPaper save and restore debugging information
*/


/**** Relationship to clsGWin ****/
/*
 Although sPaper is a descendent of clsGWin, it inherits little of
 clsGWin's behavior.  All input and translation behavior is overridden.
*/
 
#ifndef SPAPER_INCLUDED
#define SPAPER_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef UID_INCLUDED
#include <uid.h>
#endif

#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif

#ifndef WIN_INCLUDED
#include <win.h>
#endif

#ifndef VIEW_INCLUDED
#include <view.h>
#endif

// Next Up:	24	Recycled: 1, 5, 9, 12	Used:	128

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT SPAPER;

/****  Flags  ****/
/*
 * These flags are set in pNew->sPaper.flags field, and can be manipulated
 * using msgSPaperSetFlags and msgSPaperGetFlags.
 *
 *    -:  spCapture. If false, the sPaper destroys an existing scribble and
 *        creates a new one when the first stroke since the last translation
 *        is received. If true, the scribble is preserved between
 *        translations. See the "SPaper Components" section for more
 *        information.
 *
 *    -:  spProx. If true, the sPaper self sends msgSPaperComplete when
 *        msgPenOutProxUp is received. In effect, setting this flag causes
 *        the sPaper to spontaneously translate when an "out of proximity"
 *        event occurs.
 *
 *    -:  spFixedPos. If true, the sPaper keeps the top-left corner of its
 *        scribble fixed distance from the top-left corner of self during a
 *        resize operation.
 *
 *    -:  spPenCoords. If true, xlists returned by the sPaper have pen
 *        coordinate rather than LWC coordinates.
 *
 *    -:  spGrab. If true, the sPaper grabs input in response to msgPenDown
 *        and releases the grab in response to msgSPaperAbort or
 *        msgSPaperComplete. 
 *
 *    -:  spScribbleEdit. If true (the default), allows the user to delete
 *        scribbles via scratch out. sPaper implements a VERY rudimentary
 *        "scratch out" gesture. If spScribbleEdit is true and the user draws
 *        just the right "scratch out" gesture the strokes under the gesture
 *        are deleted. This does NOT use PenPoint's general gesture
 *        translation facilities.
 *		
 *    -:  spRedisplay. If true (the default), the sPaper redisplays its
 *        scribble's strokes whenever anything changes.
 *
 *    -:  spSuppressMarks. If true, the following flags are treated as 
 *        false:  spRuling, spVRuling, spGrid, spTick, and spBaseLine.
 *
 *    -:  spRuling. If true (the default), horizontal ruling lines are drawn.
 *
 *    -:  spVRuling. If true, vertical ruling lines are drawn.
 *
 *    -:  spGrid. If true, grid lines are drawn.
 *
 *    -:  spTick. If true, tick marks are drawn.
 *
 *    -:  spBaseLine. If true, and spRuling is also true, the horizontal
 *        ruling lines are drawn as a baseline.
 *
 *    -:  spDataMoveable. If true, then the sPaper's scribble is moveable.
 *
 *    -:  spDataCopyable. If true, then the sPaper's scribble is copyable.
*/

#define spCapture		flag0
#define spProx			flag4
#define spFixedPos	  	flag5
#define spPenCoords	  	flag6
#define spGrab			flag8
#define spScribbleEdit  flag11
#define spRedisplay		flag7
#define spSuppressMarks	flag12
#define spRuling		flag1
#define spVRuling		flag13
#define spGrid			flag9
#define spBaseLine		flag14
#define spTick			flag10
#define spDataMoveable	flag2
#define spDataCopyable	flag3


//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Private									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Private data structure internal to the clsSPaper. */
typedef struct SPAPER_STROKE_DATA {
    U16		count;			// number of points
    XY16	points[1];		// dynamically allocated
} SPAPER_STROKE_DATA, *P_SPAPER_STROKE_DATA;
//REFGEN ENDIGNORE

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew				takes P_SPAPER_NEW, returns STATUS 
	category: class message
	Creates an sPaper object.
*/
typedef struct SPAPER_NEW_ONLY {
	U16		flags;
	U16		lineHeight;			// Cell height (in points)
	OBJECT	translator;			// Translation object
	OBJECT	listener;			// This object is made an observer of the
								// sPaper.
	U16		rows;				// Rows for shrink wrap layout
	U16		cols;				// Cols for shrink wrap layout
	U16		charWidth;			// Cell width (in points)
	U32 	reserved;
} SPAPER_NEW_ONLY;

#define sPaperNewFields			 	\
	viewNewFields				   	\
	SPAPER_NEW_ONLY		sPaper;

typedef struct SPAPER_NEW {
   sPaperNewFields
} SPAPER_NEW, *P_SPAPER_NEW;


/****************************************************************************
 msgNewDefaults					takes P_SPAPER_NEW, returns STATUS 
	category: class message
	Initializes the NEW structure to default values.

//{
	pArgs->win.flags.input				= (inputOutProx | inputTip | 
											inputStroke | inputInk | 
											inputNoBusy | inputHWTimeout | 
								   			inputAutoTerm | inputTimeout |
								   			inputHoldTimeout);
	pArgs->win.flags.style				|= wsSendGeometry;

	pArgs->view.dataObject				= NULL;
	pArgs->view.createDataObject		= TRUE;

	pArgs->sPaper.flags					= (spRuling | spRedisplay | 
											spScribbleEdit);
	pArgs->sPaper.translator			= NULL;
	pArgs->sPaper.rows					= 0;
	pArgs->sPaper.cols					= 0;
	pArgs->sPaper.reserved				= 0;
	pArgs->sPaper.listener				= NULL;

	pArgs->sPaper.lineHeight			= 25;	// In case read fails.
	read.resId = tagPrLineHeight;
	read.heap = 0;
	read.pData = &pArgs->sPaper.lineHeight;
	read.length = SizeOf(U16);
	ObjCallRet(msgResReadData, theSystemPreferences, &read, s);

	// convert line height from hundredths of inches to points.
 	pArgs->sPaper.lineHeight = (pArgs->sPaper.lineHeight * 72) / 100;
	pArgs->sPaper.charWidth = pArgs->sPaper.lineHeight;
//}
*/


/****************************************************************************
 msgSPaperGetFlags 				takes P_U16, returns STATUS
	Passes back the sPaper's flags.
*/
#define msgSPaperGetFlags		MakeMsg(clsSPaper,19)



/****************************************************************************
 msgSPaperSetFlags				takes P_U16, returns STATUS
	Sets the sPaper's flags.

 In addition to setting the flags, the scribble self sends msgWinDirtyRect
 to force itself to redraw with the new flags.
*/
#define msgSPaperSetFlags		MakeMsg(clsSPaper,20)


/****************************************************************************
 msgSPaperGetTranslator 		takes P_OBJECT, returns STATUS
	Passes back the sPaper's translator object (may be NULL).
*/
#define msgSPaperGetTranslator  MakeMsg(clsSPaper,16)


/****************************************************************************
 msgSPaperSetTranslator 		takes P_OBJECT, returns STATUS
	Replaces the sPaper's translator passes back the old translator.

 Important: the old translator is not destroyed.  The client is responsible
 for eventually destroying it.

 In response to this message, the sPaper replaces it translator.  (The old
 translator is passed back.)  The sPaper adds itself as an observer of the
 new translator and adds the translator as the translator as an observer of
 the sPaper's scribble (if one exists).  
*/
#define msgSPaperSetTranslator  MakeMsg(clsSPaper,17)


/****************************************************************************
 msgSPaperGetScribble   		takes P_OBJECT, returns STATUS
	Passes back the sPaper scribble object (may be NULL).

 See the section "SPaper Components" for more information.
*/
#define msgSPaperGetScribble	MakeMsg(clsSPaper,14)


/****************************************************************************
 msgSPaperSetScribble   		takes P_OBJECT, returns STATUS
	Replaces the sPaper's scribble and passes back the old scribble.

 Important: the old scribble is not destroyed.  The client is responsible
 for eventually destroying it.

 In response to this message, the sPaper replaces its scribble.  (The old
 scribble is passed back.)  The sPaper makes its translator (if one exists)
 an observer of the new scribble.  This causes all strokes in the new
 scribble to be sent to the existing translator.  
*/
#define msgSPaperSetScribble	MakeMsg(clsSPaper,15)


/****************************************************************************
 msgSPaperGetCellMetrics 	takes P_SPAPER_CELL_METRICS, returns STATUS
	Passes back some of sPaper's metrics.

 In response, sPaper passes back pArgs->cellRect, pArgs->cellSize,
 pArgs->rows and pArgs->cols.

 Note that pArgs->rows and pArgs->cols are not the values passed to
 msgNew.  (The values passed to msgNew are used for shrink wrap layout.)

 See Also
	msgSPaperGetSizes
*/
#define msgSPaperGetCellMetrics MakeMsg(clsSPaper,11)

typedef struct SPAPER_CELL_METRICS {
	RECT32 	cellRect;		// centered writing area of the sPaper
	SIZE32  cellSize;		// size of an individual cell based on
							// lineHeight and charWidth
	U16	 	rows;			// number of rows displayed
	U16	 	cols;			// number of columns displayed
} SPAPER_CELL_METRICS, *P_SPAPER_CELL_METRICS;


/****************************************************************************
 msgSPaperSetCellMetrics		takes P_SPAPER_CELL_METRICS, returns STATUS
	Sets the sPaper's cell metrics.  Resizes and lays out window.

 In response, sPaper uses the new values of pArgs->cellSize, pArgs->rows and
 pArgs->cols to compute its new window size.  It then self sends
 msgWinLayout to resize and re-layout self.  The new value of the sPaper's
 cellRect is passed back in pArgs->cellRect.
*/
#define msgSPaperSetCellMetrics MakeMsg(clsSPaper,13)


/****************************************************************************
 msgSPaperGetSizes 				takes P_SIZE16, returns STATUS
	Passes back the sPaper's line height and character width sizes, 
	in points.

 The response to this message is similar to the response to 
 msgSPaperGetCellMetrics except that fewer values are returned and the
 values are in points.

 See Also
	msgSPaperGetCellMetrics
*/
#define msgSPaperGetSizes		MakeMsg(clsSPaper,21)


/****************************************************************************
 msgSPaperSetSizes				takes P_SIZE16, returns STATUS
	Sets the sPaper's line height and character width sizes, in points.

 In response, the sPaper sets its lineHeight and charWidth.  It recomputes
 other sizes that depend on those values, and repaints itself if necessary.

 See Also
	msgSPaperSetCellMetrics 	
*/
#define msgSPaperSetSizes		MakeMsg(clsSPaper,22)


/****************************************************************************
 msgSPaperClear	 				takes NULL, returns STATUS
	Destroys the sPaper's scribble.

 In response, the sPaper destroys its scribble, if it has one.
*/
#define msgSPaperClear			MakeMsg(clsSPaper,4)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Stroke Processing Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSPaperAddStroke			takes P_INPUT_EVENT, returns STATUS
	Adds a stroke to the sPaper's scribble.

 In response to msgPenStroke, the sPaper self sends this message to add a
 stroke to its scribble.  If the sPaper does not have a scribble, one is
 created.  If the sPaper is not capturing input (spCapture flag is false),
 and this is the first stroke added since the last translation, then any
 existing scribble is destroyed and a new one is created.

 The sPaper self sends msgSPaperLocate before adding the stroke to the
 scribble to allow subclasses to process the stroke.
*/
#define msgSPaperAddStroke		MakeMsg(clsSPaper,2)


/****************************************************************************
 msgSPaperLocate				takes P_SPAPER_LOCATE, returns STATUS
	Allows subclasses to process the stroke before it is added to the
	scribble.

 An sPaper's default response to this message is to return stsOK.

 See Also
	msgSPaperAddStroke
*/
#define msgSPaperLocate	  		MakeMsg(clsSPaper,6)

typedef struct SPAPER_LOCATE {
	XY32		start;			// origin of stroke
	P_UNKNOWN	pStroke;		// new stroke
} SPAPER_LOCATE, *P_SPAPER_LOCATE;


/****************************************************************************
 msgSPaperDeleteStrokes 		takes P_RECT32, returns STATUS
	Deletes strokes in the sPaper's scribble that intersect *pArgs.

 In response to this message, the sPaper sends msgScrDeleteStrokeArea to
 its scribble (after the rectangle is converted to the appropriate
 coordinate system).

 If the spRedisplay flag is true, then sPaper also dirties the specified
 rectangle in itself to cause repainting to occur.
*/
#define msgSPaperDeleteStrokes  MakeMsg(clsSPaper,18)


/****************************************************************************
 msgSPaperComplete  			takes nothing, returns STATUS
	Tells the sPaper that the current stroke is complete.

 See the "Typical Scenario" section for a description of why and when this
 message is sent.

 sPaper responds as follows.  If the sPaper has a scribble, it sends
 msgScrComplete to the scribble.  If there is no scribble, the sPaper self
 sends msgSPaperXlateCompleted to "complete" the translation, even though
 the resulting translation will be empty.
 
 If this message is received while the sPaper is handling msgInputEvent,
 the status returned from msgInputEvent will cause any grab to be released.

 See Also
	msgSPaperXlateCompleted
*/
#define msgSPaperComplete		MakeMsg(clsSPaper,3)


/****************************************************************************
 msgSPaperAbort 				takes nothing, returns STATUS
	Tells the sPaper to abort the entry of the current stroke.

 In response to this message, sPaper sends msgSPaperClear to self.
 
 If this message is received while the sPaper is handling msgInputEvent,
 the status returned from msgInputEvent will cause any grab to be released.
*/
#define msgSPaperAbort			MakeMsg(clsSPaper, 23)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Data Notification and Retrieval Messages			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSPaperXlateCompleted		takes OBJECT, returns STATUS
	Notifies observers that data is available from the sPaper.

 This message has two roles.

 Role 1:  This notification is sent to the sPaper's observers (including
 the listener) when the sPaper decides that translation is complete.  Note
 that the resulting "translation" might be empty.

 Role 2:  sPaper self sends this message when msgSPaperComplete has been
 received and there is nothing to translate.  In response to this message,
 sPaper sends the same message to its observers, as described in Role 1
 above.
*/
#define msgSPaperXlateCompleted	MakeMsg(clsSPaper,128)


/****************************************************************************
 msgSPaperGetXlateData  		takes P_XLATE_DATA, returns STATUS
	Passes back translated data.

 The sPaper's observers typically send this message in response to the
 sPaper's msgSPaperCompleted notification.  See the "Typical Scenario"
 section for more information.

 If there is no translator, or no scribbles to be translated, the sPaper
 passes back an empty xlist.

 Otherwise, the sPaper extracts the xlist from its translator.  If the
 sPaper's spPenCoords flag is true, the sPaper converts the xlist's
 coordinates to pen coordinates;  otherwise it converts the xlist's
 coordinates to local window coordinates.  Finally, the sPaper passes back
 the xlist.
 
 The client must free the passed back xlist.

 See Also
	msgSPaperGetXlateDataAndStrokes
	xlist.h
	xlfilter.h
	xlate.h
*/
#define msgSPaperGetXlateData	MakeMsg(clsSPaper,7)


/****************************************************************************
 msgSPaperGetXlateDataAndStrokes takes P_SPAPER_XDATA, returns STATUS
	Passes back translated data and its associated strokes.

 The sPaper's observers typically send this message (or
 msgSPaperGetXlateData) in response to the sPaper's msgSPaperCompleted
 notification.  See the "Typical Scenario" section for more information.

 This message is very similar in function to msgSPaperGetXlateData.  In
 fact the first two fields of pArgs for this message are the same as the
 fields of pArgs for msgSPaperGetXlateData

 The only difference between the two messages is that
 msgSPaperGetXlateDataAndStrokes also passes back the stroke information
 used to produce the translation.  The strokes are appended to the xlist as
 elements of type xtStroke16.

 If pArgs->toLWC is true, then the coordinate information in the strokes is
 converted to Local Window Coordinates (see win.h) before being passed
 back.

 The client must free the passed back xlist.

 See Also
	msgSPaperGetXlateData
*/
typedef struct SPAPER_XDATA {
	OS_HEAP_ID  heap;			// In: Heap to allocate space for stroke data
								// (Null means to use osProcessHeapId.)
	P_UNKNOWN   pXList;			// Out: pointer to xlist
	BOOLEAN	 	toLWC;		    // In: true to convert strokes to LWC 
								// coordinates
} SPAPER_XDATA, *P_SPAPER_XDATA;

#define msgSPaperGetXlateDataAndStrokes	MakeMsg(clsSPaper,8)


//REFGEN BEGINIGNORE
/****************************************************************************
 msgSPaperXlateComplete
	Obsolete.  Do not use.
*/
#define msgSPaperXlateComplete	MakeMsg(clsSPaper,10)
//REFGEN ENDIGNORE

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages Defined by Other Classes					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgFree	takes P_OBJ_KEY, returns STATUS
	Defined in clsmgr.h

 If the sPaper contains a scribble, it first removes the translator (if it
 exists) as an observer of the scribble.    It then sends msgDestroy to the
 scribble.  
 
 If the sPaper contains a translator, it first remove self as an observer
 of the translator and then send msgDestroy to the translator.
*/


/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Defined in clsmgr.h.

 An sPaper responds by sending msgResPutObject to its scribble and
 translator.  (If the scribble and/or translator is null, this effectively
 writes the "null object" id into the resource file.)
*/


/****************************************************************************
 msgRestore		takes P_OBJ_RESTORE, returns STATUS
	Defined in clsmgr.h.

 An sPaper responds by sending msgResGetObject to pArgs->file to restore 
 its scribble and translator that were saved while handling msgSave.

 If the restored translator is non-null, the sPaper makes itself an
 observer of the of the translator.  If both the translator and scribble
 are non-null, the sPaper makes the translator an observer of the scribble.
*/


/****************************************************************************
 msgSetOwner	takes P_OBJ_OWNER, returns STATUS
	Defined in clsmgr.h.

 In response, an sPaper sends this message to its translator and scribble (
 if they are non-null).  The sPaper then lets its ancestor (clsObject) set
 the sPaper's ownership.
*/


/****************************************************************************
 msgXlateCompleted	takes nothing, returns STATUS
	Defined in xlate.h.

 An sPaper receives this message because it is observing its translator. 
 The translator uses this message to indicate that translation has been
 complete and that data is available.

 In response to this message the sPaper self sends msgSPaperXlateCompleted,
 which results in msgSPaperXlateCompleted being sent to all the sPaper's
 observers.

 See Also
	msgSPaperXlateCompleted
*/


/****************************************************************************
 msgWinRepaint	takes nothing, returns STATUS
	Defined in win.h.

 An sPaper responds by (1) drawing any necessary grid lines in the window,
 and (2) if spRedisplay is true, sending msgScrRender to its scribble 
*/


/****************************************************************************
 msgWinSized	takes P_WIN_METRICS, returns STATUS
	Defined in win.h.

 If the window being resized is self, and a change in height has occurred,
 and the spFixedPos flag is true, then the sPaper's scribble's base is
 adjusted by the change in height.  This causes the  scribble to remain at
 a fixed position relative to the upper left corner of the window.  As a
 result of handling this message, msgSPaperGetCellMetrics and
 msgSPaperGetSizes will return different values.

 See Also
 	scribble.h
	msgSPaperGetCellMetrics
	msgSPaperGetSizes
*/


/****************************************************************************
 msgWinLayoutSelf	takes P_WIN_METRICS, returns STATUS
	Defined in win.h.

 If wsLayoutResize is on in pArgs->options, the sPaper picks a width of
//{
	(cols * cellWidth) + self's borderSize.w
//}
 and a height of
//{
	(rows * lineHeight) + self's borderSize.h
//}
*/


/****************************************************************************
 msgInputEvent	takes P_INPUT_EVENT, returns STATUS
	Defined in input.h.

 sPaper handles msgPenUp, msgPenDown, msgPenStroke and msgPenOutProxUp
 events.  
 
 An sPaper grabs input by returning stsInputGrabTerminate in response to
 msgPenDown.  
 
 If flags.spGrab is false, the sPaper relinquishes the grab by returning
 stsInputTerminate in response to msgPenUp.
 
 If flags.spGrab is true, the sPaper releases the grab by returning
 stsInputTerminate in response to msgPenOutProxUp.  msgPenOutProxUp also
 cause a self send of msgSPaperComplete if flags.spProx is set.

 msgPenStroke causes a self send of msgSPaperAddStroke.  
 
 All other msgInputEvent events return stsInputGrabIgnored or
 stsInputIgnored depending on the grab state the sPaper is in.

 Return Value
	stsInputTerminate
	stsInputGrabTerminate
	stsInputIgnored
	stsInputGrabIgnored
																    
 See Also
	msgSPaperComplete
	msgSPaperAddStroke
*/


/****************************************************************************
 msgSelDelete		takes U32, returns STATUS
	Defined in sel.h.

 In response to this message, the sPaper self sends msgSPaperClear.
*/


/****************************************************************************
 msgSelMoveSelection		takes P_XY32, returns STATUS
	Defined in sel.h.

 In response to this message, the sPaper first checks to see if the 
 selection owner can "speak" the xferScribbleObject data transfer type.  If
 it cannot, then the sPaper lets its ancestor process the message.

 If it can, and the selection owner is not self, then the sPaper gets the
 scribble from the selection owner, positions it as specified in pArgs, 
 self sends msgSPaperSetScribble, and finally sends msgSelDelete to the
 selection owner.
*/


/****************************************************************************
 msgSelCopySelection		takes P_XY32, returns STATUS
	Defined in sel.h.

 An sPaper's response to this message is identical to its response to
 msgSelMoveSelection except that the sPaper does not send msgSelDelete to
 the selection owner.

 See Also
	msgSelMoveSelection
*/


/****************************************************************************
 msgSelBeginMove		takes P_XY32, returns STATUS
	Defined in sel.h.

 In response to this message, the sPaper first verifies that it has a 
 scribble and that flags.spDataMoveable is true.  If either of these fail,
 the sPaper lets its ancestor process the message.

 Otherwise the sPaper computes the bounding box of the scribble and self
 sends msgEmbeddedWinBeginMove.
*/


/****************************************************************************
 msgSelBeginCopy		takes P_XY32, returns STATUS
	Defined in sel.h.

 In response to this message, the sPaper first verifies that it has a 
 scribble and that flags.spDataCopyable is true.  If either of these fail,
 the sPaper lets its ancestor process the message.

 Otherwise the sPaper computes the bounding box of the scribble and self
 sends msgEmbeddedWinBeginCopy.
*/


/****************************************************************************
 msgXferGet		takes P_UNKNOWN, returns STATUS
	Defined in xfer.h.

 If pArgs->id is xferScribbleObject, the sPaper creates a copy of its
 scribble and returns the copy in pArgs->uid.
*/


/****************************************************************************
 msgXferList		takes OBJECT, returns STATUS
	Defined in xfer.h.

 In response to this message, the sPaper adds the data transfer type
 xferScribbleObject to the list of data transfer types.
*/


/****************************************************************************
 msgTrackProvideMetrics		takes P_TRACK_METRICS, returns STATUS
	Defined in track.h.

 If pArgs->tag is tagMoveCopyIconTrack, the sPaper snaps the pen to the
 center-left of the move/copy icon.
*/


#endif
