/****************************************************************************
 File: sel.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.8  $
	 $Date:   04 Feb 1992 12:42:54  $

 This file contains the API for clsSelection.

 clsSelection inherits from clsObject.

 theSelectionManager provides management of the system-wide selection.
 theSelectionManager is the one and only instance of clsSelection.
****************************************************************************/

/**** Introduction ****/
/*
 Much of PenPoint's user interface is based on the "selection." The
 selection is often the center of the user's attention.  In general it is
 very easy for the user to set the selection -- it often just requires a
 tap.

 The precise definition of the selection is application-specific.  In text
 the selection is often a set of characters.  In a spreadsheet it might be
 a range of rows, columns, or cells.  In a Table of Contents it might be a
 set of documents.  Typically, an application "highlights" the selection
 with a grey background, handles, or some other graphic technique.

 Because the selection corresponds to the center of the user's attention,
 many user interface operations are based on the selection. Here are some
 examples:
    -:	The selection is the source of PenPoint's move and copy operations.
    -:	Typically, the selection is altered by Applying an Option Sheet.
    -:	The selection often determines which menu items are enabled and which
        are disabled.
    -:	The selection and keyboard input target are often linked together.

 Programmatically, other objects can inquire about the selection, get
 information from the selection and transfer data from the selection.
*/


/**** Road Map ****/
/*
 Use the following to take ownership of the selection:
    -:	msgSelSetOwner
    -:	msgSelSetOwnerPreserve
    -:	msgSelSelect (if object has clsEmbeddedWin in the object's ancestry)

 Selection owners must be prepared to handle the following:
    -:	msgSelDelete
    -:	msgSelYield
    -:	msgSelBeginCopy
    -:	msgSelBeginMove
	-:	msgControlProvideEnable (see section "Control Enabling")

 Use the following to inquire about the selection:
    -:	msgSelOwner
    -:	msgSelPrimaryOwner
    -:	msgSelOwners
    -:	msgSelIsSelected (if object has clsEmbeddedWin in the object's ancestry)

 theSelectionManager sends the following notifications:
    -:	msgSelChangedOwners
    -:	msgSelPromotedOwner

 Destinations of PenPoint's Move and Copy mechanism must handle the
 following:
    -:	msgSelCopySelection
    -:	msgSelMoveSelection
*/


/**** Move and Copy ****/
/*
 sel.h defines several messages that are used to implement PenPoint's Move
 and Copy operations.  These messages are used in combination with
 PenPoint's data transfer messages which are defined in xfer.h.  (PenPoint
 data transfer does not always necessarily involve the selection, but when
 it does, the messages described here are employed.)

 clsEmbeddedWin (see embedwin.h) provides the default response for several
 of the steps described below.

 Here's the typical "flow of control" for moving selected data:

    -:  The source object handles the "Press" gesture (xgsPressHold in
        xgesture.h).  The object might receive this gesture if it is a gWin
        (see gwin.h).  
 
    -:  If the Press gesture is not over the selection, the object
        typically selects what is under the gesture.  "Selecting" includes
        either (1) self sending msgSelSelect or (2) sending msgSelSetOwner
        to theSelectionManager, whichever is appropriate.

    -:	Next the object self-sends msgSelBeginMove.

    -:  msgSelBeginMove is received.  Note that msgSelBeginMove is sent in
        other cases than the Press gesture response.  For instance, the 
        standard application menu item "Move" (in the "Edit" menu) results
        in the selection owner receiving msgSelBeginMove.

    -:	In response to msgSelBeginMove, the receiver should self send
        msgEmbeddedWinBeginMove.  msgEmbeddedWinBeginMove takes, in its
        pArgs, the hot point of the gesture that kicks off the move, and
        the bounds of the selection being moved.

    -:  In response to msgEmbeddedWinBeginMove, embeddedWin creates the
        floating "move icon."  clsEmbeddedWin manages the icon.

    -:  The icon takes over at this point and manages the process of moving
        the selection.

    -:  When the icon is dropped on a destination, the icon sends
        msgMoveCopyIconDone to the source.

    -:  clsEmbeddedWin handles msgMoveCopyIconDone and sends
        msgSelMoveSelection to the destination.

    -:  In response to msgSelMoveSelection, the destination object
        retrieves the selection owner from the selection manager (using
        msgSelOwner) and engage in an xfer protocol with the selection.
        (The xfer protocols are described in xfer.h)  The data should be
        copied to the position contained in msgSelMoveSelection's pArgs,
        which is a P_XY32.

    -:	After the data has been copied from the selection owner, the
        destination should send msgSelDelete to the selection owner.

    -:	The destination object should select the data that it just absorbed.

 The "flow of control" for copying selected data is very similar, with the
 following changes:

    -:  The gesture that kicks off the protocol is "Tap-Press" (xgsTapHold
        in xgesture.h) rather than Press-Hold.

    -:	The source object self sends and handles msgSelBeginCopy rather than
        msgSelBeginMove.  The source object self sends 
        msgEmbeddedWinBeginCopy rather than msgEmbeddedWinBeginMove.

    -:	The destination receives msgSelCopySelection rather than
        msgSelMoveSelection.

    -:	The destination object should not send msgSelDelete.

 See Also
	xfer.h
	embedwin.h
*/


/**** Two Selection Owners ****/
/*
 Some objects need to own the selection, but they need to take in a fashion
 that (1) allows PenPoint to restore the original selection and (2) allows
 client code to find the original selection.  For example, Option Sheets
 apply to a selection.  But the various controls that appear within the
 option sheet might need to own the selection as well.  Both selections
 need to be maintained.

 Therefore theSelectionManager actually manages two selection owners:  a
 selection owner and a preserved selection owner.

 NOTE:  The same object cannot be both the selection owner and preserved
 selection owner.  See the detailed comments with msgSelSetOwner and
 msgSelSetOwnerPreserve for details.

 When an object needs to take the selection but allow the current selection
 to be restored, that object should take the selection via 
 msgSelSetOwnerPreserve, which "preserves" or "remembers" the original
 selection.  The preserved selection can be restored by sending
 msgSelOwnerPreserve with a pArgs of pNull to theSelectionManager.  Hence
 objects in option sheets take the selection via msgSelSetOwnerPreserve.

 Essentially all clients should operate on the selection owner.  This
 includes move and copy operations.  The only client that should operate on
 the preserved selection owner, if one exists, is option sheets.
*/


/**** Control Enabling ****/
/*
 Some controls, particularly menu items, should be disabled if there is no
 selection owner.  And some controls should be disabled based on
 application-specific details about the selection state.

 For instance, the "Move," "Copy," and "Delete" menu items should not be
 enabled if there is no selection owner.  The "Move" menu item should be
 enabled if there is a selection and the selection owner is not read-only. 
 The "Delete" menu item should be enabled if there is a selection owner and
 the contents of the selection are not empty.
 
 To support this, clsControl allows control creators to specify that the
 control should send msgControlProvideEnable to the selection owner to get
 the proper enable/disable state.

 Some standard application menus (SAMs) are set up to send
 msgControlProvideEnable to the selection owner.  See app.h for details.

 Therefore all selection owners should handle msgControlProvideEnable.
*/


/**** Relationship of Selection to the Input Target ****/
/*
 The input system's "Target" is the object to which keyboard events are
 sent.  See input.h for more information.

 Because the selection is normally the center of the user's attention, it
 often makes sense for the same object to own the selection and to be the
 input target.  For instance, PenPoint's text component always becomes the
 input target whenever it takes the selection and sets the input target to
 null when it yields the selection.

 There are, however, cases where it makes more sense to NOT link the
 selection and input target together.  For instance, some types of fields
 take the input target without taking the selection.  The decision is quite
 application-specific.

 Implementing a correspondence between the input target and selection
 ownership is the client's responsibility.
*/


/**** What to Do When the Selection Changes Within an Owner ****/
/*

 Some parts of PenPoint's UI depend on knowing when the user's center of
 attention changes.  For instance, each time that an Option Sheet is
 notified that the selection has changed it checks to be sure that the top
 card is still applicable.

 Therefore, selection owners should set the selection to self EVERY TIME
 THE SELECTION CHANGES within them, even if they are already the selection
 owner.  This lets observers take any appropriate action.
*/


/**** Only One Instance ****/
/*
 There is one and only one instance of clsSelection, and that instance is 
 the global well-known theSelectionManager.
*/



#ifndef SEL_INCLUDED
#define SEL_INCLUDED

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Status Codes ****/
/*
 * theSelectionManager returns stsSelYieldInProgress when the selection
 * manager is in the process of sending msgSelYield and therefore can't
 * respond to the message.
*/
#define stsSelYieldInProgress					MakeWarning(clsSelection,1)


/**** Types ****/
/*
 * preservedOwner is defined only if havePreservedOwner is true.  It IS
 * possible to have a null preservedOwner.
*/
typedef struct SEL_OWNERS {

	OBJECT		owner;
	OBJECT		preservedOwner;
	BOOLEAN		havePreservedOwner;

} SEL_OWNERS, *P_SEL_OWNERS;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages Sent to theSelectionManager				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

// Next Up: 26, Recycled: 9, 14, 20

/****************************************************************************
 msgSelSetOwner	takes OBJECT, returns STATUS
	Sets the selection owner.

 Send msgSelSetOwner to theSelectionManager to set the selection owner. 
 theSelectionManager responds in one of the following ways:

 If pArgs is not a valid selection owner (because it can't be called from
 other objects or is not a global object):
    -:	theSelectionManager returns stsScopeViolation.

 If pArgs is null, theSelectionManager:
    -:	sends msgSelYield to the current selection if it exists and sets the
        current selection to null.
    -:	sends msgSelYield the current preserved selection if it exists and
        sets the current preserved selection to null.
	-:	sends msgSelChangedOwners to theSelectionManager's observers.

 Otherwise, theSelectionManager:
    -:	sends msgSelYield to the current preserved selection if it exists
        and is not equal to pArgs.  theSelectionManager then sets the
        preserved selection to null and stops observing the preserved
        selection.
	-:	sends msgSelYield to the current selection if it exists and
		is not equal to pArgs.
	-:	sets the current selection to pArgs.
	-:	adds itself as an observer of the new selection.
	-:	sends msgSelChangedOwners to theSelectionManager's observers.

 Return Value
	stsScopeViolation:		pArgs is not a valid selection owner.

 See Also
	msgSelYield
	msgSelSetOwnerPreserve
	msgSelChangedOwners
*/
#define	msgSelSetOwner	MakeMsg(clsSelection,2)


/****************************************************************************
 msgSelSetOwnerPreserve	takes OBJECT, returns STATUS
	Sets the selection owner with the preserve option.

 Send msgSelSetOwnerPreserve to theSelectionManager to set the selection
 owner while preserving the current selection owner.

 See the section "Two Selection Owners" for more information.
 
 theSelectionManager's response to this message is similar to its response
 to msgSelSetOwner, with only subtle differences.

 If pArgs is null, and there is no preservedOwner:
	-:	theSelectionManager simply returns stsOK.

 If pArgs is null, and a preserved owner exists (even if it is null),
 theSelectionManager:
 	-:	sends msgSelYield to the current owner if it exists.
	-:	sends msgSelPromote to the current preserved owner if non-null.
	-:	sets the current owner to the current preserved owner if non-null.
	-:	sets the current preserved owner to null.
	-:	sets the value for SEL_OWNERS.havePreservedOwner to false.
	-:	sends msgSelPromotedOwner to theSelectionManager's observers.

 If pArgs is non-null but is not a valid selection owner (because it can't be
 called from other objects or is not a global object):
    -:	theSelectionManager returns stsScopeViolation.

 If pArgs is a valid selection owner and there is a no preserved owner:
 	-:	sends msgSelDemote to the current owner.
	-:	sets the current preserved owner to be the current owner.
	-:	sets the current owner to be pArgs.
	-:	adds itself as an observer of the new selection.
	-:	sets the value for SEL_OWNERS.havePreservedOwner to true.
	-:	sends msgSelChangedOwners to theSelectionManager's observers.

 If pArgs is a valid selection owner and there is a preserved owner:
 	-:	sends msgSelYield to the current owner if it exists and is
		not the same as pArgs.
	-:	sets the current owner to pArgs.
	-:	adds itself as an observer of the new selection.
	-:	sends msgSelChangedOwners to theSelectionManager's observers.

 Return Value
	stsScopeViolation:		pArgs is not a valid selection owner.

 See Also
	msgSelYield
	msgSelOwnerPreserve
	msgSelChangedOwners
*/
#define	msgSelSetOwnerPreserve	MakeMsg(clsSelection,5)


/****************************************************************************
 msgSelOwner	takes P_OBJECT, returns STATUS
	Passes back the selection owner.

 theSelectionManager passes back the current selection owner.  It does not
 pass back the preserved selection owner.

 Return Value
	stsSelYieldInProgress:	theSelectionManager is currently sending
							msgSelYield.
*/
#define	msgSelOwner	MakeMsg(clsSelection,1)


/****************************************************************************
 msgSelPrimaryOwner	takes P_OBJECT, returns STATUS
	Passes back the primary selection owner.

 The "primary owner" is the selection owner which an option sheet applies
 to.  If there is a preserved selection owner, the primary owner is the
 preserved owner. Otherwise, the primary selection owner is the current
 owner.

 Return Value
	stsSelYieldInProgress:	theSelectionManager is currently sending
							msgSelYield.

 See Also
	msgSelSetOwner
	msgSelOwners
*/
#define	msgSelPrimaryOwner	MakeMsg(clsSelection,7)


/****************************************************************************
 msgSelOwners	takes P_SEL_OWNERS, returns STATUS
	Passes back the selection and preserved owners.

 Return Value
	stsSelYieldInProgress:	theSelectionManager is currently sending
							msgSelYield.

 See Also
	msgSelSetOwner
	msgSelSetOwnerPreserve
*/
#define	msgSelOwners	MakeMsg(clsSelection,4)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *		  Notifications Sent to theSelectionManager's Observers			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSelChangedOwners	takes P_SEL_OWNERS, returns STATUS
	Notifies observers when either of the selection owners changes.

 theSelectionManager posts msgSelChangedOwners to its observers to inform
 the observers that the selection owner and/or preserved owner has been
 set.  (The notification is sent even if the new owner is null.)
 
 theSelectionManager sends this notification even if the old owner and new
 owner are the same.  Hence if object A is the selection owner, and
 msgSelSetOwner is sent with object A, msgSelChangedOwners IS sent to
 theSelectionManager's observers.
 
 When a preserved selection owner is promoted back to the selection owner,
 msgSelPromotedOwner is sent rather than msgSelChangedOwners.

 Example of use:  In response to this message, option sheets check the
 applicability of the top card.

 See Also
	msgSelSetOwner
	msgSelSetOwnerPreserve
	msgSelPromotedOwner
*/
#define	msgSelChangedOwners	MakeMsg(clsSelection,6)

/****************************************************************************
 msgSelPromotedOwner	takes P_SEL_OWNERS, returns STATUS
	Notifies observers when the preserved owner has been promoted back to
	the selection owner.

 theSelectionManager posts msgSelPromotedOwner to its observers to inform
 the observers that preserved selection owner has been promoted to the
 normal selection owner.

 This happens as a result of theSelectionManager handling 
 msgSelSetOwnerPreserve with a pArgs of null.
 
 See Also
	msgSelSetOwnerPreserve
*/
#define	msgSelPromotedOwner	MakeMsg(clsSelection,8)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *		  Messages Sent by theSelectionManager to Owners				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSelYield	takes BOOLEAN, returns STATUS
	theSelectionManager requires the release of the selection.

 theSelectionManager sends this message to a selection owner to inform the
 object that it is no longer the selection owner.  pArgs is true if object
 is yielding the primary selection and false when the object is yielding
 the preserved selection.   
 
 This message is not sent when an object takes the selection via
 msgSelSetOwner or msgSelSetOwnerPreserve and it already is the selection,
 or already is the preserved selection. (However, msgSelChangedOwners IS
 sent to theSelectionManager's observers.)
 
 When handling this message, be careful about sending selection manager
 messages (such as msgSelSetOwner) as deadlock can occur.

 After sending msgSelYield, theSelectionManager removes itself as an
 observer of the object.

 See Also
	msgSelSetOwner
	msgSelSetOwnerPreserve
*/
#define	msgSelYield	MakeMsg(clsSelection,11)


/****************************************************************************
 msgSelDemote	takes nothing, returns STATUS
	Informs the owner that it is becoming the preserved owner.

 theSelectionManager sends this message to a selection owner to tell the
 owner that it is becoming the preserved owner.  (This can happen when
 theSelectionManager receives msgSelSetOwnerPreserve.)

 Receivers should not do anything in response to this message.  (If for
 some reason receivers chose to handle this message, be careful about
 sending selection manager messages (such as msgSelSetOwner) as deadlock
 can occur.)

 See Also
	msgSelPromote
	msgSelSetOwner
	msgSelSetOwnerPreserve
*/
#define	msgSelDemote	MakeMsg(clsSelection,24)


/****************************************************************************
 msgSelPromote	takes nothing, returns STATUS
	Informs the  preserved owner that it is becoming the owner.

 theSelectionManager sends this message to a preserved selection owner to
 tell the owner that it is becoming the normal selection owner.  (This can
 happen when theSelectionManager receives msgSelSetOwnerPreserve.)

 Receivers should not do anything in response to this message.  (If for
 some reason receivers chose to handle this message, be careful about
 sending selection manager messages (such as msgSelSetOwner) as deadlock
 can occur.)

 See Also
	msgSelSetOwner
	msgSelSetOwnerPreserve
	msgSelDemote
*/
#define	msgSelPromote	MakeMsg(clsSelection,25)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 						Embedded Window Messages				 		   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Most subclasses of clsEmbeddedWin should use these messages.  See
 * embedwin.h for information about how and why to use them.
 *
 * The messages are defined here rather than in embedwin.h because they are
 * abstract.  Theoretically other classes can respond to these messages to
 * implement behavior analogous to that of embeddedWin (although no other
 * PenPoint system class does so).
*/

/****************************************************************************
 msgSelSelect	takes nothing, returns STATUS
	Sets self to be the selection owner.

 See the section "Embedded Window Selection Messages" for more information.

 Send this message to an object to have that object make itself be the
 selection owner or the preserved selection owner.

 Do not send this message to theSelectionManager.

 See Also
	msgSelSetOwner
	msgSelSetOwnerPreserve
	embedwin.h
*/
#define	msgSelSelect	MakeMsg(clsSelection,19)


/****************************************************************************
 msgSelIsSelected	takes nothing, returns BOOLEAN
	Returns TRUE if self is current selection owner.

 See the section "Embedded Window Selection Messages" for more information.

 Send this message to an object to inquire if it is the selection owner.
 
 Do not send this message to theSelectionManager.

 Return Value
	true:		The object is the selection owner.
	false:		The object is not the selection owner.  (The object may
				be the preserved selection owner.)

 See Also
	embedwin.h
*/
#define	msgSelIsSelected	MakeMsg(clsSelection,21)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  			Abstract Messages for Selection Move & Copy				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSelBeginCopy	takes P_XY32, returns STATUS
	Initiate a copy operation.

 See the section "Move and Copy" for information about when this message is
 sent and how it should be handled.

 pArgs will be null if this message is sent from a menu.
*/
#define	msgSelBeginCopy		MakeMsg(clsSelection, 23)


/****************************************************************************
 msgSelBeginMove	takes P_XY32, returns STATUS
	Initiates a move operation.

 See the section "Move and Copy" for information about when this message is
 sent and how it should be handled.

 pArgs will be null if this message is sent from a menu.
*/
#define	msgSelBeginMove		MakeMsg(clsSelection, 22)


/****************************************************************************
 msgSelCopySelection	takes P_XY32, returns STATUS
	The receiver should copy the selection to self at (x, y).

 See the section "Move and Copy" for information about when this message is
 sent and how it should be handled.
*/
#define	msgSelCopySelection	MsgNoError(MakeMsg(clsSelection,16))


/****************************************************************************
 msgSelMoveSelection	takes P_XY32, returns STATUS
	The receiver should move the selection to self at (x, y).

 See the section "Move and Copy" for information about when this message is
 sent and how it should be handled.
*/
#define	msgSelMoveSelection	MsgNoError(MakeMsg(clsSelection,15))


/****************************************************************************
 msgSelDelete	takes U32, returns STATUS
	The selection owner should delete the selection.

 Clients wishing to delete the selection send msgSelDelete to the selection
 owner.  Selection owners should respond to this message by deleting the
 contents of the selection.

 msgSelDelete is sent in two situations: (1) the user has hit the "Delete"
 menu item, or (2) an object has received msgSelMoveSelection, has copied
 the data (see xfer.h), and now wants to delete the original data.

 See the section "Move and Copy" for information about how msgSelDelete is
 related to moving data.

 pArgs must be one of SelDeleteReselect	or SelDeleteNoSelect.  This
 parameter is just a performance enhancement.  The sender of msgSelDelete
 should pass SelDeleteNoSelect if it plans on taking the selection after the
 msgSelDelete, and SelDeleteReselect otherwise.  The receiver of
 msgSelDelete can use pArgs as an optimization, but it is not strictly
 necessary since theSelectionManager will send a msgSelYield when the
 sender takes the selection.  (The pArgs of msgSelDelete exist primarily
 for historical reasons.  The simplest thing to do is for the sender to
 pass SelDeleteReselect and for the receiver to ignore pArgs.)
*/
#define	msgSelDelete	MakeMsg(clsSelection,3)

#define	SelDeleteReselect	0		// Display a selection after delete
#define	SelDeleteNoSelect	1		// Don't display a selection after delete


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  				Abstract Messages For Linking Protocol				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSelRememberSelection	takes P_XY32, returns STATUS
	The receiver should "remember" the selection and place the
	"remembrance" at (x, y).

 Most objects should not send or handle this message.  It might be better
 defined as a clsEmbeddedWin message.

 msgSelRememberSelection is sent to an object to ask it to "remember" the
 selection.  The response to this message is highly object specific.

 This message is not sent to the selection owner;  it is sent to any object
 to ask it remember the selection.

 An embeddedWin self sends this message in response to the "Create
 Reference Button" gesture (xgsDblCircle in xgesture.h).   In response, an
 embeddedWin creates a goto button at the specified (x,y).

 See Also
	embedwin.h
*/
#define	msgSelRememberSelection	MsgNoError(MakeMsg(clsSelection,17))


//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *		  						Obsolete								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSelOptions	takes pNull, returns STATUS
	Obsolete.  Do not use this message.
*/
#define	msgSelOptions	MsgNoError(MakeMsg(clsSelection,13))


/****************************************************************************
 msgSelOptionTagOK	takes TAG, returns STATUS
	Obsolete.  Do not use this message.
*/
#define	msgSelOptionTagOK	MsgNoError(MakeMsg(clsSelection,12))

//REFGEN ENDIGNORE

#endif
