/****************************************************************************
 ordset.h

 Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.17  $
   $Author:   msapsfor  $
     $Date:   27 Feb 1992 12:07:22  $

 This file contains the API definition for the OrderedSet interface.
 The functions described in this file are contained in MISC.LIB.
****************************************************************************/

/**** Overview ****/
/*
 An OrderedSet implements a growable, ordered set of items.  Each
 item has a key and associated data.  The ordered set knows about the
 structure of the key, but treats the data as uninterpreted bytes.
 The items in an ordered set are homogeneous: there is only one size
 for the key, and another size for the data, for all the items in the
 set.

 Keys are unsigned quantities, treated as either non-negative integers or
 indirect access identifiers.  The client specifies:
	-:	how keys are treated - direct or indirect;
	-:	for indirect keys - access and comparison functions;
	-:	whether duplicate keys are allowed;
	-:	the key size - it must be 1, 2, or 4 bytes.

 The data size (in bytes) is also specified by the client; it must be
 less than or equal to 1023.
 
 The client provides an initial estimate of the number of items in the
 ordered set when the set is created; the set will allocate more memory
 if the estimate proves to be too small.
*/

/**** Performance considerations ****/
/*
 The implementation of OrderedSet builds on the ByteArray storage
 abstraction.  This implies that either the number of elements in the
 set is small enough that it is not a problem to use a linear array
 representation for the set, or that the number of lookups dominates
 the number of insertions and deletions.
*/

/**** Indirect Keys and Two Comparison Routines ****/
/*
 Ordered sets with indirect keys have a funny property.  If you want
 to search for a key that already exists in the set, everything's
 just fine.  But if you want to do something with a key that ISN'T in
 the set (e.g. find out if the key is in the set), there is no
 indirect key to use.  (This problem also arises when clients ask
 ordered sets questions such as "What's the next entry with a key
 greater than this key k?")

 To solve this problem, indirect-keyed ordered sets must be provided
 two comparison routines by the creator.  The first routine (passed
 as the compareKey1Indirect in a called to OrderedSetExtend()) is
 used when the implementation needs to compare two keys that are both
 in the set.  The second routine (passed as compareKey1Direct in a
 call to OrderedSetExtend()) is used when the implementation needs to
 compare two keys, only one of which is in the set.

 Caution:
 
 If keys are indirect, OrderedSetFindMinMax(), 
 OrderedSetFindMaxMin(), and OrderedSetNext() return the indirect
 key, not the value the key references.
*/

/**** Known Limitations ****/
/*
 This package does not work correctly if the set has indirect keys
 and 0 (zero) is a legitimate key value.
*/
#ifndef ORDSET_INCLUDED
#define ORDSET_INCLUDED $Revision:   1.17  $ 

#include <bytarray.h>
#include <gosearch.h>		// For ACCESS/COMPARE_FUNC

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Private                                                                *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

//REFGEN BEGINIGNORE
//
// Potential Optimization: It would be possible to get back 2 of the
// bits assigned to sizeofData, while still allowing data sizes in the
// range of [0..1328] bytes, via an encoding scheme that would cause
// certain data sizes to waste up to 15 bytes.  This would require the
// client to keep track of the actual data size for sizeofData > 15.
// The encoding would be as follows:
//		sizeofData		actually means
//		[ 0.. 15]		[0..15]
//		[16.. 43]		[0..27]*4 + 16 => [16..124] in multiples of 4
//		[44..127]		[0..83]*16 + 128 => [128..1328] by 16's
//
//REFGEN ENDIGNORE

//REFGEN BEGINIGNORE
//
// Forward References
//
typedef struct ORDERED_SET * P_ORDERED_SET;
//REFGEN ENDIGNORE


typedef U32 (CDECL *READ_KEY_FUNC)(
				P_ORDERED_SET	p,
				P_UNKNOWN		pKey);

typedef struct ORDERED_SET {
	U16			 	indirectKeys	: 1;
	U16			 	uniqueKeys		: 1;	// TRUE => no duplicate keys
	U16			 	spare			: 2;	// Always set to 0
	U16			 	sizeofKeyMinus1	: 2;	// Number of bytes -1 a key needs
	U16			 	sizeofData		:10;	// Number of bytes data occupies
	P_BYTE_ARRAY 	items;					// Storage of actual items
	ACCESS_FUNC	 	access;
	COMPARE_FUNC	compareKey1Direct;
	COMPARE_FUNC	compareKey1Indirect;
	P_UNKNOWN	 	context;				// 1st arg to access() & compare()
	READ_KEY_FUNC	readKey;				// For internal use only!
} ORDERED_SET;


/****************************************************************************
 OrderedSetCountInternal returns BYTE_INDEX
	Returns the number of items currently stored in the ORDERED_SET.

 High-performance version of OrderedSetCount, but subject to change
 if the implementation of ordered sets changes.
*/
#define OrderedSetCountInternal(p) \
		(ByteArrayLength(p->items) / OrderedSetSizeofItem(p))


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Types and Constants                                                    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define stsOrdSetDuplicateKey	MakeStatus(clsMisc, 1)

//REFGEN BEGINIGNORE
// 
// Implementation depends on Nil(P_UNKNOWN) and the following two values
// having HighU16() == 0.
// 
//REFGEN ENDIGNORE
#define findNextKeyInOS		((P_UNKNOWN)1)
#define findPreviousKeyInOS	((P_UNKNOWN)2)

typedef struct OS_ITEM_INFO {
	U32			key;
	P_UNKNOWN	data;
	BOOLEAN		isDuplicate;
} OS_ITEM_INFO, *P_OS_ITEM_INFO;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Exported Functions and Macros                                          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 OrderedSetPrint returns void
	In debugging version, prints the contents of the ordered set.

 This function is undefined in the non-debugging version.
*/
#ifdef DEBUG

void EXPORTED
OrderedSetPrint(
	P_ORDERED_SET	p);

#endif // DEBUG


/****************************************************************************
 OrderedSetCreate returns STATUS
	Creates an ordered set.

 sizeofKey and sizeofData specify the size in bytes of each item's key
 and data, respectively.  The initialCount is a hint;  the ordered set
 will grow or shrink as needed.  However, if initialCount is
 approximately correct, performance will be better.  If
 initialCount=0, 1 will be assumed.  uniqueKeys should be TRUE if
 client wants all keys in the set to be unique, FALSE otherwise.  Only
 the osHeapLocal / osHeapShared flags in mode are used.

 Returns stsOK if able to create the set, in which case *pp will be the
 created set, otherwise *pp will be Nil(P_ORDERED_SET).
*/
STATUS EXPORTED
OrderedSetCreate(
	P_ORDERED_SET *	pp,
	OS_HEAP_MODE	mode,
	U8				sizeofKey,
	U8				sizeofData,
	U32				initialCount,
	BOOLEAN			uniqueKeys,
	BOOLEAN			indirectKeys);


/****************************************************************************
 OrderedSetSizeofKey returns U16
	Returns the size of a key in bytes.
*/
#define OrderedSetSizeofKey(p)	((U16)((p)->sizeofKeyMinus1 + 1))


/****************************************************************************
 OrderedSetSizeofItem returns U16
	Returns the size of an item (key plus data) in bytes.
*/
#define OrderedSetSizeofItem(p)	\
		((U16)(OrderedSetSizeofKey(p) + (p)->sizeofData))


/****************************************************************************
 OrderedSetHeapMode returns OS_HEAP_MODE
	Returns the heap mode with which the Ordered Set was created.
*/
#define OrderedSetHeapMode(p)	ByteArrayHeapMode((p)->items)


/****************************************************************************
 OrderedSetExtend returns STATUS
	Modifies the functions and context of an ordered set with indirect keys.

 Specifies access and comparison functions for an ordered set with
 indirect keys, as well as a context for those functions.

 See gosearch.h's description of binarySearch() for more information
 about the behaviors and parameters of the access and compare
 functions.
*/
void EXPORTED
OrderedSetExtend(
	P_ORDERED_SET	p,
	ACCESS_FUNC		access,
	COMPARE_FUNC	compareKey1Direct,
	COMPARE_FUNC	compareKey1Indirect,
	P_UNKNOWN		context);


/****************************************************************************
 OrderedSetContext returns P_UNKNOWN
	Get the context passed to access and compare functions.
*/
#define OrderedSetContext(_p) ((_p)->context)


/****************************************************************************
 OrderedSetModifyContext returns void
	Modify the context passed to access and compare functions.
*/
//REFGEN BEGINIGNORE
#ifdef DEBUG
#define OrderedSetModifyContext(_p, _c) \
		if ((_p)->indirectKeys) ((_p)->context = (_c)); \
		else Debugf("==> ERROR, File: %s, Line: %d\n", __FILE__, __LINE__)
#else
//REFGEN ENDIGNORE
#define OrderedSetModifyContext(_p, _c) ((_p)->context = (_c))
//REFGEN BEGINIGNORE
#endif
//REFGEN ENDIGNORE


/****************************************************************************
 OrderedSetDefaultAccess returns P_UNKNOWN
	Can be used as the client-specified access routine in OrderedSetExtend().

 In ordered sets with indirect keys the client must supply a routine
 that returns the address of the keys that are passed into the
 client-supplied comparison routine.  OrderedSetDefaultAccess computes
 the address of the key in the ordered set representation, and so may
 be used by clients as the access routine passed into
 OrderedSetExtend().
*/
P_UNKNOWN CDECL
OrderedSetDefaultAccess(
	const P_ORDERED_SET p,
	const BYTE_INDEX	index);


/****************************************************************************
 OrderedSetDestroy returns void
	Destroys an ORDERED_SET.
*/
void EXPORTED 
OrderedSetDestroy(
	P_ORDERED_SET	p);


/****************************************************************************
 OrderedSetInsert returns STATUS
	Inserts data with key into ordered set.

 Copies sizeofData bytes from the buffer pointed to by data.  Returns:
	stsOSOutOfMem: 			if no memory available, or
	stsOrdSetDuplicateKey:	if key is duplicate and unique keys required, or
	stsOK:					otherwise.

 If sizeofKey is less than 4 bytes, the least significant byte(s) of key
 are copied.
*/
STATUS EXPORTED
OrderedSetInsert(
	P_ORDERED_SET	p,
	U32		 		key,
	P_UNKNOWN 		data);


/****************************************************************************
 OrderedSetNthItem returns P_UNKNOWN
	Locates the n-th item in the ordered set (item indices begin with 0).

 Returns a pointer to ordered set's copy of the data associated with
 the Nth item.  This pointer is only valid until the next call on the
 same set.

 Upon return, the following modifications have been made to the fields
 of info:
 	key:			key of nth item
	isDuplicate:	is not set;  use OrderedSetFind() if needed;
	data:			duplicate of return value
*/
P_UNKNOWN EXPORTED
OrderedSetNthItem(
	P_ORDERED_SET	p,
	U32				n,
	P_OS_ITEM_INFO	info);


/****************************************************************************
 OrderedSetItemIndex returns BYTE_INDEX.
	Returns the index of an item
*/
#define OrderedSetItemIndex(p, pData) \
	((ByteArrayFindIndex((p)->items, ((P_U8)(pData))) \
	- OrderedSetSizeofKey(p)) / OrderedSetSizeofItem(p))


/****************************************************************************
 OrderedSetFind returns P_UNKNOWN
	Locates the data for a specified key.

 Returns a pointer to ordered set's copy of the data associated with
 info->key.  This pointer is only valid until the next call on the
 same set.  If the info->key is not in the set, the returned value is
 Nil(P_UNKNOWN).  If duplicate copies of the key exist in the set, an
 arbitrary item is found and its data returned.  All of the other
 items with the same key may be examined via use of OrderedSetNext(). 
 Upon return, the following modifications have been made to the 
 fields of info:
	isDuplicate:	0 if key is unique in set,
					1 otherwise
	data:			duplicate of return value
*/
P_UNKNOWN EXPORTED
OrderedSetFind(
	P_ORDERED_SET	p,
	P_OS_ITEM_INFO	info);


/****************************************************************************
 OrderedSetFindMinMax returns P_UNKNOWN
	Locates the data for a key >= to specified key.

 Returns a pointer to ordered set's copy of the data associated with
 the minimum key in the ordered set that is >= info->key.  If
 info->key is in the ordered set, this routine is equivalent to
 OrderedSetFind().  This pointer is only valid until the next call on
 the same set. Returns Nil(P_UNKNOWN) if info->key has no minmax in
 the set.  If duplicate copies of the minmax key exist in the set, an
 arbitrary item is found and its data returned.  All of the other
 items with the same key may be retrieved  with OrderedSetNext(). Upon
 return, the following modifications have been made to the fields of
 info:
	key:			minmax key
	isDuplicate:	0 if key is unique in set,
					1 otherwise
	data:			duplicate of return value
*/
P_UNKNOWN EXPORTED
OrderedSetFindMinMax(
	P_ORDERED_SET	p,
	P_OS_ITEM_INFO	info);


/****************************************************************************
 OrderedSetFindMaxMin returns P_UNKNOWN
	Locates the data for a key <= to specified key.

 Returns a pointer to ordered set's copy of the data associated with
 the maximum key in the ordered set that is <= info->key.  If
 info->key is in the ordered set, this routine is equivalent to
 OrderedSetFind().  This pointer is only valid until the next call on
 the same set. Returns Nil(P_UNKNOWN) if info->key has no maxmin in
 the set.  If duplicate copies of the maxmin key exist in the set, an
 arbitrary item is found and its data returned.  All of the other
 items with the same key may be retrieved  with OrderedSetNext(). Upon
 return, the following modifications have been made to the fields of
 info:
	key:			maxmin key
	isDuplicate:	0 if key is unique in set,
					1 otherwise
	data:			duplicate of return value
*/
P_UNKNOWN EXPORTED
OrderedSetFindMaxMin(
	P_ORDERED_SET	p,
	P_OS_ITEM_INFO	info);


/****************************************************************************
 OrderedSetNext returns P_UNKNOWN
	Enumerates the data for keys in the Ordered Set.

 OrderedSetNext()'s behavior depends on whether the set has unique
 keys or not.  In both cases, the enumeration is guaranteed to be
 complete provided no insertions or deletions are performed on the set
 during the enumeration.
 
 IF THE SET HAS UNIQUE KEYS:
 
 OrderedSetNext() enumerates all of the keys in the set in order.
 
 The first item in the enumeration can be found by either (1) by
 calling OrderedSetNthItem() with an "N" of 0 or (2) calling
 OrderedSetNext() with info->data set to Nil and info->key set to the
 lowest possible key value.
 
 IF THE SET DOES NOT HAVE UNIQUE KEYS:
 
 OrderedSetNext() enumerates all of the keys with the same value.  The
 order of enumeration is unspecified.
 
 The first item with a known key can be found by either (1) by calling
 OrderedSetFind with info->key set to the known key value and
 info->data set to Nil
 
 IN BOTH CASES:
 
 Further items are found by calling OrderedSetNext() with the same
 info struct until it returns Nil. OrderedSetNext() returns a pointer
 to the set's copy of the data associated with key.  This pointer is
 only valid until the next call on the same set.
 
 Returns
	Nil(P_UNKNOWN):					if specified key not in set or the
									enumeration is complete, or

	pointer to set's copy of data:	if key is in set or enumeration is 
									incomplete

 Upon return, the following modifications have been made to the 
 fields of info:
//{
	key:			next key value, iff info->data had been one of the
					three special values: Nil, next, prev.
	isDuplicate:	0 if key is unique in set,
					1 otherwise
	data:			duplicate of returned value
//}

 FOR SETS WITH DIRECT, NON-DUPLICATE KEYS ONLY:

 If the set has direct keys, setting info->data to findNextKeyInOS
 (findPreviousKeyInOS), OrderSetNext() can be used to enumerate all
 items in the set in order of increasing (decreasing) key value.  Such
 an enumeration (assuming non-unique keys) will have the structure:
//{
	info.key = 0; 
	info.data = Nil(P_UNKNOWN);
	if ((firstData = OrderedSetNext(...)) == Nil(P_UNKNOWN)) {
		info.data = findNextKeyInOS;
		if ((firstData = OrderedSetNext(...)) == Nil(P_UNKNOWN)) {
			// handle empty set
			...
		}
	}
	// firstData and info now contain first item's information

	// enumerate all keys
	do {
		// enumerate all data with the same key
		while (OrderedSetNext(...)) {
			...
		};	
		info.data = findNextKeyInOS;
	} until (!OrderedSetNext(...));
//}
*/
P_UNKNOWN EXPORTED
OrderedSetNext(
	P_ORDERED_SET	p,
	P_OS_ITEM_INFO	info);


/****************************************************************************
 OrderedSetEachItem returns P_UNKNOWN
	Helper macro to simplify the enumeration of an Ordered Set.

 This macro is only useful for sets with direct, non-duplicate keys!

 The arguments to OrderedSetEachItem() are:
	_p:		the ordered set to enumerate
	_item:	an OS_ITEM_INFO containing the enumerated item's info

 Code using these macros should look like:
		OS_ITEM_INFO	scratch;
		OrderedSetEachItem(os, scratch) {
			myPtr = (MY_PTR)scratch.data;
			...
		}
*/
#define OrderedSetEachItem(_p, _item)	\
	for ((_item).key = (U32)0, (_item).data = Nil(P_UNKNOWN); \
		 OrderedSetNext((_p), &(_item)) != Nil(P_UNKNOWN);)
		 // The condition IS the iteration step


/****************************************************************************
 OrderedSetDelete returns STATUS
	Deletes specified item from the Ordered Set.

 If duplicates are allowed, both info->key and info->data must be filled
 in by client; if keys are unique, only info->key need be filled in.

 Returns:
	stsOK: 			if item was found in set and deleted, or
	stsNoMatch:		if item not found in set, or
	STATUS < 0:		if internal error during deletion.
*/
STATUS EXPORTED
OrderedSetDelete(
	P_ORDERED_SET	p,
	P_OS_ITEM_INFO	info);


/****************************************************************************
 OrderedSetCount returns U32
	Returns the number of items currently stored in the ORDERED_SET.
*/
U32 EXPORTED
OrderedSetCount(
	P_ORDERED_SET	p);

#endif // ORDSET_INCLUDED
