/****************************************************************************
 File: modem.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.13  $
   $Author:   cleu  $
     $Date:   27 Mar 1992 18:24:28  $

 This file contains the API for clsModem.

 clsModem inherits from clsService.

 clsModem provides the interface a client uses to communicate via a
 modem.  The modem service is located, bound to, opened, and closed
 via standard PenPoint service messages.

 The object which opens a modem service becomes its client.  After
 opening a modem service, it is recommened that a client explicitly
 reset the modem firmware, initialize the modem I/O port settings,
 and then set the modem firmware to the desired state.

 The modem firware is reset by sending msgModemReset to an
 open modem service.  Refer to msgModemReset below for a
 description of the state to which the modem firmware is reset.

 A client obtains current modem I/O port settings by sending
 msgSioGetMetrics to a modem service.  I/O port settings may
 be altered by sending msgSioSetMetrics to the modem service.
 These messages in addition to msgSioInit, msgSioBreakSend,
 msgSioControlInStatus, msgSioInputBufferStatus, and
 msgSioInputBufferFlush are the only clsMILAsyncSIODevice
 messages which clsModem handles.  Refer to file "sio.h" for
 a description of these messages.

 After initializing the modem I/O port, a client may then send
 clsModem messages to initialize the modem to a desired state.
 Such initialization may be accomplished via discrete messages,
 or via msgSvcSetMetrics.

 Upon successfully initializing a modem, the client may then establish
 a connection, transmit data and/or receive data via the connection, and
 finally terminate the connection.  Clients send clsStream messages to
 read/write data from/to the modem.  Refer to file "stream.h" for a
 description of clsStream messages.

	**** PLEASE NOTE ****

	In a future release of PenPoint, the clsModem API will be augmented.
	Backward compatibility with the clsModem API described herein shall
	be maintained for at least one release.

 ---------------------------------------------------------------------------
		
	Defined within this header file for the clsModem API:

	o	defines and typedefs.
	o	messages.
	o	status values.

	See Also: "service.h", "stream.h".

****************************************************************************/
#ifndef MODEM_INCLUDED
#define MODEM_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif
#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif
#ifndef SERVICE_INCLUDED
#include <service.h>
#endif
#ifndef UID_INCLUDED
#include <uid.h>
#endif
#ifndef	DIALENV_INCLUDED
#include <dialenv.h>
#endif



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	 					Observer Notification Messages					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgModemActivity			takes MODEM_ACTIVITY,   			returns N/A
	category: observer notification
	Notification sent to observers signifying changes in modem activity.

 NOTE: A modem service needn't implement all observer notifications listed
 below.  Those marked with an asterik are the required minimum.
*/
#define msgModemActivity			MakeMsg(clsModem, 1)

Enum32(MODEM_ACTIVITY) {		// The current modem activity/state.
	mdmOpened,					//	Modem service has been opened for use. *
	mdmResetting,				//	Currently being reset.
	mdmDialing,					//	Dialing a phone number. *
	mdmAwaitingConnection,		//	Awaiting a connection/answer.
	mdmConnected,				//	Connected with remote node. *
	mdmNegotiating,				//	Negotiating session/link parms.
	mdmSending,					//	Sending data.
	mdmReceiving,				//	Receiving data.
	mdmAnswering,				//	Answering a call. *
	mdmHangingUp,				//	Terminating the connection. *
	mdmDisconnected,			//	Connection terminated. *
	mdmClosed					//	Modem service has been closed. *
};								//  * = required notifications.


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	 					Client Notification Messages					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgModemResponse			takes MODEM_RESPONSE_INFO,		returns N/A
	category: client notification
	Provides the modem's response to a command.

 Provides the response to a previous modem request/command.  msgModemReponse
 is only sent to the modem service's client if the response behavior has been
 set to mdmResponseViaMessage (RE: msgModemSetResponseBehavior).
*/
#define msgModemResponse			MakeMsg(clsModem, 2)

/*
	Possible modem responses ...

	  If a desired response isn't available below, then
		please contact GO Corporation to see that it gets
		  added as a standard modem response.  Thank you.
*/
Enum32(MODEM_RESPONSE) {		// Modem response indications.
	mdmResOK,					//	OK - command accepted.
	mdmResUnrecognized,			//	Error - Unrecognized response from modem.
	mdmResError,				//	Error - Error response from modem.
	mdmResNoCarrier,			//	Error - No line carrier detected.
	mdmResNoDialTone,			//	Error - No phone dial tone detected.
	mdmResPhoneBusy,			//	Error - Phone line busy signal detected.
	mdmResNoAnswer,				//	Error - No one answered at the other end.
	mdmResInvalidFrame,			//	Error - Invalid frame detected.
	mdmResCRCError,				//	Error - Cyclic redundancy check error.
	mdmResRing,					//	Ring indication signal detected.
	mdmResConnect,				//	Connection established.
	mdmResConnect300,			//	300 baud connection established.
	mdmResConnect600,			//	600 baud connection established.
	mdmResConnect1200,			//	1200 baud connection established.
	mdmResConnect2400,			//	2400 baud connection established.
	mdmResConnect4800,			//	4800 baud connection established.
	mdmResConnect9600,			//	9600 baud connection established.
	mdmResConnect19200,			//	19200 baud connection established.
	mdmResConnectReserved01,	//	Reserved for future expansion.
	mdmResConnectReserved02,	//	Reserved for future expansion.
	mdmResConnectReserved03,	//	Reserved for future expansion.
	mdmResConnectMNP,			//	MNP connection has been established.
	mdmResConnect1200MNP,		//	1200 baud MNP connection established.
    mdmResConnect2400MNP,		//	2400 baud MNP connection established.
    mdmResConnect1200LAPM,		//	1200 baud LAPM connection established.
    mdmResConnect2400LAPM,		//	2400 baud LAPM connection established.
	mdmResConnectReserved04,	//	Reserved for future expansion.
	mdmResConnectReserved05,	//	Reserved for future expansion.
	mdmResConnectReserved06		//	Reserved for future expansion.
};
/*	 	
	NOTE: The modem service depends upon the order
			in which the above responses are defined.
*/

#define	mdmSizeMaxResponse		63

typedef struct {				// Modem response information.
	U8	symbols[mdmSizeMaxResponse+1];//  Symbols comprising a response.
	MODEM_RESPONSE	response;	//	Response meaning as enumerated above.
	U32				spare;		//	Reserved for future expansion.
} MODEM_RESPONSE_INFO, *P_MODEM_RESPONSE_INFO;


/****************************************************************************
 msgModemConnected			takes nothing,					returns N/A
	category: client notification
	Notification sent to the client indicating the modem has connected
	with a remote node modem.

 A client may obtain information regarding the connection via
 msgModemGetConnectionInfo.
*/
#define msgModemConnected			MakeMsg(clsModem, 3)


/****************************************************************************
 msgModemDisconnected		takes nothing,					returns N/A
	category: client notification
	Notification sent to the client indicating that the current connection
	has been terminated.
*/
#define msgModemDisconnected		MakeMsg(clsModem, 4)


/****************************************************************************
 msgModemRingDetected		takes nothing,					returns N/A
	category: client notification
	Notification sent to the client indicating that a ring indication
	has been received from the modem.
*/
#define msgModemRingDetected		MakeMsg(clsModem, 5)


/****************************************************************************
 msgModemTransmissionError		takes nothing,					returns N/A
	category: client notification
	Notification sent to the client indicating that an error has been
	detected during transmission (sending or receiving) of data.

 This unsolicited message is typically sent as a result of detecting a
 data framing error, or other low-level modem link protocol generated
 error condition.
*/
#define msgModemTransmissionError	MakeMsg(clsModem, 6)


/****************************************************************************
 msgModemErrorDetected		takes nothing,					returns N/A
	category: client notification
	Notification sent to the client indicating that an unexpected error
	indication has been received from the modem.
*/
#define msgModemErrorDetected		MakeMsg(clsModem, 7)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	 						Action Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgModemSetResponseBehavior takes P_MODEM_RESPONSE_BEHAVIOR, returns STATUS
	category: modem service request
	Set the modem response mode, and command-to-response time-out values.

 Response mode mdmResponseViaStatus causes the modem service to block and
 await a response from the modem.  If the modem doesn't return a response
 within the specified time-out duration, stsTimeOut is returned.

 Response mode mdmResponseViaMessage is useful for clients that wish to
 ObjectPostAsync their modem service requests, and hence not block until
 completion (or timeout) of the request.  Modem responses are reported to
 the client via msgModemResponse.

 Response mode mdmResponseTransparent disables the modem service response
 processing sub-system.  Modem command responses are left unaltered within
 the input data stream.  The client assumes responsibility for processing
 modem responses.  All commands successfully sent to the modem return a
 status of stsOK.

 NOTE: Once a client switches to transparent mode (or sends modem register
 altering commands via msgModemSendCommand) they are responsible for the
 integrity of clsModem.  Therefore, it is the client's responsibility to
 ensure that the clsModem (and the modem) are reset to a known state upon
 switching from transparent mode to a different response mode.
*/
#define msgModemSetResponseBehavior MakeMsg(clsModem, 16)

Enum32(MODEM_RESPONSE_MODE) {	// Mode for conveying modem responses.
	mdmResponseViaStatus,		//	Report via status (Default).
	mdmResponseViaMessage,		//	Report via notification msgModemResponse.
	mdmResponseTransparent		//	Don't intercept and process modem responses.
};

#define	mdmDefaultCommandTimeout	2500	// 2 1/2 second command timeout.
#define	mdmDefaultConnectTimeout	30000	// 30 second connect timeout.

typedef struct {							// Command-to-response timeouts.
    OS_MILLISECONDS		timeoutCommand;		//	Timeout for all commands
											//	  except connect requests
											//	  (default of 2 1/2 seconds).
    OS_MILLISECONDS		timeoutConnect;		//	Timeout for connect requests
											//	  (default of 30 seconds).
} MODEM_TIMEOUT, *P_MODEM_TIMEOUT;

typedef struct {				// Modem command-response handling behavior.
	MODEM_RESPONSE_MODE	mode;	//	Mode for coveying responses
	MODEM_TIMEOUT		timeout;//	Command-to-response timeouts.
} MODEM_RESPONSE_BEHAVIOR, *P_MODEM_RESPONSE_BEHAVIOR;


/****************************************************************************
 msgModemGetResponseBehavior takes P_MODEM_RESPONSE_BEHAVIOR, returns STATUS
	category: modem service request
	Passes back the current modem response mode, and the current
	command-to-response time-out values.
*/
#define msgModemGetResponseBehavior MakeMsg(clsModem, 17)


/****************************************************************************
 msgModemSendCommand		takes P_MODEM_SEND_COMMAND,		returns STATUS
	category: modem service request
	Sends a specified command to the modem.

 The timeout value specified within MODEM_SEND_COMMAND supersedes that
 specified via msgModemSetResponseBehavior.

 NOTE: Clients should only use msgModemSendCommand to perform modem actions
 unavailable via the clsModem API described herein.

 NOTE: Clients that send commands that alter modem registers are responsible
 for the integrity of clsModem.  Therefore, it is the client's responsibility
 to ensure that such commands will not adversely affect clsModem.
*/
#define msgModemSendCommand			MakeMsg(clsModem, 18)

#define	mdmSizeMaxCommand			80		// Max command size is 80 bytes.

typedef struct {
    P_U8				pCmdStr;   			// In:  Ptr to command string
											//		(null terminated).
    OS_MILLISECONDS		timeout;			// In:  Timeout for cmd response.
    MODEM_RESPONSE_INFO	responseInfo;		// Out: The response to the cmd.
} MODEM_SEND_COMMAND, *P_MODEM_SEND_COMMAND;


/****************************************************************************
 msgModemGetConnectionInfo takes P_MODEM_CONNECTION_INFO,	returns STATUS
	category: modem service request
	Passes back information about the current connection.
*/
#define msgModemGetConnectionInfo	MakeMsg(clsModem, 19)

Enum32(MODEM_CONNECTION) {			// The type of connection established.
	mdmConnectionNone,				//	None;  Disconnected.
	mdmConnectionStandard,			//	Standard data.
	mdmConnectionMNP,				//	MNP.
	mdmConnectionLAPM				//	LAPM.
};

Enum32(MODEM_LINK_CONTROL) {			// Link and error control protocols.
	mdmLinkControlMNPClass1_4 = flag0,	//	MNP Levels 1 through 4.
	mdmLinkControlMNPClass5	= flag1,	//	MNP Level 5 data compression.
	mdmLinkControlMNPClass6	= flag2,	//	MNP Level 6.
	mdmLinkControlMNPClass7	= flag3,	//	MNP Level 7 data compression.
	mdmLinkControlV42		= flag4,	//	Physical level error detection and
										//	  correction (LAPM link control).
	mdmLinkControlV42bis	= flag5		//	V42 data compression.
};

typedef struct {					// Information about a connection.
	MODEM_CONNECTION	connection;	//	The type of connection.
	MODEM_LINK_CONTROL	linkControl;//	Link control in use, if any/known.
	S32					baudRate;	//	Baud rate of connection.
	U32					spare[2];	//	Reserved for future expansion.
} MODEM_CONNECTION_INFO, *P_MODEM_CONNECTION_INFO;


/****************************************************************************
 msgModemReset				takes nothing,					returns STATUS
	category: modem service request
	Resets the modem firmware, I/O port, and service state.

 NOTE: The modem I/O port baud rate is reset to the highest supported data
 mode baud rate.  Therefore not all implementations will reset the baud rate
 to 2400.  The client may elect to subseqently change the baud rate for auto-
 baud detecting modems.

 Reset I/O port state:

//{
	baud				= 2400;
	line.dataBits		= sioEightBits;
	line.stopBits		= sioOneStopBit;
	line.parity			= sioNoParity;
	controlOut.rts		= true;
	controlOut.dtr		= true;
	flowChar.xonChar	= 0x11;
	flowChar.xoffChar	= 0x13;
	flowType.flowControl = sioNoFlowControl;
//}

 Reset modem firmware state:
	 
	Speaker control on until carrier detected (*).
	Speaker volume medium (*).
	Dial-tone detection enabled (*).
	Busy-tone detection enabled (*).
	Dialing mode from dialing environment.
	Auto-answer disabled.
	Answer on ring zero.
	Local character echo disabled.
	Send command result codes.
	Send verbal result codes (words).
	Enable carrier upon connect.
	Escape code + (ASCII 43).
	Command termination code carriage return (ASCII 13).

	(*)		or set as per current modem option card setting.
*/
#define msgModemReset				MakeMsg(clsModem, 20)


/****************************************************************************
 msgModemOffHook			takes nothing,					returns STATUS
	category: modem service request
	Picks up the phone line.
*/
#define msgModemOffHook				MakeMsg(clsModem, 21)


/****************************************************************************
 msgModemOnline				takes nothing,					returns STATUS
	category: modem service request
	Forces the modem online into data mode.
*/
#define msgModemOnline				MakeMsg(clsModem, 22)


/****************************************************************************
 msgModemSetDialType		takes MODEM_DIAL_MODE,			returns STATUS
	category: modem service request
	Establishes the mode for dialing telephone numbers.
*/
#define msgModemSetDialType			MakeMsg(clsModem, 23)

Enum32(MODEM_DIAL_MODE) {		// Mode in which the modem is to dial.
	mdmPulseDialing,			//	Perform pulse dialing.
    mdmTouchtoneDialing,		//	Peform touch-tone dialing.
	mdmDialStringDialing,		//	Client supplies the dialing mode
								//	  embedded within the dialString.
	mdmDialingEnvironmentDialing//	If available, use the current dialing
								//	  environment dialing mode, otherwise use
								//	  current modem firmware dialing mode (Default).
};


/****************************************************************************
 msgModemDial				takes P_MODEM_DIAL,				returns STATUS
	category: modem service request
	Performs dialing and attempts to establish a connection.
*/
#define msgModemDial				MakeMsg(clsModem, 24)

typedef struct {						// Dialing and connection type.
	DIALENV_DIAL_STRING	dialString;		//	In:  Phone number to dial.
	U32					spare[2];		//  Reserved for future expansion.
} MODEM_DIAL, *P_MODEM_DIAL;


/****************************************************************************
 msgModemSetAutoAnswer		takes P_MODEM_SET_AUTO_ANSWER,	returns STATUS
	category: modem service request
	Disables or enables the modem auto-answer feature. 

 NOTE: For some modems a value of 0 for rings disables auto-answer.
*/
#define msgModemSetAutoAnswer		MakeMsg(clsModem, 25)

Enum32(MODEM_AUTO_ANSWER) { 			// Modem auto-answer capability.
	mdmAutoAnswerDisabled,				//  Disable auto-answer (Default).
	mdmAutoAnswerEnabled				//	Enable auto-answer.
};

typedef struct {						// Auto-answer settings.
	MODEM_AUTO_ANSWER	autoAnswer;		//	In: Enable/disable auto-answer.
    S32					rings;			//	In: Number of rings before answer.
} MODEM_SET_AUTO_ANSWER, *P_MODEM_SET_AUTO_ANSWER;


/****************************************************************************
 msgModemSetAnswerMode		takes MODEM_ANSWER_MODE,		returns STATUS
	category: modem service request
	Filters the type of calls to answer and connection reporting.

 NOTE: Not all modems are capable of discriminating between the type of
 incoming call.
*/
#define msgModemSetAnswerMode		MakeMsg(clsModem, 26)

Enum32(MODEM_ANSWER_MODE) {				// Modem answer filter/mode.
	mdmAnswerDataMode		= flag0,	//	Answer in data mode.
	mdmAnswerFaxMode		= flag1,	//	Answer in fax mode.
	mdmAnswerVoiceMode		= flag2		//	Answer in voice mode.
};


/****************************************************************************
 msgModemAnswer				takes nothing,					returns STATUS
	category: modem service request
	Immediately answers a telephone call.
*/
#define msgModemAnswer				MakeMsg(clsModem, 27)


/****************************************************************************
 msgModemHangUp				takes nothing,					returns STATUS
	category: modem service request
	Hang-ups and disconnects to terminate a connection.
*/
#define msgModemHangUp				MakeMsg(clsModem, 28)


/****************************************************************************
 msgModemSetSignallingModes	takes P_MODEM_SIGNALLING_MODES,	returns STATUS
	category: modem service request
	Establishes/restricts the modem to use specific signalling modes/standards.

 NOTE: Not all modems provide support for selecting signalling modes.
*/
#define msgModemSetSignallingModes	MakeMsg(clsModem, 29)

Enum32(MODEM_SIGNALLING_VOICEBAND) {	// Voice-band signalling standards.
	mdmVoiceBandBell103J	= flag0,	//	300 BPS.
	mdmVoiceBandBell212A	= flag1,	//	1200 BPS.
	mdmVoiceBandV21			= flag2,	//	300 BPS duplex modem on GSTN.
	mdmVoiceBandV22			= flag3,	//	1200 BPS duplex modem on GSTN
										//	  or P-P leased two-wire circuits.
	mdmVoiceBandV22bis		= flag4,	//	2400 BPS duplex modem on GSTN
										//	  or P-P two-wire leased circuits.
	mdmVoiceBandV23			= flag5,	//	600/1200 BPS modem on GSTN.
	mdmVoiceBandV26			= flag6,	//	2400 BPS modem on four-wire
										//	  leased circuits.
	mdmVoiceBandV26bis		= flag7,	//	2400/1200 BPS modem on GSTN.
	mdmVoiceBandV26ter		= flag8,	//	2400 BPS duplex modem on GSTN
										//	  or P-P two-wire leased circuits.
	mdmVoiceBandV27			= flag9,	//	4800 BPS on leased circuits.
	mdmVoiceBandV27bis		= flag10,	//	2400/4800 BPS on leased circuits.
	mdmVoiceBandV27ter		= flag11,	//	4800/2400 BPS modem on GSTN.
	mdmVoiceBandV29			= flag12,	//	9600 BPS FDX or HDX modem on
										//	  P-P four-wire leased circuits.
	mdmVoiceBandV32			= flag13,	//	9600/4800 BPS duplex modem on
										//	  GSTN or leased circuits.
	mdmVoiceBandV33			= flag14	//	14400 BPS modem on P-P
										//	  four-wire leased circuits.
};

Enum32(MODEM_SIGNALLING_WIDEBAND) {		// Wide-band signalling standards.
	mdmWideBandV35			= flag0,	//	48 KBPS data transmission on
										//	  60-108 KHz group band circuits.
	mdmWideBandV36			= flag1,	//	48-72 KBPS sync data transmission
										//	  on 60-108 KHz group band circuits.
	mdmWideBandV37			= flag2		//	72-168 KBPS sync data transmission
										//	  on 60-108 KHz group band circuits.
};

typedef struct {						// Modem modulation/signalling modes.
	MODEM_SIGNALLING_VOICEBAND	voiceBand;	//	Voice band signalling.
	MODEM_SIGNALLING_WIDEBAND	wideBand;	//	Wide band signalling.
} MODEM_SIGNALLING_MODES, *P_MODEM_SIGNALLING_MODES;


/****************************************************************************
 msgModemSetToneDetection	takes MODEM_TONE_DETECTION,		returns STATUS
	category: modem service request
	Enables or disables busy tone and/or dial tone detection.
*/
#define msgModemSetToneDetection	MakeMsg(clsModem, 30)

Enum32(MODEM_TONE_DETECTION) {	// Busy and dial toneCarrier signal on/off.
	mdmToneDetectDisable,		//	Detect neither busy tone or dial tone.
	mdmToneDetectBusyOnly,		//	Detect busy tone, but not dial tone.
	mdmToneDetectDialOnly,		//	Detect dial tone, but not busy tone.
	mdmToneDetectBusyAndDial	//	Detect dial tone and busy tone (Default).
};


/****************************************************************************
 msgModemSetSpeakerControl   takes MODEM_SPEAKER_CONTROL,	returns STATUS
	category: modem service request
	Enables, disables and controls modem speaker behavior.
*/
#define msgModemSetSpeakerControl	MakeMsg(clsModem, 31)

Enum32(MODEM_SPEAKER_CONTROL) {	// Specifies the modem speaker behavior.
	mdmSpeakerOn,				//	Speaker is always on.
	mdmSpeakerOff,				//	Speaker is always off.
	mdmSpeakerOnConnectOff		//	Speaker on until carrier detected (Default).
};


/****************************************************************************
 msgModemSetSpeakerVolume	takes MODEM_SPEAKER_VOLUME,		returns STATUS
	category: modem service request
	Sets the volume of the modem speaker.

 NOTE: Not all modems are capable of adjusting modem speaker volume.
*/
#define msgModemSetSpeakerVolume	MakeMsg(clsModem, 32)

Enum32(MODEM_SPEAKER_VOLUME) {	// Specifies the modem speaker volume.
	mdmSpeakerVolumeWhisper,	//	Lowest volume level.
	mdmSpeakerVolumeLow,		//	Low/reasonable volume level.
	mdmSpeakerVolumeMedium,		//	Normal/average volume level (Default).
	mdmSpeakerVolumeHigh		//	Highest volume level.
};


/****************************************************************************
 msgModemSetCommandState	takes nothing,					returns STATUS
	category: modem service request
	Sets the modem into command mode.
*/
#define msgModemSetCommandState		MakeMsg(clsModem, 33)


/****************************************************************************
 msgModemSetDuplex			takes MODEM_DUPLEX_MODE,		returns STATUS
	category: modem service request
	Sets the duplex mode for inter-modem communications while on-line.

 NOTE: Not all modems are capable of setting the duplex once on-line.
*/
#define msgModemSetDuplex			MakeMsg(clsModem, 34)

Enum32(MODEM_DUPLEX_MODE) {		// Indicates data transmission line duplex.
	mdmDuplexHalf,				// Data transmitted in one direction at a
								//   time ( the line must be turned around).
	mdmDuplexFull				// Data can be transmitted in both
								//   directions simultaneously (Default).
};


/****************************************************************************
 msgModemSetMNPMode           takes MODEM_MNP_MODE,			returns STATUS
	category: modem service request
	Sets the MNP mode of operation.

 NOTE: Not all modems provide MNP support.
*/
#define msgModemSetMNPMode			MakeMsg(clsModem, 35)

Enum32(MODEM_MNP_MODE) {		// MNP mode in which modem is to operate.
	mdmMNPModeDirect,			//	Disable MNP mode (default).
    mdmMNPModeReliable,			//	Both modems must support MNP levels
								//	 1-4 (5 if enabled) before a connection
								//	 can be made.
    mdmMNPModeAutoReliable,		//	Attempt to establish an MNP connection; if
								//	 it fails establish a direct connection.
    mdmMNPModeLAPM				//	LAPM connection.
};


/****************************************************************************
 msgModemSetMNPCompression	takes MODEM_MNP_COMPRESSION,	returns STATUS
	category: modem service request
	Sets MNP class 5 compression on or off.
*/
#define msgModemSetMNPCompression	MakeMsg(clsModem, 36)

Enum32(MODEM_MNP_COMPRESSION) {	// Type of compression to use in MNP mode.
	mdmMNPCompressionOff,		//	Disable MNP level 5 compression (default).
    mdmMNPCompressionOn			//	Enable MNP level 5 compression.
};


/****************************************************************************
 msgModemSetMNPBreakType	takes MODEM_MNP_BREAK_TYPE,		returns STATUS
	category: modem service request
	Specify how a break is handled in MNP mode.
*/
#define msgModemSetMNPBreakType		MakeMsg(clsModem, 37)

Enum32(MODEM_MNP_BREAK_TYPE) {	// How breaks are handled in MNP mode.
	mdmMNPSendNoBreak,			//	Do not send break to remote modem.
    mdmMNPEmptyBuffersThenBreak,//	Empty data buffers before sending break.
    mdmMNPImmediatelySendBreak,	//	Send break when received (default).
    mdmMNPSendBreakInSequence	//	Send break relative to data to be sent.
};


/****************************************************************************
 msgModemSetMNPFlowControl	takes MODEM_MNP_FLOW_CONTROL,	returns STATUS
	category: modem service request
	Specify the flow control to use in MNP mode.
*/
#define msgModemSetMNPFlowControl	MakeMsg(clsModem, 38)

Enum32(MODEM_MNP_FLOW_CONTROL) {// Indicates the flow control for MNP mode.
	mdmMNPFlowControlDisable,	//	No flow control used (default).
    mdmMNPFlowControlXonXoff,	//	Use Xon/Xoff flow control.
    mdmMNPFlowControlHardware	//	Use RTS/CTS flow control.
};


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *		 					Superclass Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSvcGetMetrics			takes P_SVC_GET_SET_METRICS,	returns STATUS
	category: superclass message
	Passes back the current modem metrics.

 The pMetrics field of SVC_GET_SET_METRICS is expected to point to a buffer
 capable of receiving MODEM_METRICS as described below.
*/

typedef struct MODEM_METRICS {
	MODEM_DIAL_MODE			mdmDialMode;
	MODEM_DUPLEX_MODE		mdmDuplexMode;
	MODEM_SPEAKER_CONTROL	mdmSpeakerControl;
	MODEM_SPEAKER_VOLUME	mdmSpeakerVolume;
	MODEM_TONE_DETECTION	mdmToneDetection;
	MODEM_ANSWER_MODE		mdmAnswerMode;
    MODEM_AUTO_ANSWER		mdmAutoAnswer;
	U32						mdmAutoAnswerRings;
	MODEM_MNP_MODE			mdmMNPMode;
	MODEM_MNP_COMPRESSION	mdmMNPCompression;
	MODEM_MNP_BREAK_TYPE	mdmMNPBreakType;
	MODEM_MNP_FLOW_CONTROL 	mdmMNPFlowControl;

} MODEM_METRICS, *P_MODEM_METRICS;


/****************************************************************************
 msgSvcSetMetrics			takes P_SVC_GET_SET_METRICS,	returns STATUS
	category: superclass message
	Sets current modem metrics, and re-initializes the modem with specified metrics.

 The pMetrics field of SVC_GET_SET_METRICS is expected to point to a buffer
 containing MODEM_METRICS as described above.
*/


/****************************************************************************
 msgSvcCharactersticsRequested takes P_SVC_CHARACTERISTICS,	returns STATUS
	category: superclass message
	Passes back the characteristics of the modem service.

 The pBuf field of SVC_CHARACTERISTICS is expected to point to a buffer
 capable of receiving MODEM_CHARACTERISTICS as described below.

 Implementors of clsModem services that wish to provide capabilities not
 described within MODEM_CHARACTERISTICS should contact GO Corporation to
 ensure such clsModem enhancements are standardized and noted within
 MODEM_CHARACTERISTICS.  Thank you.
*/

#define	mdmHWManufactureNameLength	15
#define	mdmHWModelNameLength		15

typedef struct {						// Modem hardware manufacturer.
	CHAR	name[mdmHWManufactureNameLength+1];	//  Name of manufacturer.
} MODEM_HARDWARE_MANUFACTURER, *P_MODEM_HARDWARE_MANUFACTURER;

typedef struct {						// Model of modem hardware.
	CHAR	name[mdmHWModelNameLength+1];//	Name of model.
} MODEM_HARDWARE_MODEL, *P_MODEM_HARDWARE_MODEL;

Enum32(MODEM_HARDWARE_FEATURES) {		// Modem hardware capabilities.
	mdmHWCapAutoDial		=	flag0,	//	Auto dialing.
	mdmHWCapAutoAnswer		=	flag1,	//	Auto answer.
	mdmHWCapAutoBaudDetect	=	flag2,	//	Auto baud detection.
	mdmHWCapCallTypeDiscrimination = flag3,	//  Call type discrimination
											//	 (Fax, Data, Voice).
	mdmHWCapPhoneJackConnectDetect = flag4,	//	Phone jack connect and
											//   disconnect event reporting.
	mdmHWCapRingSignalMachineWakeUp= flag5	//	Ring signal detection
											//   wakes up dormant machines.
};

typedef struct {						// Size of internal modem I/O buffers.
	S32			sizeInputBuffer;		//	Input buffer size.
	S32			sizeOutputBuffer;		//	Output buffer size.
} MODEM_HARDWARE_BUFFERS, *P_MODEM_HARDWARE_BUFFERS;

Enum32(MODEM_DCE_CONTROL) {				// Firmware DCE protocol/command sets.
	mdmDCEControlAT			= flag0		//	Hayes 'AT' commands.
};

typedef	struct	MODEM_CHARACTERISTICS {	// Modem hw & sw characteristics.
	MODEM_HARDWARE_MANUFACTURER	hardwareManufacturer;
	MODEM_HARDWARE_MODEL		hardwareModel;
	MODEM_HARDWARE_FEATURES		hardwareFeature;
	MODEM_HARDWARE_BUFFERS		hardwareBuffer;
	MODEM_DCE_CONTROL			dceControl;
	MODEM_SIGNALLING_MODES		signallingMode;
	MODEM_LINK_CONTROL			linkControl;
	U32							spare[4];
} MODEM_CHARACTERISTICS, *P_MODEM_CHARACTERISTICS;



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *		 					Class Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew						takes P_MODEM_NEW,				returns STATUS
	category: class message
	Creates a new instance of a modem service.

 Error Return Values:	percolated up from other classes,
*/

#define modemNewFields    serviceNewFields

typedef struct MODEM_NEW
{
    modemNewFields
		
} MODEM_NEW, *P_MODEM_NEW;


/****************************************************************************
 msgNewDefaults			takes P_MODEM_NEW,				returns STATUS
	category: class message
	Initializes the MODEM_NEW structure to default values.

 Sets:
	 
//{
    pArgs->svc.style.autoOption		=
    pArgs->svc.style.exclusiveOpen	= true;
	pArgs->svc.style.waitForTarget	= false;
	pArgs->svc.pManagerList			= pManagerList;
	pArgs->svc.numManagers			= sizeof(pManagerList)/sizeof(OBJECT);

    static OBJECT	pManagerList[] =
    {
		theModems					// clsModem is one of theModems.
    };
//}
*/


/****  clsModem error status values  ****/

/*
 This modem service doesn't (or cannot) support the
 current request due to hardware or firmware constraints.
*/
#define	stsModemNotSupported			MakeStatus(clsModem, 1)

/*
 A request to the modem service contained a parameter that
 is invalid.												
*/
#define	stsModemBadParameter			MakeStatus(clsModem, 2)

/* 
 The size of the buffer supplied to get/set modem service
 metrics or characteristics is incorrect.				
*/
#define	stsModemBufferSizeError			MakeStatus(clsModem, 3)

/*
 The modem service was unable to find and/or open its
 target service.											
*/
#define	stsModemTargetError				MakeStatus(clsModem, 4)

/* 
 The modem service is not open.  The current request
 requires that it be open.								
*/
#define	stsModemNotOpen					MakeStatus(clsModem, 5)

/* 
 The modem has responded to a modem command with an error
 response.												
*/
#define	stsModemErrorResponse			MakeStatus(clsModem, 6)

/* 
 The modem has responded to a modem command with a response
 that was unrecognized.									
*/
#define	stsModemUnrecognizedResponse	MakeStatus(clsModem, 7)

/* 
 The modem responded with a notification of carrier loss
 after dialing, attempting to go online, or being online.
*/
#define	stsModemNoCarrier				MakeStatus(clsModem, 8)

/*
 The modem didn't detect a dial tone while dialing to
 establish a connection.									
*/
#define	stsModemNoDialTone				MakeStatus(clsModem, 9)

/*
 The modem didn't detect an answer tone after dialing to
 establish a connection.									
*/
#define	stsModemNoAnswer				MakeStatus(clsModem, 10)

/* 
 The modem has been unable to successfully transmit a
 data frame to the remote node.							
*/
#define	stsModemTransmitError			MakeStatus(clsModem, 11)

/* 
 The modem has been unable to successfully receive a
 data frame to the remote node.							
 */
#define	stsModemReceiveError			MakeStatus(clsModem, 12)

/*
 The modem has detected a cyclic redundancy check error
 within a data frame received from the remote node.		
 */
#define	stsModemCRCError				MakeStatus(clsModem, 13)

/*
 The modem has detected a busy signal after dialing a
 telephone number.										
 */
#define	stsModemLineBusy				MakeStatus(clsModem, 14)

/* 
 The modem service could not locate a window within one
 of its option cards.  This is an internal error.		
 */
#define	stsModemNoSuchWindow			MakeStatus(clsModem, 255)

/****	clsModem non-error status values   ****/
/* 
 None currently defined 
 */

#endif	// MODEM_INCLUDED

