/****************************************************************************
 File: milserv.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.3  $
   $Author:   jgarrett  $
     $Date:   10 Mar 1992 16:26:16  $

 This file contains the API definition for clsMILService.  The functions
 described in this file are contained in milserv.lib.

 clsMILService inherits from clsService.
 Provides default behavior for MIL services.
 
 MIL services are PenPoint device drivers. They represent a MIL device,
 which represents a piece of hardware. A MIL service sits between a MIL 
 device and the rest of PenPoint. 

 A MIL service is typically composed of a Ring 0 part, which interfaces
 to the MIL, and a Ring 3 part, which interfaces to the rest of PenPoint.

 MIL service instances are created automatically by PenPoint. Never 
 send msgNew to a MIL Service class yourself! Each MIL device contains a 
 deviceId, which is the class of the MIL service that should be created for 
 it. PenPoint scans the MIL at power-up time and whenever a MIL service 
 installed, and creates one MIL service for each unit of each device.

 The MIL service writer can find out the logical id of the device it
 represents by self-sending msgMILSvcGetDevice. 

 A MIL service can install a MIL extension if necessary. The new MIL
 device is installed into the MIL when the MIL service is installed,
 and removed from the MIL when the MIL service is deinstalled. Use
 the InstallMILDevice() function in your DLLMain() to do this.

 You must also let the service framework know about a service by sending
 msgSvcClassInitService to your service class in DLLMain(). Here's an 
 example:
 // {
	STATUS EXPORTED DLLMain(void)
	{
		SVC_INIT_SERVICE	initService;
		STATUS				s;

		// Initalize classes.
		StsRet(ClsMILServiceInit(), s);	

		// Include if it is necessary to install MIL extensions.
		InstallMILDevice(&deviceInfo);	

		// Initialize service. This creates MIL service instances.
		memset(initService.spare, 0, sizeof(initService.spare));
		initService.autoCreate = true;			
		initService.serviceType = 0;
		initService.initServiceFlags = 0;
		ObjCallRet(msgSvcClassInitService, clsTestService, &initService, s);

		return stsOK;

	} // DllMain
 // }
 See project MILSVC for a template for creating MIL services.
****************************************************************************/
#ifndef MIL_SERVICE_INCLUDED
#define MIL_SERVICE_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef SERVICE_INCLUDED
#include <service.h>
#endif

#ifndef MIL_INCLUDED
#include <mil.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  		Common #defines and typedefs  					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Did this service install MIL devices? */
#define svcMILAttrInstalledDevice		FSMakeFix32Attr(clsMILService, 1)

/* The MIL device that this mil service is associated with. */
typedef struct MIL_SVC_DEVICE {
		TAG		unitResourceTag;		// resource tag into mil.res
		UID		conflictGroup;			// conflict group mil svc is on
		U16		logicalId;				// mil device logical id to use
		U16		unit;					// mil device unit number to use
		U8		reserved[12];
} MIL_SVC_DEVICE, *P_MIL_SVC_DEVICE;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  				  			Functions				  				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 InstallMILDevice	returns STATUS
    Install a MIL device.

 This routine should used to install one or more MIL devices. These
 devices will be automatically deinstalled when the MIL service is
 deinstalled.

 This routine *must* be called in the service's DLLMain(), after the
 classes are created but before	msgSvcClassInitService is sent.
*/
STATUS EXPORTED InstallMILDevice(
	P_MIL_DEVICE_INFO		pDeviceInfo, // Installable MIL device info.
	U32						reserved1,	 // Set this to 0
	U32						reserved2);	 // Set this to 0


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  			  	Class Messages 		  					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
 msgNew				takes P_MIL_SVC_NEW, returns STATUS
	category: class message
	Creates a new MIL service object. 

 This message should never be sent by clients. PenPoint automatically creates
 all MIL service instances by scanning the MIL.
*/

typedef struct MIL_SVC_NEW_ONLY {
	MIL_SVC_DEVICE			device;
	U32						unused1;
	U32						unused2;
	U32						unused3;
	U32						unused4;
} MIL_SVC_NEW_ONLY, *P_MIL_SVC_NEW_ONLY;

#define	milServiceNewFields	\
	serviceNewFields		\
	MIL_SVC_NEW_ONLY		milSvc;

typedef struct MIL_SVC_NEW {
	milServiceNewFields	
} MIL_SVC_NEW, *P_MIL_SVC_NEW;


/****************************************************************************
 msgNewDefaults		takes P_MIL_SVC_NEW, returns STATUS
	category: class message
    Initializes the MIL_SVC_NEW structure to default values.

 Sets	
	pArgs->svc.style.exclusiveOpen = true;	
	pArgs->svc.style.checkOwner = true;		

	Note: pArgs->svc.style.connectStyle will be set automatically to
	reflect underlying MIL device's auto-detection facilities. It will
	be set to svcAutoDetect if milDevFlagDetachable is true, 
	svcNoAutoDetect if milDevFlagDetachable is false.

	Note: pArgs->milSvc.device will be set automatically from the MIL.
*/

/****************************************************************************
 msgSvcSetConnected    	takes P_SVC_GET_SET_CONNECTED, returns STATUS
	Sets connection state of self.

 'P_SVC_GET_SET_CONNECTED' structure is defined in service.h.

 This message is self-sent whenever a MIL service thinks that it's 
 connection state has changed. This message should be sent even when a 
 mil service isn't sure if it is connected (due to possible interference
 from other mil services in its conflict group).

 If the mil service isn't in a conflict group then the message is sent to
 ancestor. If it is in a conflict group then the following will occur:

 if (pArgs->connected == true) {
    1. msgCGPollConnected is sent to the conflict group manager.

    2. The conflict group manager sends msgMILSvcAreYouConnected to all 
       services in the conflict group (including the one that self-sent 
       msgSvcSetConnected).

    3. The conflict group manager decides which service really should
       be connected and sends msgMILSvcConnectionStateResolved to all
       services. This tells which service (if any) has been chosen to
       be the connected one. MIL services should restart their connection
       detection logic if nobody is currently connected.

    4. Default behavior for msgMILSvcConnectionStateResolved is to send
       msgSvcSetConnected to ancestor if a change of state is indicated.
       MIL services must *always* send msgMILSvcConnectionStateResolved
	   to ancestor.
 } else {
	1. msgSvcSetConnected is sent to ancestor.
  
	2. msgCGInformDisconnected is sent to the conflict group manager.

    3. The conflict group manager sends msgMILSvcConnectionStateResolved
       to all mil services except the mil service that sent the
	   msgSvcSetConnected message.  MIL services should restart their
	   connection detection logic.
 }

 See Also
	msgSMConnectedChanged (servmgr.h)
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *          clsMILService Functionality Available to Subclasses			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgMILSvcGetDevice  		 	takes P_MIL_SVC_DEVICE, returns STATUS
	Returns MIL device associated with this service.

*/
#define msgMILSvcGetDevice		 			MakeMsg(clsMILService, 1)


/****************************************************************************
 msgMILSvcSetDevice  		 	takes P_MIL_SVC_DEVICE, returns STATUS
	Sets MIL device associated with this service.

 Note: This message is almost never used. Usually a MIL service is associated
 with the device that is set at msgNew time, and never changed. This 
 message is included for completeness and very special circumstances.
*/
#define msgMILSvcSetDevice		 			MakeMsg(clsMILService, 2)


/****************************************************************************
  msgMILSvcInstalledMILDevice	  	  	takes pNull, returns STATUS
	Is this MIL service targeting an installed MIL device?

 Returns stsOK if it is, stsFailed if it is not.
*/
#define msgMILSvcInstalledMILDevice			MakeMsg(clsMILService, 3)


/****************************************************************************
 msgMILSvcAddToConflictManager	takes P_MIL_SVC_ADD_TO_CONFLICT_MANAGER, returns STATUS
	Add this service instance to a conflict group manager. 

 This message is used to add a MIL service to a conflict group manager.

*/
#define msgMILSvcAddToConflictManager  		MakeMsg(clsMILService, 8)

typedef struct MIL_SVC_ADD_TO_CONFLICT_MANAGER {
	OBJECT					manager; 
} MIL_SVC_ADD_TO_CONFLICT_MANAGER, *P_MIL_SVC_ADD_TO_CONFLICT_MANAGER;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *		 			Descendant Responsibility Messages					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
  msgMILSvcPowerOff			 	  	takes pNull, returns STATUS
	The power is about to be turned off.

 This message is sent after all other power off messages are sent. MIL 
 services must *not* observe the power button to get power notification.

 MIL services should save any hardware-specific state that must be 
 restored when the power is applied.
*/
#define msgMILSvcPowerOff	 	 			MakeMsg(clsMILService, 4)


/****************************************************************************
  msgMILSvcPowerOn			 	  	takes pNull, returns STATUS
	The power has just come on.

 This message is sent before all other power on messages are sent. MIL 
 services must *not* observe the power button to get power notification.

 MIL services should restore any hardware-specific state that was saved
 when the power was disconnected.
*/
#define msgMILSvcPowerOn		 			MakeMsg(clsMILService, 5)


/****************************************************************************
 msgMILSvcAreYouConnected  	takes P_MIL_SVC_ARE_YOU_CONNECTED, returns STATUS
	Do you think you are connected?

 This message is sent to all members of a conflict group whenever any
 service thinks it has become connected. It allows all members of the
 conflict group to participate in deciding who is really connected.

 Default superclass behavior is to return msMaybe.
*/
#define msgMILSvcAreYouConnected 	   		MakeMsg(clsMILService, 6)

Enum16(MIL_SVC_ARE_YOU_CONNECTED) {
	msYes			  = 0, 
	msMaybe			  = 1, 
	msNo			  = 2
};


/****************************************************************************
 msgMILSvcConnectionStateResolved  	takes U16, returns STATUS
	Tells a MIL service who was chosen to be connected.

 The pArgs is the logical id of the service that was chosen to be
 connected. It is set to maxU16	if nobody is connected.

 Default superclass behavior is to send msgSvcSetConnected to ancestor if a 
 change of state is indicated.

 MIL services must always send msgMILSvcConnectionStateResolved
 to ancestor.

*/
#define msgMILSvcConnectionStateResolved	MakeMsg(clsMILService, 7)


/****************************************************************************
  msgMILSvcStartConnectionProcessing 	  	takes pNull, returns STATUS
	It is ok to start connection processing.

 This message is sent after booting is complete. MIL services should 
 not start their connection processing until they receive this message.
*/
#define msgMILSvcStartConnectionProcessing	\
										MsgNoError(MakeMsg(clsMILService, 9))


#endif	// MIL_SERVICE_INCLUDED

											
