/****************************************************************************
 File: grabbox.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.39  $
   $Author:   cmeyer  $
     $Date:   13 Mar 1992 19:04:22  $

 This file contains the API definition for clsGrabBox.

 clsGrabBox inherits from clsObject.
 Provides popup grab handles; uses clsTrack internally.

 GrabBoxes are used primarily by clsBorder to display resize handles,
 although other uses are possible.
****************************************************************************/
#ifndef GRABBOX_INCLUDED
#define GRABBOX_INCLUDED

													#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
													#endif

													#ifndef SYSGRAF_INCLUDED
#include <sysgraf.h>
													#endif
/* * * * * * * * *   Common #defines and typedefs   * * * * * * * * * * * */

typedef OBJECT			GRAB_BOX;

/****  Type styles  ****/
#define	gbTypeResize	0	// resize
//						1	// unused (reserved)
//						...
//						3	// unused (reserved)

/****  Locations styles  ****/
#define	gbLocULCorner		0	// upper-left corner
#define	gbLocURCorner		1 	// upper-right corner
#define	gbLocLRCorner		2 	// lower-right corner
#define	gbLocLLCorner		3	// lower-left corner
#define	gbLocLeftEdge		4	// left edge
#define	gbLocRightEdge		5 	// right edge
#define	gbLocBottomEdge		6 	// bottom edge
#define	gbLocTopEdge		7	// top edge
#define	gbLocNone			8	// no edge

typedef struct GRAB_BOX_STYLE {
	U16	type			: 2,	// type of grab box
		loc				: 4,	// location of grab box
		autoDestroy		: 1,	// destroy self on take down
		autoTakeDown	: 1,	// take down if pen is outside grab box
		spare			: 8;	// unused (reserved)
} GRAB_BOX_STYLE, *P_GRAB_BOX_STYLE;

/*
 Default GRAB_BOX_STYLE:
//{
 	type			= gbTypeResize
 	loc				= gbLocULCorner
 	autoDestroy		= true
 	autoTakeDown	= true
//}
 */

typedef struct GRAB_BOX_INFO {
	WIN	  	win;			// window over which grab box will be drawn
	U16	  	thickness; 		// thickness of visible grab area, in twips
	U16	  	length;			// length of visible grab area, in twips
	RECT32	outerMargin;	// thickness of invisible grab area, in twips
	BOOLEAN	includeOuter;	// true to include invisible area
	BOOLEAN	penIsDown;		// true if pen is down (for msgGrabBoxShow)
	XY32	downXY;			// xy on pen down in win space (for msgGrabBoxShow)
	U16		visualInset;	// amount to inset length for visual, in twips
	U16		cornerRadius;	// radius for round corners (zero for square), in twips
	U32		spare1;			// unused (reserved)
	U32		spare2;			// unused (reserved)
} GRAB_BOX_INFO, *P_GRAB_BOX_INFO;

/****************************************************************************
 msgNew						takes P_GRAB_BOX_NEW, returns STATUS
	category: class message
	Creates a grab box object.

*/
typedef struct GRAB_BOX_NEW_ONLY {
	GRAB_BOX_STYLE		style;		// overall style
	WIN					client;		// window to grab
	XY32				xy;			// unused
	WIN					xyWin;		// unused
	U8					margin;		// unused
	U32					spare;		// unused (reserved)
} GRAB_BOX_NEW_ONLY, *P_GRAB_BOX_NEW_ONLY,
  GRAB_BOX_METRICS, *P_GRAB_BOX_METRICS;

#define	grabBoxNewFields 	\
	objectNewFields			\
	GRAB_BOX_NEW_ONLY		grabBox;

typedef struct {
	grabBoxNewFields
} GRAB_BOX_NEW, *P_GRAB_BOX_NEW;

/****************************************************************************
 msgNewDefaults				takes P_GRAB_BOX_NEW, returns STATUS
	category: class message
	Initializes the GRAB_BOX_NEW structure to default values.

 Zeroes out pArgs->grabBox and sets
//{
	pArgs->grabBox.style.autoDestroy	= true;
	pArgs->grabBox.style.autoTakeDown	= true;
//}
*/

/****************************************************************************
 msgGrabBoxGetStyle		takes P_GRAB_BOX_STYLE, returns STATUS
	Passes back current style values.

*/
#define msgGrabBoxGetStyle		MakeMsg(clsGrabBox, 1)

/****************************************************************************
 msgGrabBoxSetStyle		takes P_GRAB_BOX_STYLE, returns STATUS
	Sets style values.

 Note that changing style.loc or style.type while the grab box is being
 shown is not supported.
*/
#define msgGrabBoxSetStyle		MakeMsg(clsGrabBox, 2)

/****************************************************************************
 msgGrabBoxGetMetrics		takes P_GRAB_BOX_METRICS, returns STATUS
	Passes back current metrics.

*/
#define msgGrabBoxGetMetrics 	MakeMsg(clsGrabBox, 3)

/****************************************************************************
 msgGrabBoxSetMetrics		takes P_GRAB_BOX_METRICS, returns STATUS
	Sets metrics.

 Sets the style as in msgGrabBoxSetStyle.
*/
#define msgGrabBoxSetMetrics 	MakeMsg(clsGrabBox, 4)

/****************************************************************************
 msgGrabBoxShow		takes P_GRAB_BOX_INFO, returns STATUS
	Puts up or takes down the grab box.

 If pArgs is not pNull, clsGrabBox will grab input using InputSetGrab() and
 paint the grab box.  If style.autoTakeDown is true, the grab box will be
 taken down when the pen leaves proximity or moves out of the grab box with
 the pen up.
 
 If pArgs is pNull, clsGrabBox will take down the grab box and self-send
 msgDestroy(pNull) if style.autoDestroy is true.

 The area on which the grab box was drawn will be damaged with 
 msgWinDirtyRect when the grab box is taken down.

 The grab box is drawn in the rectangle computed by GrabBoxLocToRect().
*/
#define msgGrabBoxShow			MakeMsg(clsGrabBox, 5)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Public Functions								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 GrabBoxIntersect			returns U16
	Determines where pRect is in win.
	Returns a grab box location, e.g. gbLocLRCorner.

 pRect->origin is commonly the coordinate of an event in pInfo->win's space,
 in device units.

 pInfo->thickness is the thickness (in twips) of the visible grab-sensitive
 area within pInfo->win.

 pInfo->outerMargin is the thickness (in twips) of the invisible
 grab-sensitive area within pInfo->win.  pInfo->outerMargin.{origin.x, size.w} are margins
 for the left and right, respectively.  pInfo->outerMargin.{origin.y, size.h}
 are margins for the bottom and top, respectively.

 pInfo->length is the length of each grab-sensitive area, in twips.

 If pInfo->includeOuter is true, the outer margin area is included in the rect
 for each grab box.

 This is used by clsBorder to place a grab box over the resize handles.
*/
U16 EXPORTED GrabBoxIntersect(
	P_GRAB_BOX_INFO	pInfo,	// info about grab box locations
	P_RECT32	 	pRect	// Rect to intersect
);

/****************************************************************************
 GrabBoxLocToRect			returns void
 	Computes the rectangle of the grabBox at the given location.

 pInfo is as described in GrabBoxIntersect().

 The corresponding rect for location is returned in pRect, in device units.
*/
void EXPORTED GrabBoxLocToRect(
	P_GRAB_BOX_INFO	pInfo,		// info about grab box locations
	U16				location,	// e.g. gbLocBottom
	P_RECT32 		pRect	   	// Rect to locate
);

/****************************************************************************
 GrabBoxPaint			returns STATUS
 	Paints the grab box at the specified location.

 pInfo is as described in GrabBoxIntersect().

 If dc is not objNull, it will be used for the painting.

 If pRect is pNull, the corresponding rect for location will be used; otherwise
 pRect will be used.

 If clearOuter is true, all of pRect will be cleared before painting.

 If on is true, the grab box will be painted in black, otherwise gray66.
 
 This is used by clsBorder to paint the resize handles.
*/
STATUS EXPORTED GrabBoxPaint(
	P_GRAB_BOX_INFO pInfo,
	U16				loc,
	SYSDC			dc,
	P_RECT32		pRect,
	BOOLEAN			clearOuter,
	BOOLEAN			on
);

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from other classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgInputEvent 					takes P_INPUT_EVENT, returns STATUS
	Notification of an input event. 

 clsGrabBox will respond to input events that trigger resizing.

 If pArgs->devCode is msgPenUp, msgPenOutProxUp, msgPenOutProxDown, or
 msgPenMoveUp and pArgs->xy is not in the rectangle of the grab box and
 style.autoTakeDown is true or msgPenDown has been received, the grab box
 is taken down as in msgGrabBoxShow(false).

 If pArgs->devCode is msgPenDown the following is done:

	msgTrackProvideMetrics is sent to metrics.client with the following 
	TRACK_METRICS parameters:

		msgNewDefaults is sent to clsTrack to initialize a TRACK_METRICS
		struct and then:

		style.track				= tsTrackResize;
		style.anchor			= computed from self's style.loc;
		win						= parent of metrics.client;
		client					= self;
		clientData				= window to be resized;
		initRect				= bounds of metrics.client;
		minWH					= small rectangle;
		maxWH					= limited to stay within parent of metrics.client
		tag						= tagBorderResize;

	If style.loc is gbLocLeftEdge or gbLocRightEdge, maxWH is altered to
	constrain to horizontal resize.

	If style.loc is gbLocBottomEdge or gbLocTopEdge, maxWH is altered to
	constrain to vertical resize.

	An instance of clsTrack is created and started via msgTrackStart.
*/

/****************************************************************************
 msgTrackDone					takes P_TRACK_METRICS, returns STATUS
	category: client notification
	Sent by a tracker when it's done.

 clsGrabBox responds by resizing metrics.client to pArgs->rect.size.

 If the width/height is changed, wsMaskWrapWidth/Height will be turned on in 
 WIN_METRICS.flags.style for metrics.client.

 The client window is resized by sending msgWinLayout with the following
 WIN_METRICS parameters:
	options		= 0;
	bounds		= pArgs->rect;

 If style.autoDestroy is true, msgDestroy(pNull) is self-posted.
*/

#endif	// GRABBOX_INCLUDED
