/****************************************************************************
 dynarray.h

 Copyright 1992, GO Corporation, All Rights Reserved

 $Revision:   1.6  $
	 $Date:   02 Mar 1992 12:48:56  $

 This file contains the API definition for dynarray.
 Dynarrays provide a set of dynamic array routines.
 				  
 The functions described in this file are contained in XLIST.LIB. 
													   
 Implements a dynamic array of elements.  Standard interface
 routines for indexing, inserting, deleting, and other common operations
 are provided.  This interface is primarily used internally to GO, and is
 therefore tailored to meet internal needs.

 A dynamic array is a simple data structure that contains some array
 information fields, and a pointer to a block of memory.  This block
 of memory is equal to pArray->entries * pArray->elementSize.

 The number of entries is specified at initialization time in DynArrayNew,
 and can be changed via DynArrayContract, or DynArrayExpand. These
 are implicitly called from DynArrayInsert and DynArrayDelete when inserting
 an item into a list that does not have enough entries available, or when
 deleting an item from the list.  At any time, the value returned by
 DynArrayCount, or pArray->entries, will be equal to
 the number of entries allocated in the array in pArray->pData.  The
 size of the array in pArray->pData will be equal to pArray->entries *
 pArray->elementSize.

 The maximum index set in the array at any given time, independent of the
 number of entries in the array, is referred to as maxCount.
 This is equal to the greatest array index number set via DynArraySet or
 inserted via DynArrayInsert.  It is also updated in DynArrayGetPtr, even
 if the user is getting the pointer to a cleared data pointer that has
 not been set or inserted.  DynArrayGetPtr is also used during binary
 searches, and hence that function will modify maxCount if the binary
 search expands to empty elements in the list.  This is necessary because
 client functions can modify the contents of the element via DynArrayGetPtr,
 because they have direct access to the data.  Typical users of dynamic
 arrays will not call DynArrayGetPtr in such a manner as to modify maxCount.

 In summary, entries is the amount of space allocated by the array, and
 maxCount is the number of elements set or inserted into the array.

 When memory is allocated for entries in the array, via DynArrayInsert,
 DynArrayNew, or DynArrayExpand, it is initialized to 0.
****************************************************************************/
#ifndef DYNARRAY_INCLUDED
#define DYNARRAY_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****  Dynamic Array  ****/
/*
 * This data structure is the dynamic array data structure. A dynamic
 * array created and manipulated is simply a pointer to this data structure
 * that is passed to the dynarray functions.  Accessing the fields
 * in this data structure is possible, but care should be taken as changing
 * their values could have drastic side affects. This data structure
 * is sometimes referred to as the array header.
 */
typedef struct DYNARRAY {
	OS_HEAP_ID heap;	// heap used for allocations
	U16 entries;		// total # entries in the array
	U16 elemSize;		// size in bytes of individual elements
	P_U8 pData;			// pointer to the array of values
	U16 maxCount;		// Max index accessed in the array via
						// DynArraySet, DynArrayGetPtr, DynArrayBinSearch
						// Updated when inserting, deleting, or contracting
						// array.
} DYNARRAY, *P_DYNARRAY;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Public Functions									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 DynArrayNew	returns STATUS
	Allocates a new dynamic array. Passes back the P_DYNARRAY header.

 Allocates memory for the array header, the P_DYNARRAY that is passed
 to the dynarray functions.  Allocates memory for the initial elements
 in the array.  Parameters include: the allocation heap to perform
 memory allocations, the size of an individual element, the initial
 size of the array (pArray->elements will the same as this value when this
 function returns), and any extra space to be allocated in the P_DYNARRAY
 pointer.  This space can be used by clients to store list-wide
 information or flags.  Passes back a pointer to the array data structure,
 P_DYNARRAY.
*/
STATUS DynArrayNew(
	OS_HEAP_ID heap,		// In: heap for memory allocation
							// NULL=>osProcessHeapId
	U16 elementSize,		// In: size in bytes of each array element
	U16 startSize,			// In: initial array size in number of elements
	U16 extraHeader,		// In: additional bytes to allocate in the header
	P_DYNARRAY *ppArray); 	// Out: pointer to the header pointer

/****************************************************************************
 DynArrayFree	returns STATUS
	Destroys the dynamic array and frees memory used by the array.

 Will free all memory allocated by the array to store the header information
 and the elements. Does not do anything with the entries in the array.
*/
STATUS DynArrayFree(
	P_DYNARRAY pArray);	// In: array header.  Will be freed.

/****************************************************************************
 DynArrayExpand		returns STATUS
	Expands the array by the specified number of entries.

 Expands the array by a number of entries, updating pArray->entries, the
 returned value of calling DynArrayCount, and the reallocation of
 pArray->pData to be equal to pArray->entries * pArray->elementSize. This
 function is called when calling DynArrayInsert to add space for one more
 entry.  It is also called in DynArraySet if the index is greater than the
 number of entries.

 See Also
	DynArraySet
	DynArrayCount
	DynArrayInsert
*/
STATUS DynArrayExpand(
	P_DYNARRAY pArray, 	// In: array header
	U16 add);			// In: Number of elements to add

/****************************************************************************
 DynArrayContract	returns STATUS
	Contracts the array by the number of entries.

 Will contract the number of entries in the array, and free the memory
 associated with those entries.  Will resize the amount of memory allocated
 by the array pArray->pData to be pArray->entries * pArray->elementSize.
 If the maxCount (return code of DynArrayCount) is greater than the new
 number of entries allocated, maxCount will be adjusted.  Called from
 DynArrayDelete to contract the array when deleting items.

 See Also
	DynArrayDelete
*/
STATUS DynArrayContract(
	P_DYNARRAY pArray, 	// In: array header
	U16 truncate);		// In: Number of elements to free

/****************************************************************************
 DynArrayGet	returns STATUS
	Passes back the index'th element in the array.

 Will pass back the contents of the index'th element in the array.  Will
 copy the memory of size elementSize containing the data for the element
 into pData.  It is the clients responsibility to ensure that this data
 pointer is large enough.
*/
STATUS DynArrayGet(
	P_DYNARRAY pArray,	// In: array header
	U16 index,			// In: element index
	P_UNKNOWN pData);	// Out: pointer to data buffer. Must be elementSize.

/****************************************************************************
 DynArraySet	returns STATUS
	Sets the index'th item to the given value. Update maxCount.

 Sets the contents of the index'th item to the given value.  Will copy the
 contents of the pData pointer to the memory for the index'th element in
 the array.  It is the clients responsibility to ensure that pData is correct.
 If index is greater than maxCount, it will update maxCount.  If the index
 is greater than the number of entries, the array is expanded via
 DynArrayExpand to be large enough.  Called from DynArrayInsert to set
 the value of the new index.

 See Also
	DynArrayInsert
*/
STATUS DynArraySet(
	P_DYNARRAY pArray,	// In: array header
	U16 index,			// In: element index
	P_UNKNOWN pData);	// In: pointer to data or NULL for zero fill

/****************************************************************************
 DynArrayGetPtr		returns STATUS
	Passes back a pointer to the index'th element in the array.

 Will pass back the direct pointer to the index'th element in the
 dynamic array.  Care should be taken when accessing this pointer, as
 it is memory that is allocated and managed by the array.  Accessing
 the data in this manner WILL cause the maxCount to be increased if
 maxCount < index.  This function is called during a binary search
 via DynArrayBinSearch.  Hence that function could modify maxCount.

 See Also
	DynArrayBinSearch
*/
STATUS DynArrayGetPtr(
	P_DYNARRAY pArray,	// In: array header
	U16 index,			// In: element index
	PP_UNKNOWN pData);	// Out: pointer to data buffer

/****************************************************************************
 DynArrayInsert		returns STATUS
	Inserts a new element in the array.

 The new element is indexed by index.  If the array is not big enough,
 will expand the array appropriately. Elements are copied from the
 index'th location to the next location.

 See Also
	DynArrayExpand
*/
STATUS DynArrayInsert(
	P_DYNARRAY pArray,	// array header
	U16 index,			// element index
	P_UNKNOWN pData		// new data to insert or NULL
);

/****************************************************************************
 DynArrayDelete		returns STATUS
	Deletes the index'th element from the array.

 Will delete the index'th element from the array.  If index is > entries,
 will return stsOK and do nothing.  Will move all elements greater than
 the index down by one in the array.  Will adjust maxCount if necessary.
 Will call DynArrayContract with parameter of one.
*/
STATUS DynArrayDelete(
	P_DYNARRAY pArray,	// array header
	U16 index			// element index to delete
);

/****************************************************************************
 DynArrayCount	returns STATUS
	Passes back the number of entries allocated in the array.

 Passes back the number of entries allocated in the array.  This number
 is the amount of space allocated, and not the number of items
 stored in the array.  That value is returned by DynArrayMax.
*/
STATUS DynArrayCount(
	P_DYNARRAY pArray,	// In: array header
	P_U16 pCount);		// Out: pointer to the count

/****************************************************************************
 DynArrayMax	returns STATUS
	Passes back the highest index stored.

 Will return the highest index stored via DynArraySet or DynArrayGetPtr,
 plus one. This is the "maxCount" field, and is used to indicate the
 highest array entry that has a valid value.
*/
STATUS DynArrayMax(
	P_DYNARRAY pArray, 	// In: array header
	P_U16 pMax);		// Out: pointer to the max index

/****************************************************************************
 DynArrayElemSize	returns STATUS
	Passes back the size, in bytes, of each element.

 Passes back the size allocate in the array for each element.  The
 pArray->pData size will be the value passed back by this function *
 the value passed back by DynArrayCount.
*/
STATUS DynArrayElemSize(
	P_DYNARRAY pArray,	// In: array header
	P_U16 pSize);			// Out: pointer to the size

/****************************************************************************
 DynArrayBinSearch	returns STATUS
	Performs a binary search on the array.

 Performs a binary search on the array.  Assumes that the array
 is "sorted" from lowest value to highest value.  Will access the
 value of data in the array via DynArrayGetPtr.  Hence care should
 be taken when using the data in the comparison callback routine.

 Return Value
	stsNoMatch			No matching data could be found within the range.

 See Also
	DynArrayGetPtr
*/

/*
 * P_BIN_PROC is the comparison routine callback.
 * Will be called to test items.  Called parameters
 * containing pointers to an element in the array, and
 * a pointer to a test 'element' to check for comparison.
 * Returns 0 for equal, -1 for less, 1 for greater.
 */
typedef S16 FunctionPtr(P_BIN_PROC)(P_UNKNOWN, P_UNKNOWN);

/*
 * DYNARRAY_SEARCH is the parameter into the DynArrayBinSearch function.
 * Takes the search data pointer to locate, a starting
 * index into the array to search, a stopping index into
 * the array to search, and a comparison callback function
 * to test the data pointer against elements in the array.
 * If result is 0, passes back the starting and
 * ending indices that match.  If result is -1, the
 * target data pointer was less than both the starting
 * and ending indices searched.  Similarly, if result is 1, the
 * target data pointer was greater than both indices.
 */
typedef struct DYNARRAY_SEARCH {
	P_UNKNOWN pData;	// In: Pointer to search data.
	U16 start;			// In: start index, Out: first occurrence
	U16 stop;			// In: end index, Out: last occurrence
	P_BIN_PROC pCompare;// In: routine to perform comparisons
	S16 result;		 	// Out: 0 if equal, -1 less, 1 greater
}DYNARRAY_SEARCH, *P_DYNARRAY_SEARCH;

STATUS DynArrayBinSearch(
	P_DYNARRAY pArray,			// In: array header
	P_DYNARRAY_SEARCH pSearch);	// In: search data

#endif
