/****************************************************************************
 dvdebug.h

 Copyright 1992, GO Corporation,   All Rights Reserved

 The definition of the public debug terminal device interface.

 $Revision:   1.7  $
   $Author:   thall  $
     $Date:   24 Mar 1992 15:11:12  $
****************************************************************************/
#ifndef DVDEBUG_INCLUDED
#define DVDEBUG_INCLUDED

/*
 This module contains definitions specific to the debug console device.
 This device is an output only device. At least one keyboard device must
 be provided to control debugging.

 The debug device supports optional annunciators. Examples of annunciators
 are light-emitting diodes.

 For control characters, LF (line feed) means return to the far left
 column and move down one line. CR (carriage return) means return to
 the far left column.
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 This section contains definitions specific to the debug device. 
 The definitions contained here are data structures that are necessary
 for the MIL debug specific functions along with standard
 MIL device data structures with additional fields specific to the
 debug device.
*/

/**** Debug Device Specific Function Numbers ****/

/*
 The following definitions are the device specific functions supported
 by the debug device. The function numbers start immediately after
 the last function number used by the functions that are common to all
 MIL devices. These function numbers correspond to the Function Transfer
 Table (FTT) entries.

*/

#define milDebugGetDevParameters	milLastCommonFunction+1
#define milDebugWrite				milLastCommonFunction+2
#define milDebugWriteAtRC			milLastCommonFunction+3
#define milDebugBeep				milLastCommonFunction+4
#define milDebugSetAnnunciator		milLastCommonFunction+5

#define fnCountDebug				fnCountBase+5

/**** Debug Device Function Transfer Table ****/
/*
 This structure is the declaration of the Debug Device Function
 Transfer Table. It includes one entry for each of the MIL driver
 functions common to all MIL devices along with one entry for
 each of the device specific functions for the MIL Debug device.
 Each entry contains fields that define the size of the request
 block for that function, the staging requirements of the function,
 and the address of the actual function. 
*/

typedef struct MIL_FTT_DEBUG {
	funcTransTableCommonFields
	MIL_FUNCTION_DESC functionDesc[fnCountDebug];
} MIL_FTT_DEBUG, *P_MIL_FTT_DEBUG;




/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 					Debug Device Specific Functions	   	 				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 The following function definitions are the debug device specific
 functions. Access to these functions is provided by the FTT defined
 above. All of the functions defined below will be passed 3 pointers.

	pCommonData:
		This is a pointer to the common data area.

	pRB:
		This is a pointer to the function request block.

	pDeviceBlock:
		This is a pointer to the device block for the device

 All of the functions return STATUS. Any STATUS other than staging 
 statuses will cause the request to terminate. The staging statuses
 will result in the specified staging operation to be performed.
*/


/****************************************************************************
 milDebugGetDevParameters		returns STATUS
 	Provides device specific information about the debug device.

 This function is a single stage function that provides device
 specific information about the debug device.
*/

/**** Device Specific Request Block Fields ****/
/*

 The milDebugGetDevParameters funtion request block contains the
 following fields.
	
	debugFlags:
		This field contains capability, status, and mode of
		operation bits.

	maxAnnunciator:

	currentAnnunciator:

	maxRow:
		This field contains the maximum row number on the debug console.

	maxCol:
		This field contains the maximum column on the debug console.

*/

typedef struct MIL_DEBUG_DEVICE_PARAMETERS {
	U16 debugFlags;
	U16 maxAnnunciator;
	U32 currentAnnunciators;
	U16 maxRow;
	U16 maxCol;
	U16 reservedU16[2];
	U32 reservedU32[2];
} MIL_DEBUG_DEVICE_PARAMETERS, *P_MIL_DEBUG_DEVICE_PARAMETERS;

/*
 rbPublicFieldsDebugGetDevParameters

 This define merges the device specific data structure with the standard
 MIL device parameters reserved fields.

*/

#define rbPublicFieldsDebugGetDevParameters \
	MIL_DEBUG_DEVICE_PARAMETERS debugDevParms; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milDebugGetDevParameters Request Block

*/

typedef struct MIL_RB_PUB_DEBUG_GET_DEV_PARAMETERS {
	milRequestBlockCommonFields
	rbPublicFieldsDebugGetDevParameters
} MIL_RB_PUB_DEBUG_GET_DEV_PARAMETERS, *P_MIL_RB_PUB_DEBUG_GET_DEV_PARAMETERS;


/****************************************************************************
 milDebugWrite		returns STATUS
 	Writes characters to the debug console.

 This single-stage function sends characters to the debug terminal.
 The "data" field can hold up to four charcters or serve as the pointer
 to a larger buffer. Which format is used is indicated by the BOOLEAN field
 "dataIsBuffer". See mil.h for details on the union MIL_PTR_OR_BUF.

*/

/**** Device Specific Request Block Fields ****/
/*

 The milDebugWrite funtion request block contains the
 following fields.
	
  	data:
		This field contains 1 to 4 bytes of data or a pointer to
		a buffer that contains the data to be written.

	dataIsBuffer:
		This field is a boolean that, if TRUE, indicates the data
 		field is a pointer to a data buffer.

	bufferSize:	
		This field contains the count of the number of bytes to write,

*/

#define rbPublicFieldsDebugWrite \
	MIL_PTR_OR_BUF data; \
	BOOLEAN dataIsBuffer; \
	U32 bufferSize; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milDebugWrite Request Block

*/

typedef struct MIL_RB_PUB_DEBUG_WRITE {
	milRequestBlockCommonFields
	rbPublicFieldsDebugWrite
} MIL_RB_PUB_DEBUG_WRITE, *P_MIL_RB_PUB_DEBUG_WRITE;


/****************************************************************************
 milDebugWriteAtRC		returns STATUS
 	Writes characters to the debug console at the specified row and column.

 This single-stage function sends characters to the debug terminal at the
 specified row and column. To just set the cursor position use a
 bufferSize of zero. On return the fields "row" and "col" return the
 current cursor position.

 The "data" field can hold up to four charcters or serve as the pointer
 to a larger buffer. Which format is used is indicated by the BOOLEAN field
 "dataIsBuffer". See mil.h for details on the union MIL_PTR_OR_BUF.

*/

/**** Device Specific Request Block Fields ****/
/*

 The milDebugWriteAtRC funtion request block contains the
 following fields.
	
  	data:
		This field contains 1 to 4 bytes of data or a pointer to
		a buffer that contains the data to be written.

	dataIsBuffer:
		This field is a boolean that, if TRUE, indicates the data
 		field is a pointer to a data buffer.

	bufferSize:	
		This field contains the count of the number of bytes to write,

	row:
		This field contains the destination row.

	col:
		This field contains the destination column.


*/


#define rbPublicFieldsDebugWriteAtRC \
	MIL_PTR_OR_BUF data; \
	BOOLEAN dataIsBuffer; \
	U32 bufferSize; \
	U16 row; \
	U16 col; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milDebugWriteAtRC Request Block

*/

typedef struct MIL_RB_PUB_DEBUG_WRITE_AT_ROW_AND_COL {
	milRequestBlockCommonFields
	rbPublicFieldsDebugWriteAtRC
} MIL_RB_PUB_DEBUG_WRITE_AT_ROW_AND_COL, 
	*P_MIL_RB_PUB_DEBUG_WRITE_AT_ROW_AND_COL;


/****************************************************************************
 milDebugBeep		returns STATUS
 	Beeps the debug console.

 This single-stage function causes the debug console to beep.

*/

/**** Device Specific Request Block Fields ****/
/*

 The milDebugBeep funtion request block contains no device or function
 specific fields.

*/
	
#define rbPublicFieldsDebugBeep \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milDebugBeep Request Block

*/

typedef struct MIL_RB_PUB_DEBUG_BEEP {
	milRequestBlockCommonFields
	rbPublicFieldsDebugBeep
} MIL_RB_PUB_DEBUG_BEEP, *P_MIL_RB_PUB_DEBUG_BEEP;


/****************************************************************************
 milDebugSetAnnunciator		returns STATUS
 	Sets individual annunciators.

 This single-stage function sets an annunciator.

*/

/**** Device Specific Request Block Fields ****/
/*

 The milDebugSetAnnunciator funtion request block contains the following
 fields.

	annunciatorNumber:
		This identifies the annunciator to be set.

	turnOn:
		This field is a boolean that indicates if the annunciator should
		be turned on or turned off.

*/

#define rbPublicFieldsDebugSetAnnunciator \
	U16 annunciatorNum; \
	BOOLEAN turnOn; \
	U16 funcReservedU16[2]; \
	U32 funcReservedU32[2];

/*
 
 milDebugSetAnnunciator Request Block

*/

typedef struct MIL_RB_PUB_DEBUG_SET_ANNUNCIATOR {
	milRequestBlockCommonFields
	rbPublicFieldsDebugSetAnnunciator
} MIL_RB_PUB_DEBUG_SET_ANNUNCIATOR, *P_MIL_RB_PUB_DEBUG_SET_ANNUNCIATOR;

#endif
