/****************************************************************************
 cmpstext.h

 Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.9  $
   $Author:   jbennett  $
     $Date:   03 Mar 1992 10:36:38  $

 This file contains the API definition for the compose-text package.
 
 This package is used to compose a text string that needs to have pieces
 inserted into it.  The format of the strings makes it easy to
 internationalize and localize the text.

 The functions described in this file are contained in SYSUTIL.LIB. 
****************************************************************************/

/**** Format Strings ****/
/*
 The format strings contain literal text and format codes.  A format code
 starts with '^', has a sequence of one or more digits in the middle, and a
 single letter at the end.  The digits specify which argument to the
 function to use and the letter indicates the type of the argument.  For
 instance, format code "^2s" indicates that the second argument should be
 inserted, and that the argument should be a string.

 The following fills 'buffer' with the string "a B b A c":
//{	
	SComposeText(&buffer, &size, heap, "a ^2s b ^1s c", "A", "B");
//}

 The available argument types are:
	-:	^:	Literal '^' character.  E.g. use "^^" to put a ^ in a string.
	-:	s:	String.
	-:	r:	Resource ID of a string resource.
	-:	l:	Group number and indexed list resource ID for string list.  This
			uses two arguments.
	-:	d:	U32 printed as a decimal number.
	-:	x:	U32 printed as a hexadecimal number.
    -:  {:	Singular/Plural word forms of the form "{is|are}".  When this
        argument type is used, the routine examines the specified argument.
        If its value is 1, the first string is used.  Otherwise the second
        string is used.

 The following code reads in a string from the TK group for a 'sample'
 project.
//{
	SComposeText(&buffer, &size, heap,
			"The filled in string is ^1l.", resGrpTK, sampleListResId);
//}

 As an example of the '{' format code, the following code generates the
 first string if numApples==1 and the second string if numApples==5.
//{
	SComposeText(&buffer, &size, heap,
			"There ^1{is|are} ^1d ^1{apple|apples}.", numApples);

	"There is 1 apple."
	"There are 5 apples."
//}
*/

/**** Memory Management ****/
/*
 All of the procedures fill in a buffer with the generated string.  There
 are two ways of supplying the buffer memory.

    -:  You can supply a buffer pointer and buffer length.  Do this by
        passing the pointer as *ppString, the length in *pLength, and a 
		null heapId.  If this technique is used, and the buffer is too
		small to hold the results, an error status is returned.

    -:  You can specify a heap from which memory will be allocated.  Do
        this by passing in a valid heapId.  You are obligated to free the
        memory when finished.
*/

#ifndef CMPSTEXT_INCLUDED
#define CMPSTEXT_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef RESFILE_INCLUDED
#include <resfile.h>
#endif

#include <stdarg.h>


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Common #defines and Typedefs					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define	ComposeTextMaxArguments	20		// Maximum number of parameters


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Functions									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 SComposeText  returns STATUS
 	Composes a string from a format and arguments.

 Copy the format argument into the output string, doing the appropriate
 substitutions for the format codes.

 See the section "Memory Management" for information on what values to 
 use for the first three arguments.
*/
STATUS CDECL SComposeText(
	PP_CHAR			ppString,
	P_U32			pLength,
	OS_HEAP_ID		heap,
	const P_CHAR	pFormat,
	...
);

/****************************************************************************
 VSComposeText  returns STATUS
 	Composes a string from a format and a pointer to the argument list.

 This is the same as SComposeText except the arguments are passed as a
 pointer to a list.  

 See the section "Memory Management" for information on what values to 
 use for the first three arguments.
*/
STATUS CDECL VSComposeText(
	PP_CHAR			ppString,
	P_U32			pLength,
	OS_HEAP_ID		heap,
	const P_CHAR	pFormat,
	va_list			argList
);

#endif
