/****************************************************************************
 File: clsmgr.h

 (C) Copyright 1992 by Go Corporation, All Rights Reserved.

 $Revision:   1.57  $ 
 $Author:   ctaylor  $
 $Date:   19 Mar 1992 10:45:48  $

 The Class Manager supports object-oriented programming.

 clsObject inherits from null.

 clsObject is the root of the Object System.  It defines the basic
 capabilities of all objects.

 clsClass inherits from clsObject.

 clsClass is the root of all classes.  clsClass provides class creation
 capabilities.
****************************************************************************/
#ifndef CLSMGR_INCLUDED
#define CLSMGR_INCLUDED

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							   Overview		 							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* 
 This file defines all the subroutines and messages that implement
 Object-oriented programing under PenPoint.  The most important of these
 are:

    -:  ObjectCall() and related routines, especially the Debugging
        Routines
    -:  MsgHandler() and related macros
    -:  MakeMsg() macro
	-:	clsClass, CLASS_NEW_DEFAULTS, etc.
	-:	clsObject, OBJECT_NEW_DEFAULTS, etc.
    -:  msgNew, msgNewDefaults, msgInit, msgDestroy, msgFree, msgSave,
        msgRestore

 Look at the functions starting with ClsStsToString too.

 This is one of PenPoint's key header files.  Developers should browse
 through this file and be familiar with its contents.  Other key header
 files are go.h, app.h, and win.h.

 To fully understand what's going on here, you should read the Class
 Manager section of the PenPoint Architecture Reference.
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								  Guidelines							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 Normally you should call your ancestor before processing a message.
 Possible exceptions include:
    -:  messages that are defined by your class.  Obviously, these
        shouldn't go to your ancestor at all.
    -:  messages that you want to explicitly override.  Depending on
        whether you want to override the message some of the time or all
        the time.
    -:  msgFreeOK, msgFreeing, msgFree should use objCallAncestorAfter.
    -:  protocols that requires the subclass to act on the message before
        the ancestor receives it.
*/

/**** Debugging Flags ****/
/*
 The ClsMgr debugging flag set is 'C'.  Defined values are:

    000001:	 Show calls to ObjectCall().
	000002:  Show calls to ObjectCallAncestor().
	000004:  Show calls to ObjectSend().
	000008:  Show calls to ObjectPost().

	000010:  Show indirect calls (class messages) for traced objects.
	000020:  Show object new and free calls.
	000040:  Show observer related actions: add, remove, notify and post.
	000080:  Show messages as they are dispatched.

	000100:  Show objects as they are saved and restored.
	000200:  Gather ObjectCall depth statistics.
	000400:  Show objects as they are scavenged at task termination.
	000800:  Enter Debugger(), if bad object is passed to ObjectCall().

	001000:  Show calls to ObjectCallAncestor() for traced objects.
    002000:  Enable detailed messages from ObjectValid().  These messages
             are not necessarily errors if the client code handles
             stsBadObject.  Because null objects are common they are not
             reported under C2000.
    004000:  Enable miscellaneous error/warning messages:
				Bad newStruct, Message not understood,
				WKN already exists, WKN replaced (warning).
	008000:  Enter the debugger after printing a warning.

 Temporary flags:
    010000:	 Fills the stack w/F0's before calling a method.  This is
             useful for catching uninitialized variables.
	020000:	 Show calls to extended ObjectCall() and ObjectCallAncestor().

 Implementor Flags:
	100000:	 Show all clsmgr statuses, legitimate errors are included.
*/

#ifndef GO_INCLUDED
  #include <go.h>
#endif
#ifndef OSTYPES_INCLUDED
  #include <ostypes.h>
#endif
#ifndef UID_INCLUDED
  #include <uid.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Status Values							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define	 stsBadObject				MakeStatus(clsObject, 2)
#define	 stsBadAncestor				MakeStatus(clsObject, 4)
#define	 stsBadContext				MakeStatus(clsObject, 6)
#define	 stsProtectionViolation		MakeStatus(clsObject, 8)
#define	 stsScopeViolation			MakeStatus(clsObject, 10)	// (.asm)
#define	 stsTaskTerminated			MakeStatus(clsObject, 12)
#define	 stsSizeLimit				MakeStatus(clsObject, 14)
#define	 stsBadPropTag				MakeStatus(clsObject, 16)
#define	 stsNewStructError			MakeStatus(clsObject, 18)
#define	 stsClassHasReferences		MakeStatus(clsObject, 20)
#define	 stsNotUnderstood			MakeStatus(clsObject, 22)
#define	 stsVetoed					MakeStatus(clsObject, 26)
#define	 stsWellKnownExists			MakeStatus(clsObject, 28)
#define  stsBadMethodTable			MakeStatus(clsObject, 30)	// (.asm)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						   Non-Error Status Values						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * stsMessageIgnored is equal to stsRequestForward for historical reasons.
 * MakeWarning is used to force the entry into the symbols DB.
*/
#define	 stsMessageIgnored			MakeWarning(clsGO, 2)
#define	 stsAlreadyAdded			MakeWarning(clsObject, 2)
#define	 stsAlreadyRemoved			MakeWarning(clsObject, 3)
#define	 stsSendTaskInvalid			MakeWarning(clsObject, 4)
#define	 stsWellKnownReplaced		MakeWarning(clsObject, 6)
#define	 stsTraceOn					MakeWarning(clsObject, 7)
#define	 stsTraceOff				MakeWarning(clsObject, 8)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							 Object Capabilities						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef M_I86						// 32 bit compiler

Enum32(OBJ_CAPABILITY)				// default for: OBJECT	CLASS
{									// ----------------------------
  objCapOwner		= flag1,		//				TRUE	FALSE
  objCapFree		= flag2,		//				TRUE	FALSE
  objCapSend		= flag3,		//				TRUE	FALSE
  objCapObservable	= flag4,		//				TRUE	TRUE

  objCapInherit		= flag6,		//				n/a		TRUE
  objCapScavenge	= flag7,		// enable only:	n/a		FALSE
  objCapCreate		= flag8,		//				FALSE	FALSE
  objCapProp		= flag9,		//				TRUE	TRUE
  objCapMutate		= flag10,		//				TRUE	TRUE
  objCapCall		= flag15,		//				FALSE	TRUE
  objCapCreateNotify = flag16,		// create only:	FALSE	FALSE
  objCapUnprotected = flag17,		// create only:	n/a		FALSE
  objCapNonSwappable = flag18		// create only:	FALSE	FALSE
	
};
#else								// 16 bit compiler
typedef U32 OBJ_CAPABILITY;
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				Types Derived Directly from Base Types					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* OBJECT, TAG, STATUS, etc. are defined in <go.h> */

typedef OBJECT				CLASS, *P_CLASS;

typedef TAG					MESSAGE, *P_MESSAGE;

typedef P_UNKNOWN			P_ARGS, *PP_ARGS;

typedef P_UNKNOWN			CONTEXT, *P_CONTEXT;

typedef P_UNKNOWN			P_IDATA, *PP_IDATA;

typedef U32					OBJ_KEY, *P_OBJ_KEY;

#define	 objWKNKey			((OBJ_KEY)0)

typedef const U32			*P_MSG, **PP_MSG;			// message table


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *			  Constants and Types Derived from Structs						*
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* NewArgs used to create an object. */
typedef struct OBJECT_NEW {

  U32	newStructVersion;			// Out: [msgNewDefaults] Validate msgNew
									// In:	[msgNew] Valid version
  OBJ_KEY				key;		// In:	[msgNew] Lock for the object
  OBJECT				uid;		// In:	[msgNew] Well-known uid
									// Out: [msgNew] Dynamic or Well-known uid
  OBJ_CAPABILITY		cap;		// In:	[msgNew] Initial capabilities
  CLASS					objClass;	// Out: [msgNewDefaults] Set to self
									// In:	[msgObjectNew] Class of instance
									// In:	[msg*] Used by toolkit components
  OS_HEAP_ID			heap;		// Out: [msgNewDefaults] Heap to use for
									// additional storage. If capCall then
									// OSProcessSharedHeap else OSProcessHeap
  U32					spare1;		// Unused (reserved)
  U32					spare2;		// Unused (reserved)

} OBJECT_NEW_ONLY, OBJECT_NEW, * P_OBJECT_NEW_ONLY, * P_OBJECT_NEW;

/* New defaults fields for subclassing OBJECT. */
#define objectNewFields		OBJECT_NEW_ONLY	object;

/* Fields for initializing a class. */
typedef struct CLASS_NEW_ONLY {

  P_MSG				pMsg;			// In: Can be pNull for abstract class
  CLASS				ancestor;		// In: Ancestor to inherit behavior from
  SIZEOF			size;			// In: Size of instance data, can be 0
									//	   (see comment below)
  SIZEOF			newArgsSize;	// In: Size of XX_NEW struct, can be 0
									//	   Value limited to U16
  U32				spare1;			// Unused (reserved)

} CLASS_NEW_ONLY, * P_CLASS_NEW_ONLY;

/*
 Limits on instance data size:

 Instance data for any class is limited to 64K bytes.  Instance data for an
 entire objects is limited to 64K of protected data.  Unprotected instance
 data is limited to 64K bytes per class but there is no limit for the
 object.
*/

/* New defaults fields for subclassing CLASS. */
#define classNewFields	objectNewFields		CLASS_NEW_ONLY	cls;

/* NewArgs used to create a class. */
typedef struct CLASS_NEW {

  classNewFields

} CLASS_NEW, * P_CLASS_NEW;

/* Enable/Disable capabilities */
typedef struct OBJ_CAPABILITY_SET {

  OBJ_CAPABILITY			cap;	// In: Capabilities to enable/disable
  OBJ_KEY					key;	// In: Unlocks object, e.g., objWKNKey

} OBJ_CAPABILITY_SET, * P_OBJ_CAPABILITY_SET;

/* Set/Get owner */
typedef struct OBJ_OWNER {

  OS_TASK_ID		task;			// In:	[msgSetOwner] New owner
									// Out: [msgObjectOwner] Current owner
  OBJECT			object;			// In:	[msgObjectOwner] Source object
  OBJ_KEY			key;			// In:	[msgSetOwner] If required by caps

} OBJ_OWNER, * P_OBJ_OWNER;

/* Set/Get properties */
typedef struct OBJ_PROP {

  TAG				propId;			// In:	[msgProp] Name of property
  P_IDATA			pData;			// In:	[msgProp] Pointer to data
									// In:	[msgSetProp] Data to copy to prop
  SIZEOF			length;			// In:	[msgProp] # of bytes to copy
									// Out: [msgProp] Length of data in bytes
									// In:	[msgSetProp] # of bytes to write
  OBJ_KEY			key;			// In:	[msgSetProp] If required by cap

} OBJ_PROP, * P_OBJ_PROP;

/* Add/Get observers */
typedef struct OBJ_OBSERVER_POS {

  OBJECT			observer;		// In:	[msgAddObserverAt] New observer
									// Out: [msgGetObserver] Observer at pos
  U16				position;		// In:	Position in observer list

} OBJ_OBSERVER_POS, * P_OBJ_OBSERVER_POS;

/* Notify observers */
typedef struct OBJ_NOTIFY_OBSERVERS {

  MESSAGE			msg;			// In:	Message to send/post observers
  P_ARGS			pArgs;			// In:	Args for message
  SIZEOF			lenSend;		// In:	Length of Args

} OBJ_NOTIFY_OBSERVERS, * P_OBJ_NOTIFY_OBSERVERS;

/* Buffer to hold symbol string.  Used with ClsStsToString, etc. */
#define clsSymBufSize 80
typedef char	P_CLS_SYMBUF[clsSymBufSize];

/* Array entry for OBJECT in the symbols database. */
typedef struct CLS_SYM_OBJ {

  OBJECT			obj;
  P_STRING			name;

} CLS_SYM_OBJ, *P_CLS_SYM_OBJ, * *PP_CLS_SYM_OBJ;

/* Array entry for message in symbols database. */
typedef struct CLS_SYM_MSG {

  MESSAGE			msg;
  P_STRING			name;

} CLS_SYM_MSG, *P_CLS_SYM_MSG, * *PP_CLS_SYM_MSG;

/* Array entry for STATUS in symbols database. */
typedef struct CLS_SYM_STS {

  STATUS			sts;
  P_STRING			name;

} CLS_SYM_STS, *P_CLS_SYM_STS, * *PP_CLS_SYM_STS;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				  Types Required for msgSave and msgRestore				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Resouce IDs */
typedef TAG		RES_ID, *P_RES_ID;			// Resource ID

/* System flags for save and restore. */

Enum16(RES_SAVE_RESTORE_FLAGS) {
	resDoingCopy	= flag0					// Creating a copy of object
};

typedef struct OBJ_SAVE {

  OBJECT			file;			// In:	File to save object to
  RES_ID			resId;			// In:	Resource Id of root-level object
  OBJECT			root;			// In:	Uid of root-level object
  P_UNKNOWN			pEnv;			// In:	Environment to be saved
  U16				minSysVersion;	// In/Out: Min acceptable system version
  U16				minAppVersion;	// In/Out: Min acceptable app version
  RES_SAVE_RESTORE_FLAGS sysFlags;	// In:	System flags
  U16				appFlags;		// In:	App flags
  P_UNKNOWN			pFile;			// In:	StdIO FILE* bound to file above
  U32				spare1;			// Unused (reserved)
  U32				spare2;			// Unused (reserved)

} OBJ_SAVE, * P_OBJ_SAVE;

typedef struct OBJ_RESTORE {

  OBJECT_NEW		object;			// In:	New defaults for restored object
  OBJECT			file;			// In:	File to restore object from 
  RES_ID			resId;			// In:	Resource Id of root-level object
  OBJECT			root;			// In:	Uid of root-level object
  P_UNKNOWN			pEnv;			// In:	Saved environment
  U16				sysVersion;		// In:	Sys version number of filed data
  U16				appVersion;		// In:	App version number of filed data
  RES_SAVE_RESTORE_FLAGS sysFlags;	// In:	System flags
  U16				appFlags;		// In:	App flags
  P_UNKNOWN			pFile;			// In:	StdIO FILE* bound to file above
  U32				spare1;			// Unused (reserved)
  U32				spare2;			// Unused (reserved)

} OBJ_RESTORE, * P_OBJ_RESTORE;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						   Method Definition Macros						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Definition of a pointer to a method. */
#ifdef __HIGHC__
 typedef CDECL STATUS (* P_MSG_HANDLER) (
#else
 typedef STATUS (CDECL * P_MSG_HANDLER) (
#endif
	MESSAGE		msg,
	OBJECT		self,
	P_ARGS		pArgs,
	CONTEXT		ctx,
	P_IDATA		pData
);

/* Definition of a method. */
#define	 MSG_HANDLER	STATUS CDECL

/* Shorthand used to declare a method. */
#define MsgHandler(fn) MSG_HANDLER MsgHandlerPrimitive(fn, P_ARGS, P_IDATA)

/* 
 * Shorthand used to declare a method with pArgs cast to appropriate type.
 * Note: pArgsType must be a pointer type.
*/
#define MsgHandlerArgType(fn, pArgsType) \
	MSG_HANDLER MsgHandlerPrimitive(fn, pArgsType, P_IDATA)

/*
 * Shorthand used to declare a method with casts for pArgs and instance data.
 * Note: pArgsType and pInstData must be pointer types.
*/
#define MsgHandlerWithTypes(fn, pArgsType, pInstData) \
	MSG_HANDLER MsgHandlerPrimitive(fn, pArgsType, pInstData)

/*
 * Shorthand used to declare a method.  Very fast and very dangerous.  DS is
 * NOT loaded.  Don't use strings, local functions, statics, etc.
*/
#define MsgHandlerRingCHelper(fn) \
	STATUS CDECL MsgHandlerPrimitive(fn, P_ARGS, P_IDATA)

/* Shorthand used to declare a method. */
#define MsgHandlerPrimitive(fn, pArgsType, pInstData) fn(\
	const MESSAGE	msg,	\
	const OBJECT	self,	\
	const pArgsType	pArgs,	\
	const CONTEXT	ctx,	\
	const pInstData	pData)

/* Cast pData to the appropriate type. */
#define	IDataPtr(pData, type) ((type*)pData)

/* Copy protected instance data block into local storage. */
#define	IDataDeref(pData, type) (*(type*)pData)

/* Shorthand used to ignored any unused parameters in a method. */
#define MsgHandlerParametersNoWarning \
	  Unused(msg); Unused(self); Unused(pArgs); Unused(ctx); Unused(pData)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Message Macros								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Create a message, takes a well-known class and message number.  Valid
   message numbers are between 0 and 254, inclusive.  Message number 255
   is reserved. */
#define MakeMsg(wkn,msg)		MakeTag(wkn,msg)

/* Extract the message portion of a message. */
#define MsgNum(msg)				TagNum(msg)

/* The WKNValue unique represents a class. */
#define ClsNum(msg)				WKNValue(msg)

/*
 * Messages defined with MsgNoError() will not generate a msgNotUnderstood
 * error if they reach clsObject.	 Instead, stsMessageIgnored is returned.
*/
#define MsgNoError(msg)			((msg)|msgNoErrorFlag)
#define msgNoErrorFlag			(1L<<21)

/*
 * Messages that are handled as class messages have this flag added to the
 * message value.
*/
#define msgClassMessageFlag		(1L<<22)

/* Compare two messages for equality. */
#define MsgEqual(m1,m2)			(m1==m2)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				   Object Scope Macros (Well-Known and Dynamic)			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define ObjectIsDynamic(o)		((U32)(o)&objDynamicFlag)
#define ObjectIsWellKnown(o)	(!ObjectIsDynamic(o))
#define ObjectIsWKN(o)			ObjectIsWellKnown(o)

#define ObjectIsGlobal(o)		(ObjectIsDynamic(o)||ObjectIsGlobalWKN(o))
#define ObjectIsLocal(o)		(!ObjectIsGlobal(o))

#define ObjectIsGlobalWKN(o)	(ObjectIsWKN(o) && WKNScope(o)==wknGlobal)
#define ObjectIsProcessGlobalWKN(o) \
	(ObjectIsWKN(o) && WKNScope(o)==wknProcessGlobal)
#define ObjectIsPrivateWKN(o)	(ObjectIsWKN(o) && WKNScope(o)==wknPrivate)

/* All dynamic objects have this bit set in their UID. */
#define objDynamicFlag			0x800000

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							   Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

// Recycle:
// Next available: 120

/****************************************************************************
 msgNull	takes pNull, returns STATUS
	Not a real message, just a place holder.
*/
#define	msgNull	MakeMsg(objNull, 0)

/****************************************************************************
 msgNewDefaults	takes new struct for object being created, returns STATUS
	category: class message
	Initializes new struct to default values.
*/
#define	msgNewDefaults	MakeMsg(clsObject, 2)

/****************************************************************************
 msgNew	takes new struct for object being created, returns STATUS
	category: class message
	Creates an object and sends msgInit to the new object.

 Developers normally send this message to class objects in order to create
 instances but they do NOT write code that handles msgNew.  The class
 manager does some processing on msgNew internally and finally sends
 msgInit, which developers DO need to handle.

 Return Value
    stsNewStructError:		The new struct was not properly initialized, it
                            was used more than once, or it was overwritten.
    stsBadParam:			Format of well-known UID was invalid.
    stsWellKnownExists:		Well-known UID has already been created with a
                            different key.
    stsOSOutOfMem:			Too many objects have been created or system
                            memory is exhausted.
    stsProtectionViolation:	(clsClass) objCapInherit is disabled.
    stsSizeLimit:			(clsClass) More than the maximum amount of
                            instance data has been requested.
    stsBadAncestor:			(clsClass) Ancestor is not a class.
*/
#define	msgNew	MakeMsg(clsObject, 4)

/****************************************************************************
 msgNewWithDefaults	takes new struct for object being created, returns STATUS
	category: class message
	Creates an object with default values.  
	
 Self sends msgNewDefaults followed by	msgNew.  Useful when changes to the
 new struct are NOT required.
*/
#define	msgNewWithDefaults	MakeMsg(clsObject, 5)

/****************************************************************************
 msgInit	takes new struct for object being created, returns STATUS
	Sent to the object immediately after it is created.

 When msgInit reach clsObject the capabilities and the key in the newArgs
 are set for the object.  This means that, unlike most messages, developers
 must call their ancestor AFTER processing this one.
*/
#define	msgInit	MakeMsg(clsObject, 6)

/****************************************************************************
 msgCreated	takes new struct for object being created, returns STATUS
	Sent to the object after it is fully created, i.e., after msgInit.

 This message is only sent if objCapCreateNotify is enabled.
*/
#define	msgCreated	MsgNoError(MakeMsg(clsObject, 46))

/****************************************************************************
 msgDestroy	takes OBJ_KEY, returns STATUS
	Destroys the object.

 When msgDestroy is sent to the object, clsObject sends msgFreeOK,
 msgFreeing and msgFree to self.  msgFreePending is sent to the observers.
 Only clsObject should handle msgDestroy.   (That is, like msgNew,
 developers send msgDestroy but never handle it.)

 Return Value
    stsProtectionViolation:	objCapFree is disabled and the key does not
                            open the object.
    stsClassHasReferences:	(clsClass) Instances of the class object still
                            exists.	 Only returned when the object being
                            destroyed is a class.
*/
#define	msgDestroy	MakeMsg(clsObject, 28)

/****************************************************************************
 msgFreeOK	takes OBJ_KEY, returns STATUS
	Sent as the first of three messages to destroy the object.

 There is no point in handling this message unless you have some reason
 to refuse to be freed, in which case return stsVetoed.  Note that if
 the process that owns the object or the class of the object is destroyed,
 the object will be destroyed too, regardless of what it does with
 msgFreeOK.  This is mainly useful for immortal system objects.

 See Also
	msgDestroy

 Return Value
    stsClassHasReferences:	(clsClass) Instances of the class object still
                            exists.	 Only returned when the object being
                            destroyed is a class.

*/
#define	msgFreeOK	MsgNoError(MakeMsg(clsObject, 14))

/****************************************************************************
 msgFreeing	takes OBJ_KEY, returns STATUS
	Sent as the second of three messages to destroy the object.

 Most developers never handle this message either.  If an object is part of
 a tangled web of other objects, all of which are supposed to be freed
 whenver any of them is freed, it's possible to get a loop where two
 objects respond to msgFree by trying to free each other.  The first object
 that receives msgFreeing should extract itself from any other object that
 might try to free it.  When it receives msgFree, it can then safely send
 msgDestroy to those other objects.

 See Also
	msgDestroy
*/
#define	msgFreeing	MsgNoError(MakeMsg(clsObject, 90))

/****************************************************************************
 msgFree	takes OBJ_KEY, returns STATUS
	Sent as the last of three messages to destroy the object.

 msgFree must succeed and error status should never be returned.  Any
 validation should be done during msgFreeOK.  (Like msgInit, developers
 handle this message but never send it.)

 See Also
	msgDestroy
*/
#define	msgFree	MakeMsg(clsObject, 8)

/****************************************************************************
 msgFreePending	takes OBJECT, returns STATUS
	category: observer notification
	Sent to observers immediately before the object is freed.

 If an observer really cares about the final state of the object, this is
 the last opportunity to send it a message.

 See Also
	msgDestroy
*/
#define	msgFreePending	MsgNoError(MakeMsg(clsObject, 70))

/****************************************************************************
 msgRestoreInstance	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.

 Creates an instance of the class and sends the new object msgRestore.  If
 the new object is a class, msgRestoreMsgTable is sent after msgRestore.

 Return Value
	stsRequestNotSupported:	Instances of clsClass cannot be restored.
*/
#define	msgRestoreInstance	MakeMsg(clsObject, 80)

/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.

 After a new object has been created with msgRestoreInstance it is sent
 msgRestore.  The object reads its instance data from the object file.
*/
#define	msgRestore	MakeMsg(clsObject, 10)

/****************************************************************************
 msgRestoreMsgTable	takes PP_MSG, returns STATUS
	Returns the message table for the class.
		
 Because the address of a message table is dynamic the ancestor of the
 class must provide the message table address when the class is restored. 
 The ancestor can store extra information needed to find the message table
 in the instance data or as a saved property.
*/
#define	msgRestoreMsgTable	MakeMsg(clsObject, 116)

/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Causes the object to file itself in an object file.

 clsObject files the capabilities of the object and any property that has
 tag flag1 set.  For example:
//{
	#define MY_PROP MakeTagWithFlags(clsFoo,tagNum,1)
//}

 Return Value
	stsRequestNotSupported:	(clsClass) Classes not do file.
*/
#define	msgSave	MakeMsg(clsObject, 12)

/****************************************************************************
 msgCopy                takes P_OBJ_COPY, returns STATUS
	Passes back a copy of the object.

 This message will pass back a copy of the object receiving the message.
 This object will be created by opening a temporary resource file, sending
 msgSave to the object, and then sending msgCopyRestore to the passed in
 requestor object. It will then close and destroy the temporary file. Note
 that the requestor object could be in a different task from the object
 receiving this message. In this situation, the copy of the object will
 exist in new task.

 Return Value
	stsFailed:	Could not open temporary resource file.

 See Also
	msgCopyRestore
*/
#define msgCopy	MakeMsg(clsObject, 54)

typedef struct OBJ_COPY {

	OBJECT		requestor;      // In: Object to receive msgCopyRestore
	OBJECT		object;         // Out: UID of copied object
	U32			reserved[4];    // Reserved.

} OBJ_COPY, * P_OBJ_COPY;

/****************************************************************************
 msgCopyRestore               takes P_OBJ_COPY_RESTORE, returns STATUS
    Restores the passed in object.

 This message is sent to the object with an object resource Id, and a file
 locator (a resource file). This will result in msgRestore being sent to
 the appropriate object to read in the resource object.  Sent to the
 requestor object when performing a msgCopy.

 See Also
	msgCopy
*/
#define msgCopyRestore	MakeMsg(clsObject, 56)

// This struct is copied from fs.h
typedef struct OBJ_FS_LOCATOR {
	
	OBJECT			uid;
	P_STRING		pPath;

} OBJ_FS_LOCATOR;

typedef struct OBJ_COPY_RESTORE {

	OBJ_FS_LOCATOR	locator;        // In: File locator that the object is in
	RES_ID          resId;          // In: Resource id of filed object
	OBJECT          object;         // Out: Uid of object to return
	U32             reserved[4];	// Reserved.

} OBJ_COPY_RESTORE, * P_OBJ_COPY_RESTORE;

/****************************************************************************
 msgDump	takes S32, returns STATUS
	Causes each ancestor to print interesting debugging information.

 Each class should implement a msgHandler for msgDump.	The msgHandler
 should print out interesting information for the object.

 The parameter to msgDump is used to determine how much information to
 print.

 Suggested values for pArgs:
    0:		Implementer's choice.	Print whatever information is most
            useful.
    1:		Terse.  One line only.
    -1:		Terse including embedded objects.	One line of information
            plus one line for each embedded object, e.g., a menu would
            display information about each menu item.
	maxS32:	Verbose.  All possible information about the object.
    minS32:	Verbose including embedded objects.  The maximum amount of
            information.
	other:	All other values are implementation dependent.

 If the value of the parameter is in between two defined values the action
 should be based on the smaller value.

 Suggested format:
//{
	  "msgDump(yourClassName): yourDebuggingInformation"
//}

 clsObject defines pArgs as:
    0:		The object's capabilities and internal address.
    1:		Same as 0.
    2:		Same as 1 plus owner, number of observers, number of properties,
            the size of instance data and size of property list.  maxS32:
            Same as 2 plus hex dump of instance data.
	-1:	 	Same as 0 plus msgDump to observers. ([Not implemented])
    -2:	 	Same as -1 plus owner, number of observers, number of
            properties, the size of instance data. ([Not implemented])
    minS32:	Same as -2 plus hex dump of instance data. ([Not implemented])

 clsClass defines pArgs as:
    0:		The class capabilities, size of data for instances, the number of
            instances and subclasses of the class.
	1:		Same as 0.
	2:		Same as 1 plus ancestor and newArgs size.
	maxS32:	Same as 2. ([Not implemented])
*/
#define	msgDump	MakeMsg(clsObject, 52)

/****************************************************************************
 msgException	takes P_OBJ_EXCEPTION, returns STATUS
	category: observer notification
	Sent to observers of theProcess, an object within each process, when
	an exception occurs within that process.

 If a subtask is being terminated only objects owned by the subtask are
 notified.
*/
#define	msgException	MsgNoError(MakeMsg(clsObject, 100))

typedef struct OBJ_EXCEPTION {

	OS_TASK_ERROR	errorCode;		// In: Type of exception
	OS_TASK_ID		task;			// In: Task that received the exception
	U32				spare;			// Unused (reserved)

} OBJ_EXCEPTION, *P_OBJ_EXCEPTION;

/****************************************************************************
 msgTaskTerminated	takes P_OBJ_EXCEPTION, returns STATUS
	category: observer notification
	Sent to observers of theProcess, an object within each process, after
	the task is terminated.
*/
#define	msgTaskTerminated	MsgNoError(MakeMsg(clsObject, 112))

/****************************************************************************
 msgScavenge	takes OS_TASK_ERROR, returns STATUS
	category: descendant responsibility
	Sent to the object when a class has objCapScavenge set and the object's
	task is being terminated by request or because of an error.

 This message will only be executed by class that set objCapScavenge.  Do
 not pass this message to your ancestor.
*/
#define	msgScavenge	MsgNoError(MakeMsg(clsObject, 102))

/****************************************************************************
 msgScavenged	takes OS_TASK_ERROR, returns STATUS
	category: observer notification
	Sent to the observers AFTER the object has been scavenged.
*/
#define	msgScavenged	MsgNoError(MakeMsg(clsObject, 104))

/****************************************************************************
 msgFreeSubTask	takes P_SUBTASK_FREE, returns STATUS
	Sent to theProcess to free a subtask.

 Useful for delayed termination when message is posted to theProcess.

 Return Value
	stsOSInvalidOperationForTask:  Task was not a subtask of this process.
*/
#define	msgFreeSubTask	MsgNoError(MakeMsg(clsObject, 104))

typedef struct OBJ_SUBTASK_FREE {

  OS_TASK_ID		task;			// In:	Task to be terminated
  OS_TASK_ERROR		exitCode;		// In:	Exit code for task termination

} OBJ_SUBTASK_FREE, * P_OBJ_SUBTASK_FREE;

/****************************************************************************
 msgHeap	takes P_OS_HEAP_ID, returns STATUS
	Returns the preferred heap to use when allocating storage for this object.
*/
#define	msgHeap	MakeMsg(clsObject, 96)


/****************************************************************************
 msgCan	takes OBJ_CAPABILITY, returns STATUS
	Checks the object's capabilities.

 If the capabilities in the parameter are all enabled, msgCan returns stsOK
 otherwise stsProtectionViolation is returned.

 Return Value
	stsProtectionViolation:	 Capability disabled.
*/
#define	msgCan	MakeMsg(clsObject, 36)

/****************************************************************************
 msgDisable	takes P_OBJ_CAPABILITY_SET, returns STATUS
	Disables some or all of the object's capabilities.

 Return Value
	stsProtectionViolation:	 Key does not open the object.
*/
#define	msgDisable	MakeMsg(clsObject, 16)

/****************************************************************************
 msgEnable	takes P_OBJ_CAPABILITY_SET, returns STATUS
	Enables some or all of the object's capabilities.

 Return Value
	stsProtectionViolation:	 Key does not open the object.
*/
#define	msgEnable	MakeMsg(clsObject, 18)

/****************************************************************************
 msgIsA	takes CLASS, returns STATUS
	Tests if the object's class inherits from the class.

 Return Value
 	stsOK:				Class is an ancestor of the object's class.
	stsBadAncestor:	 Class is not an ancestor of the object's class.
*/
#define	msgIsA	MakeMsg(clsObject, 30)

/****************************************************************************
 msgAncestorIsA	takes CLASS, returns STATUS
	Tests if self inherits from the	class.

 This is a clsClass message and can only be sent to a class.  Consider
 using msgIsA if the object is not a class.

 Return Value
 	stsOK:				Class parameter is an ancestor.
	stsBadObject:		Class parameter is not an object.
	stsBadAncestor:		Class parameter is not an ancestor.
*/
#define	msgAncestorIsA	MakeMsg(clsObject, 32)

/****************************************************************************
 msgClass	takes P_CLASS, returns STATUS
	Passes back the class of the object.
*/
#define	msgClass	MakeMsg(clsObject, 34)

/****************************************************************************
 msgAncestor	takes P_CLASS, returns STATUS
	Passes back the ancestor of the class.

 This is a clsClass message and can only be sent to a class.  Consider
 using msgClass if the object is not a class.
*/
#define	msgAncestor	MakeMsg(clsObject, 20)

/****************************************************************************
 msgSetLock	takes OBJ_LOCK_SET, returns STATUS
	Sets or changes the key of the object.

 Return Value
	stsProtectionViolation:	 Old key does not open the object.
*/
#define	msgSetLock	MakeMsg(clsObject, 106)

typedef struct OBJ_LOCK_SET {

  OBJ_KEY			oldKey;			// In:	Required to set lock
  OBJ_KEY			newKey;			// In:	New key, if successful

} OBJ_LOCK_SET, * P_OBJ_LOCK_SET;

/****************************************************************************
 msgUnlocks	takes OBJ_KEY, returns STATUS
	Tests if a key will unlock the object.

 Return Value
	stsProtectionViolation:	 Key does not open the object.
*/
#define	msgUnlocks	MakeMsg(clsObject, 38)

/****************************************************************************
 msgDuplicateLock	takes OBJECT, returns STATUS
	Locks the pArgs object with the same key as object.

 Return Value
	stsBadObject:  Parameter is not an object.
*/
#define	msgDuplicateLock	MakeMsg(clsObject, 40)

/****************************************************************************
 msgVersion	takes pNull, returns STATUS
	Returns the version of the object.

 Return Value
	stsScopeViolation:	Object was dynamic, request is nonsense.
*/
#define	msgVersion	MakeMsg(clsObject, 82)

/****************************************************************************
 msgNewArgsSize	takes pNull, returns STATUS
	Returns the size of the new struct required to create an instance of
	this class.

 This is a clsClass message and can only be sent to a class.
*/
#define	msgNewArgsSize	MakeMsg(clsObject, 92)

/****************************************************************************
 msgOwner	takes P_OS_TASK_ID, returns STATUS
	Passes back the task that owns this object.
*/
#define	msgOwner	MakeMsg(clsObject, 22)

/****************************************************************************
 msgSetOwner	takes P_OBJ_OWNER, returns STATUS
	Changes the owner task.

 Return Value
	stsProtectionViolation:	 Key does not open the object.
*/
#define	msgSetOwner	MakeMsg(clsObject, 24)

/****************************************************************************
 msgProp	takes P_OBJ_PROP, returns STATUS
	Passes back the value of a property for the object.

 Return Value
	stsBadPropTag:	Tag value was not in the proper range.
*/
#define	msgProp	MakeMsg(clsObject, 108)

/****************************************************************************
 msgSetProp	takes P_OBJ_PROP, returns STATUS
	Sets a property on the object.

 clsObject files any property that has tag flag 1 turned on.  For example:
//{
	#define MY_PROP MakeTagWithFlags(clsFoo,tagNum,1)
//}

 Return Value
	stsBadPropTag:				Tag value was not in the proper range.
	stsProtectionViolation:		Key does not open the object.
*/
#define	msgSetProp	MakeMsg(clsObject, 110)


/****  msgObjectXXX	 ****/
/*
 * These msgObjectXXX messages can be used with ObjectCall() to get
 * information about all objects, regardless of their task.  Functionally
 * they are equivalent to msgXXX, when applicable.
*/

/****************************************************************************
 msgObjectIsA	takes P_OBJ_IS_A, returns STATUS
	Using the object and the class in the pArgs. Tests if the object's class
	inherits from the class.

 Return Value
	stsBadObject:		Parameter is not an object.
	stsBadAncestor:		Class is not an ancestor of the object's class.
*/
#define	msgObjectIsA	MakeMsg(clsObject, 84)

typedef struct OBJ_IS_A {

  OBJECT			object;			// In:	Source object
  CLASS				objClass;		// In:	Ancestor of source object's class

} OBJ_IS_A, * P_OBJ_IS_A;

/****************************************************************************
 msgObjectAncestorIsA	takes P_OBJ_ANCESTOR_IS_A, returns STATUS
	Tests if the descendant class inherits from the ancestor.

 This is a clsClass message and can only be sent to a class.

 Return Value
	stsBadObject:	One of the parameters is not a class.
	stsBadAncestor:	Ancestor parameter is not an ancestor.
*/
#define	msgObjectAncestorIsA	MakeMsg(clsObject, 86)

typedef struct OBJ_ANCESTOR_IS_A {

  CLASS				descendant;		// In:	Source class (always a class)
  CLASS				ancestor;		// In:	Ancestor of the descendant

} OBJ_ANCESTOR_IS_A, * P_OBJ_ANCESTOR_IS_A;

/****************************************************************************
 msgObjectClass	takes P_OBJ_CLASS, returns STATUS
	Passes back the class for the object in pArgs.

 Return Value
	stsBadObject:  Object or class parameters are not objects.
*/
#define	msgObjectClass	MakeMsg(clsObject, 88)

typedef struct OBJ_CLASS {

  OBJECT			object;			// In:	Source object
  CLASS				objClass;		// Out: Class of source object

} OBJ_CLASS, * P_OBJ_CLASS;

/****************************************************************************
 msgObjectOwner	takes P_OBJ_OWNER, returns STATUS
	Passes back the owning task for the object in pArgs.

 Return Value
	stsBadObject:  Parameter is not an object.
*/
#define	msgObjectOwner	MakeMsg(clsObject, 26)

/****************************************************************************
 msgObjectValid	takes OBJECT, returns STATUS
	Tests that the object in pArgs exists.

 Return Value
	stsBadObject:	Parameter is not an object.
	stsBadAncestor:	Invalid ancestor.
*/
#define	msgObjectValid	MakeMsg(clsObject, 42)

/****************************************************************************
 msgObjectVersion	takes OBJECT, returns STATUS
	Returns the version of the object in pArgs.

 Return Value
	stsBadObject:		Parameter is not an object.
	stsScopeViolation:	Parameter was dynamic, request is nonsense.
*/
#define	msgObjectVersion	MakeMsg(clsObject, 44)

/****************************************************************************
 msgObjectNew	takes newArgs, returns STATUS
	Creates a new object in the same context as the object that
	receives this message.

 Return Value
	stsProtectionViolation:	objCapCreate is disabled.
	stsScopeViolation:		Must be executed in the owner task of the
							receiving object.
*/
#define	msgObjectNew	MakeMsg(clsObject, 98)

/****************************************************************************
 msgTrace	takes TAG, returns STATUS
	Turn tracing on for classes and objects.  Return value is stsTraceOn if
	tracing	was on and stsTraceOff if tracing was off.

 When tracing is turned on for the object, every ObjectCall() to the object
 causes a 3-line message to be printed.  The format of the output is:
//{
	C> Trace ObjectCall: @ cls="ancestor name"				  task="task"
	C> object="object name"									  depth="D"
	C> msg="message name", pArgs="address", pData="address"
//}

 On return from the ObjectCall() a 2-line message is printed.  The format
 of the output is:
//{
	C> Trace ObjectCall: returns="status value"				  task="task"
	C> object="object name"									  depth="D/C"
//}

 where task is the task id in hex, depth is the number of recursive
 dispatch loops.  All names are printed symbolically when symbols are
 available.

 ObjectCallAncestor() calls are traced for objects if tracing is on for the
 object and the debug flag /DC1000 is set.

 When tracing is turned on for a class, the class is traced as an object.
 In addition, all ObjectCallAncestor() calls that pass through the class
 are traced.
*/
#define	msgTrace	MakeMsg(clsObject, 48)

#define objTraceOn	(P_ARGS)stsTraceOn
#define objTraceOff	(P_ARGS)stsTraceOff


/****************************************************************************
 msgMutate	takes P_OBJ_MUTATE, returns STATUS
	Changes the ancestor of the object to be the newAncestor class.

 The total size of the instance data for the new and old ancestors must be
 equal, this is the sum for all the ancestors up to clsObject.  This
 message is NOT intended for general use.	Use it when the behavior of an
 existing object needs to be overridden.

 Return Value
	stsBadAncestor:	The newAncestor class is not a valid class.
	stsSizeLimit:	The sizes of new and old instance data don't match.
*/
#define	msgMutate	MakeMsg(clsObject, 46)

typedef struct OBJ_MUTATE {

  CLASS				newClass;		// In:	Object's new class
  OBJ_KEY			key;			// In:	If required by caps

} OBJ_MUTATE, * P_OBJ_MUTATE;


/****************************************************************************
 msgAddObserver	takes OBJECT, returns STATUS
	Adds an observer to the end of the object's observer list.

 Return Value
	stsBadObject:  			Parameter is not an object.
	stsProtectionViolation:	objCapObservable is disabled.
    stsScopeViolation:		Observer is local and has a different owner
                            than the observed object or the observed object
                            is callable.
    stsAlreadyAdded:  		The same observer has been added twice.  This
                            is only a warning, the observers are ref
                            counted. Two adds require two removes.
*/
#define	msgAddObserver	MakeMsg(clsObject, 58)

/****************************************************************************
 msgAddObserverAt	takes P_OBJ_OBSERVER_POS, returns STATUS
	Adds an observer at the specified position in the observer list.

 Return Value
	stsBadObject:  			Parameter is not an object.
	stsProtectionViolation:	objCapObservable is disabled.
    stsScopeViolation:		Observer is local and has a different owner
                            than the observed object or the observed object
                            is callable.
    stsAlreadyAdded:		The same observer has been added twice.  This is
                            only a warning, the observers are ref counted.
                            Two adds require two removes.
*/
#define	msgAddObserverAt	MakeMsg(clsObject, 78)

/****************************************************************************
 msgRemoveObserver	takes OBJECT, returns STATUS
	Removes an observer from the object's observer list.

 msgRemoved is sent to the observer after it is removed.

 Return Value
	stsProtectionViolation:		objCapObservable disabled.
	stsAlreadyRemoved:			Observer was not on the list.
*/
#define	msgRemoveObserver	MakeMsg(clsObject, 60)

/****************************************************************************
 msgNotifyObservers	takes P_OBJ_NOTIFY_OBSERVERS, returns STATUS
	Sends a message to the observers.

 Any observer that returns stsBadObject is removed from the observer list.
*/
#define	msgNotifyObservers	MakeMsg(clsObject, 62)

/****************************************************************************
 msgPostObservers	takes P_OBJ_NOTIFY_OBSERVERS, returns STATUS
	Posts a message to the observers.

 Any observer that returns stsBadObject is removed from the observer list.
*/
#define	msgPostObservers	MakeMsg(clsObject, 94)

/****************************************************************************
 msgEnumObservers	takes P_OBJ_ENUM_OBSERVERS, returns STATUS
	Passes back the observer list.

 Return Value
	stsEndOfData:  The size of the array is greater than or equal to the
				   number of observer.
*/
#define	msgEnumObservers	MakeMsg(clsObject, 64)

typedef struct OBJ_ENUM_OBSERVERS {

  U16				max,		// In:	Size of pObservers[] 
					count;		// In:	# to pass back in pObservers[].
								//		If count > max memory may be allocated
								// Out: # of valid entries in pObservers[]
  P_OBJECT			pObservers; // In:	ptr to array
								// Out: If memory was allocated 
								//		client should free the memory
  U16				next;		// In:	Set to 0 for the first call
								// Out: Next available entry

} OBJ_ENUM_OBSERVERS, * P_OBJ_ENUM_OBSERVERS;

/****************************************************************************
 msgGetObserver	takes P_OBJ_OBSERVER_POS, returns STATUS
	Passes back the observer at the specified position in the observer list.

 objNull is returned if the position is not in the observer list.
*/
#define	msgGetObserver	MakeMsg(clsObject, 74)

/****************************************************************************
 msgNumObservers	takes P_U16, returns STATUS
	Passes back the number of observers for this object.
*/
#define	msgNumObservers	MakeMsg(clsObject, 72)

/****************************************************************************
 msgAdded	takes OBJECT, returns STATUS
	category: observer notification
	Sent to the observer when it is added to an object's observer list.
*/
#define	msgAdded	MsgNoError(MakeMsg(clsObject, 66))

/****************************************************************************
 msgRemoved	takes OBJECT, returns STATUS
	category: observer notification
	Sent to the observer when it is removed from an object's observer list.
*/
#define	msgRemoved	MsgNoError(MakeMsg(clsObject, 68))

/****************************************************************************
 msgChanged	takes OBJECT, returns STATUS
	category: observer notification
	Generic message that can be used to notify observers that a change
	has occurred.
*/
#define	msgChanged	MsgNoError(MakeMsg(clsObject, 76))

/****************************************************************************
 msgNotUnderstood	takes P_MSG_NOT_UNDERSTOOD, returns STATUS
	Sent by clsObject when an unrecognized message is received.

 Return Value
	stsNotUnderstood:  Always returned by clsObject when this message 
					   reaches clsObject.
*/
#define	msgNotUnderstood	MakeMsg(clsObject, 50)

typedef struct MSG_NOT_UNDERSTOOD {

  MESSAGE			msg;			// In:	Message not understood
  P_ARGS			pArgs;			// In:	Args of message

} MSG_NOT_UNDERSTOOD, * P_MSG_NOT_UNDERSTOOD;

/****  Message wild cards ****/
/* Used to define a class wild card and as a table wild card. */
#define objWildCard -1

/* Wild card for clsObject. */
#define clsObjWildCard		MakeMsg(clsObject, objWildCard)

/* Wild card for clsClass. */
#define clsClsWildCard		MakeMsg(clsClass, objWildCard)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							  Functions									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 ObjectCall	returns STATUS
	Maps the message to the object's method (MsgHandler) and calls it 
	with pArgs.

 Return Value
	stsBadObject:		Object was invalid.
    stsScopeViolation:	Object owned by a different task and does not have
                        objCapCall set.
*/
STATUS EXPORTED ObjectCall(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs
);

/****************************************************************************
 ObjectCallAncestorCtx	returns STATUS
	Calls the next ancestor in the class chain.

 Developers usually can avoid calling this explicitly by specifying
 objCallAncestorBefore or (for a few messages) objCallAncestorAfter in the
 method table.  Occasionally, you need to call your ancestor in the middle
 of things, and this is the call you do it with.

 See Also
	ObjectCallAncestor

 Return Value
	stsBadContext:	if ctx parameter is bad.
*/
STATUS EXPORTED ObjectCallAncestorCtx(
	CONTEXT		ctx
);

/****************************************************************************
 ObjectCallAncestor	returns STATUS
	Calls the ancestor with the parameters supplied.

 In general you should use ObjectCallAncestorCtx().

 Return Value
	stsBadContext:	if ctx parameter is bad.
*/
STATUS EXPORTED ObjectCallAncestor(
	MESSAGE		msg, 
	OBJECT		self, 
	P_ARGS		pArgs,
	CONTEXT		ctx
);

/****************************************************************************
 ObjectSend	returns STATUS
	Generalized version of ObjectCall() that works across tasks boundaries.

 The pArgs block is copied into the address space of the task that owns the
 object and an ObjectCall() is executed in that task's context.  If lenArgs
 equals 0, pArgs block is not copied and the pointer is passed directly.
 In this case, pArgs must point to global storage.
	 
 While the current task is waiting for ObjectSend() to return, the task will
 continue to dispatch messages sent to objects owned by the task.  This allows
 sending to an object in another task, which in turns sends to an object owned
 by the current task, without deadlock.

 Return Value
	stsProtectionViolation:	objCapSend is disabled.
	stsSendTaskInvalid:	 	Object's owning task is invalid.
	stsTaskTerminated:		While waiting for a reply the object's task died.
*/
STATUS EXPORTED ObjectSend(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs,			// In only: Not updated
	SIZEOF		lenArgs
);

/****************************************************************************
 ObjectSendUpdate	returns STATUS
	Same as ObjectSend(), additionally the pArgs block is copied back
	to the current task.
*/
STATUS EXPORTED ObjectSendUpdate(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs,			// In/Out: Updated 
	SIZEOF		lenArgs
);

/****************************************************************************
 ObjectSendU32	returns STATUS
	Same as ObjectSend() without the length arg, lenArgs = 0.
*/
STATUS EXPORTED ObjectSendU32(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs			// In only: Not updated
);

/****************************************************************************
 ObjectSendTask	returns STATUS
	Same as ObjectSend() except the task is specified explicitly.

 For experts only: Use this routine with care, the task of the object is
 ignored.  ObjectSendTask() allows sending to well-known process-globals
 from outside the process, such as, theProcess.	 You might use this to
 communicate with theUndoManager in an embedded application.
*/
STATUS EXPORTED ObjectSendTask(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs,			// In only: Not updated
	SIZEOF		lenArgs,
	OS_TASK_ID	task
);

/****************************************************************************
 ObjectSendUpdateTask	returns STATUS
	Same as ObjectSendTask(), additionally the pArgs are updated.

 Experts only, use this routine with care.
*/
STATUS EXPORTED ObjectSendUpdateTask(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs,			// In/Out: Updated 
	SIZEOF		lenArgs,
	OS_TASK_ID	task
);

/****************************************************************************
 ObjectPost	returns STATUS
	Posts a message to the object via the system input queue.

ObjectPost() is similar to ObjectSend() but the message delivery is
deferred and the current task continues to run.  Because the current
task does not wait, it is not possible to return a status value or
pArgs.

The most common use of ObjectPost() is to delay the effect of a
msgDestroy.  For example, if a button sends you a message when it is
pressed, and you want to destroy the button at that point, you cannot
use ObjectCall() to send msgDestroy to it until after you have returned
from processing the message the button sent.  If you ObjectPost() the
msgDestroy, this guarantees the button won't receive it until you have
returned.

ObjectPost() is synchronized with respect to the input system.  A
posted message is placed in the system input queue.  When the message
reaches the head of the queue it is sent to the object in the context
of the task that owns the object.  A posted message is typically
dispatched by a task's top-level dispatch loop.  If the task is
already processing a message or waiting for a reply to a sent message
the posted message is queued.  The one exception is when the input
system is running system modal, in this case the posted messages are
delivered to any dispatch loop.  Dispatch loops are created whenever
an ObjectSend() is waiting for a reply.  The side effect is that any
task that is running concurrently may receive a posted message at any
time.
*/
STATUS EXPORTED ObjectPost(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs,
	SIZEOF		lenArgs
);

/****************************************************************************
 ObjectPostU32	returns STATUS
	Same as ObjectPost() without the length arg, lenArgs = 0.
*/
STATUS EXPORTED ObjectPostU32(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs
);

/****************************************************************************
 ObjectPostTask	returns STATUS
	Same as ObjectPost() except the task is specified explicitly.

 For experts only: Use this routine with care, the owning task of the object
 is ignored.  ObjectPostTask() allows posting to WKN process-globals from
 outside the process, such as, theProcess.
*/
STATUS EXPORTED ObjectPostTask(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs,
	SIZEOF		lenArgs,
	OS_TASK_ID	task
);

/****************************************************************************
 ObjectPostAsync	returns STATUS
	Similar to ObjectPost() but not synchronized with the input system.

 This call causes concurrency and all the difficulties associated with it.

 One of these difficulties, described in detail under ObjectPost, is
 the handling of posted messages when the input system is running
 system modal.
*/
STATUS EXPORTED ObjectPostAsync(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs,
	SIZEOF		lenArgs
);

/****************************************************************************
 ObjectPostAsyncTask	returns STATUS
	Same as ObjectPostAsync() except the task is specified explicitly.

 This call causes concurrency and all the difficulties associated with it.

 For experts only: Use this routine with care, the owning task of the object
 is ignored.  ObjectPostAsyncTask() allows posting to WKN process-globals from
 outside the process, such as, theProcess.
*/
STATUS EXPORTED ObjectPostAsyncTask(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs,
	SIZEOF		lenArgs,
	OS_TASK_ID	task
);

/****************************************************************************
 ObjectPostDirect	returns STATUS
	Similar to ObjectPostAsync() but can be dispatched by any dispatch loop.

 This call causes concurrency and all the difficulties associated with it.

 One of these difficulties, described in detail under ObjectPost, is
 the handling of posted messages when the input system is running
 system modal.
*/
STATUS EXPORTED ObjectPostDirect(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs,
	SIZEOF		lenArgs
);

/****************************************************************************
 ObjectPostDirectTask	returns STATUS
	Same as ObjectPostDirect() except the task is specified explicitly.

 This call causes concurrency and all the difficulties associated with it.

 For experts only: Use this routine with care, the owning task of the object
 is ignored.  ObjectPostDirectTask() allows posting to WKN process-globals
 from outside the process, such as, theProcess.
*/
STATUS EXPORTED ObjectPostDirectTask(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs,
	SIZEOF		lenArgs,
	OS_TASK_ID	task
);

/****************************************************************************
 ObjectWrite	returns STATUS
	Writes the instance data for self in a protected area.

 Return Value
	stsBadContext:	Invalid context.
*/
STATUS EXPORTED0 ObjectWrite(
	OBJECT		self, 
	CONTEXT		ctx,
	P_UNKNOWN	pData
);

/****************************************************************************
 ObjectWritePartial	returns STATUS
	Updates part of the instance data for self in a protected area.

 Return Value
	stsBadContext:	Invalid context.
*/
STATUS EXPORTED0 ObjectWritePartial(
	OBJECT		self, 
	CONTEXT		ctx,
	P_UNKNOWN	pData,
	SIZEOF		offset,
	SIZEOF		length
);

/****************************************************************************
 ObjectRead	returns STATUS
	Copies the instance data from protected storage into pBuf.

 The pData pointer passed into the MsgHandler is a faster way to read the
 protected data.
*/
STATUS EXPORTED ObjectRead(
	OBJECT		self,
	CONTEXT		ctx,
	P_UNKNOWN	pBuf
);

/****************************************************************************
 ObjectPoke	returns STATUS
	Writes the object's instance data.

 Copies pBuf into the instance data block for the class specified.

 Return Value
	stsBadAncestor:			ClassMsgTable did not correspond to an ancestor.
	stsProtectionViolation:	Key does not open the object.
*/
STATUS EXPORTED0 ObjectPoke(
	OBJECT		object,	 
	P_MSG		classMsgTable,		// Address of the class's table
	OBJ_KEY		key,				// Key for the class
	P_UNKNOWN	pBuf
);

/****************************************************************************
 ObjectPeek	returns STATUS
	Reads the object's instance data.

 Copies the instance data block for the class specified into pBuf.

 Return Value
	stsBadAncestor:			ClassMsgTable did not correspond to an ancestor.
	stsProtectionViolation:	Key does not open the object.
*/
STATUS EXPORTED ObjectPeek(
	OBJECT		object,	 
	P_MSG		classMsgTable,
	OBJ_KEY		key,
	P_UNKNOWN	pBuf
);

/****************************************************************************
 ObjectOwner	returns STATUS
	Returns the object's owner.

*/
OS_TASK_ID EXPORTED ObjectOwner(
	OBJECT		object
);

/****************************************************************************
 ObjectValid	returns STATUS
	Returns stsOK if the object is validate, otherwise an error is returned.

*/
STATUS EXPORTED ObjectValid(
	OBJECT		object
);


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							 Default MsgHandlers						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Default MsgHandler that always returns stsOK. */
MsgHandler(StsOKMsgHandler);

/* Default MsgHandler that always returns stsFailed. */
MsgHandler(StsFailedMsgHandler);

/* Default MsgHandler that always returns stsReqNotSupported. */
MsgHandler(StsReqNotSupportedMsgHandler);

/* Default MsgHandler that always returns stsNotYetImplemented. */
MsgHandler(StsNotYetImplemented);

/* Default MsgHandler that always returns stsMessageIgnored. */
MsgHandler(StsMessageIgnoredMsgHandler);


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				   Functions for Generating Symbolic Names				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * These routines are very useful for debugging.  It is MUCH more useful to
 * be able to print "stsBadParameter" instead of some 32-bit hex number.
*/

/****************************************************************************
 ClsStsToString	returns P_STRING
	Takes a STATUS and returns its symbolic name or [wkn=num:sts=num].

 Returns either an internal pointer to a symbolic name or the pArgs buffer.
 If a symbolic name is not found, a string [wkn=num:sts=num] is constructed
 in the pArgs buffer.

 Symbolic names are added via ClsMgrSymbolsInit().
*/
P_STRING EXPORTED ClsStsToString(
	STATUS sts,
	P_STRING pStr
);

/****************************************************************************
 ClsMsgToString	returns P_STRING
	Takes a message and returns its symbolic name or [wkn=num:msg=num].

 Returns either an internal pointer to a symbolic name or the pArgs buffer.
 If a symbolic name is not found, a string [wkn=num:msg=num] is constructed
 in the pArgs buffer.

 Symbolic names are added via ClsMgrSymbolsInit().
*/
P_STRING EXPORTED ClsMsgToString(
	MESSAGE		msg,
	P_STRING	pStr
);

/****************************************************************************
 ClsTagToString	returns P_STRING
	Takes a message and returns its symbolic name or [wkn=num:tag=num].

 Returns either an internal pointer to a symbolic name or the pArgs buffer.
 If a symbolic name is not found, a string [wkn=num:tag=num] is constructed
 in the pArgs buffer.
	 
 Currently, TAGs and MSGs are kept in the same list.  If a TAG and MSG have
 the same value then first one found will be displayed.  This may change in
 the future.

 Symbolic names are added via ClsMgrSymbolsInit().
*/
P_STRING EXPORTED ClsTagToString(
	TAG			tag,
	P_STRING	pStr
);

/****************************************************************************
 ClsObjToString	returns P_STRING
	Takes an OBJECT and returns its symbolic name or [type:num:num].

 Returns either an internal pointer to a symbolic name or the pArgs buffer.
 If a symbolic name is not found, a string [type=num:num] is constructed in
 the pArgs buffer.

 Symbolic names are added via ClsMgrSymbolsInit().
*/
P_STRING EXPORTED ClsObjToString(
	OBJECT		object,
	P_STRING	pStr
);

/****************************************************************************
 ObjectInfoString	returns P_STRING
	Takes an OBJECT and returns its symbolic name and additional information.

 Formats is the first if the name is found, and the second if not:
//{
	name (cls=name or [type=num:num])
	[type=num:num] (cls=name or [type=num:num])
//}

 Return Value
	stsBadObject:  Parameter is not an object.
*/
P_STRING EXPORTED ObjectInfoString(
	OBJECT		object,
	P_STRING	pStr
);

/****************************************************************************
 ClsStringToSts	returns STATUS
	Takes a symbolic name as a string and returns the corresponding STATUS.
*/
STATUS	 EXPORTED ClsStringToSts(
	P_STRING	sts
);

/****************************************************************************
 ClsStringToMsg	returns MESSAGE
	Takes a symbolic name as a string and returns the corresponding message.
*/
MESSAGE	 EXPORTED ClsStringToMsg(
	P_STRING	msg
);

/****************************************************************************
 ClsStringToTag	returns TAG
	Takes a symbolic name as a string and returns the corresponding tag.

 Currently, TAGs and MSGs are kept in the same list.  If a TAG and MSG
 have the same value then first one found will be displayed.  This may change
 in the future.

*/
MESSAGE	 EXPORTED ClsStringToTag(
	P_STRING	tag
);

/****************************************************************************
 ClsStringToObj	returns OBJECT
	Takes a symbolic name as a string and returns the corresponding OBJECT.
*/
OBJECT	 EXPORTED ClsStringToObj(
	P_STRING	object
);

/****************************************************************************
 ClsSymbolsInit	returns STATUS
	Adds three arrays of symbolic names (OBJECT, MSG, STATUS) to the database.

 Each group of arrays is labelled with a tag.  If two groups have the same
 tag, the last group to be added replaces the earlier group.  The arrays
 must be in shared, user visible memory.

 Return Value
	stsBadParam:	symbols were not in shared, user visible memory
*/
STATUS EXPORTED0 ClsSymbolsInit(
  P_STRING			type,
  P_CLS_SYM_OBJ		objSymbols,
  P_CLS_SYM_MSG		msgSymbols,
  P_CLS_SYM_STS		stsSymbols
);


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					   Low-Level Task Dispatch Routines					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 ObjectMsgLoop	returns STATUS
	Receives and dispatches object messages forever.

 If you create a sub-task with OSSubTaskCreate(), and you want that subtask
 to be able to receive messages, then you have to make it call this
 routine.  ObjectMsgLoop() never returns.  It just sits there waiting for
 messages generated by input events or sent from other processes and
 calling the appropriate local message handler for each one in turn.  Even
 if you never use this directly, knowing that it exists makes it much
 easier to understand the difference between ObjectCall, ObjectPost, and
 ObjectSend.

 Return Value
	stsBadParam:  Bad ITMSG_INFO parameter.
*/
#define ObjectMsgLoop() ObjectMsgDispatch(pNull)


/****************************************************************************
 ObjectMsgDispatch	returns STATUS
	Dispatches object message received by OSITMsgReceive().

 Return Value
	stsBadParam:	Bad ITMSG_INFO parameter.  ITMsg type must be one of
					osClsmgrSend or osClsmgrPost.
*/
STATUS EXPORTED ObjectMsgDispatch(P_OS_ITMSG_INFO pITMsg);


/****************************************************************************
 ObjectMsgDispatchInfo	returns STATUS
	Passes back information on the current ObjectMsgDispatch frame.

 Return Value
	stsBadParam:	Bad ITMSG_INFO parameter.
	stsFailed:		Not inside a dispatch loop or invalid frame number
*/
STATUS EXPORTED ObjectMsgDispatchInfo(
  P_OS_ITMSG_INFO	pInfo,			// Out: ITMSG_INFO for requested frame
  P_U32				pLevel			// In/Out: requested frame
									// In: requested dispatch frame,
									//     maxU32 = current, 1 = top level
									// Out: actual level of dispatch frame.
);


Enum32(SEND_TYPE)			// (.asm)
{
	objSendNoUpdate	= flag0,
	objSendUpdate	= flag1,
	objPostAsync	= flag2,
	objPostDirect	= flag3,
	objSendMax		= flag10

};

/*
 * Used by ObjectMsgExtract() and ObjectmsgAlter().  All fields are out
 * parameters for ObjectMsgExtract and in parameters of ObjectMsgAlter.  The
 * token field is currently not used and not settable by ObjectMsgAlter.
*/
typedef struct OBJ_DISPATCH_INFO {

  MESSAGE			msg;
  OBJECT			object;
  P_ARGS			pArgs;
  U32				length;
  U32				token;
  SEND_TYPE			type;

} OBJ_DISPATCH_INFO, *P_OBJ_DISPATCH_INFO;	

/****************************************************************************
 ObjectMsgExtract	returns STATUS
	Extracts the interesting ObjectSend fields from the ITMsg packet.

 Return Value
	stsBadParam:  Bad ITMSG_INFO parameter.
*/
STATUS EXPORTED ObjectMsgExtract(
  P_OS_ITMSG_INFO		pITMsg,
  P_OBJ_DISPATCH_INFO	pInfo
);


/****************************************************************************
 ObjectMsgAlter	returns STATUS
	Alters the ObjectSend fields of the ITMsg packet.

 Return Value
	stsBadParam:  Bad ITMSG_INFO parameter.
*/
STATUS EXPORTED ObjectMsgAlter(
  P_OS_ITMSG_INFO		pITMsg,
  P_OBJ_DISPATCH_INFO	pInfo
);


/* 
 * These structs are used by the method compiler, outside of Penpoint.
*/

Enum16(MSG_HANDLER_FLAGS) {

	objCallAncestorBefore	= flag0,	// Call ancestor before this handler
	objCallAncestorAfter	= flag1,	// Call ancestor after this handler
	objDerefIData			= flag2,	// No-op
	objInheritMethod		= flag3,	// No-op
	objClassMessage			= flag4,	// Handle messages sent to a class
	objSaveSpace			= flag5,	// Optimize for space
	objSaveTime				= flag6		// Optimize for time
};

typedef struct MSG_INFO {

	MESSAGE				msg;
	P_U8	 			functionName;
	MSG_HANDLER_FLAGS	flags;

} MSG_INFO, * P_MSG_INFO;

typedef struct CLASS_INFO {

	P_U8			tableName;			// name to use for compiled table
	P_MSG_INFO		msgTable;			// message table to compile
	U32				flags;				// no flags, must be set to zero

} CLASS_INFO, * P_CLASS_INFO;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							  Debugging Support							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 ObjectCallNoDebug	returns STATUS
	Same as ObjectCall() but prevents tracing (i.e., no debug output for /DC1)
*/
STATUS EXPORTED ObjectCallNoDebug(
	MESSAGE		msg, 
	OBJECT		object, 
	P_ARGS		pArgs
);

#define objMaxCallsDepth	10

typedef struct OBJ_STATISTICS {
	U32 numObjReads;
	U32 numObjWrites;
	U32 numObjPeeks;
	U32 numObjPokes;
	U32 numObjCalls;
	U32 numObjSends;
	U32	numObjPosts;
	U32 depthObjCalls[objMaxCallsDepth];
	U32 numObjMaxDepth;

} OBJ_STATISTICS, *P_OBJ_STATISTICS;

/****************************************************************************
 ClsClearStatistics	returns STATUS
	Zeros the statistics gathering counters.
*/
STATUS EXPORTED ClsClearStatistics(void);

/****************************************************************************
 ClsDumpStatistics	returns STATUS
	Prints the current value of the statistics.
*/
STATUS EXPORTED ClsDumpStatistics(void);

/****************************************************************************
 ClsStatistics	returns STATUS
	Passes back the current value of the statistics in stats parameter.
*/
STATUS EXPORTED ClsStatistics(P_OBJ_STATISTICS stats);

/****************************************************************************
 ClsSetStatistics	returns STATUS
	Resets the value of the statistics to stats parameter.

 By calling ClsStatistics() at the beginning of a routine and 
 ClsSetStatistics() at the end selected routines can be exempted from
 statistics gathering.
*/
STATUS EXPORTED ClsSetStatistics(P_OBJ_STATISTICS stats);


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							   Debugging Macros							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * The debugging macros are short-hand for a call to the appropriate function
 * followed by a conditional test and action.	 All the message passing
 * functions have macros that: return if there is an error (Ret), jump to a
 * label on an error (Jmp) and test for an error and return the value (OK). 
 * ObjectCall and ObjectCallAncestor have two additional macros, Failed and
 * Chk.
 *
 * Standard GO error recovery is done by using the Ret() form as long as
 * there's nothing to clean up and then using the Jmp() form to jump to a
 * label at the bottom of the routine that knows how to clean up.  Note that
 * both Ret() and Jmp() forms use Warn() forms of their respective calls, so
 * any sts < stsOK generates an error message if DEBUG is set.
*/

/* ObjectCall */

#define ObjCallRet(m,o,p,s) \
	if (((s) = ObjCallWarn(m,o,p)) < stsOK) return s; else

#define ObjCallJmp(m,o,p,s,x) \
	if (((s) = ObjCallWarn(m,o,p)) < stsOK) goto x; else

#define ObjCallOK(m,o,p,s) ((s = ObjCallWarn(m,o,p)) >= stsOK)

#define ObjCallFailed(m,o,p,s) ((s = ObjCallWarn(m,o,p)) < stsOK)

#define ObjCallChk(m,o,p,s) ((s = ObjectCall(m,o,p)) < stsOK)


/* ObjectCallAncestor */

#define ObjCallAncestorRet(m,o,p,c,s) \
	if (((s) = ObjCallAncestorWarn(m,o,p,c)) < stsOK) return s; else

#define ObjCallAncestorJmp(m,o,p,c,s,x) \
	if (((s) = ObjCallAncestorWarn(m,o,p,c)) < stsOK) goto x; else

#define ObjCallAncestorOK(m,o,p,c,s) \
	((s = ObjCallAncestorWarn(m,o,p,c)) >= stsOK)

#define ObjCallAncestorFailed(m,o,p,c,s) \
	((s = ObjCallAncestorWarn(m,o,p,c)) < stsOK)

#define ObjCallAncestorChk(m,o,p,c,s) \
	((s = ObjectCallAncestor(m,o,p,c)) < stsOK)


#define ObjCallAncestorCtxRet(c,s) \
	if (((s) = ObjCallAncestorCtxWarn(c)) < stsOK) return s; else

#define ObjCallAncestorCtxJmp(c,s,x) \
	if (((s) = ObjCallAncestorCtxWarn(c)) < stsOK) goto x; else

#define ObjCallAncestorCtxOK(c,s) \
	((s = ObjCallAncestorCtxWarn(c)) >= stsOK)


/* ObjectSend */

#define ObjSendRet(m,o,p,l,s) \
	if (((s) = ObjSendWarn(m,o,p,l)) < stsOK) return s; else

#define ObjSendJmp(m,o,p,l,s,x) \
	if (((s) = ObjSendWarn(m,o,p,l)) < stsOK) goto x; else

#define ObjSendOK(m,o,p,l,s) ((s = ObjSendWarn(m,o,p,l)) >= stsOK)


/* ObjectSendUpdate */

#define ObjSendUpdateRet(m,o,p,l,s) \
	if (((s) = ObjSendUpdateWarn(m,o,p,l)) < stsOK) return s; else

#define ObjSendUpdateJmp(m,o,p,l,s,x) \
	if (((s) = ObjSendUpdateWarn(m,o,p,l)) < stsOK) goto x; else

#define ObjSendUpdateOK(m,o,p,l,s) ((s = ObjSendUpdateWarn(m,o,p,l)) >= stsOK)


/* ObjectSendTask */

#define ObjSendTaskRet(m,o,p,l,t,s) \
	if (((s) = ObjSendTaskWarn(m,o,p,l,t)) < stsOK) return s; else

#define ObjSendTaskJmp(m,o,p,l,t,s,x) \
	if (((s) = ObjSendTaskWarn(m,o,p,l,t)) < stsOK) goto x; else

#define ObjSendTaskOK(m,o,p,l,t,s) ((s = ObjSendTaskWarn(m,o,p,l,t)) >= stsOK)


/* ObjectSendUpdateTask */

#define ObjSendUpdateTaskRet(m,o,p,l,t,s) \
	if (((s) = ObjSendUpdateTaskWarn(m,o,p,l,t)) < stsOK) return s; else

#define ObjSendUpdateTaskJmp(m,o,p,l,t,s,x) \
	if (((s) = ObjSendUpdateTaskWarn(m,o,p,l,t)) < stsOK) goto x; else

#define ObjSendUpdateTaskOK(m,o,p,l,t,s) \
	((s = ObjSendUpdateTaskWarn(m,o,p,l,t)) >= stsOK)


/* ObjectSendU32 */

#define ObjSendU32Ret(m,o,p,s) \
	if (((s) = ObjSendU32Warn(m,o,p)) < stsOK) return s; else

#define ObjSendU32Jmp(m,o,p,s,x) \
	if (((s) = ObjSendU32Warn(m,o,p)) < stsOK) goto x; else

#define ObjSendU32OK(m,o,p,s) ((s = ObjSendU32Warn(m,o,p)) >= stsOK)


/* ObjectPost */

#define ObjPostRet(m,o,p,l,s) \
	if (((s) = ObjPostWarn(m,o,p,l)) < stsOK) return s; else

#define ObjPostJmp(m,o,p,l,s,x) \
	if (((s) = ObjPostWarn(m,o,p,l)) < stsOK) goto x; else

#define ObjPostOK(m,o,p,l,s) ((s = ObjPostWarn(m,o,p,l)) >= stsOK)


/* ObjectPostAsync */

#define ObjPostAsyncRet(m,o,p,l,s) \
	if (((s) = ObjPostAsyncWarn(m,o,p,l)) < stsOK) return s; else

#define ObjPostAsyncJmp(m,o,p,l,s,x) \
	if (((s) = ObjPostAsyncWarn(m,o,p,l)) < stsOK) goto x; else

#define ObjPostAsyncOK(m,o,p,l,s) ((s = ObjPostAsyncWarn(m,o,p,l)) >= stsOK)


/* ObjectPostDirect */

#define ObjPostDirectRet(m,o,p,l,s) \
	if (((s) = ObjPostDirectWarn(m,o,p,l)) < stsOK) return s; else

#define ObjPostDirectJmp(m,o,p,l,s,x) \
	if (((s) = ObjPostDirectWarn(m,o,p,l)) < stsOK) goto x; else

#define ObjPostDirectOK(m,o,p,l,s) ((s = ObjPostDirectWarn(m,o,p,l)) >= stsOK)


/* ObjectPostU32 */

#define ObjPostU32Ret(m,o,p,s) \
	if (((s) = ObjPostU32Warn(m,o,p)) < stsOK) return s; else

#define ObjPostU32Jmp(m,o,p,s,x) \
	if (((s) = ObjPostU32Warn(m,o,p)) < stsOK) goto x; else

#define ObjPostU32OK(m,o,p,s) ((s = ObjPostU32Warn(m,o,p)) >= stsOK)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				  Debugging Helper Functions (with /DDEBUG)				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#if defined DEBUG || defined CLSMGR_COMPILE

/****************************************************************************
 ObjectCallWarning	returns STATUS
	Same as ObjectCall(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjCallWarn macro should be used to call this routine.
*/
STATUS EXPORTED ObjectCallWarning(
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectCallNoDebugWarning	returns STATUS
	Same as ObjectCallNoDebug(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjCallNoDebugWarn macro should be used to call this routine.
*/
STATUS EXPORTED ObjectCallNoDebugWarning(
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectCallAncestorCtxWarning	returns STATUS
	Same as ObjectCallAncestorCtx(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjCallAncestorCtxWarn macro should be used.
*/
STATUS EXPORTED ObjectCallAncestorCtxWarning(
	CONTEXT		ctx,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectCallAncestorWarning	returns STATUS
	Same as ObjectCallAncestor(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjCallAncestorWarn macro should be used.
*/
STATUS EXPORTED ObjectCallAncestorWarning(
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	CONTEXT		ctx,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectSendWarning	returns STATUS
	Same as ObjectSend(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjectSendWarn macro should be used.
*/
STATUS EXPORTED ObjectSendWarning(
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	SIZEOF		lenArgs,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectSendUpdateWarning	returns STATUS
	Same as ObjectSendUpdate(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjectSendUpdateWarn macro should be used.
*/
STATUS EXPORTED ObjectSendUpdateWarning(
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	SIZEOF		lenArgs,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectSendTaskWarning	returns STATUS
	Same as ObjectSendTask(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjectSendTaskWarn macro should be used.
*/
STATUS EXPORTED ObjectSendTaskWarning(
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	SIZEOF		lenArgs,
	OS_TASK_ID	task,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectSendUpdateTaskWarning	returns STATUS
	Same as ObjectSendUpdateTask(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjectSendUpdateTaskWarn macro should be used.
*/
STATUS EXPORTED ObjectSendUpdateTaskWarning(
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	SIZEOF		lenArgs,
	OS_TASK_ID	task,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectPostWarning	returns STATUS
	Same as ObjectPost(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjectPostWarn macro should be used.
*/
STATUS EXPORTED ObjectPostWarning(
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	SIZEOF		lenArgs,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectPostAsyncWarning	returns STATUS
	Same as ObjectPostAsync(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjectPostAsyncWarn macro should be used.
*/
STATUS EXPORTED ObjectPostAsyncWarning(
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	SIZEOF		lenArgs,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectPostDirectWarning	returns STATUS
	Same as ObjectPostDirect(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjectPostDirectWarn macro should be used.
*/
STATUS EXPORTED ObjectPostDirectWarning(
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	SIZEOF		lenArgs,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectPostTaskWarning	returns STATUS
	Same as ObjectPostTask(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjectPostTaskWarn macro should be used.
*/
STATUS EXPORTED ObjectPostTaskWarning(
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	SIZEOF		lenArgs,
	OS_TASK_ID	task,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectPostAsyncTaskWarning	returns STATUS
	Same as ObjectPostAsyncTask(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjectPostAsyncTaskWarn macro should be used.
*/
STATUS EXPORTED ObjectPostAsyncTaskWarning(
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	SIZEOF		lenArgs,
	OS_TASK_ID	task,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectPostDirectTaskWarning	returns STATUS
	Same as ObjectPostDirectTask(), additionally prints a debugging message
	if status less than stsOK.

 In general, ObjectPostDirectTaskWarn macro should be used.
*/
STATUS EXPORTED ObjectPostDirectTaskWarning(
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	SIZEOF		lenArgs,
	OS_TASK_ID	task,
	P_STRING	fn,
	U16			ln
);

/****************************************************************************
 ObjectWarning returns nothing
	Prints object warning message.	Low-level routine.
*/
void EXPORTED ObjectWarning(
	P_STRING	label,
	MESSAGE		msg,
	OBJECT		object,
	P_ARGS		pArgs,
	STATUS		sts,
	P_STRING	fn,
	U16			ln
);

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Debugging Helper Macros (with /DDEBUG)				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 * Conditional macros.  Under /DDEBUG generates indirect calls via debugging
 * functions, without /DDEBUG generates direct calls.
 *
 * The only difference between the Warn() form and the plain form of these
 * calls is that Warn() prints an error message if sts < stsOK AND the module
 * was compiled for DEBUG.  Use of the Warn() form is strongly encouraged.
*/

/* ObjectCall */

#define ObjCallWarn(m,o,p)	ObjectCallWarning(m,o,p,__FILE__,__LINE__)
#define ObjCallNoDebugWarn(m,o,p) \
	ObjectCallNoDebugWarning(m,o,p,__FILE__,__LINE__)
#define ObjCallAncestorCtxWarn(c) \
	ObjectCallAncestorCtxWarning(c,__FILE__,__LINE__)
#define ObjCallAncestorWarn(m,o,p,c) \
	ObjectCallAncestorWarning(m,o,p,c,__FILE__,__LINE__)


/* ObjectSend */

#define ObjSendWarn(m,o,p,l) ObjectSendWarning(m,o,p,l,__FILE__,__LINE__)
#define ObjSendUpdateWarn(m,o,p,l) \
	ObjectSendUpdateWarning(m,o,p,l,__FILE__,__LINE__)
#define ObjSendTaskWarn(m,o,p,l,t) \
	ObjectSendTaskWarning(m,o,p,l,t,__FILE__,__LINE__)
#define ObjSendUpdateTaskWarn(m,o,p,l,t) \
	ObjectSendUpdateTaskWarning(m,o,p,l,t,__FILE__,__LINE__)
#define ObjSendU32Warn(m,o,p) ObjectSendWarning(m,o,p,0L,__FILE__,__LINE__)


/* ObjectPost */

#define ObjPostWarn(m,o,p,l) ObjectPostWarning(m,o,p,l,__FILE__,__LINE__)
#define ObjPostAsyncWarn(m,o,p,l) \
	ObjectPostAsyncWarning(m,o,p,l,__FILE__,__LINE__)
#define ObjPostDirectWarn(m,o,p,l) \
	ObjectPostDirectWarning(m,o,p,l,__FILE__,__LINE__)
#define ObjPostTaskWarn(m,o,p,l,t) \
	ObjectPostTaskWarning(m,o,p,l,t,__FILE__,__LINE__)
#define ObjPostAsyncTaskWarn(m,o,p,l,t) \
	ObjectPostAsyncTaskWarning(m,o,p,l,t,__FILE__,__LINE__)
#define ObjPostDirectTaskWarn(m,o,p,l,t) \
	ObjectPostDirectTaskWarning(m,o,p,l,t,__FILE__,__LINE__)
#define ObjPostU32Warn(m,o,p) ObjectPostWarning(m,o,p,0L,__FILE__,__LINE__)


#else // DEBUG

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				  Debugging Helper Macros (without /DDEBUG)				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* ObjectCall */

#define ObjCallWarn(m,o,p) ObjectCall(m,o,p)
#define ObjCallNoDebugWarn(m,o,p) ObjectCall(m,o,p)
#define ObjCallAncestorCtxWarn(c) ObjectCallAncestorCtx(c)
#define ObjCallAncestorWarn(m,o,p,c) ObjectCallAncestor(m,o,p,c)

/* ObjectSend */

#define ObjSendWarn(m,o,p,l) ObjectSend(m,o,p,l)
#define ObjSendUpdateWarn(m,o,p,l) ObjectSendUpdate(m,o,p,l)
#define ObjSendTaskWarn(m,o,p,l,t) ObjectSendTask(m,o,p,l,t)
#define ObjSendUpdateTaskWarn(m,o,p,l,t) ObjectSendUpdateTask(m,o,p,l,t)
#define ObjSendU32Warn(m,o,p) ObjectSendU32(m,o,p)


/* ObjectPost */

#define ObjPostWarn(m,o,p,l) ObjectPost(m,o,p,l)
#define ObjPostAsyncWarn(m,o,p,l) ObjectPostAsync(m,o,p,l)
#define ObjPostDirectWarn(m,o,p,l) ObjectPostDirect(m,o,p,l)
#define ObjPostTaskWarn(m,o,p,l,t) ObjectPostTask(m,o,p,l,t)
#define ObjPostAsyncTaskWarn(m,o,p,l,t) ObjectPostAsyncTask(m,o,p,l,t)
#define ObjPostDirectTaskWarn(m,o,p,l,t) ObjectPostDirectTask(m,o,p,l,t)
#define ObjPostU32Warn(m,o,p) ObjectPost(m,o,p,0L)

#endif // DEBUG


//REFGEN BEGINIGNORE
#if defined CLSMGR_PRIVATE

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *			   Debugging Helper Functions used by Mini-Debugger			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef NO_DEBUGGERS

/****************************************************************************
 ObjectToMsgProc	returns P_MEM
	Used by the debuggers() to print stack traces.
*/
P_MEM EXPORTED ObjectToMsgProc(OBJECT object, CONTEXT ctx);

/****************************************************************************
 ObjectWknToDyn	returns OBJECT
	Used by the debuggers() to convert well-known objects to dynamic objects.
*/
OBJECT EXPORTED ObjectWKNToDyn(OBJECT object);

/****************************************************************************
 ObjectTableBase	returns P_OBJECT
	Returns the base of the object table.  Used only for debugging.
*/
P_OBJECT EXPORTED ObjectTableBase(void);

//	Used by DB to single step thru CallMethod(), etc.
extern U32 ClsDBExits[];

#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Private Messages and Functions					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNewInstance	takes P_OBJ_RESTORE, returns STATUS
	Creates an object that can be restored from an object file.

 Sent to a class causing an instance of the class to be created. After the
 new object is created it must be sent msgRestore.

 Return Value
	stsRequestNotSupported:  Instances of clsClass cannot be restored.
*/
#define	msgNewInstance	\
								MakeMsg(clsObject, 114)

/****************************************************************************
 msgDestroyNoVeto	takes OBJ_KEY, returns STATUS
	Destroys the object.

 When msgDestroyNoVeto is sent to the object, clsObject sends msgFreeing
 and msgFree to self.  msgFreePending is sent to the observers.  Only
 clsObject should handle msgDestroyNoVeto.

 Return Value
    stsProtectionViolation:	objCapFree is disabled and the key does not
                            open the object.
    stsClassHasReferences:	(clsClass) Instances of the class object still
                            exists.	 Only returned when the object being
                            destroyed is a class.
*/
#define	msgDestroyNoVeto	\
						MakeMsg(clsObject, 118)

#endif

#endif /* CLSMGR_INCLUDED */
