/****************************************************************************
 File: appwin.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.26  $
   $Author:   pstahl  $
     $Date:   03 Mar 1992 11:46:46  $

 This file contains the API definition for clsAppWin.

 clsAppWin inherits from clsCustomLayout.
 Provides support for embedded applications.

 "AppWin" stands for Application Window.
****************************************************************************/

/**** Introduction ****/
/*
 clsAppWin is an embedded window that manages an embedded document.
 It shrink-wraps around a clsIcon object to display an icon to the user,
 like those on the bookshelf or embedded in a document. When an icon with
 style awOpenInPlace is tapped, the application window destroys the icon
 and opens the associated document into itself.  The application window
 then shrink-wraps around the document's main window.

 Application Windows live in the process space and are filed with the
 embeddor document.

 An application window reads its icon bitmap from metrics.resList of 
 OSThisApp() in response to msgIconProvideBitmap (see icon.h).
 It uses the following resID (see apptag.h):

//{
	MakeWknResIdX(read.resId, appResId, tagAppIconBitmap);
//}

 This bitmap is usually found in the app.res file of the application class
 for the associated document. The document can override this bitmap
 by filing a resource with the above resId into its doc.res file.

 AppWins can also store their own private bitmaps.  Use
 msgAppWinSetIconBitmap to give an application window a bitmap.  This
 bitmap object will be filed by the application window.  If an application
 window has its own bitmap object, it will not read from the resList.
*/

#ifndef APPWIN_INCLUDED
#define APPWIN_INCLUDED

#ifndef CLAYOUT_INCLUDED
#include <clayout.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT APP_WIN, *P_APP_WIN;

/****  Application Window States  ****/
/*
 * These are the valid states for an application window.
 */
#define awClosed					0
#define awOpenedFloating			1
#define awOpenedInPlace				2
#define awOpenedInPlaceFloating		3

/****  Application Window Open Styles  ****/
/*
 * These are the valid styles for directing an application window
 * how to open.
 */
#define awOpenInPlace				0
#define awOpenFloating				1

/****  Application Window Icon Types  ****/
/*
 * These are the valid icon types.
 */
#define awPictAndTitle				0
#define awPictOnly					1
#define awSmallPictAndTitle			2
#define awSmallPictOnly				3
#define awSmallPictOverTitle		4

/****  Application Window Style Structure ****/
/*
 * This structure defines the various application window styles.
 */
typedef struct APP_WIN_STYLE {
	U16	open		 	: 2;	// Open style.
	U16	type			: 4;	// Icon type.
	U16 openStyleLock	: 1;	// True = cannot change open style.
	U16 private1		: 1;    // Reserved.
	U16 private2		: 1;    // Reserved.
	U16	reserved		: 7;	// Reserved.
} APP_WIN_STYLE, *P_APP_WIN_STYLE;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew     takes P_APP_WIN_NEW, returns STATUS
    category: class message
	Creates a new Application Window.
*/
typedef struct APP_WIN_NEW_ONLY {
	UUID			appUUID;				// App uuid.
	APP_WIN_STYLE	style;					// Application Window style.
	U16				state;					// Application Window state.	
	char			label[nameBufLength];	// Icon label.
	U32				reserved[4];            // Reserved.
} APP_WIN_NEW_ONLY, *P_APP_WIN_NEW_ONLY;

#define	appWinNewFields			\
	customLayoutNewFields		\
	APP_WIN_NEW_ONLY			appWin;

typedef struct APP_WIN_NEW {
	appWinNewFields
} APP_WIN_NEW, *P_APP_WIN_NEW;

/****************************************************************************
 msgNewDefaults     takes P_APP_WIN_NEW, returns STATUS
    category: class message
	Initializes the APP_WIN_NEW structure to default values.

 Zeroes out pArgs->appWin and sets
//{
	pArgs->win.flags.style	           |= wsCaptureGeometry 
                                        | wsSendGeometry 
                                        | wsShrinkWrapWidth  
                                        | wsShrinkWrapHeight;
	pArgs->win.flags.input	           |= inputHoldTimeout;
	pArgs->embeddedWin.style.embeddee	= true;
	pArgs->embeddedWin.style.moveable	= true;
	pArgs->embeddedWin.style.copyable	= true;
	pArgs->border.style.previewAlter	= bsAlterNone;
	pArgs->border.style.selectedAlter	= bsAlterNone;
	pArgs->appWin.style.open		    = awOpenInPlace;
	pArgs->appWin.style.type		    = awSmallPictAndTitle;
	pArgs->appWin.state				    = awClosed;
//}
*/

/****************************************************************************
 msgAppWinGetMetrics takes P_APP_WIN_METRICS, returns STATUS
	Passes back an application window's metrics.
*/
#define msgAppWinGetMetrics				MakeMsg(clsAppWin, 1)

typedef struct APP_WIN_METRICS {

	UUID			appUUID;				// Application uuid.
	OBJECT			icon;					// Application Window icon.
	OBJECT			iconBitmap;				// Icon bitmap.
	OBJECT			smallIconBitmap;		// Small icon bitmap.
	OBJECT			appClass;				// Application class.
	APP_WIN_STYLE	style;					// Application Window style.
	U16				state;					// Application Window state.
	char			label[nameBufLength];	// Icon label.
	U32				reserved[4];			// Reserved.
	
} APP_WIN_METRICS, *P_APP_WIN_METRICS;

/****************************************************************************
 msgAppWinGetState	takes P_U16, returns STATUS
	Passes back an application window's state.

 Possible values are described in Application Window States, above.
*/
#define msgAppWinGetState				MakeMsg(clsAppWin, 2)

/****************************************************************************
 msgAppWinSetState	takes U16, returns STATUS
	Specifies an application window's state.

 Possible values are described in Application Window States, above.
*/
#define msgAppWinSetState				MakeMsg(clsAppWin, 3)

/****************************************************************************
 msgAppWinGetStyle	takes P_APP_WIN_STYLE, returns STATUS
	Passes back an application window's style.
*/
#define msgAppWinGetStyle				MakeMsg(clsAppWin, 4)

/****************************************************************************
 msgAppWinSetStyle	takes APP_WIN_STYLE, returns STATUS
	Specifies an application window's style.
*/
#define msgAppWinSetStyle				MakeMsg(clsAppWin, 5)

/****************************************************************************
 msgAppWinSetLabel	takes P_STRING, returns STATUS
	Specifies an application window's label.
*/
#define msgAppWinSetLabel 				MakeMsg(clsAppWin, 6)

/****************************************************************************
 msgAppWinSetIconBitmap	takes BITMAP, returns STATUS
	Specifies an application window's large icon bitmap.
*/
#define msgAppWinSetIconBitmap 			MakeMsg(clsAppWin, 7)

/****************************************************************************
 msgAppWinSetSmallIconBitmap	takes BITMAP, returns STATUS
	Specifies an application window's small icon bitmap.
*/
#define msgAppWinSetSmallIconBitmap		MakeMsg(clsAppWin, 8)

/****************************************************************************
 msgAppWinOpen	takes nothing, returns STATUS
	Opens the document associated with an application window.
*/
#define msgAppWinOpen	 				MakeMsg(clsAppWin, 9)

/****************************************************************************
 msgAppWinClose	takes nothing, returns STATUS
	Closes the document associated with an application window.
*/
#define msgAppWinClose	 				MakeMsg(clsAppWin, 10)

/****************************************************************************
 msgAppWinDelete	takes BOOLEAN, returns STATUS
	Deletes an application window.  
	
 If pArgs is true, msgAppWinDelete also deletes the associated document.
 If pArgs is false, msgAppWinDelete does not delete the document.
*/
#define msgAppWinDelete	 				MakeMsg(clsAppWin, 11)

/****************************************************************************
 msgAppWinSetUUID	takes P_UUID, returns STATUS
	Specifies the UUID of the document to which an application window 
    is linked.
*/
#define msgAppWinSetUUID				MakeMsg(clsAppWin, 12)

/****************************************************************************
 msgAppWinCreateIcon	takes P_UUID, returns STATUS
	Creates an application window's icon.
*/
#define msgAppWinCreateIcon				MakeMsg(clsAppWin, 13)

/****************************************************************************
 msgAppWinDestroyIcon	takes P_UUID, returns STATUS
	Destroys an application window's icon.
*/
#define msgAppWinDestroyIcon			MakeMsg(clsAppWin, 14)

/****************************************************************************
 msgAppWinStyleChanged	takes OBJECT, returns STATUS
	Notification that an application window style changed.

 Application windows send this message to their observers whenever they
 receive msgAppWinSetStyle.  Note that application icon option cards
 will send msgAppWinSetStyle to application windows whenever they cause the
 application window's icon style to change.
*/
#define msgAppWinStyleChanged			MakeMsg(clsAppWin, 15)

/****************************************************************************
 msgAppWinEditName	takes nothing, returns STATUS
	Pops up an edit pad to allow the user to rename the document
    associated with an application window.
*/
#define msgAppWinEditName				MakeMsg(clsAppWin, 16)

#endif	// APPWIN_INCLUDED
