/****************************************************************************
 File: appmon.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.31  $
   $Author:   sisaac  $
     $Date:   04 Mar 1992 13:49:10  $

 This file contains the API definition for clsAppMonitor.

 clsAppMonitor inherits from clsApp. 
 Provides the standard behavior for an application's monitor object.

 You create an application monitor when you call AppMonitorMain from
 your main routine, when processCount is zero. An application monitor
 drives application installation and helps with deinstallation. It also
 controls displaying global application options, maintaining global state,
 and importing files. 

 You should subclass clsAppMonitor if your application needs to do a more 
 sophisticated installation (such as installing shared dictionaries or data
 files), to support file import, to set and save global application 
 configurations, and to provide file converters. See the section below on 
 Subclassing.
****************************************************************************/

/****  clsAppMonitor's Lifecycle  ****/
/*
 Every application has a single instance of its application
 monitor class alive as long as the application is installed. The
 app monitor object is owned by the application's processCount 0 process. 
 Clients can get the uid of the app monitor object by sending 
 msgAppMgrGetMetrics to an application's class.

 clsAppMonitor is a descendant of clsApp. It makes use of the standard
 Application Framework lifecycle messages to perform some of its functions:

	msgAppInit:			Install the application.
	msgAppRestore:		Reinitialize the application after a warm-boot.
	msgAppOpenTo:		Display global application option sheet.
	msgAppCloseTo:		Take down global application option sheet.

 Note: msgAppTerminate must *never* be sent directly to the app monitor.
 Use msgAMTerminateOK and msgAMTerminate instead.
*/

/****  Application Installation  ****/
/*
 Application installation is performed as follows:

	1.:	Somebody sends msgIMInstall to theInstalledApps. theInstalledApps
		creates an application directory in the selected volume under 
		\penpoint\sys\app, copies the application's resource file into the 
		application directory, and installs the application's code. 
		See appimgr.h for details.

	2.:	When the code is installed, the operating system creates
		the application's first process (processCount = 0) and begins 
		execution of the main() routine. The application installs its 
		classes and calls AppMonitorMain(). AppMonitorMain never returns; it
		creates the	app monitor object and goes into an object dispatch 
		loop (see clsmgr.h).

	3.:	The Application Framework sends msgAppInit to the app monitor.
		This initiates the app monitor's installation sequence. 

	4.:	The app monitor self sends msgAMLoadInitDll. This causes an
		optional initialization .dll to be run and then be unloaded.

	5.: The app monitor self sends msgAMPopupOptions. If a descendant
		wants to pop up the app monitor global option sheet, it must 
		handle this message and set pArgs to true, then pass it on to
		its ancestor. This will cause the option sheet protocol 
		(msgOptionAddCards, etc) to be sent to the app monitor. 

	6.:	The app monitor self sends msgAMLoadMisc. This causes any files
		that the application has in the MISC directory to be
		copied into the app directory in the selected volume.

	7.:	The app monitor self sends msgLoadAuxNotebooks, which causes 
		msgLoadStationery and msgLoadHelp to be sent to self. 
		msgLoadStationery causes all the stationery and accessory  
		templates that do not have an anmAttrNoLoad attribute on them to be 
		loaded into the machine. Stationery is stored in the STATNRY 
		directory; Accessories are stored in the ACESSRY directory.
		msgLoadHelp causes all Help Notebook documents and templates that
		do not have the anmAttrNoLoad attribute set to be loaded into the
		Help Notebook.

	8.:	The app monitor self sends msgAMLoadFormatConverters and 
		msgAMLoadOptionalDlls. These messages are currently not
		implemented by clsAppMonitor; descendants can deal with them if 
		desired. There might be default superclass behavior in the future.
*/

/****  Stationery, Accessory, and Help Documents  ****/
/*
 Stationery and Accessory documents can either be saved document 
 instances (typically copied out to a distribution disk with the Connections
 Notebook), or plain directories containing files that the application 
 knows about. 

 Help documents can be directories containing ASCII or RTF files, or 
 PenPoint documents.

 These items must be located in the application's installation directory
 in subdirectories called STATNRY, ACCESSRY, and HELP.
*/

/****  Subclassing clsAppMonitor  ****/
/*
 The app monitor is an excellent place to add global application control
 and syncronization functions, since it is always around and easily 
 accessable. For instance, if an application wants its documents
 to access some application-specific shared data (such as a list of 
 worldwide telephone country codes), the app monitor for the application 
 could manage this data and provide an API to access it.

 Applications can have a global application option sheet automatically
 displayed when the application is installed by handling msgAMPopupOptions. 
 A special resource is written into the application's resource file after 
 this occurs, inhibiting subsequent popups if the resource file is copied to
 the application's installation directory. clsAppMon does not provide any 
 default cards; you must provide at least one if you handle 
 msgAMPopupOptions. 

 If you display a user interface from your app monitor you will probably
 have to turn on the fullEnvironment app manager flag when you create
 your main application class. If this flag is false then the app 
 monitor will run in a stripped down process environment. This saves a
 substantial amount of memory, but does not have process-local resources
 such as theProcessResList.

 If you subclass clsAppMonitor, you must specify your descendant's class
 name when you call AppMonitorMain. The first parameter to this routine
 is the global well-known name of your application class. The second is	the
 global well-known name of your descendant of clsAppMonitor. If you do not
 subclass clsAppMonitor, pass objNull for the second parameter. The
 AppMonitorMain routine will know to create a default application monitor.
*/
#ifndef APPMON_INCLUDED
#define APPMON_INCLUDED

#ifndef FS_INCLUDED
#include <fs.h>
#endif

#ifndef OPTION_INCLUDED
#include <option.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  		Common #defines and typedefs  					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* 
 This attribute represents the last modified date that a piece of stationery 
 had when it was installed on the machine. 
*/
#define amAttrDateTimeLoaded	  		FSMakeFix32Attr(clsAppMonitor, 2)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *     	   		       Application Framework Messages			  		   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgAppInit     		takes DIR_HANDLE, returns STATUS
	Installs the application.

 This message is sent once and only once by the system, when the application
 is first installed from disk. 

 The app monitor initializes its instance data, runs the installation 
 protocol (msgAMLoadInitDLL, msgAMLoadStationery, msgAMLoadMisc, etc), 
 adds this application to system lists, and signals the installation 
 process to continue running. 

 Descendants: You can handle this message to perform any first-time 
 initialization. The ancestor must be called before your handler.
*/


/****************************************************************************
 msgAppRestore	   		takes nothing, returns STATUS
	Reinitializes the application after a warm-boot.

 This message is sent by the system when a warm-boot occurs. 
 The app monitor initializes its instance data and signals the 
 system warm-boot process to proceed.

 Descendants: You can handle this message and perform any first-time 
 initialization. The ancestor must be called before your handler.
*/


/****************************************************************************
 msgAppOpen		takes P_APP_OPEN, returns STATUS
	Displays the global configuration option sheet.

 This message is self-sent by msgAMPopupOptions. It can also be sent 
 by anyone else.

 The app monitor displays the application configuration option sheet 
 (tagAppDocOptSheet).

 Descendants: You normally do not handle this message. To provide an
 option sheet, see msgAMPopupOptions.
*/


/****************************************************************************
 msgAppClose	takes nothing, returns STATUS
	Removes the global configuration option sheet.

 This message is self-sent by msgAMPopupOptions. It can also be sent 
 by anyone else.

 Descendants: You normally do not handle this message.
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *            	   	  			 Import Messages		  				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgImportQuery     takes P_IMPORT_QUERY, returns STATUS
	Determines if a file can be imported by the application.

 The app monitor forwards msgImportQuery to its class as a class message. 
 If it isn't handled there, the app monitor sends back "No" to all import
 requests. In the future there will be support to run 
 through any of the file translators that the application has loaded.

 Descendants: You normally do not handle this message.

 See Also
	import.h
*/


/****************************************************************************
 msgImport	     takes P_IMPORT_DOC, returns STATUS
	Imports a file.

 The app monitor first creates a new document object and activates it. It
 then forwards msgImport to the document. Next, it sends msgAppMgrShutdown
 to both save the document and shut it down.

 Descendants: You normally do not handle this message.

 See Also
	import.h
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *            			       App Monitor Messages		  				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgAMGetMetrics			takes P_AM_METRICS, returns STATUS
	Gets the app monitor's metrics. 

 Descendants: You normally do not handle this message.
*/
#define msgAMGetMetrics		 				MakeMsg(clsAppMonitor, 1)

typedef struct AM_METRICS {
	CLASS				appClass;  // Main application class.
	OBJECT				handle;	   // This app's handle in theInstalledApps.
	U32					unused2;
	U32					unused3;
	U32					unused4;
	U16					unused; 
} AM_METRICS, *P_AM_METRICS;


/****************************************************************************
 msgAMGetInstallDir			takes P_OBJECT, returns STATUS
	Creates a directory handle on the application's installation directory. 

 The app monitor creates a clsDirHandle object which references the 
 location on external media that the application was installed from.
 If the external volume is not connected, the user is asked to attach it. 

 If this application was bundled with PenPoint then there is no valid
 external volume beyond installation time. stsFailed is returned in
 this case.

 NOTE: CALLER IS RESPONSIBLE FOR DESTROYING THE DIR HANDLE WHEN DONE.

 Return Value
	stsOK					The external volume is attached.
	stsFSVolDisconnected	The user tapped the Cancel button when prompted
							 to attach the external volume.
	stsFailed				The external volume cannot be determined because
							 this application was bundled with PenPoint.

 Descendants: You normally do not handle this message.
*/
#define msgAMGetInstallDir	 					MakeMsg(clsAppMonitor, 2)


/****************************************************************************
 msgAMLoadInitDll			takes OBJECT, returns STATUS
	Loads, runs, and unloads an optional dll initialization routine.

 The app monitor looks for an init.dll file in the application's
 directory (which is specified in pArgs). If it is found, the 
 DllMain routine for this dll is run. The dll is then unloaded.

 Descendants: You normally do not handle this message.

 Return Value
	stsOK	Either the dll initialization was not found or it was found and 
			run successfully.
*/
#define msgAMLoadInitDll					MakeMsg(clsAppMonitor, 4)


/****************************************************************************
 msgAMLoadMisc			takes nothing, returns STATUS
	Load the application's miscellaneous files.

 If a directory called MISC exists, the app monitor copies this directory
 into the in-memory application directory.

 Descendants: You normally do not handle this message. However, you can
 create the MISC directory and place in it files that all of your documents
 need to use. For example, your documents may need to reference a file 
 that contains all of the postal/zip codes for a country.

 Return Value
	stsOK					Either the MISC directory was not found or was 
							found and copied successfully.

*/
#define msgAMLoadMisc						MakeMsg(clsAppMonitor, 5)


/****************************************************************************
 msgAMLoadStationery			takes nothing, returns STATUS
	Loads stationery and accessory templates.

 The app monitor looks for stationery in a directory named STATNRY and
 accessories in a directory named ACCESSRY in the app's directory. 
 It copies any templates that are not marked with the noLoad attribute 
 from these directories to the Stationery and Accessories notebooks.

 A template is a subdirectory with a either a complete, saved document or 
 any kind of file that the application can read.

 If appMgrMetrics.flags.stationery is true, the app monitor creates a 
 default piece of stationery (an empty document of its application type).
 Similarly, if appMgrMetrics.flags.accessory is true, the app monitor
 places an empty document in the Accessories notebook.

 Descendants: You normally do not handle this message.
*/
#define msgAMLoadStationery					MakeMsg(clsAppMonitor, 6)


/****************************************************************************
 msgAMRemoveStationery			takes nothing, returns STATUS
	Removes all the stationery and accessory templates for this application.

 The app monitor removes the stationery notebook section for this application,
 which removes the stationery loaded in msgAMLoadStationery and any user-
 defined stationery. It then removes all of this application's documents
 from the Accessories notebook (thereby removing templates loaded in
 msgAMLoadStationery and any documents that the user placed there).

 Descendants: You normally do not handle this message.
*/
#define msgAMRemoveStationery				MakeMsg(clsAppMonitor, 7)


/****************************************************************************
 msgAMLoadHelp					takes nothing, returns STATUS
	Loads the application's help into the Help Notebook.
 
 The app monitor looks for a HELP subdirectory in the application's 
 directory. If HELP exists, the app monitor copies all of the help 
 templates that are not marked with the noLoad attribute to the
 Help Notebook. Help templates can be directories with ASCII, RTF or
 saved MiniText documents in them.

 Descendants: You normally do not handle this message.
*/
#define msgAMLoadHelp						MakeMsg(clsAppMonitor, 8)


/****************************************************************************
 msgAMRemoveHelp				takes nothing, returns STATUS
	Removes all Help Notebook items for this application.

 The app monitor removes all of this application's items from the Help 
 Notebook.

 Descendants: You normally do not handle this message.
*/
#define msgAMRemoveHelp						MakeMsg(clsAppMonitor, 9)


/****************************************************************************
 msgAMPopupOptions					takes P_BOOLEAN, returns STATUS
	Pops up a global option sheet the first time the app is installed.

 If pArgs is false, the app monitor does not do anything. If it is true,
 the app monitor pops up the global option sheet, then writes a resource
 in the application's resource file which inhibits subsequent popups. 

 Descendants: If you want to allow the user to configure (or check the 
 configuration of) the application as it is being installed, you need
 to handle this message. In your handler, you should set pArgs to true
 and then call the ancestor. You also need to create an option sheet 
 resource with a tag of tagAppDocOptSheet (in your application's 
 msgAppAddCards handler).

 You can have the option sheet to always pop up (even after the first time
 the user installs the application) by not calling the ancestor and 
 popping up the option sheet yourself with:

    ObjCallRet(msgAppOpenTo, self, (P_ARGS) appOpenToFloating, s);

*/
#define msgAMPopupOptions				MakeMsg(clsAppMonitor, 17)


/****************************************************************************
 msgAMLoadAuxNotebooks			takes nothing, returns STATUS
	Loads items into auxilliary notebooks.

 The app monitor self sends msgAMLoadStationery and msgAmLoadHelp to 
 load the application's stationery, accessory, and help templates.

 Descendants: You normally do not handle this message.
*/
#define msgAMLoadAuxNotebooks				MakeMsg(clsAppMonitor, 14)


/****************************************************************************
 msgAMLoadFormatConverters		takes nothing, returns STATUS
	Loads file format converter .dlls.

 Currently, the app monitor does not do anything in response to this message.
 It will do something in the future.

 Descendants: You normally do not handle this message.
*/
#define msgAMLoadFormatConverters			MakeMsg(clsAppMonitor, 10)


/****************************************************************************
 msgAMUnloadFormatConverters		takes nothing, returns STATUS
	Unloads file format converter .dlls.

 Currently, the app monitor does not do anything in response to this message.
 It will do something in the future.

 Descendants: You normally do not handle this message.
*/
#define msgAMUnloadFormatConverters			MakeMsg(clsAppMonitor, 11)


/****************************************************************************
 msgAMLoadOptionalDlls		takes nothing, returns STATUS
	Loads an application's optional .dlls.

 Currently, the app monitor does not do anything in response to this message.
 It will do something in the future.

 Descendants: You normally do not handle this message.
*/
#define msgAMLoadOptionalDlls				MakeMsg(clsAppMonitor, 12)


/****************************************************************************
 msgAMUnloadOptionalDlls		takes nothing, returns STATUS
	Unloads an application's optional .dlls.

 Currently, the app monitor does not do anything in response to this message.
 It will do something in the future.

 Descendants: You normally do not handle this message.
*/
#define msgAMUnloadOptionalDlls				MakeMsg(clsAppMonitor, 13)


/****************************************************************************
 msgAMTerminateOK	takes P_OBJECT, returns STATUS
	Asks if this application is willing to terminate.

 Deinstallation is a two phase process. All applications and services
 that are to be deinstalled together get the chance to veto. This message is
 sent to an application monitor to see if it wishes to veto.

 By default, the app monitor unconditionally terminates all of its 
 application's instances. To do so, it sends msgAppMgrShutdown to its 
 application class for each of its active documents. 

 Descendants: If you want to be given the chance to terminate the application,
 you should handle this message. In your handler, if you decide
 that you want to terminate, you simply pass the message on to your 
 ancestor. 

 You can veto the termination by returning anything other than stsOK and by
 not passing the message on to your ancestor. If you veto, you must set 
 pArgs to the uid of the object that was responsible for the veto, which
 is typically self.

 See Also
	msgAMTerminate
	msgAMTerminateVetoed
*/
#define msgAMTerminateOK			 		MakeMsg(clsAppMonitor, 20)


/****************************************************************************
 msgAMTerminate     			takes nothing, returns STATUS
	Terminates this application.

 Deinstallation is a two phase process. All applications and services
 that are to deinstalled together get the chance to veto. This message is
 sent to an application monitor after everyone has agreed to the 
 deinstallation.

 This message unconditionally terminates the application in the final 
 phase of deinstallation. The app monitor self sends msgAMRemoveStationery
 and msgAMRemoveHelp, and then calls OSTaskTerminate to kill the 
 application's processCount 0 task.

 Descendants: You should handle this message to remove anything you have
 loaded. The ancestor must be called after your handler.

 See Also
	msgAMTerminateOK
	msgAMTerminateVetoed
*/
#define msgAMTerminate				 		MakeMsg(clsAppMonitor, 21)


/****************************************************************************
 msgAMTerminateVetoed		takes P_AM_TERMINATE_VETOED, returns STATUS
	Sent when the application termination sequence is vetoed.

 When one of the applications or services that are deinstalled together
 vetoes termination, the Application Framework sends this message to
 those applications and services.

 pArgs->vetoer gives the uid of the object or class that vetoed the
 deinstallation. pArgs->status gives the return status of the veto.
 The app monitor does not do anything in response to this message.

 Descendants: You can handle this message if you wish. If you handled
 msgTerminateOK, and changed anything because you thought you were 
 about to be terminated, you should handle this message to change things
 back to the way they were.

 See Also
	msgAMTerminateOK
	msgAMTerminate 
*/
#define msgAMTerminateVetoed		 		MakeMsg(clsAppMonitor, 22)

typedef struct AM_TERMINATE_VETOED {
	OBJECT	   	vetoer;		// Object or class that vetoed the deinstallation.
	STATUS	   	status;		// Veto status.
} AM_TERMINATE_VETOED, *P_AM_TERMINATE_VETOED;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *      	      	   			     Tags								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
#define tagAMFirstTime						MakeTag(clsAppMonitor, 2) 

#endif	// APPMON_INCLUDED

