/****************************************************************************
 $Workfile:   PIXWIN.C  $

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 You may use this Sample Code any way you please provided you 
 do not resell the code and that this notice (including the above 
 copyright notice) is reproduced on all copies.  THIS SAMPLE CODE 
 IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, AND GO CORPORATION 
 EXPRESSLY DISCLAIMS ALL IMPLIED WARRANTIES, INCLUDING BUT NOT 
 LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE. IN NO EVENT WILL GO CORPORATION BE LIABLE TO YOU 
 FOR ANY CONSEQUENTIAL,INCIDENTAL,OR INDIRECT DAMAGES ARISING OUT OF 
 THE USE OR INABILITY TO USE THIS SAMPLE CODE.

 $Revision:   1.4  $
   $Author:   kcatlin  $
     $Date:   18 Mar 1992 09:07:36  $

 This file contains the class definition and methods for clsPixWin.
****************************************************************************/

#include <go.h>
#include <string.h>
#include <stdio.h>
#include <input.h>		// to set win.flags.input = inputTransparent
#include <os.h>			// for memory allocs
#include <fs.h>			// for read/write
#include <debug.h>

#include "cutil.h"
#include "pixwin.h"
#include "pixelmap.h"
#include "methods.h"

/****************************************************************************/
/*					D E F I N E S   A N D   T Y P E D E F S					*/
/****************************************************************************/

typedef struct INSTANCE_DATA
{
	// filed data
	P_STRING	pFileName;		// name of the source file

	// state data
	OBJECT		pixelMap;		// Pixelmap to hold image
	OBJECT		file;			// memory mapped file that holds pixmap
	PP_UNKNOWN	pPlanes;		// pointer to array of planes
	U16			planeCount;		// depth of pixelmap

	// these are not really needed as state variables
	PIX_DEV_ORIENT	orient;		// orientation of the pixelMap
	U32			width, height;	// desired width and height
	U32			planeSize;		// bytes in a plane of pixelmap
} INSTANCE_DATA, *P_INSTANCE_DATA;

typedef struct PIXELMAP_DATA
{
	U32			version;		// filed version number
	U32			width, height;	// desired width and height
	U16			planeCount;		// depth of pixelmap
	SIZEOF		planeSize;		// size of one plane
	PIX_DEV_ORIENT	orient;		// orientation of the pixelMap
} PIXELMAP_DATA, *P_PIXELMAP_DATA;

#define CURRENT_VERSION 1
#define CURRENT_PIXELMAP_VERSION 1

/****************************************************************************/
/*					F O R W A R D   D E C L A R A T I O N S					*/
/****************************************************************************/

/****************************************************************************/
/*			L O C A L   F U N C T I O N S   &   M E T H O D S			    */
/****************************************************************************/

/****************************************************************************
	SetupUpPlanes
****************************************************************************/
STATUS SetUpPlanes(
	OBJECT file,
	P_INSTANCE_DATA pData)
{
	P_U8 pMem;
	U32 i;

	// allocate memory for pPlanes
	StsWarn(OSHeapBlockAlloc(osProcessSharedHeapId, \
		pData->planeCount * sizeof(P_UNKNOWN), &pData->pPlanes));

	ObjCallWarn(msgFSMemoryMap, file, &pMem);
	for(i = 0; i < pData->planeCount; i++)
	  {
		pData->pPlanes[i] = \
			pMem + sizeof(PIXELMAP_DATA) + (i * pData->planeSize);
//Debugf("pMem[%p] plane[%d]=[%p]", pMem, (int)i, pData->pPlanes[i]);
	  }
	return stsOK;
}

/****************************************************************************
	CreatePixelmapFile
		In:  pData-> width, height, depth, orient, pFileName
****************************************************************************/
STATUS CreatePixelmapFile(
	OBJECT self,
	P_INSTANCE_DATA pData)
{
	STATUS	s;
	P_PIXELMAP_DATA pPixelmapData;
	WIN_DEV_PIXELMAP pm;
	OBJECT file;
	PIX_DEV_ORIENT	orient;

//Debugf("Creating pixelmap file '%s'", pData->pFileName);
	pm.size.w = pData->width;
	pm.size.h = pData->height;
	pm.planeCount = pData->planeCount;

	// set planeCount to agree with hardware
	{
		WIN_METRICS wm;
		PIX_DEV_METRICS pdMetrics;
		ObjCallWarn(msgWinGetMetrics, self, &wm);
		ObjCallWarn(msgPixDevGetMetrics, wm.device, &pdMetrics);
		if(pm.planeCount != 1)
			pm.planeCount = pdMetrics.planeNormalCount;
		orient = pdMetrics.orient;
//Debugf("Set pixwin's planeCount to %d", (int)pm.planeCount);
	}

	// determine file size
	SizePixelMap(self, &pm);

	// create/size file
	{
	FS_NEW new;
	FS_SET_SIZE fsSetSize;
	P_MEM	pMem;
	ObjCallRet(msgNewDefaults, clsFileHandle, &new, s);
	new.fs.locator.pPath = pData->pFileName;
	new.fs.mode = fsSharedMemoryMap;
	ObjCallRet(msgNew, clsFileHandle, &new, s);
	file = new.object.uid;
	fsSetSize.newSize = \
		sizeof(PIXELMAP_DATA) + (pm.planeCount * pm.planeSize);
	ObjCallRet(msgFSSetSize, file, &fsSetSize, s);

	ObjCallWarn(msgFSMemoryMap, file, &pMem);
//Debugf("CreatePixelMap file of size %ld:  %ld + (%ld * %ld)", (long)fsSetSize.newSize,(long)sizeof(PIXELMAP_DATA), (long)pm.planeCount, (long)pm.planeSize);
	memset(pMem, (int)0xFF, (size_t)fsSetSize.newSize);
	}

	// set up file header
	ObjCallWarn(msgFSMemoryMap, file, &pPixelmapData);
	pPixelmapData->version = CURRENT_PIXELMAP_VERSION;
	pPixelmapData->orient = orient;
	pPixelmapData->width = pData->width;
	pPixelmapData->height = pData->height;
	pPixelmapData->planeCount = pm.planeCount;
	pPixelmapData->planeSize = pm.planeSize;

	ObjCallRet(msgDestroy, file, pNull, s);
	return stsOK;
	//return SetUpPlanes(file, pData);
}
/****************************************************************************
	ReadPixelmapData
****************************************************************************/
STATUS ReadPixelmapData(
	OBJECT file,
	P_INSTANCE_DATA pData)
{
	STATUS	s;
	STREAM_READ_WRITE stream;
	PIXELMAP_DATA pixelmapData;

	stream.numBytes = sizeof(pixelmapData);
	stream.pBuf = &pixelmapData;
	s = ObjectCall(msgStreamRead, file, &stream);
	if(stream.count != stream.numBytes)
		return s;

	if(pixelmapData.version != CURRENT_PIXELMAP_VERSION)
		return stsIncompatibleVersion;
	if(pixelmapData.orient != pData->orient)
	  {
	  	Dbg(Debugf("The filed pixwin is orientaion %d, we need %d", \
			(int)pixelmapData.orient, (int)pData->orient);)
		return stsIncompatibleVersion;
	  }
	pData->width = pixelmapData.width;
	pData->height = pixelmapData.height;
	pData->planeCount = pixelmapData.planeCount;
	pData->planeSize = pixelmapData.planeSize;

	return SetUpPlanes(file, pData);
}
/****************************************************************************
	PixWinSetup
	Called during msgRestore and msgInit
****************************************************************************/
static STATUS PixWinSetup(
	OBJECT self,
	P_INSTANCE_DATA pData)
{
	STATUS s;
	WIN_DEV_PIXELMAP pm;
	FS_NEW fNew;

	ObjCallRet(msgNewDefaults, clsFileHandle, &fNew, s);
	fNew.fs.locator.pPath = pData->pFileName;
	//fNew.fs.mode = fsReadOnly | fsSharedMemoryMap | fsDenyWriters;
	fNew.fs.mode = fsSharedMemoryMap;
	fNew.fs.exist = fsExistOpen | fsNoExistGenError;
	if(ObjCallChk(msgNew, clsFileHandle, &fNew, s))
		return s;

	// get self's orientation
	{
	WIN_METRICS wm;
	PIX_DEV_METRICS pm;
	ObjCallWarn(msgWinGetMetrics, self, &wm);
	ObjCallWarn(msgPixDevGetMetrics, wm.device, &pm);
	pData->orient = pm.orient;
//Debugf("PixWin's device's orientation is %d", (int)pData->orient);
	}

	if(StsChk(ReadPixelmapData(fNew.object.uid, pData), s))
		return s;

	//ObjCallWarn(msgDestroy, fNew.object.uid, pNull);
	pm.size.w = pData->width;
	pm.size.h = pData->height;
	pm.planeCount = pData->planeCount;
	pm.pPlanes = pData->pPlanes;
	pData->pixelMap = CreatePixelMap(self, &pm);
//Debugf("PixWin:  created pixelMap[%p]", pData->pixelMap);
	pData->file = fNew.object.uid;
	return stsOK;
}

/****************************************************************************
	msgNewDefaults
****************************************************************************/
MsgHandlerWithTypes(PixWinNewDefaults, P_PIXWIN_NEW, P_UNKNOWN)
{
	STATUS s;
	ObjCallAncestorRet(msgNewDefaults, self, pArgs, ctx, s);
	pArgs->pixwin.pName = 0;
	pArgs->pixwin.size.w = 0;
	pArgs->pixwin.size.h = 0;
	pArgs->pixwin.planeCount = 0;
	pArgs->pixwin.bCreate = false;
	pArgs->win.flags.input = inputTransparent;
	pArgs->win.flags.style |= wsShrinkWrapWidth;
	pArgs->win.flags.style |= wsShrinkWrapHeight;
	return stsOK;
	MsgHandlerParametersNoWarning;
}
/****************************************************************************
	msgInit.
	Initialize pixWin
****************************************************************************/
MsgHandlerWithTypes(PixWinInit, P_PIXWIN_NEW, P_INSTANCE_DATA)
{
	STATUS				s;
	INSTANCE_DATA inst;

	//  pass message to ancestor
	ObjCallAncestorRet(msgInit, self, pArgs, ctx, s);

	// clear out the instance data
	memset(&inst, 0, sizeof(INSTANCE_DATA));

	inst.pFileName = CreateStringCopy(pArgs->pixwin.pName);
	inst.width = pArgs->pixwin.size.w;
	inst.height = pArgs->pixwin.size.h;
	inst.planeCount = (U16)pArgs->pixwin.planeCount;
	ObjectWrite(self, ctx, &inst);

	if(pArgs->pixwin.bCreate)
		StsRet(CreatePixelmapFile(self, &inst), s);

	if(s >= stsOK)
		s = PixWinSetup(self, &inst);

	ObjectWrite(self, ctx, &inst);
	return s;
	MsgHandlerParametersNoWarning;
}
/****************************************************************************
	msgSave
****************************************************************************/
MsgHandlerWithTypes(PixWinSave, P_OBJ_SAVE, P_INSTANCE_DATA)
{
	STATUS s;
	U32	version = CURRENT_VERSION;

	StsRet(ObjectCallAncestor(msgSave, self, pArgs, ctx), s);

	StsRet(WriteData(pArgs->file, sizeof(version), &version), s);
	StsRet(WriteString(pArgs->file, pData->pFileName), s);

	return stsOK;
	MsgHandlerParametersNoWarning;
}
/****************************************************************************
	msgRestore
****************************************************************************/
MsgHandlerWithTypes(PixWinRestore, P_OBJ_RESTORE, P_INSTANCE_DATA)
{
	STATUS s;
	INSTANCE_DATA inst = *pData;
	U32 version;

	StsRet(ObjectCallAncestor(msgRestore, self, pArgs, ctx), s);

	StsRet(ReadData(pArgs->file, sizeof(version), &version), s);
	if(version != CURRENT_VERSION)
		return stsIncompatibleVersion;
	StsRet(ReadString(pArgs->file, &inst.pFileName), s);

	s = PixWinSetup(self, &inst);
	ObjectWrite(self, ctx, &inst);
	return s;
	MsgHandlerParametersNoWarning;
}

/****************************************************************************
	msgFree.
		
	Destroy/free self and its internals
****************************************************************************/
MsgHandlerWithTypes(PixWinFree, P_ARGS, P_INSTANCE_DATA)
{
	U32 i;
	INSTANCE_DATA inst = *pData;

	if(inst.pixelMap)
	  {
		ObjectCall(msgDestroy, pData->pixelMap, Nil(P_ARGS));
		inst.pixelMap = 0;
		// don't free the planes if we're using a memory mapped file
		if(inst.file == 0)
		  {
			for(i = 0; i < inst.planeCount; i++)
		  	{
				OSHeapBlockFree(inst.pPlanes[i]);
				inst.pPlanes[i] = 0;
		  	}
		  }
		OSHeapBlockFree(inst.pPlanes);
		inst.pPlanes = 0;
	  }
	if(inst.file)
	  {
		ObjCallWarn(msgDestroy, inst.file, pNull);
		inst.file = 0;
	  }

	ObjectWrite(self, ctx, &inst);

	return ObjectCallAncestor(msgFree, self, pArgs, ctx);
	MsgHandlerParametersNoWarning;
}

/****************************************************************************
	msgPixWinPixelmap.
****************************************************************************/
MsgHandlerWithTypes(PixWinPixelmap, P_OBJECT, P_INSTANCE_DATA)
{
	*pArgs = pData->pixelMap;
	return stsOK;
	MsgHandlerParametersNoWarning;
}


/****************************************************************************
	msgWinRepaint.
	Repaint/redraws the pixWin
****************************************************************************/
MsgHandlerWithTypes(PixWinRepaint, P_ARGS, P_INSTANCE_DATA)
{
	STATUS				s;
	RECT32				dirtyRect;

	ObjCallRet(msgWinBeginRepaint, self, &dirtyRect, s);

	if(pData->planeCount ==  1)
	  {
		StsWarn(DrawPixelMap(pData->pixelMap, self));
	  }
	else if(pData->planeCount >  1)
	  {
		StsWarn(CopyPixelMap(pData->pixelMap, self));
	  }

	ObjCallRet(msgWinEndRepaint, self, Nil(P_ARGS), s);

	return stsOK;
	MsgHandlerParametersNoWarning;
}

/****************************************************************************
	Respond to msgWinGetDesiredSize.

	Return "natural size" of receiver
****************************************************************************/
MsgHandlerWithTypes(PixWinGetDesiredSize, P_WIN_METRICS, P_INSTANCE_DATA)
{
	STATUS s;

	s = ObjectCall(msg, pData->pixelMap, pArgs);
//Debugf("PixWin:  Desired size of pixelmap '%s' is (%ld x %ld)", pData->pFileName, pArgs->bounds.size.w, pArgs->bounds.size.h);
	return s;
	MsgHandlerParametersNoWarning;
} 

/****************************************************************************
	ClsPixWinInit
	
	Install/register the class clsPixWin.
****************************************************************************/
STATUS EXPORTED ClsPixWinInit(void)
{
	CLASS_NEW		new;
	STATUS			s;

	//
	//  Install/register clsPixWin.
	//
	ObjCallRet(msgNewDefaults, clsClass, &new, s);
	new.object.uid		= clsPixWin;
	new.object.key		= objWKNKey;
	new.cls.pMsg		= PixWinMethodTable;
	new.cls.ancestor	= clsWin;
	new.cls.size		= SizeOf(INSTANCE_DATA);
	new.cls.newArgsSize	= SizeOf(PIXWIN_NEW);
	ObjCallRet(msgNew, clsClass, &new, s);

	return stsOK;
}

