/****************************************************************************
 File: milsvc.c

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.21  $
   $Author:   jgarrett  $
     $Date:   30 Mar 1992 15:07:36  $

 You may use this Sample Code any way you please provided you 
 do not resell the code and that this notice (including the above 
 copyright notice) is reproduced on all copies.  THIS SAMPLE CODE 
 IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, AND GO CORPORATION 
 EXPRESSLY DISCLAIMS ALL IMPLIED WARRANTIES, INCLUDING BUT NOT 
 LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE. IN NO EVENT WILL GO CORPORATION BE LIABLE TO YOU 
 FOR ANY CONSEQUENTIAL,INCIDENTAL,OR INDIRECT DAMAGES ARISING OUT OF 
 THE USE OR INABILITY TO USE THIS SAMPLE CODE.

 This file contains the class definition and methods for clsTestMILService.
****************************************************************************/

#include <go.h>
#include <clsmgr.h>
#include <system.h>
#include <os.h>
#include <ospriv.h>
#include <debug.h>
#include <milserv.h>
#include <stream.h>
#include <string.h>
#include <list.h>
#include <servmgr.h>

#include <dvparall.h>

#include <method.h>
#include <milsvc.h>
#include <milsvc0.h>


/****************************************************************************
	This service is to be placed on the service managers listed.
****************************************************************************/

static const UID  logicalParallelPorts[] = { theParallelDevices,
												thePrinterDevices };


/****************************************************************************
	The following routines provide the entry points to the ring 0
	portion of test mil service.  First a prototype of the ring 0
	function is	given then the actual transition occurs by calling
	OSSupervisorCall using the function in the prototype as the
	function to call in the ring 0 code.  In the ring 0 linker
	definition file the function defined with a leading 'X' is
	aliased with the actual function in the ring 0 code without
	the 'X'.
****************************************************************************/

STATUS EXPORTED XTestMILSvcGetReqBlocks(P_INSTANCE_DATA pInst);

STATUS EXPORTED TestMILSvcGetReqBlocks(P_INSTANCE_DATA pInst)
{
	return (STATUS)(OSSupervisorCall(XTestMILSvcGetReqBlocks, &pInst, 1));
}


STATUS EXPORTED XTestMILSvcInitialize(P_INSTANCE_DATA pInst);

STATUS EXPORTED TestMILSvcInitialize(P_INSTANCE_DATA pInst)
{
	return (STATUS)(OSSupervisorCall(XTestMILSvcInitialize, &pInst, 1));
}


STATUS EXPORTED XTestMILSvcGetMetrics(P_TEST_MIL_SVC_METRICS pMetrics,
									P_INSTANCE_DATA pInst);

STATUS EXPORTED TestMILSvcGetMetrics(P_TEST_MIL_SVC_METRICS pMetrics,
									P_INSTANCE_DATA pInst)
{
	Unused(pInst);
	return (STATUS)(OSSupervisorCall(XTestMILSvcGetMetrics, &pMetrics, 2));
}


STATUS EXPORTED XTestMILSvcSetMetrics(P_TEST_MIL_SVC_METRICS pMetrics,
									P_INSTANCE_DATA pInst);

STATUS EXPORTED TestMILSvcSetMetrics(P_TEST_MIL_SVC_METRICS pMetrics,
									P_INSTANCE_DATA pInst)
{
	Unused(pInst);
	return (STATUS)(OSSupervisorCall(XTestMILSvcSetMetrics, &pMetrics, 2));
}


STATUS EXPORTED XTestMILSvcPrint(P_STREAM_READ_WRITE pStream,
							P_INSTANCE_DATA pInst);

STATUS EXPORTED TestMILSvcPrint(P_STREAM_READ_WRITE pStream,
							P_INSTANCE_DATA pInst)
{
	Unused(pInst);
	return (STATUS)(OSSupervisorCall(XTestMILSvcPrint, &pStream, 2));
}


STATUS EXPORTED XTestMILSvcGetStatus(P_TEST_MIL_SVC_STATUS pTestMILSvcStatus,
									P_INSTANCE_DATA pInst);

STATUS EXPORTED TestMILSvcGetStatus(P_TEST_MIL_SVC_STATUS pTestMILSvcStatus,
									P_INSTANCE_DATA pInst)
{
	Unused(pInst);
	return (STATUS)(OSSupervisorCall(XTestMILSvcGetStatus,
									 &pTestMILSvcStatus,
									 2));
}


STATUS EXPORTED XTestMILSvcSetInterrupt(P_INSTANCE_DATA pInst);

STATUS EXPORTED TestMILSvcSetInterrupt(P_INSTANCE_DATA pInst)
{
	return (STATUS)(OSSupervisorCall(XTestMILSvcSetInterrupt,
									 &pInst,
									 1));
}


STATUS EXPORTED XTestMILSvc0Destroy(P_INSTANCE_DATA pInst);

STATUS EXPORTED TestMILSvc0Destroy(P_INSTANCE_DATA pInst)
{
	return (STATUS)(OSSupervisorCall(XTestMILSvc0Destroy, &pInst, 1));
}


STATUS EXPORTED XTestMILSvcCancelPrint(P_INSTANCE_DATA pInst);

STATUS EXPORTED TestMILSvcCancelPrint(P_INSTANCE_DATA pInst)
{
	return (STATUS)(OSSupervisorCall(XTestMILSvcCancelPrint, &pInst, 1));
}


STATUS EXPORTED XTestMILSvcStartConnectionDetection(P_INSTANCE_DATA pInst);

STATUS EXPORTED TestMILSvcStartConnectionDetection(P_INSTANCE_DATA pInst)
{
	return (STATUS)(OSSupervisorCall(XTestMILSvcStartConnectionDetection,
									 &pInst,
									 1));
}


STATUS EXPORTED XTestMILSvcStopConnectionDetection(P_INSTANCE_DATA pInst);

STATUS EXPORTED TestMILSvcStopConnectionDetection(P_INSTANCE_DATA pInst)
{
	return (STATUS)(OSSupervisorCall(XTestMILSvcStopConnectionDetection,
									 &pInst,
									 1));
}


/****************************************************************************
	void LOCAL TestMILSvcSwitchForConnection(P_INSTANCE_DATA pInst)

	This routine sends the message msgTestMILSvcDoConnection, a private
	message, to self in process context.  See msgTestMILSvcDoConnection
	for more details.
****************************************************************************/

void LOCAL TestMILSvcSwitchForConnectionDetection(P_INSTANCE_DATA pInst)
{
	OS_TASK_ID		taskId;

	
	ObjectCall(msgOwner, pInst->self, &taskId);
	ObjectSendTask(msgTestMILSvcDoConnection,
					pInst->self,
					&taskId,
					SizeOf(OS_TASK_ID),
					taskId);

}	// TestMILSvcSwitchForConnectionDetection


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcDoConnection, P_ARGS, P_INSTANCE_DATA)

	This routine handles the message msgTestMILSvcDoConnection, a private
	message.  This message was self sent to change process to testMILSvc's
	original process context.  This process keeps the continuous mil
	request from terminating when the process whose context the service
	set connected message was sent.  We don't know if a sub-task will be
	created for the continuous request.  We don't need to do this for
	terminating connection detection.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcDoConnection, P_ARGS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA	pInst;

	
	MsgHandlerParametersNoWarning;

	pInst = *(P_INSTANCE_DATA *)pData;
	return TestMILSvcStartConnectionDetection(pInst);

} // TestMILSvcDoConnection

/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcPenpointBooted, P_ARGS, P_INSTANCE_DATA)

	This routine handles the message msgMILSvcConnectionProcessing.
	This message is received when the PenPoint operating system is fully
	booted.  We wait to start connection processing until the system is
	booted and all services are running.  When a connection is detected,
	all services will have a chance to respond to the queries of the
	conflict group manager.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcPenpointBooted, P_ARGS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;


	MsgHandlerParametersNoWarning;


	pInst = *(P_INSTANCE_DATA *)pData;

	// Start connection detection if we have auto detecting hardware
	if (pInst->connectStyle == svcAutoDetect)
	{
		pInst->connected = false;
		/* Start connection detection */
		TestMILSvcSwitchForConnectionDetection(pInst);
	}
	return stsOK;

}	/* TestMILSvcPenpointBooted */


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcNewDefaults,
						P_TEST_MIL_SVC_NEW,
						P_INSTANCE_DATA)

	This routine handles the message msgNewDefaults.  The routine sets
	default parameters for msgNew.  All style bits for mil service are
	set/cleared.  Some may be default values.
*****************************************************************************/

MsgHandlerWithTypes(TestMILSvcNewDefaults, P_TEST_MIL_SVC_NEW, P_INSTANCE_DATA)
{

	MsgHandlerParametersNoWarning;

	/*
	 * Since a mil service targets a mil device not another
	 * service, any style bit dealing with a target should be
	 * false.  These bits are 'waitForTarget', 'autoOwnTarget',
	 * 'autoOpen', 'autoMsgPass', and 'autoOption'.
	 *
	 * Since our mil service can only be openned by one client,
	 * 'exclusiveOpen' and 'checkOwner' style bits should be true.
	 */
	pArgs->svc.style.waitForTarget		=	false;
    pArgs->svc.style.exclusiveOpen      =   true;
	pArgs->svc.style.autoOwnTarget		=	false;
    pArgs->svc.style.autoOpen           =   false;
	pArgs->svc.style.autoMsgPass		=	false;
	pArgs->svc.style.checkOwner			=	true;
	pArgs->svc.style.autoOption			=	false;
	pArgs->svc.style.openClass			=	objNull;

	/*
	 * Indicate which service managers we're to be placed on
	 */
	pArgs->svc.numManagers              =   SizeOf(logicalParallelPorts) /
											SizeOf(UID);
    pArgs->svc.pManagerList             =   logicalParallelPorts;

	return stsOK;

}	/* TestMILSvcNewDefaults */


/****************************************************************************
	BOOLEAN LOCAL IsAnybodyConnected(P_INSTANCE_DATA pInst)

	This routine scans the conflict group we're in checking if a
	mil service in the conflict group is connected.
****************************************************************************/

BOOLEAN LOCAL IsAnybodyConnected(P_INSTANCE_DATA pInst)
{
	LIST_ENTRY				le;
	OBJECT					list;
	U16						n;
	SM_QUERY_LOCK			query;
	STATUS					s;
	SVC_GET_SET_CONNECTED	serviceConnected;


	// Create a list of mil services in our conflict group
	ObjCallRet(msgIMGetList, pInst->conflictGroup, &list, s);

	// Get the number of mil services in our conflict group
	ObjectCall(msgListNumItems, list, &n);

	// Query each mil service in our group
	for (le.position = 0; le.position < n; le.position++)
	{
		// Get the next mil service in our group
		ObjCallWarn(msgListGetItem, list, &le);
		query.handle = (OBJECT)le.item;
		ObjCallWarn(msgSMQuery, pInst->conflictGroup, &query);

		// Get the connected state of the mil service
		ObjCallWarn(msgSvcGetConnected, query.service, &serviceConnected);
		if (serviceConnected.connected != false)
		{
			// if connected - we're done
			break;
		}
	}
	// Destroy list
	ObjCallWarn(msgDestroy, list, pNull);

	// return connected state
	return serviceConnected.connected;

}	/* IsAnybodyConnected */

/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcNew, P_TEST_MIL_SVC_NEW, P_INSTANCE_DATA)
	
	This routine handles the message msgInit.  This routine creates a
	new testMILSvc object.  'msgMILSvcGetDevice' is called to obtain	the
	mil device we are to use for printing.  Connection detection is
	not started until the PenPoint is booted.  A check is provided
	here in case we're loaded after the system is booted.  In which
	case, we start connection detection only if no other mil service
	in our conflict group is in the connected state.  It's normal for
	all	mil services other than the mil service being connected to
	terminate their connection detection functions when the service
	is connected.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcNew, P_TEST_MIL_SVC_NEW, P_INSTANCE_DATA)
{
	MIL_SVC_DEVICE				device;
	OBJECT						myself;
	P_INSTANCE_DATA				pInst;
	TEST_MIL_SVC_TIME_DELAYS	timeDelays;
	STATUS						s;
	SYS_BOOT_STATE				sysBootState;


	MsgHandlerParametersNoWarning;


	/* Create self.	*/

	// Save who we are
	myself = pArgs->object.uid;

	// Get mil device we are to use
	ObjCallRet(msgMILSvcGetDevice, myself, &device, s);

	Dbg(Debugf("TEST_MIL_SVC: creating instance for logicalId %d: unit %d",
											device.logicalId, device.unit);)

	// Allocate our instance data
	StsRet(OSHeapBlockAlloc(osProcessSharedHeapId,
							SizeOf(INSTANCE_DATA),
							&pInst), s);

	// Store pointer to our instance data
	ObjectWrite(myself, ctx, &pInst);

	// Initialize our instance data
	memset(pInst, 0, SizeOf(INSTANCE_DATA));

	// Save who we are and who our mil device is
	pInst->self = myself;
	pInst->logicalId = device.logicalId;
	pInst->conflictGroup = device.conflictGroup;
	pInst->unit = device.unit;

	// Get whether we can detect connection 
	pInst->connectStyle = pArgs->svc.style.connectStyle;

	// Initialize print in progress, connected and open flag
	pInst->printInProgress = false;
	pInst->connected = false;
	pInst->open = false;

	// Allocate necessary ring 0 memory
	StsRet(TestMILSvcGetReqBlocks(pInst), s);

	// Enable interrupt
	StsRet(TestMILSvcSetInterrupt(pInst), s);

	// Initialize parallel printer device timeouts
	timeDelays.initDelay = 500000;
	timeDelays.interruptTimeOut = 30000;
	ObjCallRet(msgTestMILSvcSetTimeDelays, self, &timeDelays, s);
	StsRet(TestMILSvcGetMetrics(&(pInst->testMILSvcMetrics), pInst), s);

	// Get the booted state of the system
	ObjCallRet(msgSysGetBootState, theSystem, &sysBootState, s);

	// Don't start if we don't have auto connecting hardware
	if (pInst->connectStyle == svcAutoDetect)
	{
		// If the system is booted and no other mil service is connected
		// start connection detection
		if (sysBootState.booted != false && IsAnybodyConnected(pInst) == false)
		{
			TestMILSvcSwitchForConnectionDetection(pInst);
		}
	}
	return s;

}  /* TestMILSvcNew */


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcFree, P_ARGS, P_INSTANCE_DATA)
	
	This routine handles the message msgFree.  This routine destroys
	a testMILSvc object.  We destroy ourself by terminating connection
	detection and freeing all system resources we've allocated.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcFree, P_ARGS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;
	STATUS				s;

	MsgHandlerParametersNoWarning;

	pInst = *(P_INSTANCE_DATA *)pData;

	if (pInst != 0)
	{
		StsRet(TestMILSvcStopConnectionDetection(pInst), s);
		StsRet(TestMILSvc0Destroy(pInst), s);
		StsRet(OSHeapBlockFree(pInst), s);
	}

	return stsOK;

}  /* TestMILSvcFree  */


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcGetSvcMetrics,
						P_SVC_GET_SET_METRICS,
						P_INSTANCE_DATA)

	This routine handles the message msgSvcGetMetrics.  This routine
	gets the parallel printer port metrics.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcGetSvcMetrics,
					P_SVC_GET_SET_METRICS,
					P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;

	MsgHandlerParametersNoWarning;


	// Get pointer to our instance data
	pInst = *(P_INSTANCE_DATA *)pData;

	/*
	 * If the len field of the metrics structure passed in is zero
	 * return the size of data area needed to store the metrics;
	 * otherwise, if  the length field equals the size of the
	 * metrics, return the metrics.
	 */
	if (pArgs->len == 0)
	{
		return pArgs->len = SizeOf(TEST_MIL_SVC_METRICS);
	}
	else if (pArgs->len != SizeOf(TEST_MIL_SVC_METRICS))
	{
		return stsBadParam;
	}
	else
	{
		memcpy(pArgs->pMetrics,
				&(pInst->testMILSvcMetrics),
				SizeOf(TEST_MIL_SVC_METRICS));
	}

}	/* TestMILSvcGetMetrics */

/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcSetSvcMetrics,
						P_SVC_GET_SET_METRICS,
						P_INSTANCE_DATA)

	This routine handles the message msgSvcSetMetrics.  This routine
	sets the parallel printer port metrics.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcSetSvcMetrics,
					P_SVC_GET_SET_METRICS,
					P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;

	MsgHandlerParametersNoWarning;

	
	pInst = *(P_INSTANCE_DATA *)pData;

	// Only set the metrics if the size of the data area is correct
	if (pArgs->len != SizeOf(TEST_MIL_SVC_METRICS))
	{
		return stsBadParam;
	}
	// Call the ring 0 code to set the mil device metrics
	return TestMILSvcSetMetrics(pArgs->pMetrics, pInst);

}	/* TestMILSvcSetMetrics */


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcOpen, P_ARGS, P_INSTANCE_DATA)

	This routine handles the message msgSvcOpenRequested/
	msgQueryLockRequested.  Either of these messages is received when
	a client wishes to open our mil service.  We save our open state
	since when we receive messages from our client they don't go through
	the service open checks.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcOpen, P_ARGS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;

	MsgHandlerParametersNoWarning;
	
	
	pInst = *(P_INSTANCE_DATA *)pData;

	pInst->open = true;

	return stsOK;

}	/* TestMILSvcOpen */


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcClose, P_ARGS, P_INSTANCE_DATA)

	This routine handles the message msgSvcCloseRequested/
	msgSvcQueryUnlockRequested.  Either of these messages is received
	when a client wishes to close our mil service.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcClose, P_ARGS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;

	MsgHandlerParametersNoWarning;
	
	pInst = *(P_INSTANCE_DATA *)pData;
	
	pInst->open = false;

	return stsOK;

}	/* TestMILSvcClose */


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcHWInitialize, P_ARGS, P_INSTANCE_DATA)

	This routine handles the message msgTestMILSvcHWInitialize.  This routine
	initializes the printer attached by toggling the initialize signal to
	the printer.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcHWInitialize, P_ARGS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;

	MsgHandlerParametersNoWarning;
	
	
	pInst = *(P_INSTANCE_DATA *)pData;

	// Ignore if printer is not connected or we are not open
	if (pInst->connected == false || pInst->open == false)
	{
		return stsFailed;
	}
	// Must call the mil device from ring 0
	return TestMILSvcInitialize(pInst);

}	/* TestMILSvcHWInitialize */


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcGetHWStatus, P_TEST_MIL_SVC_STATUS, P_INSTANCE_DATA)

	This routine handles the message msgTestMILSvcGetStatus.  The routine
	gets the contents of the parallel port status register.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcGetHWStatus, P_TEST_MIL_SVC_STATUS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;

	MsgHandlerParametersNoWarning;
	
	
	pInst = *(P_INSTANCE_DATA *)pData;
	
	// Must call the mil device from ring 0
	return TestMILSvcGetStatus(pArgs, pInst);

}	/* TestMILSvcGetHWStatus */


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcAutoLineFeedOn, P_ARGS, P_INSTANCE_DATA)

	This routine handles the message msgTestMILSvcAutoLineFeedOn.  The routine
	sets the auto line feed signal to the printer active.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcAutoLineFeedOn, P_ARGS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;

	MsgHandlerParametersNoWarning;
	
	
	pInst = *(P_INSTANCE_DATA *)pData;
	
	// Set the auto line feed flag in our metrics
	pInst->testMILSvcMetrics.unitFlags |= parallelAutoLineFeed;

	// Set the auto line feed flag in the mil devices parameters
	return TestMILSvcSetMetrics(&(pInst->testMILSvcMetrics), pInst);

}	/* TestMILSvcAutoLineFeedOn */


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcAutoLineFeedOff, P_ARGS, P_INSTANCE_DATA)

	This routine handles the message msgTestMILSvcAutoLineFeedOn.  The routine
	sets the auto line feed signal to the printer active.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcAutoLineFeedOff, P_ARGS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;

	MsgHandlerParametersNoWarning;
	
	
	pInst = *(P_INSTANCE_DATA *)pData;

	// Clear the auto line feed flag in our metrics
	pInst->testMILSvcMetrics.unitFlags &= ~parallelAutoLineFeed;

	// Clear the auto line feed flag in the mil devices parameters
	return TestMILSvcSetMetrics(&(pInst->testMILSvcMetrics), pInst);

}	/* TestMILSvcAutoLineFeedOff */


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcGetTimeDelays,
						P_TEST_MIL_SVC_TIME_DELAYS,
						P_INSTANCE_DATA)

	This routine handles the message msgTestMILSvcGetTimeDelays.  This routine
	gets the time delays used by the mil device.  'initDelay' is the width
	in microseconds of the initialization pulse used to hardware initialize
	the printer.  'interruptTimeOut' is the time in milliseconds to wait
	for an interrupt to occur after writing a character to the printer.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcGetTimeDelays, P_TEST_MIL_SVC_TIME_DELAYS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;

	MsgHandlerParametersNoWarning;
	
	
	pInst = *(P_INSTANCE_DATA *)pData;

	// Return the time periods from our metrics
	pArgs->initDelay = pInst->testMILSvcMetrics.initDelay;
	pArgs->interruptTimeOut = pInst->testMILSvcMetrics.interruptTimeOut;
					
	return stsOK;

}	/* TestMILSvcGetTimeDelays */

/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcSetTimeDelays,
						P_TEST_MIL_SVC_TIME_DELAYS,
						P_INSTANCE_DATA)

	This routine handles the message msgTestMILSvcSetTimeDelays.  This routine
	sets the time delays used by the mil device.  'initDelay' is the width
	in microseconds of the initialization pulse used to hardware initialize
	the printer.  'interruptTimeOut' is the time in milliseconds to wait
	for an interrupt to occur after writing a character to the printer.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcSetTimeDelays, P_TEST_MIL_SVC_TIME_DELAYS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;

	MsgHandlerParametersNoWarning;
	
	
	pInst = *(P_INSTANCE_DATA *)pData;

	// Neither time period can be zero
	if (pArgs->initDelay == 0 || pArgs->interruptTimeOut == 0)
	{
		return stsBadParam;
	}
	else
	{
		// Set our metrics to the new values
		pInst->testMILSvcMetrics.initDelay =	pArgs->initDelay;
		pInst->testMILSvcMetrics.interruptTimeOut = pArgs->interruptTimeOut;

		// Tell the mil device of the new values
		return TestMILSvcSetMetrics(&(pInst->testMILSvcMetrics), pInst);
	}

}	/* TestMILSvcSetTimeDelays */

/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcCancelPrintBuffer, P_ARGS,	P_INSTANCE_DATA)

	This routine handles the message msgTestMILSvcCancelPrintBuffer.  The
	routine cancels the printing of the buffer currently being printed.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcCancelPrintBuffer, P_ARGS,	P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;

	MsgHandlerParametersNoWarning;

	
	pInst = *(P_INSTANCE_DATA *)pData;

	// Transition to ring 0 to do the work
	return TestMILSvcCancelPrint(pInst);

}	/* TestMILSvcCancelPrintBuffer */

/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcFlush, P_ARGS, P_INSTANCE_DATA)

	This routine handles the message msgFSFlush.  This routine allows
	a client to send a flush to our mil service without causing an error.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcFlush, P_ARGS, P_INSTANCE_DATA)
{
	MsgHandlerParametersNoWarning;
	
	
	// So stdio can use this without warnings being generated
	return stsOK;

}	/* TestMILSvcFlush */

/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcPrintBuffer, P_ARGS, P_INSTANCE_DATA)

	This routine handles the message msgStreamWrite.  The routine causes
	a clients buffer to be sent to the printer connected.  An error is
	returned and the buffer is not printed if we are not open and no
	printer is connected.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcPrintBuffer, P_ARGS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;

	MsgHandlerParametersNoWarning;
	
	
	pInst = *(P_INSTANCE_DATA *)pData;

	// Return error if not connected or openned
	if (pInst->connected == false || pInst->open == false)
	{
		return stsFailed;
	}

	// Must call mil device from ring 0
	return TestMILSvcPrint((P_STREAM_READ_WRITE)pArgs, pInst);

}	/* TestMILSvcPrintBuffer */


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcAreYouConnected,
						P_MIL_SVC_ARE_YOU_CONNECTED,
						P_INSTANCE_DATA)

	This routine handles the message msgMILSvcAreYouConnected.  The
	message is sent by our conflict group manager when a mil service
	in our conflict group reports connected.  More than one mil
	service can report connected.  The valid responses are msYES,
	msMaybe, or msNO.  'msYES' indicates we are certain our device is
	connected. 'msMaybe' indicates something is connected, but we are
	not sure if it is our device.  'msMaybe' can also indicate we don't
	know if anything is connected.  'msNO' indicates our device is not
	connected.  The mil device for the printer can determine absolutely
	if a printer is connected.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcAreYouConnected,
					P_MIL_SVC_ARE_YOU_CONNECTED,
					P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;

	
	MsgHandlerParametersNoWarning;
	

	pInst = *(P_INSTANCE_DATA *)pData;

	/*
	 *	Answer maybe if we can't determine
	 *	if anything is connected.  If only one
	 *	other mil service indicates it is connected
	 *	with msYes, then that mil service will get
	 *	the connection.  If more than one mil service
	 *	indicates it is connected with msYes or msMaybe,
	 *	then a dialog box will be displayed with all the
	 *	mil services indicating connected.  The user
	 *	then selects which mil service will actually be
	 *	connected.  If we're the only mil service indicating
	 *	connected with msMaybe, then we will be connected by
	 *	default.
	 */
	if (pInst->connectStyle == svcNoAutoDetect)
	{
		*pArgs = msMaybe;
	}
	else if (pInst->connected == false)
	{
		Dbg(Debugf("TEST_MIL_SVC: responding to 'msgMILSvcAreConnected'"
					" with msNo");)
		*pArgs = msNo;
	}
	else
	{
		Dbg(Debugf("TEST_MIL_SVC: responding to 'msgMILSvcAreConnected'"
					" with msYes");)
		*pArgs = msYes;
	}
	return stsOK;

}	/* TestMILSvcAreYouConnected */


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcConnectionStateResolved, P_ARGS, P_INSTANCE_DATA)

	This routine handles the messge msgMILSvcConnectionStateResolved.
	This message is sent by our conflict manager when a mil service's
	connection state changes.  This message indicates which mil service
	on the conflict group is being set connected.  The logical id
	of the corresponding mil device is used to indicate who received the
	connection.  A logical id of maxU16 indicates a mil service was
	disconnected.  When another mil service receives connection, we
	should terminate our connection detection function to prevent our
	connection function from interferring with the operation of the
	other mil service and mil device.  We restart our connection
	detection function when the other mil service is disconnected.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcConnectionStateResolved, P_ARGS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;


	MsgHandlerParametersNoWarning;


	pInst = *(P_INSTANCE_DATA *)pData;

	/*
	 * Start our connection detection function if
	 * no other mil service is connected, if our
	 * mil device can detect connection, and if
	 * our connection detection function is not
	 * already started.
	 *
	 * Terminate our connection detection function
	 * if another mil service in our conflict group
	 * is connected and if we have a connection
	 * detection function started.
	 *
	 * Otherwise if we are connected, perform any
	 * initialization operations.
	 */
	if ((U16)pArgs == maxU16)
	{
		if (pInst->connectStyle == svcAutoDetect && pInst->pRBConnect == 0)
		{
			TestMILSvcSwitchForConnectionDetection(pInst);
		}
	}
	else if ((U16)pArgs != pInst->logicalId)
	{
		if (pInst->connectStyle == svcAutoDetect && pInst->pRBConnect != 0)
		{
			TestMILSvcStopConnectionDetection(pInst);
			pInst->pRBConnect = 0;
		}
	}
	else if ((U16)pArgs == pInst->logicalId)
	{
	}
	return stsOK;

}	/* TestMILSvcConnectionStateResolved */


/****************************************************************************
	MsgHandlerWithTypes(TestMILSvcPowerOff, P_ARGS, P_INSTANCE_DATA)

	This message handles the message msgMILSvcPowerOff.  This messge
	is received when the power to the tablet is about to be shut off.
	We perform any clean up operations here.  There is a corresponding
	message, msgMILSvcPowerOn, which can be handled when power is
	applied.  At which time, we can perform any operations necessary
	when power is applied.  When power is shut off we want to cancel
	printing if a buffer is currently being printed.
****************************************************************************/

MsgHandlerWithTypes(TestMILSvcPowerOff, P_ARGS, P_INSTANCE_DATA)
{
	P_INSTANCE_DATA		pInst;


	MsgHandlerParametersNoWarning;


	pInst = *(P_INSTANCE_DATA *)pData;
	return TestMILSvcCancelPrint(pInst);

}	/* TestMILSvcPowerOff */


/****************************************************************************
	ClsTestMILServiceInit(void)
	
	Install the class.
****************************************************************************/
STATUS EXPORTED ClsTestMILServiceInit(void)
{
	CLASS_NEW		classNew;
	STATUS			s;


	/* Create the class. */

	ObjCallWarn(msgNewDefaults, clsClass, &classNew);

	classNew.object.cap		   |= objCapCall;
	classNew.object.uid			= clsMILParallelPortDevice;
	classNew.cls.pMsg			= clsTestMILSvcServiceTable;
	classNew.cls.ancestor		= clsMILService;
	classNew.cls.size			= SizeOf(INSTANCE_DATA);
	classNew.cls.newArgsSize	= SizeOf(TEST_MIL_SVC_NEW);

	ObjCallRet(msgNew, clsClass, &classNew, s);

	return (stsOK);

}	/* ClsTestMILServiceInit */


/****************************************************************************
	DLLMain
	
	Initialize milsvc.dll

****************************************************************************/

STATUS EXPORTED DLLMain(void)
{
	STATUS				s;
	SVC_INIT_SERVICE	initService;


	// Install classes
	StsRet(ClsTestMILServiceInit(), s);

	// Let system know about our service
	// and set global service characteristics
	memset(initService.spare, 0, sizeof(initService.spare));
	initService.autoCreate = true;	  		
	initService.serviceType = 0;
	initService.initServiceFlags = svcNoShow;
	ObjCallRet(msgSvcClassInitService, 
				clsMILParallelPortDevice,
				&initService,
				s);

	return stsOK;

}	/* DLLMain */
