/****************************************************************************
 File: hellowin.c

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 You may use this Sample Code any way you please provided you 
 do not resell the code and that this notice (including the above 
 copyright notice) is reproduced on all copies.  THIS SAMPLE CODE 
 IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, AND GO CORPORATION 
 EXPRESSLY DISCLAIMS ALL IMPLIED WARRANTIES, INCLUDING BUT NOT 
 LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 PARTICULAR PURPOSE. IN NO EVENT WILL GO CORPORATION BE LIABLE TO YOU 
 FOR ANY CONSEQUENTIAL,INCIDENTAL,OR INDIRECT DAMAGES ARISING OUT OF 
 THE USE OR INABILITY TO USE THIS SAMPLE CODE.

 $Revision:   1.8  $
   $Author:   kcatlin  $
     $Date:   18 Mar 1992 08:55:36  $

 This file contains a simple "Hello World" window subclass.

 It creates a drawing context to paint a welcome message in self.
 Since clsHelloWin doesn't use the DC anywhere else but msgWinRepaint,
 it could create it on the fly during msgWinRepaint processing, but
 instead clsHelloWin saves the DC in its instance data.
 Since clsHelloWorld frees the hello window upon receiving msgAppClose,
 the DC doesn't take up space when the application is "closed down."

 The repainting routine jumps through some geometry/drawing context hoops
 to ensure that the drawing fits in the window yet remains proportionately
 sized.

 If you turn on the "F40" debugging flag (e.g. by putting
	DEBUGSET=/DF0040
 in \penpoint\boot\environ.ini), then drawing takes places with thick lines
 so that drawing operations are more visible.
 If you turn on the "F20" debugging flag, messages to clsHelloWin will be
 traced.   

****************************************************************************/
#include <debug.h>
#include <win.h>
#include <sysgraf.h>
#include <sysfont.h>
#include <string.h>				// for memset().
#include <gomath.h>				// for scale calc. in fixed point.
#include <helwtbl.h>			// method definitions
#include <hellowin.h>			// clsHelloWin's UID and msgNew args.

typedef struct INSTANCE_DATA {
	SYSDC			dc;
} INSTANCE_DATA, *P_INSTANCE_DATA;


// Scale font to 100 units to begin with.
#define initFontScale	100
// Line thickness a twelfth of the font scale.
#define lineThickness	8


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                Methods								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
	HelloWinInit
	
	Create a new window object.
****************************************************************************/
MsgHandler(HelloWinInit)
{
	SYSDC_NEW			dn;
	INSTANCE_DATA		data;
	SYSDC_FONT_SPEC		fs;
	SCALE				fontScale;
	STATUS				s;


	// Null the instance data.
	memset(&data, 0, sizeof(INSTANCE_DATA));

	// Create a dc.
	ObjCallWarn(msgNewDefaults, clsSysDrwCtx, &dn);
	ObjCallRet(msgNew, clsSysDrwCtx, &dn, s);
	data.dc = dn.object.uid;

	// Rounded lines, thickness of zero.
	ObjectCall(msgDcSetLineThickness, data.dc, (P_ARGS)0);
	if (DbgFlagGet('F', 0x40L)) {
		Dbg(Debugf("Use a non-zero line thickness.");)
		ObjectCall(msgDcSetLineThickness, data.dc, (P_ARGS)2);
	}

	// Open a font.  Use the "user input" font (whatever the user has
	// chosen for this in System Preferences.
	fs.id				= 0;
	fs.attr.group		= sysDcGroupUserInput;
	fs.attr.weight		= sysDcWeightNormal;
	fs.attr.aspect		= sysDcAspectNormal;
	fs.attr.italic		= 0;
	fs.attr.monospaced	= 0;
	fs.attr.encoding	= sysDcEncodeGoSystem;
	ObjCallJmp(msgDcOpenFont, data.dc, &fs, s, Error);

	//
	// Scale the font.  The entire DC will be scaled in the repaint
	// to pleasingly fill the window.
	fontScale.x = fontScale.y = FxMakeFixed(initFontScale,0);
	ObjectCall(msgDcScaleFont, data.dc, &fontScale);

	// Bind the window to the dc.
	ObjectCall(msgDcSetWindow, data.dc, (P_ARGS)self);

	// Update the instance data.
	ObjectWrite(self, ctx, &data);

	return stsOK;
	MsgHandlerParametersNoWarning;				// suppress compiler warnings about unused parameters

Error:

	ObjCallWarn(msgDestroy, data.dc, Nil(OBJ_KEY));

	return s;

}  /* HelloWinInit */


/****************************************************************************
	HelloWinFree
	
	Free self.
****************************************************************************/
MsgHandlerWithTypes(HelloWinFree, P_ARGS, P_INSTANCE_DATA)
{
	// Destroy the dc.  (Assumes that this will not fail.)
	// Note that pData is now invalid.
	ObjCallWarn(msgDestroy, pData->dc, Nil(P_ARGS));

	// Ancestor will eventually free self.
	return stsOK;
	MsgHandlerParametersNoWarning;
} /* HelloWinFree */


/****************************************************************************
	HelloWinRepaint
	
	Repaint the window.  This is the only paint routine needed; clsHelloWin
	relies on the window system to tell it when it needs (re)painting.
****************************************************************************/
MsgHandlerWithTypes(HelloWinRepaint, P_ARGS, P_INSTANCE_DATA)
{
	SYSDC_TEXT_OUTPUT	tx;
	S32					textWidth;
	S32					helloAdjust, worldAdjust;
	SYSDC_FONT_METRICS	fm;
	SIZE32				drawingSize;
	WIN_METRICS			wm;
	FIXED				drawingAspect, winAspect;
	SCALE				scale;
	RECT32				dotRect;
	XY32				bezier[4];
	STATUS				s;

	//
	// Determine size of drawing in 100 units to a point coord. system.
	// The words "Hello" and "World" have no descenders (in most fonts!!).
	// Height is font height (initFontScale) * 2 - the descender size.
	// Width is max of the two text widths plus em.width (width of
	// the exclamation point.
	//

	// Figure out the widths of the two text strings.
	
	// Init tx.
	memset(&tx, 0, sizeof(SYSDC_TEXT_OUTPUT));
	tx.underline	= 0;
	tx.alignChr		= sysDcAlignChrBaseline;
	tx.stop			= maxS32;
	tx.spaceChar	= 32;

	// Set the overall text width to whichever text string is wider.
	tx.cp.x			= 0;
	tx.cp.y			= 0;
	tx.pText		= "World";
	tx.lenText		= strlen(tx.pText);
	ObjectCall(msgDcMeasureText, pData->dc, &tx);
	textWidth = tx.cp.x;

	tx.cp.x			= 0;
	tx.cp.y			= 0;
	tx.pText		= "Hello";
	tx.lenText		= strlen(tx.pText);
	ObjectCall(msgDcMeasureText, pData->dc, &tx);

	if (tx.cp.x > textWidth) {
		// "Hello" is wider
		helloAdjust = 0;
		worldAdjust = (tx.cp.x - textWidth) / 2;
		textWidth = tx.cp.x;
	} else {
		// "World" was wider
		worldAdjust = 0;
		helloAdjust = (textWidth - tx.cp.x) / 2;
	}

	// Get font metrics.
	ObjectCall(msgDcGetFontMetrics, pData->dc, &fm);

	drawingSize.w = textWidth + fm.em.w;
	// Remember, descenderPos is negative.
	drawingSize.h = (2 * initFontScale) + fm.descenderPos;

	//
	// Must bracket all repainting with msgWinBegin/EndRepaint.
	// The window system figures out which part of the window needs
	// repainting, and restricts all painting operations to that update
	// area.
	//
	ObjCallRet(msgWinBeginRepaint, pData->dc, pNull, s);

	// Fill the background with white to start.
	ObjectCall(msgDcFillWindow, pData->dc, pNull);

	//
	// We have determined the size of the drawing in points.
	// But if the window is much smaller than this the drawing will
	// be cropped.  So, we must scale it to fit the window.
	// You can scale a DC to match the width and height of a window using
	// dcUnitsWorld, but then the text would be stretched strangely.
	//
	// Instead, we'll compute a consistent scaling factor for the drawing.
	//

	//
	// We need to first determine the size of the window.
	// We send the message to the DC to get the size in DC units.
	//
	ObjCallJmp(msgWinGetMetrics, pData->dc, &wm, s, exit);

	// Now decide whether to scale by the x or y coordinate.
	// Have to hassle with Fixed Point!
	drawingAspect = FxDivIntsSC(drawingSize.h, drawingSize.w);
	winAspect = FxDivIntsSC(wm.bounds.size.h, wm.bounds.size.w);

	if (winAspect > drawingAspect ) {
		//
		// The window is "taller" than the drawing.  Scale so the
		// drawing fills the window horizontally.
		//
		Dbg(Debugf("Window is taller than drawing!  Still must calculate vertical offset!");)
		scale.x = scale.y = FxDivIntsSC(wm.bounds.size.w, drawingSize.w);
	} else {
		//
		// The window is "wider" than the drawing.  Scale so the
		// drawing fills the window vertically.
		//
		Dbg(Debugf("Window is wider than drawing!  Still must calculate horizontal offset!");)
		scale.x = scale.y = FxDivIntsSC(wm.bounds.size.h, drawingSize.h);
	}
	ObjectCall(msgDcScale, pData->dc, &scale);



	//
	// At this point a more sophisticated program would figure out
	// which parts need redrawing based on the boundaries of the
	// dirty area.
	//

	// Display the text.

	// Display "Hello". tx was set to do this from before, but need to
	// reset tx.lenText because msgDcMeasureText passes back in it the 
	// offset of the last character that would be drawn in it.
	tx.cp.x			= helloAdjust;
	tx.cp.y			= initFontScale;
	tx.lenText		= strlen(tx.pText);
	ObjectCall(msgDcDrawText, pData->dc, &tx);

	// Display "World".
	tx.cp.x			= worldAdjust;
	tx.cp.y			= 0;
	tx.pText		= "World";
	tx.lenText		= strlen(tx.pText);
	ObjectCall(msgDcDrawText, pData->dc, &tx);

	// Paint the exclamation point.

	ObjectCall(msgDcSetForegroundRGB, pData->dc, (P_ARGS)sysDcRGBGray66);
	// Want Foreground color of Gray for edges of Exclamation Point.
	ObjectCall(msgDcSetBackgroundRGB, pData->dc, (P_ARGS)sysDcRGBGray33);
	ObjectCall(msgDcSetLineThickness, pData->dc, (P_ARGS)lineThickness);

	// Paint the teardrop.
	// First the left half...
	bezier[0].x = textWidth + (fm.em.w / 2);
	bezier[0].y = fm.ascenderPos;
	bezier[1].x = textWidth;
	bezier[1].y = initFontScale * 3 / 2;
	bezier[2].x = bezier[1].x;
	bezier[2].y = initFontScale + fm.ascenderPos;
	bezier[3].x = bezier[0].x;
	bezier[3].y = bezier[2].y;
	ObjectCall(msgDcDrawBezier, pData->dc, bezier);
	
	// Then the right half...
	bezier[1].x = textWidth + fm.em.w;
	bezier[2].x = bezier[1].x;
	ObjectCall(msgDcDrawBezier, pData->dc, bezier);

	// Paint the dot.
	dotRect.origin.x = textWidth + (fm.em.w - fm.xPos) / 2;
	dotRect.origin.y = lineThickness / 2;
	dotRect.size.w = dotRect.size.h = fm.xPos;
	ObjectCall(msgDcDrawEllipse, pData->dc, &dotRect);


	// Fall through to return.
	s = stsOK;


exit:
	ObjCallWarn(msgWinEndRepaint, self, Nil(P_ARGS));

	// Need to restore state if no errors, so might as well do it always.
	ObjectCall(msgDcSetForegroundRGB, pData->dc, (P_ARGS)sysDcRGBBlack);
	ObjectCall(msgDcSetBackgroundRGB, pData->dc, (P_ARGS)sysDcRGBWhite);
	ObjectCall(msgDcSetLineThickness, pData->dc, (P_ARGS)0);
	if (DbgFlagGet('F', 0x40)) {
		Dbg(Debugf("Use a non-zero line thickness.");)
		ObjectCall(msgDcSetLineThickness, pData->dc, (P_ARGS)2);
	}

	return s;
	MsgHandlerParametersNoWarning;
} /* HelloWinRepaint */


/****************************************************************************
	ClsHelloWinInit
	
	Install the class.
****************************************************************************/
STATUS ClsHelloWinInit (void)
{
	CLASS_NEW		new;
	STATUS			s;

	// Create the class.
	ObjCallWarn(msgNewDefaults, clsClass, &new);
	new.object.uid			= clsHelloWin;
	new.cls.pMsg			= clsHelloWinTable;
	new.cls.ancestor		= clsWin;
	new.cls.size			= SizeOf(INSTANCE_DATA);
	new.cls.newArgsSize		= SizeOf(HELLO_WIN_NEW);
	ObjCallRet(msgNew, clsClass, &new, s);

	if (DbgFlagGet('F', 0x20)) {
		Dbg(Debugf("Turning on message tracing for clsHelloWin");)
		(void)ObjCallWarn(msgTrace, clsHelloWin, (P_ARGS) true);
	}

	return stsOK;

} /* ClsHelloWinInit */
