/****************************************************************************
 stdmsg.h

 Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.43  $
   $Author:   sisaac  $
     $Date:   02 Mar 1992 10:06:56  $

 This file contains the API definition for the standard message package. 

 The functions described in this file are contained in SYSUTIL.LIB. 
****************************************************************************/

/**** Introduction ****/
/*
 The standard message package makes it easy to display error messages,
 modal dialog boxes, and progress notes.  The package hides many of the
 details of finding resources and creating UI objects.  The package uses
 clsNote to display messages.  (See note.h.)

 Message are stored as strings in string array resources.  A 32 bit value
 identifies the proper resource.  For error messages the value is a STATUS;
 for dialog boxes and progress notes the value is a TAG constructed using
 the MakeDialogTag macro.
*/


/**** Road Map ****/
/*
 To display a dialog box, use:
	-:	StdMsg

 To display an error message when you know about the error, use:
	-:	StdError

 To display an error message when you don't know about the error, use:
	-:	StdUnknownError

 To display a progress note, use:
	-:	StdProgressUp
	-:	StdProgressDown

 To display messages extracted from a specified resource file or path, use:
	-:	StdMsgRes
	-:	StdErrorRes

 To construct a customized message, use:
	-:	StdMsgCustom

 PenPoint-internal use only:
 	-:	StdSystemError
*/


/**** A Typical Scenario ****/
/*
 [This scenario illustrates some features of the package that haven't been
 described yet.  See the sections "Button Definition" and "Text
 Substitution and Formatting" for more information.]

 The first step in using the standard message package is to define a tag or
 status for each string:

//{
	//	#define stsFooError1 		MakeStatus(clsFoo, 0)
	//	#define stsFooError2 		MakeStatus(clsFoo, 1)

	//	#define tagFooDialog1 		MakeDialogTag(clsFoo, 0) 
	//	#define tagFooDialog2 		MakeDialogTag(clsFoo, 1) 
//}

 The next thing to do is to construct resources which contain the text
 strings.  Standard message strings live in string array resources (see
 resfile.h).  Application string arrays should reside in the application's
 app.res file.  (PenPoint's string arrays reside in penpoint.res.)

 There is one string array for error strings and another separate array for
 dialog box and progress strings. A single string array resource holds all
 of the strings for a given class. 

 Typically the string arrays are defined in a .rc file which is compiled
 with the PenPoint SDK's resource compiler.  The position of each string in
 the string array resource must match its tag or status index (starting
 from 0).

//{
	static P_STRING errorClsFoo[] = {
		"This is the first error message.",  
		"[Retry] [Cancel] This is the second error message. count: ^1d"};

	static P_STRING dialogClsFoo[] = {
		"This is the first dialog message.",  
		"[Go] [Stop] This is the second dialog message. str: ^1s"};

	static RC_INPUT errorTabClsFoo = {
		resForStdMsgError(clsFoo), errorClsFoo, 0, resStringArrayResAgent};

	static RC_INPUT dialogTabClsFoo = {
		resForStdMsgDialog(clsFoo), dialogClsFoo, 0, resStringArrayResAgent};

	P_RC_INPUT  resInput [] = {
		&errorTabClsFoo,		// String array for std msg error strings
		&dialogTabClsFoo,		// String array for other std msg strings
		pNull};
//}

 Finally create a note by simply calling one of the appropriate function.
 This example uses StdMsg(), StdError() and StdUnknownError().

//{
	buttonHit = StdMsg(tagFooDialog2, "String");
	s = ObjectCall(...);
	if (s < stsOK) {
		if (s == stsFooError1) {
			StdError(stsFooError1);
		} else {
			StdUnknownError(s);
		}
	}
//}
*/


/**** Button Definition ****/
/*
 Message strings may contain button definitions.  A button definition is a
 substring enclosed in square brackets at the beginning of the message
 string.  Any number of buttons may be defined but all buttons must appear
 at the front of the string.  If no buttons are defined then a single "OK"
 button is created.

 StdMsg(), StdError(), StdMsgRes(), StdErrorRes() and StdSystemError()
 return the button number that the user hit to dismiss the note.  Button
 numbers start with 0.  For example, a note constructed with the following
 string:

//{
 "[Button0] [Button1] [Button2] Here's your message!"
//}

 returns the value 1 if the user hits Button1.  These functions might also
 return a negative error status if a problem occurred inside the function.

 See the section "A Typical Scenario" for an example.
*/

/**** Text Substitution and Formatting ****/
/*
 Message strings may contain parameter substitutions, as defined in
 cmpstext.h.  Text substitution also works inside the button substrings.  

 See the section "A Typical Scenario" for an example.

 You can break your message up into paragraphs by putting 2 newlines at the
 paragraph breaks.  For example:

//{
 "Here's the first paragraph.\n\nHere's the second one."
//}
*/


/**** Progress Notes ****/
/*
 Clients can put up a progress note to inform the user that a lengthy
 operation has begun, and take down the progress note to indicate that the
 operation has been completed.
 
 Cancellation of the operation is not supported in PenPoint 1.0.  Progress
 notes do not have buttons.  Here's an example of progress indication
 usage:

//{
	SP_TOKEN 	token;

	StdProgressUp(tagFooProgress1, &token, param1, param2);
	... Lengthy operation ...
	StdProgressDown(&token);
//}
*/


/**** Searching for Resources ****/
/*
 Most of the functions in this package search for resources as follows:
    -:  If the process is an application process (OSThisApp() returns
        non-null), then the application's resource list is searched. 
        Otherwise theSystemResFile is searched.
    -:  If the desired resource is not found in the above resource files or
        lists, then theServiceResList is searched.

 The exceptions to this rule are:
	-:	StdSystemError(), which only checks theSystemResFile.
    -:  StdMsgRes(), which takes as one of its parameters the resource file
        or list to search.
    -:  StdErrorRes(), which takes as one of its parameters the resource
        file or list to search.
*/


/**** Note Titles and Reference Field ****/
/*
 Notes will be titled "Note from {App}..." if the string was found in 
 the app resource file, or "Note from PenPoint..." if the string was 
 found in the system resource file.

 You can use StdMsgCustom() if you want to have some other title.

 Error messages will also have an additional line at the bottom of the
 note of the form:

 Reference: xxx-xxx

 where xxx-xxx is the status code that generated the error.
*/


/**** Customization of Standard Message Package Notes ****/
/*
 StdMsgCustom() allows clients to customize a standard message package
 note. It returns the note object, without displaying it. Developers can
 modify this object as they wish and then display it themselves.
*/ 

#ifndef STDMSG_INCLUDED
#define STDMSG_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef OSTYPES_INCLUDED
#include <ostypes.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Common #defines								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/**** Constructing Standard Message Tags ****/
/*
 Use MakeStatus() (defined in go.h) to construct string tags for errors.

 Use MakeDialogTag() to construct string tags for dialog and progress
 strings.
*/
#define MakeDialogTag(wkn, index)	MakeIndexedResId(wkn, 1, index)


/**** Constructing Standard Message Resource Ids ****/
/*
 In a .rc file, use resForStdMsgDialog to construct the resource id for a
 class's dialog string array.  Use resForStdMsgError to construct the
 resource id for a class's error string array.

 See the section "A Typical Scenario" for an example.
*/
#define resForStdMsgDialog(wkn)	MakeListResId(wkn, resGrpStdMsg, 1)
#define resForStdMsgError(wkn) 	MakeListResId(wkn, resGrpStdMsg, 0)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Public Functions							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 StdUnknownError	                   	returns STATUS
	Displays an error message when the client doesn't recognize the error.

 Use this function to display an error message when the error status is one
 that you don't pay special attention to.

 StdUnknownError searches for an error message that matches the status
 parameter.  If the specified status isn't found then a note with just the
 error code is displayed.

 StdUnknownError does not allow parameter substitution or multiple command
 buttons.  Any parameter substitution specifications in the text are
 replaced with "???".  A single "OK" command button is always displayed.

 See the section "A Typical Scenario" for an example.  See the section 
 "Searching for Resources" for a description of which resource files are 
 searched.
*/
STATUS CDECL StdUnknownError(
	STATUS	status);


/****************************************************************************
 StdMsg				                   	returns S32
	Displays a dialog box from a resource file.

 Use this function to display a dialog box.

 StdMsg searches for an dialog string that matches the tag parameter.  A
 dialog box with the message and buttons defined in the message string is
 displayed.

 See the section "A Typical Scenario" for an example.  See the section 
 "Searching for Resources" for a description of which resource files are 
 searched.

 Like printf, this function takes a variable number of parameters.  There
 is no error checking on the number	and type of the parameters.

 Return Value
	stsResResourceNotFound:		the specified tag is not found.  
	< stsOK:					some other error occurred. 
	>= stsOK:					number of button the user hit (0 based).
*/
S32 CDECL StdMsg(
	const TAG tag, 
	...);


/****************************************************************************
 StdError			                   	returns S32
	Displays an error message from a resource file.

 Use this function to display an error message.

 StdError searches for an error message string that matches the status
 parameter.  A note with the message and buttons defined in the error
 message string is displayed. The note also contains an Error Code 
 number line.

 See the section "A Typical Scenario" for an example.  See the section 
 "Searching for Resources" for a description of which resource files are 
 searched.

 Like printf, this function takes a variable number of parameters.  There
 is no error checking on the number	and type of the parameters.

 Return Value
	stsResResourceNotFound:		the specified tag is not found.  
	< stsOK:					some other error occurred. 
	>= stsOK:					number of button the user hit (0 based).
*/
S32 CDECL StdError(
	const STATUS status,
	...);


/****************************************************************************
 StdSystemError		                   	returns S32
	For PenPoint internal use only.  Displays an error message for a standard
	PenPoint error.

 StdSystemError searches theSystemResFile (penpoint.res) for an error
 message string that matches the status parameter.  A note with the message
 and buttons defined in the string is displayed.

 Like printf, this function takes a variable number of parameters.  There
 is no error checking on the number	and type of the parameters.

 Return Value
	stsResResourceNotFound:		the specified tag is not found.  
	< stsOK:					some other error occurred. 
	>= stsOK:					number of button the user hit (0 based).
*/
S32 CDECL StdSystemError(
	const STATUS status,
	...);


/****************************************************************************
 StdProgressUp                  	returns STATUS
	Displays a progress note from a resource file.

 Use this function to inform the user that a lengthy operation has started.

 StdProgressUp searches for an dialog message that matches the tag
 parameter.  A dialog box with the message string is displayed.  This
 dialog box stay ups until StdProgressDown is called.

 The pToken parameter, as filled in by StdProgressUp, must be passed to
 StdProgressDown.  The client shouldn't touch it!
 
 Example:
 
//{
	SP_TOKEN 	token;

	StdProgressUp(tagFoo, &token, param1, param2);
	...
	StdProgressDown(&token);
//}

 Progress notes do not contain a command bar.  Any button definitions are
 ignored.

 Like printf, this function takes a variable number of parameters.  There
 is no error checking on the number	and type of the parameters.

 See the section "Progress Notes" for more information.  See the section 
 "Searching for Resources" for a description of which resource files are 
 searched.

 Return Value
	stsResResourceNotFound:		The specified tag was not found

 See Also
	StdProgressDown
*/
typedef struct SP_TOKEN {

	OBJECT				uid;
	OS_MILLISECONDS		startTime;
	U32					spare[8];

} SP_TOKEN, *P_SP_TOKEN;

STATUS CDECL StdProgressUp(
	const 	TAG 		tag, 
			P_SP_TOKEN 	pToken,
			...);


/****************************************************************************
 StdProgressDown                  	returns STATUS
	Brings down a progress note that was put up with StdProgressUp().

 The pToken parameter must be the same as that passed to StdProgressUp().

 See the section "Progress Notes" for more information.

 See Also
	StdProgressUp
*/
STATUS CDECL StdProgressDown(
	P_SP_TOKEN	pToken);


/****************************************************************************
 StdMsgRes			                   	returns S32
	Just like StdMsg() except that the resource path is specified.

 Use StdMsgRes when you need the functionality of StdMsg, but need to look
 up the string in a specified resource file or resource list.

 StdMsgRes searches the specified resource file or list for a dialog
 message string that matches the tag parameter.  A dialog box with the
 message and buttons defined in the message string is displayed.

 Like printf, this function takes a variable number of parameters.  There
 is no error checking on the number	and type of the parameters.

 Return Value
	stsResResourceNotFound:		the specified tag is not found.  
	< stsOK:					some other error occurred. 
	>= stsOK:					number of button the user hit (0 based).

 See Also
	StdMsg
*/
S32 CDECL StdMsgRes(
			OBJECT	resFile, 
	const 	TAG 	tag,
			...);


/****************************************************************************
 StdErrorRes		                   	returns S32
	Just like StdError() except that the resource path is specified.
	
 Use StdMsgError when you need the functionality of StdError, but need to
 look up the string in a specified resource file or resource list.

 StdErrorRes searches the specified resource file or list for an error
 message string that matches the status parameter.  A note with the message
 and buttons defined in the error message string is displayed.

 Like printf, this function takes a variable number of parameters.  There
 is no error checking on the number	and type of the parameters.

 Return Value
	stsResResourceNotFound:		the specified tag is not found.  
	< stsOK:					some other error occurred. 
	>= stsOK:					number of button the user hit (0 based).

 See Also
	StdError
*/
S32 CDECL StdErrorRes(
			OBJECT	resFile, 
	const	STATUS	status,
			...);


/****************************************************************************
 StdMsgCustom			            	returns OBJECT
	 Creates a note object in the manner of StdMsg().

 Use StdMsgCustom when you want to create a note using the facilities of
 the standard message package but need to customize the note before
 displaying it.

 The client is responsible for displaying the note object.  The note has
 autoDestroy on, so it self-destructs when dismissed.

 StdMsgCustom allows the specification of a resource file or list to
 search.  If resFile is objNull then searching occurs as described in the
 section "Searching for Resources." The tag parameter can either be a
 dialog tag (created with MakeDialogTag()) or an error status (created with
 MakeStatus()).
 
 Here's an example:
//{
	#define tagFooDialog1		MakeDialogTag(clsFoo, 0) 

	S32		buttonHit;
	OBJECT	note;

	note = StdMsgCustom(objNull, tagFooDialog1, arg1, arg2);
	if (note == objNull) {
		...  // Handle error, probably resource not found.
		goto error;
	}
	...  // Customize the note.
	ObjCallRet(msgNoteShow, note, &buttonHit, s);
//}

 Like printf, this function takes a variable number of parameters.  There
 is no error checking on the number	and type of the parameters.

 Return Value
	objNull:	No match, or some other error occurred.
*/
OBJECT CDECL StdMsgCustom(
			OBJECT	resFile, 
	const	TAG 	tag,
			...);


#endif
